"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var TabBarToolbar_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TabBarToolbar = exports.TabBarToolbarFactory = exports.TabBarToolbarRegistry = exports.TabBarToolbarItem = exports.TabBarDelegator = exports.TabBarToolbarContribution = void 0;
const debounce = require("lodash.debounce");
const React = require("react");
const inversify_1 = require("inversify");
const widgets_1 = require("../widgets");
const label_parser_1 = require("../label-parser");
const contribution_provider_1 = require("../../common/contribution-provider");
const command_1 = require("../../common/command");
const disposable_1 = require("../../common/disposable");
const context_key_service_1 = require("../context-key-service");
const event_1 = require("../../common/event");
const context_menu_renderer_1 = require("../context-menu-renderer");
const menu_1 = require("../../common/menu");
/**
 * Clients should implement this interface if they want to contribute to the tab-bar toolbar.
 */
exports.TabBarToolbarContribution = Symbol('TabBarToolbarContribution');
var TabBarDelegator;
(function (TabBarDelegator) {
    TabBarDelegator.is = (candidate) => {
        if (candidate) {
            const asDelegator = candidate;
            return typeof asDelegator.getTabBarDelegate === 'function';
        }
        return false;
    };
})(TabBarDelegator = exports.TabBarDelegator || (exports.TabBarDelegator = {}));
var TabBarToolbarItem;
(function (TabBarToolbarItem) {
    /**
     * Compares the items by `priority` in ascending. Undefined priorities will be treated as `0`.
     */
    TabBarToolbarItem.PRIORITY_COMPARATOR = (left, right) => {
        // The navigation group is special as it will always be sorted to the top/beginning of a menu.
        const compareGroup = (leftGroup = 'navigation', rightGroup = 'navigation') => {
            if (leftGroup === 'navigation') {
                return rightGroup === 'navigation' ? 0 : -1;
            }
            if (rightGroup === 'navigation') {
                return leftGroup === 'navigation' ? 0 : 1;
            }
            return leftGroup.localeCompare(rightGroup);
        };
        const result = compareGroup(left.group, right.group);
        if (result !== 0) {
            return result;
        }
        return (left.priority || 0) - (right.priority || 0);
    };
    function is(arg) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        return !!arg && 'command' in arg && typeof arg.command === 'string';
    }
    TabBarToolbarItem.is = is;
})(TabBarToolbarItem = exports.TabBarToolbarItem || (exports.TabBarToolbarItem = {}));
/**
 * Main, shared registry for tab-bar toolbar items.
 */
let TabBarToolbarRegistry = class TabBarToolbarRegistry {
    constructor() {
        this.items = new Map();
        this.onDidChangeEmitter = new event_1.Emitter();
        this.onDidChange = this.onDidChangeEmitter.event;
        // debounce in order to avoid to fire more than once in the same tick
        this.fireOnDidChange = debounce(() => this.onDidChangeEmitter.fire(undefined), 0);
    }
    onStart() {
        const contributions = this.contributionProvider.getContributions();
        for (const contribution of contributions) {
            contribution.registerToolbarItems(this);
        }
    }
    /**
     * Registers the given item. Throws an error, if the corresponding command cannot be found or an item has been already registered for the desired command.
     *
     * @param item the item to register.
     */
    registerItem(item) {
        const { id } = item;
        if (this.items.has(id)) {
            throw new Error(`A toolbar item is already registered with the '${id}' ID.`);
        }
        this.items.set(id, item);
        this.fireOnDidChange();
        const toDispose = new disposable_1.DisposableCollection(disposable_1.Disposable.create(() => this.fireOnDidChange()), disposable_1.Disposable.create(() => this.items.delete(id)));
        if (item.onDidChange) {
            toDispose.push(item.onDidChange(() => this.fireOnDidChange()));
        }
        return toDispose;
    }
    /**
     * Returns an array of tab-bar toolbar items which are visible when the `widget` argument is the current one.
     *
     * By default returns with all items where the command is enabled and `item.isVisible` is `true`.
     */
    visibleItems(widget) {
        if (widget.isDisposed) {
            return [];
        }
        const result = [];
        for (const item of this.items.values()) {
            const visible = TabBarToolbarItem.is(item)
                ? this.commandRegistry.isVisible(item.command, widget)
                : (!item.isVisible || item.isVisible(widget));
            if (visible && (!item.when || this.contextKeyService.match(item.when, widget.node))) {
                result.push(item);
            }
        }
        return result;
    }
    unregisterItem(itemOrId) {
        const id = typeof itemOrId === 'string' ? itemOrId : itemOrId.id;
        if (this.items.delete(id)) {
            this.fireOnDidChange();
        }
    }
};
__decorate([
    inversify_1.inject(command_1.CommandRegistry),
    __metadata("design:type", command_1.CommandRegistry)
], TabBarToolbarRegistry.prototype, "commandRegistry", void 0);
__decorate([
    inversify_1.inject(context_key_service_1.ContextKeyService),
    __metadata("design:type", context_key_service_1.ContextKeyService)
], TabBarToolbarRegistry.prototype, "contextKeyService", void 0);
__decorate([
    inversify_1.inject(contribution_provider_1.ContributionProvider),
    inversify_1.named(exports.TabBarToolbarContribution),
    __metadata("design:type", Object)
], TabBarToolbarRegistry.prototype, "contributionProvider", void 0);
TabBarToolbarRegistry = __decorate([
    inversify_1.injectable()
], TabBarToolbarRegistry);
exports.TabBarToolbarRegistry = TabBarToolbarRegistry;
/**
 * Factory for instantiating tab-bar toolbars.
 */
exports.TabBarToolbarFactory = Symbol('TabBarToolbarFactory');
/**
 * Tab-bar toolbar widget representing the active [tab-bar toolbar items](TabBarToolbarItem).
 */
let TabBarToolbar = TabBarToolbar_1 = class TabBarToolbar extends widgets_1.ReactWidget {
    constructor() {
        super();
        this.inline = new Map();
        this.more = new Map();
        this.toDisposeOnSetCurrent = new disposable_1.DisposableCollection();
        this.showMoreContextMenu = (event) => {
            event.stopPropagation();
            event.preventDefault();
            this.renderMoreContextMenu(event.nativeEvent);
        };
        this.executeCommand = (e) => {
            e.preventDefault();
            e.stopPropagation();
            const item = this.inline.get(e.currentTarget.id);
            if (TabBarToolbarItem.is(item)) {
                this.commands.executeCommand(item.command, this.current);
            }
            this.update();
        };
        this.onMouseDownEvent = (e) => {
            if (e.button === 0) {
                e.currentTarget.classList.add('active');
            }
        };
        this.onMouseUpEvent = (e) => {
            e.currentTarget.classList.remove('active');
        };
        this.addClass(TabBarToolbar_1.Styles.TAB_BAR_TOOLBAR);
        this.hide();
    }
    updateItems(items, current) {
        this.inline.clear();
        this.more.clear();
        for (const item of items.sort(TabBarToolbarItem.PRIORITY_COMPARATOR).reverse()) {
            if ('render' in item || item.group === undefined || item.group === 'navigation') {
                this.inline.set(item.id, item);
            }
            else {
                this.more.set(item.id, item);
            }
        }
        this.setCurrent(current);
        if (!items.length) {
            this.hide();
        }
        this.onRender.push(disposable_1.Disposable.create(() => {
            if (items.length) {
                this.show();
            }
        }));
        this.update();
    }
    updateTarget(current) {
        const operativeWidget = TabBarDelegator.is(current) ? current.getTabBarDelegate() : current;
        const items = operativeWidget ? this.toolbarRegistry.visibleItems(operativeWidget) : [];
        this.updateItems(items, operativeWidget);
    }
    setCurrent(current) {
        this.toDisposeOnSetCurrent.dispose();
        this.toDispose.push(this.toDisposeOnSetCurrent);
        this.current = current;
        if (current) {
            const resetCurrent = () => {
                this.setCurrent(undefined);
                this.update();
            };
            current.disposed.connect(resetCurrent);
            this.toDisposeOnSetCurrent.push(disposable_1.Disposable.create(() => current.disposed.disconnect(resetCurrent)));
        }
    }
    render() {
        return React.createElement(React.Fragment, null,
            this.renderMore(),
            [...this.inline.values()].map(item => TabBarToolbarItem.is(item) ? this.renderItem(item) : item.render(this.current)));
    }
    renderItem(item) {
        let innerText = '';
        const classNames = [];
        if (item.text) {
            for (const labelPart of this.labelParser.parse(item.text)) {
                if (typeof labelPart !== 'string' && label_parser_1.LabelIcon.is(labelPart)) {
                    const className = `fa fa-${labelPart.name}${labelPart.animation ? ' fa-' + labelPart.animation : ''}`;
                    classNames.push(...className.split(' '));
                }
                else {
                    innerText = labelPart;
                }
            }
        }
        const command = this.commands.getCommand(item.command);
        const iconClass = (typeof item.icon === 'function' && item.icon()) || item.icon || (command && command.iconClass);
        if (iconClass) {
            classNames.push(iconClass);
        }
        const tooltip = item.tooltip || (command && command.label);
        const toolbarItemClassNames = this.getToolbarItemClassNames(command === null || command === void 0 ? void 0 : command.id);
        return React.createElement("div", { key: item.id, className: toolbarItemClassNames, onMouseDown: this.onMouseDownEvent, onMouseUp: this.onMouseUpEvent, onMouseOut: this.onMouseUpEvent },
            React.createElement("div", { id: item.id, className: classNames.join(' '), onClick: this.executeCommand, title: tooltip }, innerText));
    }
    getToolbarItemClassNames(commandId) {
        const classNames = [TabBarToolbar_1.Styles.TAB_BAR_TOOLBAR_ITEM];
        if (commandId) {
            if (this.commandIsEnabled(commandId)) {
                classNames.push('enabled');
            }
            if (this.commandIsToggled(commandId)) {
                classNames.push('toggled');
            }
        }
        return classNames.join(' ');
    }
    renderMore() {
        return !!this.more.size && React.createElement("div", { key: '__more__', className: TabBarToolbar_1.Styles.TAB_BAR_TOOLBAR_ITEM + ' enabled' },
            React.createElement("div", { id: '__more__', className: 'fa fa-ellipsis-h', onClick: this.showMoreContextMenu, title: 'More Actions...' }));
    }
    /* eslint-disable-next-line @typescript-eslint/no-explicit-any */
    renderMoreContextMenu(anchor) {
        var _a;
        const menuPath = ['TAB_BAR_TOOLBAR_CONTEXT_MENU'];
        const toDisposeOnHide = new disposable_1.DisposableCollection();
        this.addClass('menu-open');
        toDisposeOnHide.push(disposable_1.Disposable.create(() => this.removeClass('menu-open')));
        for (const item of this.more.values()) {
            // Register a submenu for the item, if the group is in format `<submenu group>/<submenu name>/.../<item group>`
            if ((_a = item.group) === null || _a === void 0 ? void 0 : _a.includes('/')) {
                const split = item.group.split('/');
                const paths = [];
                for (let i = 0; i < split.length - 1; i += 2) {
                    paths.push(split[i], split[i + 1]);
                    // TODO order is missing, items sorting will be alphabetic
                    toDisposeOnHide.push(this.menus.registerSubmenu([...menuPath, ...paths], split[i + 1]));
                }
            }
            // TODO order is missing, items sorting will be alphabetic
            toDisposeOnHide.push(this.menus.registerMenuAction([...menuPath, ...item.group.split('/')], {
                label: item.tooltip,
                commandId: item.command,
                when: item.when
            }));
        }
        return this.contextMenuRenderer.render({
            menuPath,
            args: [this.current],
            anchor,
            onHide: () => toDisposeOnHide.dispose()
        });
    }
    shouldHandleMouseEvent(event) {
        return event.target instanceof Element && this.node.contains(event.target);
    }
    commandIsEnabled(command) {
        return this.commands.isEnabled(command, this.current);
    }
    commandIsToggled(command) {
        return this.commands.isToggled(command, this.current);
    }
};
__decorate([
    inversify_1.inject(command_1.CommandRegistry),
    __metadata("design:type", command_1.CommandRegistry)
], TabBarToolbar.prototype, "commands", void 0);
__decorate([
    inversify_1.inject(label_parser_1.LabelParser),
    __metadata("design:type", label_parser_1.LabelParser)
], TabBarToolbar.prototype, "labelParser", void 0);
__decorate([
    inversify_1.inject(menu_1.MenuModelRegistry),
    __metadata("design:type", menu_1.MenuModelRegistry)
], TabBarToolbar.prototype, "menus", void 0);
__decorate([
    inversify_1.inject(context_menu_renderer_1.ContextMenuRenderer),
    __metadata("design:type", context_menu_renderer_1.ContextMenuRenderer)
], TabBarToolbar.prototype, "contextMenuRenderer", void 0);
__decorate([
    inversify_1.inject(TabBarToolbarRegistry),
    __metadata("design:type", TabBarToolbarRegistry)
], TabBarToolbar.prototype, "toolbarRegistry", void 0);
TabBarToolbar = TabBarToolbar_1 = __decorate([
    inversify_1.injectable(),
    __metadata("design:paramtypes", [])
], TabBarToolbar);
exports.TabBarToolbar = TabBarToolbar;
(function (TabBarToolbar) {
    let Styles;
    (function (Styles) {
        Styles.TAB_BAR_TOOLBAR = 'p-TabBar-toolbar';
        Styles.TAB_BAR_TOOLBAR_ITEM = 'item';
    })(Styles = TabBarToolbar.Styles || (TabBarToolbar.Styles = {}));
})(TabBarToolbar = exports.TabBarToolbar || (exports.TabBarToolbar = {}));
exports.TabBarToolbar = TabBarToolbar;
//# sourceMappingURL=tab-bar-toolbar.js.map