/*******************************************************************************
 * Copyright (c) 2010, 2015 Wind River Systems, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Wind River Systems - initial API and implementation
 *******************************************************************************/
package org.eclipse.tcf.internal.debug.ui.commands;

import java.math.BigInteger;
import java.util.Map;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tcf.internal.debug.ui.model.TCFModel;
import org.eclipse.tcf.protocol.JSON;
import org.eclipse.tcf.services.IMemoryMap;

class MemoryMapItemDialog extends Dialog {

    private final Map<String,Object> props;
    private final boolean enable_editing;
    private final Image image;

    private Text addr_text;
    private Text size_text;
    private Text offset_text;
    private Text query_text;
    private Text file_text;
    private Button rd_button;
    private Button wr_button;
    private Button ex_button;

    MemoryMapItemDialog(Shell parent, Image image, Map<String,Object> props, boolean enable_editing) {
        super(parent);
        this.image = image;
        this.props = props;
        this.enable_editing = enable_editing;
    }

    @Override
    protected void configureShell(Shell shell) {
        super.configureShell(shell);
        shell.setText("Symbol File");
        shell.setImage(image);
    }

    @Override
    protected void createButtonsForButtonBar(Composite parent) {
        createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
        createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
        updateButtons();
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite)super.createDialogArea(parent);
        createFileNameFields(composite);
        createPropsFields(composite);
        createQueryFields(composite);
        setData();
        composite.setSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
        return composite;
    }

    private void createFileNameFields(Composite parent) {
        Font font = parent.getFont();
        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(3, false);
        composite.setFont(font);
        composite.setLayout(layout);
        composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        Label file_label = new Label(composite, SWT.WRAP);
        file_label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        file_label.setFont(font);
        file_label.setText("File name:");

        file_text = new Text(composite, SWT.SINGLE | SWT.BORDER);
        file_text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        file_text.setFont(font);
        file_text.setEditable(enable_editing);

        file_text.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                updateButtons();
            }
        });

        Button button = new Button(composite, SWT.PUSH);
        button.setFont(font);
        button.setText("...");
        button.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        button.setEnabled(enable_editing);
        button.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                FileDialog file_dialog = new FileDialog(getShell(), SWT.NONE);
                file_dialog.setFileName(file_text.getText());
                String path = file_dialog.open();
                if (path != null) file_text.setText(path);
            }
        });
    }

    private void createPropsFields(Composite parent) {
        Font font = parent.getFont();
        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        composite.setFont(font);
        composite.setLayout(layout);
        composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        createTextFields(composite);
        createFlagsGroup(composite);
    }

    private void createTextFields(Composite parent) {
        Font font = parent.getFont();
        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        composite.setFont(font);
        composite.setLayout(layout);
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.widthHint = 250;
        composite.setLayoutData(gd);

        Label addr_label = new Label(composite, SWT.WRAP);
        addr_label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        addr_label.setFont(font);
        addr_label.setText("Address:");

        addr_text = new Text(composite, SWT.SINGLE | SWT.BORDER);
        addr_text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        addr_text.setFont(font);
        addr_text.setEditable(enable_editing);

        addr_text.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                updateButtons();
            }
        });

        Label size_label = new Label(composite, SWT.WRAP);
        size_label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        size_label.setFont(font);
        size_label.setText("Size:");

        size_text = new Text(composite, SWT.SINGLE | SWT.BORDER);
        size_text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        size_text.setFont(font);
        size_text.setEditable(enable_editing);

        Label offset_label = new Label(composite, SWT.WRAP);
        offset_label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        offset_label.setFont(font);
        offset_label.setText("File offset:");

        offset_text = new Text(composite, SWT.SINGLE | SWT.BORDER);
        offset_text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        offset_text.setFont(font);
        offset_text.setEditable(enable_editing);
    }

    private void createFlagsGroup(Composite parent) {
        Font font = parent.getFont();

        Group group = new Group(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.verticalSpacing = 0;
        layout.numColumns = 1;
        group.setLayout(layout);
        group.setLayoutData(new GridData(GridData.FILL_BOTH));
        group.setFont(font);
        group.setText("Flags");

        rd_button = new Button(group, SWT.CHECK);
        rd_button.setFont(font);
        rd_button.setText("Data read");
        rd_button.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        rd_button.setEnabled(enable_editing);

        wr_button = new Button(group, SWT.CHECK);
        wr_button.setFont(font);
        wr_button.setText("Data write");
        wr_button.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        wr_button.setEnabled(enable_editing);

        ex_button = new Button(group, SWT.CHECK);
        ex_button.setFont(font);
        ex_button.setText("Instructions read");
        ex_button.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        ex_button.setEnabled(enable_editing);
    }

    private void createQueryFields(Composite parent) {
        Font font = parent.getFont();

        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        composite.setFont(font);
        composite.setLayout(layout);
        composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        Label query_label = new Label(composite, SWT.WRAP);
        query_label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
        query_label.setFont(font);
        query_label.setText("Context query:");

        query_text = new Text(composite, SWT.SINGLE | SWT.BORDER);
        query_text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        query_text.setFont(font);
        query_text.setEditable(enable_editing);
    }

    private String toHex(Number n) {
        if (n == null) return null;
        BigInteger x = JSON.toBigInteger(n);
        String s = x.toString(16);
        int l = 16 - s.length();
        if (l < 0) l = 0;
        if (l > 16) l = 16;
        return "0x0000000000000000".substring(0, 2 + l) + s;
    }

    private void setText(Text text, String str) {
        if (str == null) str = "";
        text.setText(str);
    }

    private void setData() {
        setText(addr_text, toHex((Number)props.get(IMemoryMap.PROP_ADDRESS)));
        setText(size_text, toHex((Number)props.get(IMemoryMap.PROP_SIZE)));
        if (props.get(IMemoryMap.PROP_SECTION_NAME) != null) {
            setText(offset_text, (String)props.get(IMemoryMap.PROP_SECTION_NAME));
        }
        else {
            setText(offset_text, toHex((Number)props.get(IMemoryMap.PROP_OFFSET)));
        }
        setText(query_text, (String)props.get(IMemoryMap.PROP_CONTEXT_QUERY));
        setText(file_text, (String)props.get(IMemoryMap.PROP_FILE_NAME));
        int flags = 0;
        Number n = (Number)props.get(IMemoryMap.PROP_FLAGS);
        if (n != null) flags = n.intValue();
        rd_button.setSelection((flags & IMemoryMap.FLAG_READ) != 0);
        wr_button.setSelection((flags & IMemoryMap.FLAG_WRITE) != 0);
        ex_button.setSelection((flags & IMemoryMap.FLAG_EXECUTE) != 0);
        updateButtons();
    }

    private void getNumber(Text text, String key) {
        String s = text.getText().trim();
        if (s == null || s.length() == 0) {
            props.remove(key);
        }
        else if (s.startsWith("0x")) {
            props.put(key, new BigInteger(s.substring(2), 16));
        }
        else {
            props.put(key, new BigInteger(s));
        }
    }

    private void getText(Text text, String key) {
        String s = text.getText().trim();
        if (s == null || s.length() == 0) {
            props.remove(key);
        }
        else {
            props.put(key, s);
        }
    }

    private void getData() {
        getNumber(addr_text, IMemoryMap.PROP_ADDRESS);
        getNumber(size_text, IMemoryMap.PROP_SIZE);
        String s = offset_text.getText().trim();
        if (s.length() > 0 && !Character.isDigit(s.charAt(0))) {
            props.put(IMemoryMap.PROP_SECTION_NAME, s);
            props.remove(IMemoryMap.PROP_OFFSET);
        }
        else {
            getNumber(offset_text, IMemoryMap.PROP_OFFSET);
            props.remove(IMemoryMap.PROP_SECTION_NAME);
        }
        getText(query_text, IMemoryMap.PROP_CONTEXT_QUERY);
        getText(file_text, IMemoryMap.PROP_FILE_NAME);
        int flags = 0;
        if (rd_button.getSelection()) flags |= IMemoryMap.FLAG_READ;
        if (wr_button.getSelection()) flags |= IMemoryMap.FLAG_WRITE;
        if (ex_button.getSelection()) flags |= IMemoryMap.FLAG_EXECUTE;
        props.put(IMemoryMap.PROP_FLAGS, flags);
    }

    private void updateButtons() {
        Button btn = getButton(IDialogConstants.OK_ID);
        if (btn != null) {
            btn.setEnabled(!enable_editing ||
                    file_text != null && file_text.getText().trim().length() > 0 ||
                    addr_text != null && addr_text.getText().trim().length() > 0);
        }
    }

    @Override
    protected void okPressed() {
        if (enable_editing) {
            try {
                getData();
            }
            catch (Throwable x) {
                MessageBox mb = new MessageBox(getShell(), SWT.ICON_ERROR | SWT.OK);
                mb.setText("Invalid data");
                mb.setMessage(TCFModel.getErrorMessage(x, true));
                mb.open();
                return;
            }
        }
        super.okPressed();
    }
}
