"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createTypeScriptServerProject = void 0;
const language_service_1 = require("@volar/language-service");
const typescript_1 = require("@volar/typescript");
const path = require("path-browserify");
const vscode = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const uriMap_1 = require("../utils/uriMap");
async function createTypeScriptServerProject(ts, tsLocalized, tsconfig, server, serviceEnv, getLanguagePlugins) {
    let parsedCommandLine;
    let projectVersion = 0;
    let languageService;
    const sys = (0, typescript_1.createSys)(ts, serviceEnv, server.uriConverter.uriToFileName(serviceEnv.workspaceFolder));
    const host = {
        ...sys,
        configFileName: typeof tsconfig === 'string' ? tsconfig : undefined,
        getSystemVersion() {
            return sys.version;
        },
        syncSystem() {
            return sys.sync();
        },
        getCurrentDirectory() {
            return server.uriConverter.uriToFileName(serviceEnv.workspaceFolder);
        },
        getProjectVersion() {
            return projectVersion.toString();
        },
        getScriptFileNames() {
            return rootFiles;
        },
        getScriptSnapshot(fileName) {
            const uri = server.uriConverter.fileNameToUri(fileName);
            askedFiles.set(vscode_uri_1.URI.parse(uri), true);
            const document = server.documents.get(uri);
            if (document) {
                return document.getSnapshot();
            }
        },
        getCompilationSettings() {
            return parsedCommandLine.options;
        },
        getLocalizedDiagnosticMessages: tsLocalized ? () => tsLocalized : undefined,
        getProjectReferences() {
            return parsedCommandLine.projectReferences;
        },
        fileNameToScriptId: serviceEnv.typescript.fileNameToUri,
        scriptIdToFileName: serviceEnv.typescript.uriToFileName,
    };
    const languagePlugins = await getLanguagePlugins(serviceEnv, {
        configFileName: typeof tsconfig === 'string' ? tsconfig : undefined,
        host,
        sys,
    });
    const askedFiles = (0, uriMap_1.createUriMap)();
    const docChangeWatcher = server.documents.onDidChangeContent(() => {
        projectVersion++;
    });
    const fileWatch = serviceEnv.onDidChangeWatchedFiles?.(params => {
        onWorkspaceFilesChanged(params.changes);
    });
    let rootFiles = await getRootFiles(languagePlugins);
    return {
        askedFiles,
        getLanguageService,
        getLanguageServiceDontCreate: () => languageService,
        tryAddFile(fileName) {
            if (!rootFiles.includes(fileName)) {
                rootFiles.push(fileName);
                projectVersion++;
            }
        },
        dispose,
        getParsedCommandLine: () => parsedCommandLine,
    };
    async function getRootFiles(languagePlugins) {
        parsedCommandLine = await createParsedCommandLine(ts, sys, server.uriConverter.uriToFileName(serviceEnv.workspaceFolder), tsconfig, languagePlugins.map(plugin => plugin.typescript?.extraFileExtensions ?? []).flat());
        return parsedCommandLine.fileNames;
    }
    function getLanguageService() {
        if (!languageService) {
            const language = (0, typescript_1.createTypeScriptLanguage)(ts, [
                {
                    getLanguageId(uri) {
                        return server.documents.get(uri)?.languageId;
                    },
                },
                ...languagePlugins,
            ], host);
            languageService = (0, language_service_1.createLanguageService)(language, server.languageServicePlugins, serviceEnv);
        }
        return languageService;
    }
    async function onWorkspaceFilesChanged(changes) {
        const createsAndDeletes = changes.filter(change => change.type !== vscode.FileChangeType.Changed);
        if (createsAndDeletes.length) {
            rootFiles = await getRootFiles(languagePlugins);
        }
        projectVersion++;
    }
    function dispose() {
        sys.dispose();
        languageService?.dispose();
        fileWatch?.dispose();
        docChangeWatcher.dispose();
    }
}
exports.createTypeScriptServerProject = createTypeScriptServerProject;
async function createParsedCommandLine(ts, sys, workspacePath, tsconfig, extraFileExtensions) {
    let content = {
        errors: [],
        fileNames: [],
        options: {},
    };
    let sysVersion;
    let newSysVersion = await sys.sync();
    while (sysVersion !== newSysVersion) {
        sysVersion = newSysVersion;
        try {
            if (typeof tsconfig === 'string') {
                const config = ts.readJsonConfigFile(tsconfig, sys.readFile);
                content = ts.parseJsonSourceFileConfigFileContent(config, sys, path.dirname(tsconfig), {}, tsconfig, undefined, extraFileExtensions);
            }
            else {
                content = ts.parseJsonConfigFileContent({ files: [] }, sys, workspacePath, tsconfig, workspacePath + '/jsconfig.json', undefined, extraFileExtensions);
            }
            // fix https://github.com/johnsoncodehk/volar/issues/1786
            // https://github.com/microsoft/TypeScript/issues/30457
            // patching ts server broke with outDir + rootDir + composite/incremental
            content.options.outDir = undefined;
            content.fileNames = content.fileNames.map(fileName => fileName.replace(/\\/g, '/'));
        }
        catch {
            // will be failed if web fs host first result not ready
        }
        newSysVersion = await sys.sync();
    }
    if (content) {
        return content;
    }
    return content;
}
//# sourceMappingURL=typescriptProject.js.map