/**
 * Copyright (c) 2010-2015, Balazs Grill, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Joiner;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import org.apache.log4j.Appender;
import org.apache.log4j.Level;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.viatra.query.testing.core.IMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.InitializedSnapshotMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.MatchSetRecordDiff;
import org.eclipse.viatra.query.testing.core.SnapshotHelper;
import org.eclipse.viatra.query.testing.core.TestingSeverityAggregatorLogAppender;
import org.eclipse.viatra.query.testing.core.api.JavaObjectAccess;
import org.eclipse.viatra.query.testing.core.api.MatchRecordEquivalence;
import org.eclipse.viatra.query.testing.core.internal.DefaultMatchRecordEquivalence;
import org.eclipse.viatra.query.testing.snapshot.MatchRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSetRecord;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.ComparisonFailure;

/**
 * @author Grill Balazs
 */
@SuppressWarnings("all")
public class ViatraQueryTestCase {
  private static final String SEVERITY_AGGREGATOR_LOGAPPENDER_NAME = (ViatraQueryTestCase.class.getName() + 
    ".severityAggregatorLogAppender");
  
  public static final String UNEXPECTED_MATCH = "Unexpected match";
  
  public static final String EXPECTED_NOT_FOUND = "Expected match not found";
  
  private EMFScope scope = new EMFScope(new ResourceSetImpl());
  
  private boolean isScopeSet = false;
  
  private final List<IMatchSetModelProvider> modelProviders;
  
  @Extension
  private final SnapshotHelper snapshotHelper;
  
  private final TestingSeverityAggregatorLogAppender appender;
  
  public ViatraQueryTestCase() {
    this(new SnapshotHelper());
  }
  
  public ViatraQueryTestCase(final Map<String, JavaObjectAccess> accessMap) {
    this(new SnapshotHelper(accessMap, new HashMap<EClass, Function<EObject, String>>()));
  }
  
  /**
   * @since 2.2
   */
  public ViatraQueryTestCase(final SnapshotHelper helper) {
    LinkedList<IMatchSetModelProvider> _linkedList = new LinkedList<IMatchSetModelProvider>();
    this.modelProviders = _linkedList;
    this.snapshotHelper = helper;
    final Appender a = ViatraQueryLoggingUtil.getLogger(ViatraQueryEngine.class).getAppender(
      ViatraQueryTestCase.SEVERITY_AGGREGATOR_LOGAPPENDER_NAME);
    TestingSeverityAggregatorLogAppender _xifexpression = null;
    if ((a instanceof TestingSeverityAggregatorLogAppender)) {
      TestingSeverityAggregatorLogAppender _xblockexpression = null;
      {
        ((TestingSeverityAggregatorLogAppender)a).clear();
        _xblockexpression = ((TestingSeverityAggregatorLogAppender)a);
      }
      _xifexpression = _xblockexpression;
    } else {
      TestingSeverityAggregatorLogAppender _xblockexpression_1 = null;
      {
        final TestingSeverityAggregatorLogAppender na = new TestingSeverityAggregatorLogAppender();
        na.setName(ViatraQueryTestCase.SEVERITY_AGGREGATOR_LOGAPPENDER_NAME);
        ViatraQueryLoggingUtil.getLogger(ViatraQueryEngine.class).addAppender(na);
        _xblockexpression_1 = na;
      }
      _xifexpression = _xblockexpression_1;
    }
    this.appender = _xifexpression;
  }
  
  public SnapshotHelper getSnapshotHelper() {
    return this.snapshotHelper;
  }
  
  public void assertLogSeverityThreshold(final Level severity) {
    int _int = this.appender.getSeverity().toInt();
    int _int_1 = severity.toInt();
    boolean _greaterThan = (_int > _int_1);
    if (_greaterThan) {
      String _string = this.appender.getSeverity().toString();
      String _plus = (_string + " message on log: ");
      String _renderedMessage = this.appender.getEvent().getRenderedMessage();
      String _plus_1 = (_plus + _renderedMessage);
      Assert.fail(_plus_1);
    }
  }
  
  public void assertLogSeverity(final Level severity) {
    Assert.assertEquals(severity, this.appender.getSeverity());
  }
  
  public boolean loadModel(final URI uri) {
    boolean _xblockexpression = false;
    {
      final ResourceSetImpl resourceSet = new ResourceSetImpl();
      resourceSet.getResource(uri, true);
      EMFScope _eMFScope = new EMFScope(resourceSet);
      this.scope = _eMFScope;
      _xblockexpression = this.isScopeSet = true;
    }
    return _xblockexpression;
  }
  
  /**
   * Sets the scope of the VIATRA Query test case
   * 
   * @since 1.5.2
   */
  public boolean setScope(final EMFScope scope) {
    boolean _xblockexpression = false;
    {
      this.scope = scope;
      _xblockexpression = this.isScopeSet = true;
    }
    return _xblockexpression;
  }
  
  public void dispose() {
    final Consumer<IMatchSetModelProvider> _function = (IMatchSetModelProvider it) -> {
      it.dispose();
    };
    this.modelProviders.forEach(_function);
    this.modelProviders.clear();
    Set<? extends Notifier> _scopeRoots = this.scope.getScopeRoots();
    for (final Notifier n : _scopeRoots) {
      boolean _matched = false;
      if (n instanceof ResourceSet) {
        _matched=true;
        final Consumer<Resource> _function_1 = (Resource it) -> {
          it.unload();
        };
        ((ResourceSet)n).getResources().forEach(_function_1);
      }
      if (!_matched) {
        if (n instanceof Resource) {
          _matched=true;
          ((Resource)n).unload();
        }
      }
      if (!_matched) {
        if (n instanceof EObject) {
          _matched=true;
          ((EObject)n).eResource().unload();
        }
      }
    }
  }
  
  public <T extends EObject> void modifyModel(final Class<T> clazz, final Predicate<T> condition, final Consumer<T> operation) {
    final Function1<IMatchSetModelProvider, Boolean> _function = (IMatchSetModelProvider it) -> {
      boolean _updatedByModify = it.updatedByModify();
      return Boolean.valueOf((!_updatedByModify));
    };
    final Iterable<IMatchSetModelProvider> nonIncrementals = IterableExtensions.<IMatchSetModelProvider>filter(this.modelProviders, _function);
    CollectionExtensions.<IMatchSetModelProvider>removeAll(this.modelProviders, nonIncrementals);
    final Consumer<IMatchSetModelProvider> _function_1 = (IMatchSetModelProvider it) -> {
      it.dispose();
    };
    nonIncrementals.forEach(_function_1);
    final LinkedList<T> elementsToModify = CollectionLiterals.<T>newLinkedList();
    Set<? extends Notifier> _scopeRoots = this.scope.getScopeRoots();
    for (final Notifier n : _scopeRoots) {
      {
        Iterator<? extends Notifier> iterator = null;
        boolean _matched = false;
        if (n instanceof ResourceSet) {
          _matched=true;
          iterator = ((ResourceSet)n).getAllContents();
        }
        if (!_matched) {
          if (n instanceof Resource) {
            _matched=true;
            iterator = ((Resource)n).getAllContents();
          }
        }
        if (!_matched) {
          if (n instanceof EObject) {
            _matched=true;
            iterator = ((EObject)n).eAllContents();
          }
        }
        while (iterator.hasNext()) {
          {
            final Notifier element = iterator.next();
            boolean _isInstance = clazz.isInstance(element);
            if (_isInstance) {
              final T cast = clazz.cast(element);
              boolean _test = condition.test(cast);
              if (_test) {
                T _cast = clazz.cast(element);
                elementsToModify.add(_cast);
              }
            }
          }
        }
      }
    }
    for (final T element : elementsToModify) {
      operation.accept(element);
    }
    this.appender.clear();
  }
  
  public boolean addMatchSetModelProvider(final IMatchSetModelProvider matchSetModelProvider) {
    return this.modelProviders.add(matchSetModelProvider);
  }
  
  public <Match extends IPatternMatch> void assumeMatchSetsAreAvailable(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification) {
    final Consumer<IMatchSetModelProvider> _function = (IMatchSetModelProvider it) -> {
      try {
        it.<Match>getMatchSetRecord(this.scope, querySpecification, null);
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          Assume.assumeNoException(e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    };
    this.modelProviders.forEach(_function);
  }
  
  public <Match extends IPatternMatch> void assertMatchSetsEqual(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification) {
    DefaultMatchRecordEquivalence _defaultMatchRecordEquivalence = new DefaultMatchRecordEquivalence(this.snapshotHelper);
    this.<Match>assertMatchSetsEqual(querySpecification, _defaultMatchRecordEquivalence);
  }
  
  /**
   * Checks if the match sets of the given query specification are equivalent, based on a user specified equivalence logic.
   * 
   * @since 1.6
   */
  public <Match extends IPatternMatch> void assertMatchSetsEqual(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification, final MatchRecordEquivalence equivalence) {
    this.validateTestCase();
    final IMatchSetModelProvider reference = IterableExtensions.<IMatchSetModelProvider>head(this.modelProviders);
    final Consumer<IMatchSetModelProvider> _function = (IMatchSetModelProvider it) -> {
      final MatchSetRecordDiff matchDiff = this.<Match>getMatchSetDiff(querySpecification, reference, it, equivalence);
      boolean _isEmpty = matchDiff.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final Joiner joiner = Joiner.on("\n");
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Differences found between reference ");
        String _name = reference.getClass().getName();
        _builder.append(_name);
        _builder.append(" and match set provider ");
        String _name_1 = it.getClass().getName();
        _builder.append(_name_1);
        final Function1<MatchRecord, String> _function_1 = (MatchRecord it_1) -> {
          return this.snapshotHelper.prettyPrint(it_1);
        };
        String _join = joiner.join(IterableExtensions.<MatchRecord, String>map(matchDiff.getAdditions(), _function_1));
        final Function1<MatchRecord, String> _function_2 = (MatchRecord it_1) -> {
          return this.snapshotHelper.prettyPrint(it_1);
        };
        String _join_1 = joiner.join(IterableExtensions.<MatchRecord, String>map(matchDiff.getRemovals(), _function_2));
        throw new ComparisonFailure(_builder.toString(), _join, _join_1);
      }
    };
    IterableExtensions.<IMatchSetModelProvider>tail(this.modelProviders).forEach(_function);
  }
  
  public void assertMatchSetsEqual(final IQueryGroup queryGroup) {
    DefaultMatchRecordEquivalence _defaultMatchRecordEquivalence = new DefaultMatchRecordEquivalence(this.snapshotHelper);
    this.assertMatchSetsEqual(queryGroup, _defaultMatchRecordEquivalence);
  }
  
  /**
   * Checks if the match sets of the queries contained in the provided query group are equivalent in the scope of added {@link IMatchSetModelProvider} instances.
   * 
   * @since 1.6
   */
  public void assertMatchSetsEqual(final IQueryGroup queryGroup, final MatchRecordEquivalence equivalence) {
    final Consumer<IQuerySpecification<?>> _function = (IQuerySpecification<?> it) -> {
      this.<IPatternMatch>assertMatchSetsEqual(((IQuerySpecification<? extends ViatraQueryMatcher<IPatternMatch>>) it), equivalence);
    };
    queryGroup.getSpecifications().forEach(_function);
  }
  
  /**
   * Calculates the differences between the match sets of a given {@link IQuerySpecification} based on the specified {@link IMatchSetModelProvider} instances.
   * 
   * @since 1.6
   */
  public <Match extends IPatternMatch> MatchSetRecordDiff getMatchSetDiff(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification, final IMatchSetModelProvider expectedProvider, final IMatchSetModelProvider actualProvider, final MatchRecordEquivalence equivalence) {
    MatchSetRecordDiff _xblockexpression = null;
    {
      this.validateTestCase();
      Match filter = null;
      MatchSetRecord expected = expectedProvider.<Match>getMatchSetRecord(this.scope, querySpecification, filter);
      MatchRecord _filter = expected.getFilter();
      boolean _tripleNotEquals = (_filter != null);
      if (_tripleNotEquals) {
        filter = this.snapshotHelper.<Match>createMatchForMatchRecord(querySpecification, expected.getFilter());
      }
      final MatchSetRecord actual = actualProvider.<Match>getMatchSetRecord(this.scope, querySpecification, filter);
      MatchRecord _filter_1 = actual.getFilter();
      boolean _tripleNotEquals_1 = (_filter_1 != null);
      if (_tripleNotEquals_1) {
        if ((filter != null)) {
          throw new IllegalArgumentException(
            ((("Filter is provided by more than one sources: " + expectedProvider) + ", ") + actualProvider));
        } else {
          filter = this.snapshotHelper.<Match>createMatchForMatchRecord(querySpecification, actual.getFilter());
          expected = expectedProvider.<Match>getMatchSetRecord(this.scope, querySpecification, filter);
        }
      }
      _xblockexpression = MatchSetRecordDiff.compute(expected, actual, equivalence);
    }
    return _xblockexpression;
  }
  
  public <Match extends IPatternMatch> MatchSetRecordDiff getMatchSetDiff(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification, final IMatchSetModelProvider expectedProvider, final IMatchSetModelProvider actualProvider) {
    DefaultMatchRecordEquivalence _defaultMatchRecordEquivalence = new DefaultMatchRecordEquivalence(this.snapshotHelper);
    return this.<Match>getMatchSetDiff(querySpecification, expectedProvider, actualProvider, _defaultMatchRecordEquivalence);
  }
  
  /**
   * Validate the created test configuration before calculating and comparing the query results
   * 
   * @since 1.6
   */
  private void validateTestCase() {
    int _size = this.modelProviders.size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      throw new IllegalArgumentException("At least two model providers shall be set");
    }
    if (((!this.isScopeSet) && (!IterableExtensions.<IMatchSetModelProvider>exists(this.modelProviders, 
      ((Function1<IMatchSetModelProvider, Boolean>) (IMatchSetModelProvider it) -> {
        return Boolean.valueOf((it instanceof InitializedSnapshotMatchSetModelProvider));
      }))))) {
      throw new IllegalArgumentException("Always include a model in the test specification");
    }
  }
}
