/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLParserImpl.java,v 1.4 2007/03/18 09:07:04 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource.parser.impl;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.eodm.OWLFactory;
import org.eclipse.eodm.exceptions.AddingTripleException;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.owlbase.OWLGraph;
import org.eclipse.eodm.owl.owlbase.OWLOntology;
import org.eclipse.eodm.owl.resource.parser.OWLDocument;
import org.eclipse.eodm.owl.resource.parser.OWLParser;
import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.resource.parser.RDFXMLParser;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserException;
import org.eclipse.eodm.rdf.resource.parser.xml.RDFXMLParserImpl;
import org.eclipse.eodm.util.Triple;
import org.eclipse.eodm.vocabulary.OWL;
import org.eclipse.eodm.vocabulary.RDF;
import org.xml.sax.InputSource;

/**
 * OWLParserImpl - OWL Parser implementation
 */
public class OWLParserImpl implements OWLParser {

    /**
     * OWL documents added in the parser.
     */
    private ArrayList documents = new ArrayList();

    /**
     * Parsed OWL documents
     */
    private Hashtable parsedResults = new Hashtable();
    
    /**
     * Hashtable to keep mapping information 
     * between ontology urls and local file names
     */
    private Hashtable mapping=new Hashtable();

    /**
     * Construct an OWL Parser instance.
     */
    public OWLParserImpl() {
        super();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#addOWLDocument(org.eclipse.eodm.owl.resource.parser.OWLDocument)
     */
    public void addOWLDocument(OWLDocument document) throws OWLParserException {
        /** Document's local URI and public URI can not be null at the same time */
        if (document == null
            || document.getLocalURI() == null
            && document.getPublicURI() == null)
            throw new OWLParserException("Document is not valid.");

        /** If an existing document has been added, nothing will be done. */
        if (getOWLDocument(document) != null) {
            System.out.print("Document already exists.");
            return;
        }

        documents.add(document);

        OntologyTripleAnalyzer tripleAnalyser = new OntologyTripleAnalyzer(
                document);
        RDFXMLParser parser = new RDFXMLParserImpl();
        // Set the namespace and statement handler.
        OWLParserResultHandler resultHandler = new OWLParserResultHandler(
                tripleAnalyser);
        parser.setNamespaceHandler(resultHandler);
        parser.setStatementHandler(resultHandler);

        InputSource input = new InputSource(document.readInput());
        if (document.getCharset() != null)
            input.setEncoding(document.getCharset());
        /** Set the default base uri to be document's uri. */
        String defaultBaseURI = document.getDefaultBaseURI();
        if (defaultBaseURI == null) {
            if (document.isLocalFile()) {
                StringBuffer buffer = new StringBuffer("file:///");
                for (int i = 0; i < document.getLocalURI().length(); i++)
                    if (document.getLocalURI().charAt(i) == '\\')
                        buffer.append('/');
                    else if (document.getLocalURI().charAt(i) != ' ')
                        buffer.append(document.getLocalURI().charAt(i));
                defaultBaseURI = buffer.toString();
            } else
                defaultBaseURI = document.getPublicURI();
            document.setDefaultBaseURI(defaultBaseURI);
        }
        /** Start the parsing process: deal with ontology import firstly */
        try {
            parser.parse(input, defaultBaseURI);
        } catch (ParserException e) {
            throw new OWLParserException("RDF parsing error", e);
        }
        input = null;
        try {
            if (!document.getLocalURI().equals("OWLInputStream")){
            document.readInput().close();
            document.setInputStream(null);
            }
        } catch (OWLParserException e) {
            e.printStackTrace();
            throw new OWLParserException("Parser IO exception: "
                                         + e.getMessage());
        } catch (IOException e) {
            e.printStackTrace();
            throw new OWLParserException("Parser IO exception: "
                                         + e.getMessage());
        }
        parser = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#addOWLDocument(java.lang.String)
     */
    public void addOWLDocument(String localFileName) throws OWLParserException {
        if (localFileName == null || localFileName.length() == 0)
            return;
        OWLDocument doc = new OWLDocumentImpl(localFileName, null, true);
        addOWLDocument(doc);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#parseOWLDocument(org.eclipse.eodm.owl.resource.parser.OWLDocument)
     */
    public OWLOntologyGraph[] parseOWLDocument(OWLDocument document)
            throws OWLParserException {
        String documentURI;
        OWLDocument childdoc;
        List importingList;
        /** Add the document if it does not exist */
        OWLDocument doc = getOWLDocument(document);
        if (doc == null)
            addOWLDocument(document);
        else {
            if (doc.isParsed())
                return (OWLOntologyGraph[]) parsedResults.get(doc
                        .getPublicURI());
        }
        // store all imported ontology uris
        LinkedList docList = new LinkedList();
        docList.addFirst(document.getPublicURI());
        Iterator iter = document.getImportOntologyURIs(true).iterator();
        while (iter.hasNext()) {
            String str = (String) iter.next();
            if (!isEixstingURI(str, docList))
                docList.addLast(str);
        }

        OWLParserTripleAnalyzer tripleAnalyser = new OWLParserTripleAnalyzer();

        RDFXMLParser parser;
        // Set the namespace and statement handler.
        OWLParserResultHandler resultHandler = new OWLParserResultHandler(
                tripleAnalyser);

        for (int i = 0; i < docList.size(); i++) {
            childdoc = null;
            documentURI = (String) docList.get(i);
            childdoc = getOWLDocumentbyPublicURI(documentURI);
            // if the document does not exist, create a new OWLDocument
            if (childdoc == null) {
                URL url = null;
                try {
                    url = new URL(documentURI);
                } catch (MalformedURLException e1) {
                    e1.printStackTrace();
                    throw new OWLParserException("Parser exception: "
                                                 + e1.getMessage());
                }
                if (url != null && url.getProtocol().equalsIgnoreCase("file")) {
                    try {
                        String file = new File(url.getPath())
                                .getCanonicalPath();
                        childdoc = new OWLDocumentImpl(file, null, true);
                    } catch (IOException e) {
                        e.printStackTrace();
                        throw new OWLParserException("Parser exception: "
                                                     + e.getMessage());
                    }
                } else{
                    String name=(String)mapping.get(documentURI);  
                    if (name==null)
                        childdoc = new OWLDocumentImpl(null, documentURI, false);
                    else
                        childdoc = new OWLDocumentImpl(name, documentURI, true);
                }
                addOWLDocument(childdoc);
            }

            // if the document is not parsed, parse it.
            if (!childdoc.isParsed()) {
                parser = new RDFXMLParserImpl();
                // Set the namespace and statement handler.
                parser.setNamespaceHandler(resultHandler);
                parser.setStatementHandler(resultHandler);
                LinkedList links = childdoc.getImportOntologyURIs(true);
                for (int jj = 0; jj < links.size(); jj++) {
                    String str = (String) links.get(jj);
                    if (!isEixstingURI(str, docList))
                        docList.addLast(str);
                    importingList = getOWLDocumentbyImportingURI(childdoc
                            .getPublicURI());
                    if (importingList != null) {
                        for (int j = 0; j < importingList.size(); j++) {
                            ((OWLDocument) importingList.get(j))
                                    .addImportOntologyURIs(str);
                        }
                    }
                }
                tripleAnalyser.initialize(childdoc.getPublicURI());
                InputSource input = new InputSource(childdoc.readInput());
                if (childdoc.getCharset() != null)
                    input.setEncoding(childdoc.getCharset());
                try {
                    parser.parse(input, childdoc.getDefaultBaseURI());
                } catch (ParserException e) {
                    throw new OWLParserException("RDF parsing error: "+ e.getCause());
                }
                tripleAnalyser.postProcessing();
                parsedResults.put(childdoc.getPublicURI(), tripleAnalyser
                        .getOWLOntologyGraph());
                childdoc.setParsed();
                childdoc.setOntologyGraph(tripleAnalyser.getOWLOntologyGraph());
                parser = null;
            }
        }

        // generate parsed results
        for (int i = 0; i < docList.size(); i++) {
            childdoc = getOWLDocumentbyPublicURI((String) docList.get(i));
            if (childdoc == null) {
                docList.remove(i);
                i--;
            }
        }
        OWLOntologyGraph[] ontologygraph = new OWLOntologyGraph[docList.size()];
        OWLOntologyGraph g;
        OWLGraph igraph = null;
        if (docList.size() > 1) {
            try {
                igraph = OWLFactory.eINSTANCE.createOWLGraph(document
                        .getPublicURI());
            } catch (URISyntaxException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());
            }
            igraph = mergeGraph(igraph, document.getOntologyGraph().getgraph());
            for (int i = 0; i < docList.size(); i++) {
                childdoc = getOWLDocumentbyPublicURI((String) docList.get(i));
                g = childdoc.getOntologyGraph();
                iter = null;
                try {
                    iter = g.getgraph().getTypeResources(
                            OWL.NAMESPACE_URI + OWL.ONTOLOGY).iterator();
                } catch (UnsupportedViewTypeException e) {
                    e.printStackTrace();
                    throw new OWLParserException("Parser exception: "
                                                 + e.getMessage());
                }
                while (iter != null && iter.hasNext())
                    try {
                        g.setOntology((OWLOntology) (((RDFSResource) iter
                                .next()).asType(OWLOntology.class)));
                    } catch (UnsupportedViewTypeException e) {
                        e.printStackTrace();
                        throw new OWLParserException("Parser exception: "
                                                     + e.getMessage());
                    }
                ontologygraph[i] = g;
                if (i > 0)
                    igraph = mergeGraph(igraph, g.getgraph());
            }
            ontologygraph[0].setgraph(igraph);
            parsedResults.put(document.getPublicURI(), ontologygraph);
            return ontologygraph;
        } else {
            g = document.getOntologyGraph();
            try {
                iter = g.getgraph().getTypeResources(
                        OWL.NAMESPACE_URI + OWL.ONTOLOGY).iterator();
            } catch (UnsupportedViewTypeException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());
            }
            while (iter != null && iter.hasNext())
                try {
                    g.setOntology((OWLOntology) (((RDFSResource) iter.next())
                            .asType(OWLOntology.class)));
                } catch (UnsupportedViewTypeException e) {
                    e.printStackTrace();
                    throw new OWLParserException("Parser exception: "
                                                 + e.getMessage());
                }
        }
        ontologygraph[0] = g;
        parsedResults.put(document.getPublicURI(), ontologygraph);
        return ontologygraph;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#clear()
     */
    public void clear() {
        documents.clear();
        parsedResults.clear();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#clearDocument(java.lang.String)
     */
    public void clearDocument(String localFileName) {
        OWLDocument doc;
        for (int i = 0; i < documents.size(); i++) {
            doc = (OWLDocument) documents.get(i);
            if (doc.getLocalURI().equals(localFileName)) {
                documents.remove(i);
                return;
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#getOWLDocuments()
     */
    public List getOWLDocuments() {
        return documents;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#getOWLDocument(org.eclipse.eodm.owl.resource.parser.OWLDocument)
     */
    public OWLDocument getOWLDocument(OWLDocument document) {
        OWLDocument doc;
        if (documents.size() == 0)
            return null;
        for (int i = 0; i < documents.size(); i++) {
            doc = (OWLDocument) documents.get(i);
            if (document.getPublicURI() != null && doc.getPublicURI() != null)
                if (document.getPublicURI().equals(doc.getPublicURI()))
                    return doc;
            if (document.getLocalURI() != null && doc.getLocalURI() != null)
                if (document.getLocalURI().equals(doc.getLocalURI()))
                    return doc;
        }
        return null;
    }

    private OWLDocument getOWLDocumentbyPublicURI(String uri) {
        OWLDocument doc;
        if (documents.size() == 0)
            return null;
        for (int i = 0; i < documents.size(); i++) {
            doc = (OWLDocument) documents.get(i);
            if (doc.getPublicURI() != null)
                if (uri.equals(doc.getPublicURI()))
                    return doc;
        }
        return null;
    }

    private boolean isEixstingURI(String uri, LinkedList docList) {
        for (int i = 0; i < docList.size(); i++)
            if (uri.equals(docList.get(i)))
                return true;
        return false;
    }

    private List getOWLDocumentbyImportingURI(String URI) {
        OWLDocument document;
        ArrayList list = new ArrayList();
        Iterator iter;
        for (int i = 0; i < documents.size(); i++) {
            document = (OWLDocument) documents.get(i);
            iter = document.getImportOntologyURIs(true).iterator();
            while (iter.hasNext()) {
                if (URI.equals((String) iter.next())) {
                    list.add(document);
                    break;
                }
            }
        }
        if (list.size() > 0)
            return list;
        else
            return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLParser#parseOWLDocument(java.lang.String)
     */
    public OWLOntologyGraph[] parseOWLDocument(String localFileName)
            throws OWLParserException {
        if (localFileName == null || localFileName.length() == 0)
            return null;
        OWLDocument doc = new OWLDocumentImpl(localFileName, null, true);
        return parseOWLDocument(doc);
    }

    private OWLGraph mergeGraph(OWLGraph g1, OWLGraph g2) {
        Iterator triples = g2.exportTriples().iterator();
        Triple triple;
        while (triples.hasNext()) {
            triple = (Triple) triples.next();
            try {
                if (triple.getPredicateUriRef().getURIString().equals(RDF.P_TYPE_STR))
                    g1.addTriple(triple);
            } catch (UnsupportedViewTypeException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());
            } catch (URISyntaxException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());
            } catch (AddingTripleException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());            	
            }
        }
        
        Iterator triples2 = g2.exportTriples().iterator();
        Triple triple2;
        while (triples2.hasNext()) {
            triple2 = (Triple) triples2.next();
            try {
                //if (triple.getPredicateUriRef().getURIString().equals(RDF.P_TYPE_STR))
                    g1.addTriple(triple2);
            } catch (UnsupportedViewTypeException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());
            } catch (URISyntaxException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());
            } catch (AddingTripleException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser exception: "
                                             + e.getMessage());             
            }
        }
        return g1;
    }

    public void addURL2FileMapping(String url, String filename) {
        String name=(String)mapping.get(url);
        if (name==null)
            mapping.put(url, filename);        
    }

}