/**
 * Copyright (c) 2021 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.designer.transformation.tracing.barectf.library;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.papyrus.designer.languages.common.base.StringConstants;
import org.eclipse.papyrus.designer.transformation.base.utils.OperationUtils;
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.Activator;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.CTFTypeUtils;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.flatten.Flattener;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.flatten.NameType;
import org.eclipse.papyrus.designer.transformation.tracing.library.EventNames;
import org.eclipse.papyrus.designer.transformation.tracing.library.utils.TraceUtils;
import org.eclipse.papyrus.moka.tracepoint.service.TraceActions;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Event;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.SignalEvent;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Create a BareCTF config.yaml file containing the information what to trace
 */
@SuppressWarnings("all")
public class BareCTFconfig {
  protected static IFolder srcCTF;
  
  protected static final String METADATA = "metadata";
  
  public static Map<Type, Boolean> types;
  
  /**
   * Create the YAML header. Must be called after createYaml, as it needs the list of declared
   * types
   */
  public static CharSequence createYamlHdr() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# Needed YAML tag for the configuration object");
    _builder.newLine();
    _builder.append("--- !<tag:barectf.org,2020/3/config>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# Configuration\'s trace");
    _builder.newLine();
    _builder.append("trace:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Type of the trace");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("type:");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("# Add standard field type aliases");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("$include:");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("- stdint.yaml");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("- stdreal.yaml");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("- stdmisc.yaml");
    _builder.newLine();
    {
      Set<Type> _keySet = BareCTFconfig.types.keySet();
      for(final Type type : _keySet) {
        _builder.append("\t\t\t");
        _builder.append("- ");
        String _name = type.getName();
        _builder.append(_name, "\t\t\t");
        _builder.append(".yaml");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    return _builder;
  }
  
  public static CharSequence createYaml(final List<Element> elemsToTrace) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("\t\t");
    _builder.append("# Native byte order is little-endian");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("native-byte-order: little-endian");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("# One clock type");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("clock-types:");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("# The Linux FS platform requires a clock type named `default`");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("# which has a 1-GHz frequency and the `uint64_t` C type.");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("default:");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("frequency: 1000000000");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("$c-type: uint64_t");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("# One data stream type");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("data-stream-types:");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("# Stream type named `default`");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("default:");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("# Default data stream type");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("$is-default: true");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("# Default clock type: `default`");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("$default-clock-type-name: default");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("# Two event record types");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("event-record-types:");
    _builder.newLine();
    {
      for(final Element elemToTrace : elemsToTrace) {
        {
          if ((elemToTrace instanceof Port)) {
            _builder.append("\t\t\t\t\t");
            CharSequence _createPortEvent = BareCTFconfig.createPortEvent(((Port) elemToTrace));
            _builder.append(_createPortEvent, "\t\t\t\t\t");
            _builder.newLineIfNotEmpty();
          } else {
            if ((elemToTrace instanceof State)) {
              _builder.append("\t\t\t\t\t");
              CharSequence _createStateEvent = BareCTFconfig.createStateEvent(((State) elemToTrace));
              _builder.append(_createStateEvent, "\t\t\t\t\t");
              _builder.newLineIfNotEmpty();
            } else {
              if ((elemToTrace instanceof Transition)) {
                _builder.append("\t\t\t\t\t");
                CharSequence _createTransitionEvent = BareCTFconfig.createTransitionEvent(((Transition) elemToTrace));
                _builder.append(_createTransitionEvent, "\t\t\t\t\t");
                _builder.newLineIfNotEmpty();
              } else {
                if ((elemToTrace instanceof Operation)) {
                  _builder.append("\t\t\t\t\t");
                  CharSequence _createOperationEvent = BareCTFconfig.createOperationEvent(((Operation) elemToTrace), null);
                  _builder.append(_createOperationEvent, "\t\t\t\t\t");
                  _builder.newLineIfNotEmpty();
                } else {
                  if ((elemToTrace instanceof DataType)) {
                    _builder.append("\t\t\t\t\t");
                    CharSequence _createDataTypeEvent = BareCTFconfig.createDataTypeEvent(((DataType) elemToTrace));
                    _builder.append(_createDataTypeEvent, "\t\t\t\t\t");
                    _builder.newLineIfNotEmpty();
                  }
                }
              }
            }
          }
        }
        _builder.newLine();
      }
    }
    return _builder;
  }
  
  /**
   * Create config.yaml entries for a datatype
   */
  public static CharSequence createDataTypeEvent(final DataType dt) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# UML datatype named `");
    String _name = dt.getName();
    _builder.append(_name);
    _builder.append("`");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    String _name_1 = dt.getName();
    _builder.append(_name_1, "\t");
    _builder.append(":");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("payload-field-type:");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("class: structure");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("members:");
    _builder.newLine();
    {
      boolean _hasHint = TraceUtils.hasHint(dt);
      if (_hasHint) {
        _builder.append("\t\t\t\t");
        String _declarationHint = TraceUtils.declarationHint(dt);
        _builder.append(_declarationHint, "\t\t\t\t");
        _builder.newLineIfNotEmpty();
      } else {
        {
          List<NameType> _flattenDecl = Flattener.<Property>flattenDecl(dt.getAttributes());
          for(final NameType attribute : _flattenDecl) {
            _builder.append("\t\t\t\t");
            _builder.append("- ");
            String _name_2 = attribute.getName();
            _builder.append(_name_2, "\t\t\t\t");
            _builder.append(": ");
            CharSequence _ctfType = CTFTypeUtils.ctfType(attribute.getType());
            _builder.append(_ctfType, "\t\t\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    return _builder;
  }
  
  /**
   * Create entries for an interface exposed by a port. Create one event for each
   * operation
   */
  public static CharSequence createPortEvent(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<Interface> _provideds = port.getProvideds();
      for(final Interface intf : _provideds) {
        {
          EList<Operation> _operations = intf.getOperations();
          for(final Operation iOp : _operations) {
            final Operation cOp = OperationUtils.getSameOperation(iOp, port.getClass_());
            _builder.newLineIfNotEmpty();
            CharSequence _createOperationEvent = BareCTFconfig.createOperationEvent(cOp, port);
            _builder.append(_createOperationEvent);
            _builder.newLineIfNotEmpty();
            _builder.newLine();
          }
        }
      }
    }
    return _builder;
  }
  
  /**
   * Create entries for an operation.
   */
  public static CharSequence createOperationEvent(final Operation operation, final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# operation ");
    String _name = operation.getName();
    _builder.append(_name);
    _builder.append(" of classifier ");
    String _name_1 = operation.getClass_().getName();
    _builder.append(_name_1);
    _builder.newLineIfNotEmpty();
    CharSequence _operationStartsEventName = EventNames.operationStartsEventName(operation, port);
    _builder.append(_operationStartsEventName);
    _builder.append(":");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("payload-field-type:");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("class: structure");
    _builder.newLine();
    _builder.append("\t\t");
    final CharSequence members = BareCTFconfig.operationMembers(operation, port);
    _builder.newLineIfNotEmpty();
    {
      int _length = members.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _builder.append("\t\t");
        _builder.append("members:");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("\t");
        _builder.append(members, "\t\t\t");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      if ((TraceUtils.traceOpOrPortAction(operation, port, TraceActions.TAOperation.MethodEnds) && (operation.getType() == null))) {
        CharSequence _operationEndsEventName = EventNames.operationEndsEventName(operation, port);
        _builder.append(_operationEndsEventName);
        _builder.append(":");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("payload-field-type:");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("class: structure");
        _builder.newLine();
        {
          boolean _needInstanceName = TraceUtils.needInstanceName(operation);
          if (_needInstanceName) {
            _builder.append("\t\t");
            _builder.append("members:");
            _builder.newLine();
            _builder.append("\t\t");
            _builder.append("\t");
            _builder.append("- instanceId : string");
            _builder.newLine();
          }
        }
      }
    }
    return _builder;
  }
  
  /**
   * Calculate the list of members for an operation
   */
  public static CharSequence operationMembers(final Operation operation, final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _needInstanceName = TraceUtils.needInstanceName(operation);
      if (_needInstanceName) {
        _builder.append("- instanceId : string");
        _builder.newLine();
      }
    }
    {
      boolean _hasHint = TraceUtils.hasHint(operation);
      if (_hasHint) {
        String _declarationHint = TraceUtils.declarationHint(operation);
        _builder.append(_declarationHint);
        _builder.newLineIfNotEmpty();
      } else {
        boolean _traceOpOrPortAction = TraceUtils.traceOpOrPortAction(operation, port, TraceActions.TAOperation.ParameterValues);
        if (_traceOpOrPortAction) {
          {
            List<NameType> _flattenDecl = Flattener.<Parameter>flattenDecl(TraceUtils.getInAndInout(operation));
            for(final NameType param : _flattenDecl) {
              _builder.append("- ");
              String _name = param.getName();
              _builder.append(_name);
              _builder.append(" : ");
              CharSequence _ctfType = CTFTypeUtils.ctfType(param.getType());
              _builder.append(_ctfType);
              _builder.newLineIfNotEmpty();
            }
          }
        }
      }
    }
    return _builder;
  }
  
  /**
   * Create entries for a transition.
   */
  public static CharSequence createTransitionEvent(final Transition t) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# transition ");
    String _name = t.getName();
    _builder.append(_name);
    _builder.append(" of state-machine ");
    String _name_1 = t.containingStateMachine().getName();
    _builder.append(_name_1);
    _builder.newLineIfNotEmpty();
    CharSequence _transitionEventName = EventNames.transitionEventName(t);
    _builder.append(_transitionEventName);
    _builder.append(":");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("payload-field-type:");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("class: structure");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("members:");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("- instanceId : string");
    _builder.newLine();
    {
      boolean _traceTransitionAction = TraceUtils.traceTransitionAction(t, TraceActions.TATransition.TriggerValues);
      if (_traceTransitionAction) {
        {
          EList<Trigger> _triggers = t.getTriggers();
          for(final Trigger trigger : _triggers) {
            {
              Event _event = trigger.getEvent();
              if ((_event instanceof SignalEvent)) {
                _builder.append("\t\t\t");
                Event _event_1 = trigger.getEvent();
                final SignalEvent sigEvent = ((SignalEvent) _event_1);
                _builder.newLineIfNotEmpty();
                {
                  List<NameType> _flattenDecl = Flattener.<Property>flattenDecl(sigEvent.getSignal().getAttributes());
                  for(final NameType attr : _flattenDecl) {
                    _builder.append("\t\t\t");
                    _builder.append("- ");
                    String _name_2 = attr.getName();
                    _builder.append(_name_2, "\t\t\t");
                    _builder.append(" : ");
                    CharSequence _ctfType = CTFTypeUtils.ctfType(attr.getType());
                    _builder.append(_ctfType, "\t\t\t");
                    _builder.newLineIfNotEmpty();
                  }
                }
              }
            }
          }
        }
      }
    }
    _builder.newLine();
    return _builder;
  }
  
  /**
   * Create entries for a state.
   */
  public static CharSequence createStateEvent(final State state) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _traceStateAction = TraceUtils.traceStateAction(state, TraceActions.TAState.StateEnter);
      if (_traceStateAction) {
        _builder.append("# enter state ");
        String _name = state.getName();
        _builder.append(_name);
        _builder.append(" of state-machine ");
        String _name_1 = state.containingStateMachine().getName();
        _builder.append(_name_1);
        _builder.newLineIfNotEmpty();
        CharSequence _enterStateEventName = EventNames.enterStateEventName(state);
        _builder.append(_enterStateEventName);
        _builder.append(":");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("payload-field-type:");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("class: structure");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("members:");
        _builder.newLine();
        _builder.append("\t\t\t");
        _builder.append("- instanceId : string");
        _builder.newLine();
        _builder.newLine();
      }
    }
    {
      boolean _traceStateAction_1 = TraceUtils.traceStateAction(state, TraceActions.TAState.StateLeave);
      if (_traceStateAction_1) {
        _builder.append("# exit state ");
        String _name_2 = state.getName();
        _builder.append(_name_2);
        _builder.append(" of state-machine ");
        String _name_3 = state.containingStateMachine().getName();
        _builder.append(_name_3);
        _builder.newLineIfNotEmpty();
        CharSequence _exitStateEventName = EventNames.exitStateEventName(state);
        _builder.append(_exitStateEventName);
        _builder.append(":");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("payload-field-type:");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("class: structure");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("members:");
        _builder.newLine();
        _builder.append("\t\t\t");
        _builder.append("- instanceId : string");
        _builder.newLine();
        _builder.newLine();
      }
    }
    return _builder;
  }
  
  /**
   * Write a barectf configuration file (config.yaml)
   */
  public static void writeConfig(final IFolder srcCTF, final List<Element> elemsToTrace) {
    BareCTFconfig.srcCTF = srcCTF;
    LinkedHashMap<Type, Boolean> _linkedHashMap = new LinkedHashMap<Type, Boolean>();
    BareCTFconfig.types = _linkedHashMap;
    final CharSequence yamlEnd = BareCTFconfig.createYaml(elemsToTrace);
    CharSequence _createYamlHdr = BareCTFconfig.createYamlHdr();
    final StringBuffer yaml = new StringBuffer(_createYamlHdr).append(yamlEnd);
    BareCTFconfig.writeFile("config.yaml", yaml);
  }
  
  public static void writeFile(final String fileName, final CharSequence content) {
    try {
      final IFile configFile = BareCTFconfig.srcCTF.getFile(fileName);
      TransformationContext.current.keepFiles.add(configFile.getFullPath().toString());
      String _trim = CTFTypeUtils.confTabs(content).trim();
      final String contentStr = (_trim + StringConstants.EOL);
      byte[] _bytes = contentStr.getBytes();
      final ByteArrayInputStream contentStream = new ByteArrayInputStream(_bytes);
      boolean _exists = configFile.exists();
      if (_exists) {
        configFile.setContents(contentStream, true, true, null);
      } else {
        configFile.create(contentStream, true, null);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Inject the URIs of the elements to trace into the meta-data file. This is
   * required, as barectf itself does not support such an attribute in the config.yaml
   * file.
   */
  public static void injectURIs(final List<Element> elemsToTrace) {
    try {
      final IFile metaData = BareCTFconfig.srcCTF.getFile("metadata");
      boolean _exists = metaData.exists();
      boolean _not = (!_exists);
      if (_not) {
        Activator.log.debug("Cannot read CTF metadata file");
        return;
      }
      byte[] _readAllBytes = metaData.getContents().readAllBytes();
      String contentStr = new String(_readAllBytes, StandardCharsets.UTF_8);
      for (final Element elemToTrace : elemsToTrace) {
        {
          CharSequence eventName = null;
          if ((elemToTrace instanceof Port)) {
            EList<Interface> _provideds = ((Port)elemToTrace).getProvideds();
            for (final Interface intf : _provideds) {
              EList<Operation> _operations = intf.getOperations();
              for (final Operation iOp : _operations) {
                {
                  final Operation cOp = OperationUtils.getSameOperation(iOp, ((Port)elemToTrace).getClass_());
                  contentStr = BareCTFconfig.injectURI(contentStr, elemToTrace, EventNames.operationStartsEventName(cOp, ((Port) elemToTrace)));
                }
              }
            }
          } else {
            if ((elemToTrace instanceof State)) {
              eventName = EventNames.enterStateEventName(((State) elemToTrace));
              contentStr = BareCTFconfig.injectURI(contentStr, elemToTrace, eventName);
              eventName = EventNames.exitStateEventName(((State) elemToTrace));
            } else {
              if ((elemToTrace instanceof Transition)) {
                eventName = EventNames.transitionEventName(((Transition) elemToTrace));
              } else {
                if ((elemToTrace instanceof Operation)) {
                  eventName = EventNames.operationStartsEventName(((Operation) elemToTrace), null);
                  contentStr = BareCTFconfig.injectURI(contentStr, elemToTrace, eventName);
                  eventName = EventNames.operationEndsEventName(((Operation) elemToTrace), null);
                } else {
                  if ((elemToTrace instanceof DataType)) {
                  }
                }
              }
            }
          }
          if ((eventName != null)) {
            contentStr = BareCTFconfig.injectURI(contentStr, elemToTrace, eventName);
          }
        }
      }
      byte[] _bytes = contentStr.getBytes();
      final ByteArrayInputStream contentStream = new ByteArrayInputStream(_bytes);
      metaData.setContents(contentStream, true, true, null);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Inject the URI of an element to trace into the meta-data file. The
   * place where it needs to be inserted is identified via the assignment of the
   * event name.
   */
  public static String injectURI(final String contentStr, final Element elemToTrace, final CharSequence eventName) {
    final EObject sourceElemToTrace = TraceUtils.getSourceElement(elemToTrace);
    if ((sourceElemToTrace != null)) {
      final Resource r = sourceElemToTrace.eResource();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("name = \"");
      String _string = eventName.toString();
      _builder.append(_string);
      _builder.append("\";");
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("name = \"");
      _builder_1.append(eventName);
      _builder_1.append("\";");
      _builder_1.newLineIfNotEmpty();
      _builder_1.append("\t");
      _builder_1.append("model.emf.uri = \"");
      URI _uRI = r.getURI();
      _builder_1.append(_uRI, "\t");
      _builder_1.append("#");
      String _uRIFragment = r.getURIFragment(sourceElemToTrace);
      _builder_1.append(_uRIFragment, "\t");
      _builder_1.append("\";");
      return contentStr.replaceFirst(_builder.toString(), _builder_1.toString());
    }
    return null;
  }
}
