/*******************************************************************************
 * Copyright (c) 2007 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.ISafeRunnable;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.mint.CodeGenStatus;
import org.eclipse.emf.mint.IMemberAnnotationListener;
import org.eclipse.emf.mint.IMemberAnnotationManager;
import org.eclipse.emf.mint.MemberAnnotationChangedEvent;
import org.eclipse.emf.mint.MintCore;
import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IElementChangedListener;
import org.eclipse.jdt.core.IInitializer;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

public class MemberAnnotationManager implements IMemberAnnotationManager,
		IElementChangedListener {

	private static final Pattern GENERATED = Pattern
			.compile("@generated([\\s]+NOT)?"); //$NON-NLS-1$

	private static int MAX_CACHE_SIZE = 64;

	private static long CACHE_INTERVAL = 2000L;

	private final ListenerList listeners = new ListenerList();

	private final Map<IMember, CachedStatus> statusCache = new LinkedHashMap<IMember, CachedStatus>();

	private final CacheCleaner cacheCleaner = new CacheCleaner();

	public MemberAnnotationManager() {
		JavaCore.addElementChangedListener(this,
				ElementChangedEvent.POST_RECONCILE);
	}

	public void addMemberAnnotationListener(IMemberAnnotationListener listener) {
		listeners.add(listener);
	}

	public void removeMemberAnnotationListener(
			IMemberAnnotationListener listener) {
		listeners.remove(listener);
	}

	public CodeGenStatus getCodeGenStatus(IMember member) {
		synchronized (statusCache) {
			CachedStatus status = statusCache.remove(member);
			if (status == null) {
				CodeGenStatus value = getCodeGenStatusChecked(member);
				status = new CachedStatus(value);
			}

			status.timestamp = System.currentTimeMillis();
			statusCache.put(member, status);
			cacheCleaner.schedule();
			return status.value;
		}
	}

	private CodeGenStatus getCodeGenStatusChecked(IMember member) {
		CodeGenStatus value = CodeGenStatus.NONE;
		try {
			value = computeCodeGenStatus(member);
		} catch (JavaModelException e) {
			MintCore.getInstance().logError(
					Messages.MemberAnnotationManager_ErrorCodegenStatus, e);
		} catch (IOException e) {
			MintCore.getInstance().logError(
					Messages.MemberAnnotationManager_ErrorCodegenStatus, e);
		}

		return value;
	}

	public void elementChanged(ElementChangedEvent event) {
		IJavaElementDelta delta = event.getDelta();
		if ((delta.getKind() & IJavaElementDelta.CHANGED) == 0)
			return;

		IJavaElement element = delta.getElement();
		int flags = delta.getFlags();
		if (element instanceof ICompilationUnit) {
			if ((flags & IJavaElementDelta.F_AST_AFFECTED) != 0
					&& (flags & IJavaElementDelta.F_FINE_GRAINED) != 0
					&& (flags & (IJavaElementDelta.F_CONTENT | IJavaElementDelta.F_CHILDREN)) != 0) {
				ArrayList<IMember> members = new ArrayList<IMember>();
				collectMembers(element, members);

				Map<IMember, CodeGenStatus> changes = new HashMap<IMember, CodeGenStatus>();
				computeChanges(members, changes);

				if (!changes.isEmpty())
					fireMemberAnnotationChangedEvent(new MemberAnnotationChangedEvent(
							this, changes));
			}
		}
	}

	private void collectMembers(IJavaElement element,
			Collection<IMember> collector) {
		if (element instanceof IMember)
			collector.add((IMember) element);

		if (!(element instanceof IParent))
			return;

		IJavaElement[] children;
		try {
			children = ((IParent) element).getChildren();
		} catch (JavaModelException e) {
			MintCore.getInstance().logError(
					Messages.MemberAnnotationManager_ErrorJavaModel, e);
			return;
		}

		for (IJavaElement child : children) {
			if (child instanceof IInitializer)
				continue;

			collectMembers(child, collector);
		}
	}

	private void computeChanges(Collection<IMember> members,
			Map<IMember, CodeGenStatus> changes) {
		synchronized (statusCache) {
			for (IMember member : members) {
				CachedStatus status = statusCache.remove(member);
				if (status == null)
					continue;

				CodeGenStatus value = getCodeGenStatusChecked(member);
				if (value != status.value) {
					status.value = value;
					changes.put(member, value);
				}

				status.timestamp = System.currentTimeMillis();
				statusCache.put(member, status);
				cacheCleaner.schedule();
			}
		}
	}

	private void fireMemberAnnotationChangedEvent(
			final MemberAnnotationChangedEvent event) {
		for (Object obj : listeners.getListeners()) {
			final IMemberAnnotationListener listener = (IMemberAnnotationListener) obj;
			SafeRunner.run(new ISafeRunnable() {

				public void run() throws Exception {
					listener.memberAnnotationChanged(event);
				}

				public void handleException(Throwable e) {
					MintCore
							.getInstance()
							.logError(
									Messages.MemberAnnotationManager_ErrorNotifyListener,
									e);
				}
			});
		}
	}

	private CodeGenStatus computeCodeGenStatus(IMember member)
			throws JavaModelException, IOException {
		CodeGenStatus result = CodeGenStatus.NONE;
		if (!member.exists())
			return result;

		String javadoc = getJavadoc(member);
		if (javadoc == null)
			return result;

		Matcher m = GENERATED.matcher(javadoc);
		if (m.find())
			result = m.group(1) == null ? CodeGenStatus.GENERATED
					: CodeGenStatus.GENERATED_NOT;

		return result;
	}

	private String getJavadoc(IMember member) throws JavaModelException {
		IBuffer buf = member.getOpenable().getBuffer();
		if (buf == null)
			return null;

		ISourceRange range = member.getJavadocRange();
		if (range != null)
			return buf.getText(range.getOffset(), range.getLength());

		return null;
	}

	public void dispose() {
		cacheCleaner.cancel();
		JavaCore.removeElementChangedListener(this);
		listeners.clear();
	}

	private static class CachedStatus {

		public CodeGenStatus value;

		public long timestamp;

		public CachedStatus(CodeGenStatus value) {
			this.value = value;
		}
	}

	private class CacheCleaner extends Job {

		public CacheCleaner() {
			super(Messages.MemberAnnotationManager_JobCacheCleaner);
			setSystem(true);
			setPriority(DECORATE);
		}

		protected IStatus run(IProgressMonitor monitor) {
			synchronized (statusCache) {
				long threshold = System.currentTimeMillis() - CACHE_INTERVAL;
				for (Iterator<Map.Entry<IMember, CachedStatus>> i = statusCache
						.entrySet().iterator(); i.hasNext();) {
					Map.Entry<IMember, CachedStatus> entry = i.next();
					CachedStatus status = (CachedStatus) entry.getValue();
					if (status.timestamp < threshold) {
						if (statusCache.size() <= MAX_CACHE_SIZE)
							break;

						i.remove();
					} else {
						break;
					}
				}
			}

			return Status.OK_STATUS;
		}
	}
}
