/**
 * <copyright>
 *
 * Copyright (c) 2005, 2006, 2007, 2008 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal
 * </copyright>
 *
 * $Id: FeatureMapMapping.java,v 1.6 2008/02/28 07:09:02 mtaal Exp $
 */

package org.eclipse.emf.teneo.jpox.elist;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.jdo.spi.PersistenceCapable;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.teneo.jpox.JpoxStoreException;
import org.eclipse.emf.teneo.type.FeatureMapEntry;
import org.eclipse.emf.teneo.util.StoreUtil;
import org.jpox.StateManager;
import org.jpox.store.mapping.CollectionMapping;
import org.jpox.store.scostore.CollectionStore;

/**
 * Mapping class around the FeatureMap class. Is used to return the elver.jpox specific feature map implementation. In
 * addition when a FeatureMap is store it replaces the featuremap entries with the ones which are know to jpox. These
 * are the feature map entries generated by the OR Mapper.
 * 
 * @author <a href="mailto:mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.6 $ $Date: 2008/02/28 07:09:02 $
 */

public class FeatureMapMapping extends CollectionMapping {

	/**
	 * Accessor for an instance of the wrapper class.
	 * 
	 * @param sm
	 *            StateManager of the owner
	 * @param fldName
	 *            name of the field
	 * @return Instance of the wrapper
	 */
	protected Object newWrapper(StateManager sm, String fldName) {
		return new FeatureMapWrapper(sm, fieldName, new ArrayList<FeatureMap.Entry>());
	}

	/**
	 * Accessor for an instance of the wrapper class.
	 * 
	 * @param sm
	 *            StateManager of the owner
	 * @param fieldName
	 *            name of the field
	 * @return Instance of the wrapper
	 */
	protected Object newDefaultWrapper(StateManager sm, String fieldName) {
		return new FeatureMapWrapper(sm, fieldName, new ArrayList<FeatureMap.Entry>());
	}

	// ---------------- Implementation of MappingCallbacks --------------------

	/**
	 * Method to be called after any fetch of the owner class element.
	 * 
	 * @param sm
	 *            StateManager of the owner
	 */
	@SuppressWarnings("unchecked")
	public void postFetch(StateManager sm) {
		// Retrieve the type of the class
		Object value = sm.provideField(fmd.getAbsoluteFieldNumber());
		if (value == null) {
			if (fmd.getOrderMetaData() != null && fmd.getType() == java.util.Collection.class) {
				// We have a collection and the user has defined an ordering, so give them a List as a minimum
				instantiatedType = java.util.ArrayList.class;
			} else {
				instantiatedType = null;
			}
		} else {
			instantiatedType = value.getClass();
		}

		if (value == null) {
			sm.replaceField(fmd.getAbsoluteFieldNumber(), new FeatureMapWrapper(sm, fieldName, new ArrayList<FeatureMap.Entry>()));
		} else if (!(value instanceof FeatureMapWrapper)) {
			sm.replaceField(fmd.getAbsoluteFieldNumber(), new FeatureMapWrapper(sm, fieldName, new ArrayList<FeatureMap.Entry>(
					(List)value)));
		}
	}

	/**
	 * Is overridden from superclass to handle replacing of normal FeatureMapEntries by emf-jpox feature map entries.
	 */
	@SuppressWarnings("unchecked")
	public void postInsert(StateManager sm) {
		Collection value = (Collection) sm.provideField(fmd.getAbsoluteFieldNumber());

		// here replace the values
		// first get the owner!
		final ArrayList list;
		if (value != null) {
			EObject owner = (EObject) sm.getObject();

			Class elementClass;
			// todo optimise this with a cache
			final EStructuralFeature feature = StoreUtil.getEStructuralFeature(owner, fmd.getName());
			if (StoreUtil.isWildCard(feature) || StoreUtil.isMixed(feature)) {
				elementClass = AnyFeatureMapEntry.class;
			} else {
				elementClass = GenericFeatureMapEntry.class; // FeatureMapEntry.getEntryClass(owner.getClass(),
				// fieldName);
			}
			list = replaceEntryAll(elementClass, value);

			// now persist any new values
			if (elementClass == AnyFeatureMapEntry.class) {
				final Iterator it = list.iterator();
				while (it.hasNext()) {
					((AnyFeatureMapEntry) it.next()).persistValue(sm.getPersistenceManager());
				}
			}
		} else {
			list = new ArrayList();
		}

		getBackingStore(sm.getPersistenceManager().getClassLoaderResolver()).addAll(sm, list);

		sm.replaceField(fmd.getAbsoluteFieldNumber(), new FeatureMapWrapper(sm, fieldName, list));
	}

	/**
	 * Gets an 'normal' FeatureMap.Entry and if it is not a FeatureMapEntry replaces it with a specific implementation.
	 */
	private FeatureMapEntry replaceEntry(Class<?> elementClass, Object obj) {
		// always replace
		if (true || !(obj instanceof FeatureMapEntry)) {
			try {
				FeatureMapEntry entry = (FeatureMapEntry) elementClass.newInstance();
				entry.setEntry((FeatureMap.Entry) obj);
				return entry;
			} catch (Exception e) {
				throw new JpoxStoreException(
						"Exception while instantiating for elementClass " + elementClass.getName(), e);
			}
		} else {
			return (FeatureMapEntry) obj;
		}
	}

	/**
	 * Replaces for a collection
	 */
	@SuppressWarnings("unchecked")
	private ArrayList replaceEntryAll(Class elementClass, Collection c) {
		final ArrayList newEntries = new ArrayList();
		final Iterator it = c.iterator();
		while (it.hasNext()) {
			newEntries.add(replaceEntry(elementClass, it.next()));
		}
		return newEntries;
	}

	/**
	 * Method to delete all dependent objects (the contents of the Collection).
	 * 
	 * @param sm
	 *            State Manager of the owner
	 */
	public void deleteDependent(StateManager sm) {
		doDelete(sm);
	}

	/**
	 * Method to be called before any delete of the owner class element, if the field in the owner is dependent
	 * 
	 * @param sm
	 *            StateManager of the owner
	 */
	@SuppressWarnings("unchecked")
	public void doDelete(StateManager sm) {
		Collection value = (Collection) sm.provideField(fmd.getAbsoluteFieldNumber());
		final CollectionStore cs = getBackingStore(sm.getPersistenceManager().getClassLoaderResolver());
		Iterator it;
		if (value == null) {
			it = cs.iterator(sm);
		} else {
			it = value.iterator();
		}

		final ArrayList toRemove = new ArrayList();
		while (it.hasNext()) {
			final FeatureMapEntry entry = (FeatureMapEntry) it.next();
			if (entry.getEStructuralFeature() instanceof EReference) {
				final EReference eref = (EReference) entry.getEStructuralFeature();
				if (eref.isContainment() && entry.getValue() instanceof PersistenceCapable) {
					toRemove.add(entry.getValue());
				}
			}
		}
		cs.clear(sm);
		sm.getPersistenceManager().deletePersistentAll(toRemove);
		sm.getPersistenceManager().flush();
	}
}