/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package distributions.math;

import org.apache.commons.math3.distribution.IntegerDistribution;
import org.apache.commons.math3.distribution.RealDistribution;
import org.apache.commons.math3.exception.MathIllegalNumberException;

public interface ModeDistribution {
    /**
     * The ‘mode’ is the value that occurs most often.
     *
     * @return the mode, if applicable
     * @throws MathIllegalNumberException if the mode cannot be determined for the distributiuon
     */
    double getMode() throws ModeNotSupportedException;

    /**
     * Returns the default value to use for scheduling for this distribution.<br>
     * The default value can either be specified by the end user or it typically defaults to either the 'mode' or 'mean'
     * of this distribution.
     */
    double getDefault();

    /**
     * Use this method to get the numerical value of the mean of this distribution.
     *
     * @return the mean or {@code Double.NaN} if it is not defined
     */
    double getNumericalMean();

    /**
     * Access the lower bound of the support.<br>
     * See {@link RealDistribution}.getSupportLowerBound()
     * or {@link IntegerDistribution}.getSupportLowerBound()
     */
    double getLowerBound();

    /**
     * Access the upper bound of the support.<br>
     * See {@link RealDistribution}.getSupportUpperBound()
     * or {@link IntegerDistribution}.getSupportUpperBound()
     */
    double getUpperBound();

    /**
     * Generate a random value sampled from this distribution.
     *
     * @return a random value.
     */
    double nextSample();

    /**
     * Generate a random sample from the distribution.
     *
     * @param sampleSize the number of random values to generate
     * @return an array representing the random sample
     * @throws org.apache.commons.math3.exception.NotStrictlyPositiveException if {@code sampleSize} is not positive
     */
    double[] nextSamples(int sampleSize);

    /**
     * Returns the probability density function (PDF)
     * See {@link RealDistribution}.density()
     */
    double density(double x);
}
