package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NetworkCreation}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNetworkCreation.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNetworkCreation implements NetworkCreation {
  private final String id;
  private final @Nullable String warnings;

  private ImmutableNetworkCreation(String id, @Nullable String warnings) {
    this.id = id;
    this.warnings = warnings;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("Id")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code warnings} attribute
   */
  @JsonProperty("Warnings")
  @Override
  public @Nullable String warnings() {
    return warnings;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkCreation#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNetworkCreation withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableNetworkCreation(newValue, this.warnings);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkCreation#warnings() warnings} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for warnings (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNetworkCreation withWarnings(@Nullable String value) {
    if (Objects.equals(this.warnings, value)) return this;
    return new ImmutableNetworkCreation(this.id, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNetworkCreation} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNetworkCreation
        && equalTo(0, (ImmutableNetworkCreation) another);
  }

  private boolean equalTo(int synthetic, ImmutableNetworkCreation another) {
    return id.equals(another.id)
        && Objects.equals(warnings, another.warnings);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code warnings}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + Objects.hashCode(warnings);
    return h;
  }

  /**
   * Prints the immutable value {@code NetworkCreation} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NetworkCreation{"
        + "id=" + id
        + ", warnings=" + warnings
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NetworkCreation} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NetworkCreation instance
   */
  public static ImmutableNetworkCreation copyOf(NetworkCreation instance) {
    if (instance instanceof ImmutableNetworkCreation) {
      return (ImmutableNetworkCreation) instance;
    }
    return ImmutableNetworkCreation.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNetworkCreation ImmutableNetworkCreation}.
   * <pre>
   * ImmutableNetworkCreation.builder()
   *    .id(String) // required {@link NetworkCreation#id() id}
   *    .warnings(String | null) // nullable {@link NetworkCreation#warnings() warnings}
   *    .build();
   * </pre>
   * @return A new ImmutableNetworkCreation builder
   */
  public static ImmutableNetworkCreation.Builder builder() {
    return new ImmutableNetworkCreation.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNetworkCreation ImmutableNetworkCreation}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private long initBits = 0x1L;

    private String id;
    private String warnings;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NetworkCreation} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NetworkCreation instance) {
      Objects.requireNonNull(instance, "instance");
      this.id(instance.id());
      @Nullable String warningsValue = instance.warnings();
      if (warningsValue != null) {
        warnings(warningsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkCreation#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Id")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkCreation#warnings() warnings} attribute.
     * @param warnings The value for warnings (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Warnings")
    public final Builder warnings(@Nullable String warnings) {
      this.warnings = warnings;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNetworkCreation ImmutableNetworkCreation}.
     * @return An immutable instance of NetworkCreation
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNetworkCreation build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNetworkCreation(id, warnings);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      return "Cannot build NetworkCreation, some of required attributes are not set " + attributes;
    }
  }
}
