//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.varorder.hyperedges;

import static org.eclipse.escet.common.java.Sets.set;

import java.util.List;
import java.util.Set;

import org.eclipse.escet.cif.common.CifEquationUtils;
import org.eclipse.escet.cif.metamodel.cif.declarations.AlgVariable;
import org.eclipse.escet.cif.metamodel.cif.expressions.AlgVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.java.CifWithArgWalker;

/**
 * Comparison binary expression collector.
 *
 * <p>
 * Each referenced algebraic variable is only considered once.
 * </p>
 */
class ComparisonCollector extends CifWithArgWalker<Set<BinaryExpression>> {
    /** The already considered objects that don't have to be considered again. */
    private final Set<AlgVariable> alreadyConsidered = set();

    /**
     * Collect and return the comparison binary expressions in the given expression.
     *
     * @param expr The expression in which to collect, recursively.
     * @param comparisons The comparisons collected so far. Is extended in-place.
     * @return The collected comparisons, i.e., {@code comparisons}.
     */
    Set<BinaryExpression> collectComparisons(Expression expr, Set<BinaryExpression> comparisons) {
        walkExpression(expr, comparisons);
        return comparisons;
    }

    @Override
    protected void preprocessBinaryExpression(BinaryExpression expr, Set<BinaryExpression> comparisons) {
        switch (expr.getOperator()) {
            case EQUAL:
            case UNEQUAL:
            case LESS_EQUAL:
            case LESS_THAN:
            case GREATER_EQUAL:
            case GREATER_THAN:
                comparisons.add(expr);
                break;
            default:
                break;
        }
    }

    @Override
    protected void preprocessAlgVariableExpression(AlgVariableExpression expr, Set<BinaryExpression> comparisons) {
        // Skip already processed algebraic variables.
        AlgVariable algVar = expr.getVariable();
        if (alreadyConsidered.contains(algVar)) {
            return;
        }
        alreadyConsidered.add(algVar);

        // Get the possible values of the variable.
        List<Expression> values = CifEquationUtils.getValuesForAlgVar(algVar, false);

        // Collect for each possible value.
        for (Expression value: values) {
            walkExpression(value, comparisons);
        }
    }
}
