//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.spec;

import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.Arrays;

import org.eclipse.escet.common.java.Assert;

import com.github.javabdd.BDDFactory;
import com.github.javabdd.BDDVarSet;

/**
 * A CIF/BDD domain. It represents a collection of BDD variables that together can represent a value of a
 * {@link CifBddVariable}. There can be multiple domains for a {@link CifBddVariable}, for instance to represent its
 * {@link CifBddVariable#domain current value} or its {@link CifBddVariable#domainNew value after an assignment}.
 */
public class CifBddDomain {
    /** The BDD factory of this domain. */
    private final BDDFactory factory;

    /** The index of this BDD domain in the domains of the {@link CifBddSpec CIF/BDD specification}. */
    private final int index;

    /** The BDD variable indices of the BDD variables that make up this domain. Is not empty. */
    private final int[] vars;

    /** The BDD variable set for the domain. Is {@code null} until lazily created in {@link #makeVarSet}. */
    private BDDVarSet varSet;

    /**
     * Constructor for the {@link CifBddDomain} class. Should only be invoked by {@link CifBddSpec#createDomains}.
     *
     * @param factory The BDD factory of this domain.
     * @param index The index of this BDD domain in the domains of the {@link CifBddSpec CIF/BDD specification}.
     * @param vars The BDD variable indices of the BDD variables that make up this domain. Must not be empty.
     */
    CifBddDomain(BDDFactory factory, int index, int[] vars) {
        Assert.notNull(factory);
        Assert.check(index >= 0);
        Assert.notNull(vars);
        Assert.check(vars.length > 0);
        Assert.check(Arrays.stream(vars).allMatch(v -> v >= 0));

        this.factory = factory;
        this.index = index;
        this.vars = vars;
    }

    /**
     * Returns the BDD factory of this domain.
     *
     * @return The BDD factory of this domain.
     */
    public BDDFactory getFactory() {
        return factory;
    }

    /**
     * Returns the index of this BDD domain in the domains of the {@link CifBddSpec CIF/BDD specification}.
     *
     * @return The index of this BDD domain.
     */
    public int getIndex() {
        return index;
    }

    /**
     * Returns the number of BDD variables in this domain.
     *
     * @return The number of BDD variables in this domain.
     */
    public int getVarCount() {
        return vars.length;
    }

    /**
     * Returns the BDD variable indices of the BDD variables that make up this domain.
     *
     * @return The BDD variable indices. The array must not be modified by the caller.
     */
    public int[] getVarIndices() {
        return vars;
    }

    /**
     * Returns a new BDD variable set for this domain. Each invocation returns a new instance.
     *
     * @return The BDD variable set.
     */
    public BDDVarSet makeVarSet() {
        // If no varset has been created yet, create one and cache it.
        if (varSet == null) {
            varSet = factory.makeSet(vars);
        }

        // Return unique copies of the varset, to prevent callers from freeing our cached one.
        return varSet.id();
    }

    /** Frees the CIF/BDD domain, releasing its BDD-related resources. It should not be used anymore afterwards. */
    public void free() {
        Arrays.fill(vars, -1); // Prevent against accidental use after freeing by setting invalid variable indices.
        if (varSet != null) {
            varSet.free();
        }
    }

    @Override
    public String toString() {
        return fmt("Domain #%d: vars=%s", index, Arrays.toString(vars));
    }
}
