/**
 * Copyright (c) 2018 Kiel University and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.elk.alg.packing.rectangles.options;

import java.util.EnumSet;
import org.eclipse.elk.alg.packing.rectangles.util.PackingStrategy;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

@SuppressWarnings("all")
public class RectPackingMetaDataProvider implements ILayoutMetaDataProvider {
  /**
   * Default value for {@link #STRATEGY}.
   */
  private static final PackingStrategy STRATEGY_DEFAULT = PackingStrategy.MAX_SCALE_DRIVEN;
  
  /**
   * Strategy for approximation of the bounding box given by the first iteration. Determines whether layout is
   * sorted by scale measure, aspect ratio or area.
   */
  public static final IProperty<PackingStrategy> STRATEGY = new Property<PackingStrategy>(
            "org.eclipse.elk.rectpacking.strategy",
            STRATEGY_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #LAST_PLACE_SHIFT}.
   */
  private static final boolean LAST_PLACE_SHIFT_DEFAULT = true;
  
  /**
   * When placing a rectangle behind or below the last placed rectangle in the first iteration, it is sometimes
   * possible to shift the rectangle further to the left or right, resulting in less whitespace. True (default)
   * enables the shift and false disables it. Disabling the shift produces a greater approximated area by the first
   * iteration and a layout, when using ONLY the first iteration (default not the case), where it is sometimes
   * impossible to implement a size transformation of rectangles that will fill the bounding box and eliminate
   * empty spaces.
   */
  public static final IProperty<Boolean> LAST_PLACE_SHIFT = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.lastPlaceShift",
            LAST_PLACE_SHIFT_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #ONLY_FIRST_ITERATION}.
   */
  private static final boolean ONLY_FIRST_ITERATION_DEFAULT = false;
  
  /**
   * Determines whether only the first iteration should be executed or not. If set to true, only the first iteration
   * will be executed. When using only the first iteration, rectangles will not be resized to fill their bounding
   * box and eliminate unused whitespace.
   */
  public static final IProperty<Boolean> ONLY_FIRST_ITERATION = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.onlyFirstIteration",
            ONLY_FIRST_ITERATION_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #CHECK_FOR_SPECIAL_CASE}.
   */
  private static final boolean CHECK_FOR_SPECIAL_CASE_DEFAULT = false;
  
  /**
   * Determines whether the algorithm should check for the case that one big rectangle is present with a lot of
   * smaller rectangles of the same size. If the check is true, a special algorithm for this case is used.
   */
  public static final IProperty<Boolean> CHECK_FOR_SPECIAL_CASE = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.checkForSpecialCase",
            CHECK_FOR_SPECIAL_CASE_DEFAULT,
            null,
            null);
  
  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.strategy")
        .group("")
        .name("Packing Strategy")
        .description("Strategy for approximation of the bounding box given by the first iteration. Determines whether layout is sorted by scale measure, aspect ratio or area.")
        .defaultValue(STRATEGY_DEFAULT)
        .type(LayoutOptionData.Type.ENUM)
        .optionClass(PackingStrategy.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.ADVANCED)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.lastPlaceShift")
        .group("")
        .name("Shift when placing at last placed rectangle.")
        .description("When placing a rectangle behind or below the last placed rectangle in the first iteration, it is sometimes possible to shift the rectangle further to the left or right, resulting in less whitespace. True (default) enables the shift and false disables it. Disabling the shift produces a greater approximated area by the first iteration and a layout, when using ONLY the first iteration (default not the case), where it is sometimes impossible to implement a size transformation of rectangles that will fill the bounding box and eliminate empty spaces.")
        .defaultValue(LAST_PLACE_SHIFT_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.ADVANCED)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.onlyFirstIteration")
        .group("")
        .name("Only first iteration should be executed.")
        .description("Determines whether only the first iteration should be executed or not. If set to true, only the first iteration will be executed. When using only the first iteration, rectangles will not be resized to fill their bounding box and eliminate unused whitespace.")
        .defaultValue(ONLY_FIRST_ITERATION_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.HIDDEN)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.checkForSpecialCase")
        .group("")
        .name("Check for special case one big rectangle.")
        .description("Determines whether the algorithm should check for the case that one big rectangle is present with a lot of smaller rectangles of the same size. If the check is true, a special algorithm for this case is used.")
        .defaultValue(CHECK_FOR_SPECIAL_CASE_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.HIDDEN)
        .create()
    );
    new org.eclipse.elk.alg.packing.rectangles.options.RectPackingOptions().apply(registry);
  }
}
