/**
 ********************************************************************************
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 *******************************************************************************/

package org.eclipse.app4mc.amalthea.visualization.hw;

import org.eclipse.app4mc.amalthea.model.HWModel;
import org.eclipse.app4mc.amalthea.model.HwStructure;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;

public class DiagramLocation {

	private IFolder diagramFolder = null;
	private String plantUmlFileName = null;
	private String plantUmlFilePath = null;
	private String diagramFileName = null;
	private String diagramFilePath = null;

	public DiagramLocation(EObject eObject) {
		IFile file = getResourceFile(eObject);
		if (file == null) return;

		IFolder folder = getOrCreateDiagramFolder(file, "hwModelVisualization");
		if (folder == null) return;

		initialize(folder, buildFilename(file, eObject));
	}

	public DiagramLocation(IFolder folder, String baseFilename) {
		if (folder == null || !folder.exists() || baseFilename == null || baseFilename.isEmpty()) return;

		initialize(folder, baseFilename);
	}

	private void initialize(IFolder folder, String baseFilename) {
		diagramFolder = folder;
		plantUmlFileName = baseFilename + ".plantuml";
		plantUmlFilePath = folder.getFile(plantUmlFileName).getLocation().toString();
		diagramFileName = baseFilename + ".svg";
		diagramFilePath = folder.getFile(diagramFileName).getLocation().toString();
	}


	private IFile getResourceFile(EObject eObject) {
		if (eObject == null) return null;

		Resource eResource = eObject.eResource();
		if (eResource == null) return null;

		URI uri = eResource.getURI();
		IPath path = new Path(uri.toPlatformString(true));
		return ResourcesPlugin.getWorkspace().getRoot().getFile(path);
	}

	private IFolder getOrCreateDiagramFolder(IFile file, String folderName) {
		if (file == null) return null;

		IContainer parentFolder = file.getParent();
		if (parentFolder == null) return null;

		// **** Find visualization folder (create if necessary)

		IFolder folder = parentFolder.getFolder(new Path(folderName));

		if (!folder.exists()) {
			try {
				folder.create(IResource.NONE, true, null);
			} catch (CoreException e) {
				return null;
			}
		}

		return folder;
	}

	private String buildFilename(IFile file, EObject eObject) {
		String modelName = "model"; // default

		if (file != null && file.getName() != null) {
			modelName = file.getName().replaceFirst("[.][^.]+$", ""); // remove extension
		}

		String extendedName = "";
		if (eObject instanceof HWModel) {
			extendedName = "_hw";
		} else if (eObject instanceof HwStructure) {
			extendedName = "_structure";
		}

		return modelName + extendedName;
	}

	public IFolder getDiagramFolder() {
		return diagramFolder;
	}

	public String getPlantUmlFileName() {
		return plantUmlFileName;
	}

	public String getPlantUmlFilePath() {
		return plantUmlFilePath;
	}

	public String getDiagramFileName() {
		return diagramFileName;
	}

	public String getDiagramFilePath() {
		return diagramFilePath;
	}

	public boolean isValid() {
		return diagramFolder != null;
	}
}

