/**
 * Copyright (c) 2021 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch - initial API and implementation
 */

package org.eclipse.app4mc.amalthea.model.predefined;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

@SuppressWarnings("null")
public class StandardSchedulers {

	// Suppress default constructor
	private StandardSchedulers() {
		throw new IllegalStateException("Utility class");
	}

	public enum Algorithm {
		// *** ADD PREDEFINED SCHEDULER DEFINITIONS HERE ***
		GROUPING_SERVER( // only for migration
				"GroupingServer",
				"This is not a scheduler algorithm. Schedulers using this definition\n"
				+ "act as a logical grouping of tasks/child-schedulers, e.g. a partition\n"
				+ "for some tasks for budget accounting reasons.\n\n"
				+ "This scheduler does not take any scheduling decisions,\n"
				+ "and a parent scheduler is mandatory.\n\n"
				+ "Algorithm parameters\n"
				+ " - capacity [1] Time\n"
				+ "      The fixed budget that can be used by processes.\n"
				+ "      It will be replenished periodically.\n"
				+ " - period [1] Time\n"
				+ "      Amount of time after which the capacity will be replenished.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " - passes parameters upwards\n"
				+ " - requires parent scheduler\n",
				new Parameter[] { Parameter.CAPACITY, Parameter.PERIOD },
				new Parameter[] {},
				false, true, true),
		PRIORITY_BASED(
				"PriorityBased",
				"???\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - priority [1] Integer\n"
				+ "      The priority of the process (a higher value means a higher priority).\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.PRIORITY },
				false, false, false),

		OSEK(
				"OSEK",
				"OSEK compliant Scheduling. A fixed priority preemptive scheduling algorithm\n"
				+ "with task groups. Tasks belonging to the same task group are scheduled\n"
				+ "cooperatively (they do not preempt each other), preemptive otherwise.\n"
				+ "Tasks with the same priority also behave cooperatively.\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - priority [1] Integer\n"
				+ "      The priority of the process (a higher value means a higher priority).\n"
				+ " - taskGroup [1] Integer\n"
				+ "      The OSEK task group number (if for two processes the number is equal,\n"
				+ "      that means they are in the same task group).\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.PRIORITY, Parameter.TASK_GROUP },
				false, false, false),
		FIXED_PRIORITY_PREEMPTIVE(
				"FixedPriorityPreemptive",
				"Fixed Priority Preemptive Scheduling (e.g. AUTOSAR),\n"
				+ "same as OSEK but without task groups.\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - priority [1] Integer\n"
				+ "      The priority of the process (a higher value means a higher priority).\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.PRIORITY },
				false, false, false),
		FIXED_PRIORITY_PREEMPTIVE_WITH_BUDGET_ENFORCEMENT( // only for migration
				"FixedPriorityPreemptiveWithBudgetEnforcement",
				"Works like the Fixed Priority Preemptive Scheduling. But it is possible\n"
				+ "to put budget boundaries on the execution. Prevents low priority tasks \n"
				+ "from starving if a higher priority task is trying to constantly occupy the\n"
				+ "CPU (safety insurance, bounding the execution time of sporadic loads).\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - priority [1] Integer\n"
				+ "      The priority of the process (a higher value means a higher priority).\n"
				+ " - minBudget [1] Time\n"
				+ "      The guaranteed amount of budget available to the process within the replenishment period.\n"
				+ " - maxBudget [1] Time\n"
				+ "      The upper bound of the budget available to the process within the replenishment period.\n"
				+ "      Sets a limit to the usage of a process even if the CPU would be idle\n"
				+ "      (important for algorithms with payback mechanisms).\n"
				+ " - replenishment [1] Time\n"
				+ "      The periodic time interval after which the budget is set to the configured value.\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.PRIORITY, Parameter.MIN_BUDGET, Parameter.MAX_BUDGET, Parameter.REPLENISHMENT },
				false, false, false),
		DEADLINE_MONOTONIC( // only for migration
				"DeadlineMonotonic",
				"This is not a scheduling algorithm, it only describes how to derive priorities for a\n"
				+ "fixed priority scheduler: Task with the shortest deadline gets the highest priority.\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - deadline [1] Time\n"
				+ "      The time after each activation at which the process must finish.\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.DEADLINE },
				false, false, false),
		RATE_MONOTONIC( // only for migration
				"RateMonotonic",
				"This is not a scheduling algorithm, it only describes how to derive priorities for a\n"
				+ "fixed priority scheduler: Task with the shortest period gets the highest priority.\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - period [1] Integer\n"
				+ "      The time span after the previous activation at which\n"
				+ "      the next instance of the process shall be activated.\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.PERIOD },
				false, false, false),

		EARLIEST_DEADLINE_FIRST(
				"EarliestDeadlineFirst",
				"Earliest Deadline First (EDF): The task with the closest deadline in relation\n"
				+ "to the current point in time will be scheduled next.\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - deadline [1] Time\n"
				+ "      The time after each activation at which the process must finish.\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - First introduced in: Liu, Chung Laung, and James W. Layland.\n"
				+ "   \"Scheduling algorithms for multiprogramming in a hard-real-time environment.\"\n"
				+ "   Journal of the ACM (JACM) 20.1 (1973): 46-61.",
				new Parameter[] {},
				new Parameter[] { Parameter.DEADLINE },
				false, false, false),
		LEAST_LOCAL_REMAINING_EXECUTION_TIME_FIRST( // only for migration
				"LeastLocalRemainingExecutionTimeFirst",
				"Least Local Remaining Execution-time First (LLREF): Task with the\n"
				+ "smallest local remaining execution time will be scheduled next.\n\n"
				+ "Algorithm parameters\n"
				+ " -\n\n"
				+ "Process parameters\n"
				+ " - executionTime [1] Time\n"
				+ "      The time which the process will use until it is finished\n"
				+ "      (usually the worst case execution time is used here in order\n"
				+ "      to guarantee that the process can finish).\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] {},
				new Parameter[] { Parameter.EXECUTION_TIME },
				false, false, false),
		PRIORITY_BASED_ROUND_ROBIN(
				"PriorityBasedRoundRobin",
				"Round Robin scheduling algorithm assigns equally sized time slices\n"
				+ "to each process that it schedules. The priority describes the order\n"
				+ "in which the processes will be executed. If two processes have the\n"
				+ "same priority, the order of these two is random (non-deterministic).\n\n"
				+ "Algorithm parameters\n"
				+ " - timeSliceLength [1] Time\n"
				+ "      Length of each time slice.\n\n"
				+ "Process parameters\n"
				+ " - priority [1] Integer\n"
				+ "      The priority of the process (a higher value means a higher priority).\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.TIME_SLICE_LENGTH },
				new Parameter[] { Parameter.PRIORITY },
				false, false, false),

		P_FAIR_PD2(
				"PFairPD2",
				"Proportionate Fair PD2 Scheduling (Pfair-PD2).\n\n"
				+ "Algorithm parameters\n"
				+ " - quantSize [0..1] Time = 1ns\n"
				+ "      Length of the minimum schedulable time slot used in Pfair scheduling.\n"
				+ "      It is assumed that execution times are an integer multiple of this\n"
				+ "      time slot length.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.QUANT_SIZE },
				new Parameter[] {},
				false, false, false),
		PARTLY_P_FAIR_PD2(
				"PartlyPFairPD2",
				"Partly Proportionate Fair PD2 Scheduling (PPfair-PD2).\n\n"
				+ "Algorithm parameters\n"
				+ " - quantSize [0..1] Time = 1ns\n"
				+ "      Length of the minimum schedulable time slot used in Pfair scheduling.\n"
				+ "      It is assumed that execution times are an integer multiple of this\n"
				+ "      time slot length.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.QUANT_SIZE },
				new Parameter[] {},
				false, false, false),
		EARLY_RELEASE_FAIR_PD2(
				"EarlyReleaseFairPD2",
				"Early Release Fair PD2 Scheduling (ERfair-PD2).\n\n"
				+ "Algorithm parameters\n"
				+ " - quantSize [0..1] Time = 1ns\n"
				+ "      Length of the minimum schedulable time slot used in Pfair scheduling.\n"
				+ "      It is assumed that execution times are an integer multiple of this\n"
				+ "      time slot length.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.QUANT_SIZE },
				new Parameter[] {},
				false, false, false),
		PARTLY_EARLY_RELEASE_FAIR_PD2(
				"PartlyEarlyReleaseFairPD2",
				"Partly Early Release Fair PD2 Scheduling (P-ERfair-PD2).\n\n"
				+ "Algorithm parameters\n"
				+ " - quantSize [0..1] Time = 1ns\n"
				+ "      Length of the minimum schedulable time slot used in Pfair scheduling.\n"
				+ "      It is assumed that execution times are an integer multiple of this\n"
				+ "      time slot length.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " -\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.QUANT_SIZE },
				new Parameter[] {},
				false, false, false),

		DEFERRABLE_SERVER(
				"DeferrableServer",
				"Deferrable Server (DS): provides a fixed budget,\n"
				+ "in which the budget replenishment is done periodically.\n\n"
				+ "Algorithm parameters\n"
				+ " - capacity [1] Time\n"
				+ "      The fixed budget that can be used by processes.\n"
				+ "      It will be replenished periodically.\n"
				+ " - period [1] Time\n"
				+ "      Amount of time after which the capacity will be replenished.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " - has exactly one child\n"
				+ " - requires parent scheduler\n\n"
				+ "Reference:\n"
				+ " - First introduced in: Strosnider, Jay K., John P. Lehoczky, and Lui Sha.\n"
				+ "   \"The deferrable server algorithm for enhanced aperiodic responsiveness in hard real-time environments.\"\n"
				+ "   IEEE Transactions on Computers 44.1 (1995): 73-91.",
				new Parameter[] { Parameter.CAPACITY, Parameter.PERIOD },
				new Parameter[] {},
				true, false, true),
		POLLING_SERVER(
				"PollingServer",
				"Polling Server (PS): provides a fixed budget periodically that is only\n"
				+ "available at pre-defined times. If the process is not using the budget\n"
				+ "at that point in time the budget is lost.\n\n"
				+ "Algorithm parameters\n"
				+ " - capacity [1] Time\n"
				+ "      The fixed budget that can be used by processes (usually directly\n"
				+ "      after it has been replenished). The capacity will be consumed even\n"
				+ "      if there is no process using it. It will be replenished periodically.\n"
				+ " - period [1] Time\n"
				+ "      Amount of time after which the capacity will be replenished.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " - has exactly one child\n"
				+ " - requires parent scheduler\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.CAPACITY, Parameter.PERIOD },
				new Parameter[] {},
				true, false, true),
		SPORADIC_SERVER(
				"SporadicServer",
				"Sporadic Server (SS): provides a fixed budget, in which the budget replenishment\n"
				+ "is performed with a pre-defined replenishment delay after it was consumed.\n\n"
				+ "Algorithm parameters\n"
				+ " - capacity [1] Time\n"
				+ "      The fixed budget that can be used by processes. It will be replenished after\n"
				+ "      the specified amount of time has passed since it has last been consumed.\n"
				+ " - replenishmentDelay [1] Time\n"
				+ "      Amount of time after which the capacity will be replenished\n"
				+ "      after it has last been consumed.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " - has exactly one child\n"
				+ " - requires parent scheduler\n\n"
				+ "Reference:\n"
				+ " - First introduced in: Sprunt, Brinkley, Lui Sha, and John Lehoczky.\n"
				+ "   \"Aperiodic task scheduling for hard-real-time systems.\"\n"
				+ "   Real-Time Systems 1.1 (1989): 27-60.",
				new Parameter[] { Parameter.CAPACITY, Parameter.REPLENISHMENT_DELAY },
				new Parameter[] {},
				true, false, true),
		CONSTANT_BANDWIDTH_SERVER(
				"ConstantBandwidthServer",
				"Constant Bandwidth Server (CBS): provides a fixed utilization for\n"
				+ "executing jobs, in which the deadline for execution is independent\n"
				+ "on the execution time of jobs.\n\n"
				+ "Algorithm parameters\n"
				+ " - capacity [1] Time\n"
				+ "      The fixed budget that can be used by processes.\n"
				+ "      It will be replenished periodically.\n"
				+ " - period [1] Time\n"
				+ "      Amount of time after which the capacity will be replenished.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " - has exactly one child\n"
				+ " - requires parent scheduler\n\n"
				+ "Reference:\n"
				+ " - First introduced in: Abeni, Luca, and Giorgio Buttazzo.\n"
				+ "   \"Integrating multimedia applications in hard real-time systems.\"\n"
				+ "   Proceedings 19th IEEE Real-Time Systems Symposium (Cat. No. 98CB36279). IEEE, 1998.",
				new Parameter[] { Parameter.CAPACITY, Parameter.PERIOD },
				new Parameter[] {},
				true, false, true),
		CONSTANT_BANDWIDTH_SERVER_WITH_CAPACITY_SHARING(
				"ConstantBandwidthServerWithCapacitySharing",
				"Constant Bandwidth Server (CBS) with capacity sharing (CASH).\n"
				+ "Consumes residual slack from other servers (work conserving).\n\n"
				+ "Algorithm parameters\n"
				+ " - capacity [1] Time\n"
				+ "      The fixed budget that can be used by processes.\n"
				+ "      It will be replenished periodically.\n"
				+ " - period [1] Time\n"
				+ "      Amount of time after which the capacity will be replenished.\n\n"
				+ "Process parameters\n"
				+ " -\n\n"
				+ "Options:\n"
				+ " - has exactly one child\n"
				+ " - requires parent scheduler\n\n"
				+ "Reference:\n"
				+ " - TODO",
				new Parameter[] { Parameter.CAPACITY, Parameter.PERIOD },
				new Parameter[] {},
				true, false, true);

		private final String algorithmName;
		private final String description;
		private final List<Parameter> algorithmParameters;
		private final List<Parameter> processParameters;
		private final boolean hasExactlyOneChild;
		private final boolean passesParametersUpwards;
		private final boolean requiresParentScheduler;

		// private enum constructor
		private Algorithm(String algorithmName, String description,
				Parameter[] algorithmParameters, Parameter[] processParameters,
				boolean hasExactlyOneChild, boolean passesParametersUpwards, boolean requiresParentScheduler) {
			this.algorithmName = algorithmName;
			this.description = description;
			this.algorithmParameters = Arrays.asList(algorithmParameters);
			this.processParameters = Arrays.asList(processParameters);
			this.hasExactlyOneChild = hasExactlyOneChild;
			this.passesParametersUpwards = passesParametersUpwards;
			this.requiresParentScheduler = requiresParentScheduler;
		}

		public String getAlgorithmName() {
			return algorithmName;
		}

		public String getDescription() {
			return description;
		}

		public List<Parameter> getAlgorithmParameters() {
			return algorithmParameters;
		}

		public List<String> getAlgorithmParameterNames() {
			return algorithmParameters.stream().map(Parameter::getParameterName).collect(Collectors.toList());
		}

		public List<Parameter> getProcessParameters() {
			return processParameters;
		}

		public List<String> getProcessParameterNames() {
			return processParameters.stream().map(Parameter::getParameterName).collect(Collectors.toList());
		}

		public boolean hasExactlyOneChild() {
			return hasExactlyOneChild;
		}

		public boolean passesParametersUpwards() {
			return passesParametersUpwards;
		}

		public boolean requiresParentScheduler() {
			return requiresParentScheduler;
		}
	}

	public enum Parameter {
		// *** ADD PREDEFINED PARAMETER DEFINITIONS HERE ***
		PRIORITY("priority", Type.INTEGER, false, true),
		TASK_GROUP("taskGroup", Type.INTEGER, false, true),
		MIN_BUDGET("minBudget", Type.TIME, false, true),
		MAX_BUDGET("maxBudget", Type.TIME, false, true),
		REPLENISHMENT("replenishment", Type.TIME, false, true),
		DEADLINE("deadline", Type.TIME, false, true),
		PERIOD("period", Type.TIME, false, true),
		EXECUTION_TIME("executionTime", Type.TIME, false, true),
		QUANT_SIZE("quantSize", Type.TIME, false, false, "1 ns"),
		TIME_SLICE_LENGTH("timeSliceLength", Type.TIME, false, true),
		CAPACITY("capacity", Type.TIME, false, true),
		REPLENISHMENT_DELAY("replenishmentDelay", Type.TIME, false, true);

		private final String parameterName;
		private final Type type;
		private final boolean many;
		private final boolean mandatory;
		private final String defaultValue;

		// private enum constructors
		private Parameter(String parameterName, Type type, boolean many, boolean mandatory) {
			this(parameterName, type, many, mandatory, null);
		}

		private Parameter(String parameterName, Type type, boolean many, boolean mandatory, String defaultValue) {
			this.parameterName = parameterName;
			this.type = type;
			this.many = many;
			this.mandatory = mandatory;
			this.defaultValue = defaultValue;
		}

		public String getParameterName() {
			return parameterName;
		}

		public Type getType() {
			return type;
		}

		public boolean isMany() {
			return many;
		}

		public boolean isMandatory() {
			return mandatory;
		}

		public String getDefaultValue() {
			return defaultValue;
		}
	}

	public enum Type {
		INTEGER("Integer"),
		FLOAT("Float"),
		BOOL("Bool"),
		TIME("Time"),
		STRING("String");

		private final String typeName;

		// private enum constructor
		private Type(String typeName) {
			this.typeName = typeName;
		}

		public String getTypeName() {
			return typeName;
		}
	}

	private static final Map<String, Parameter> PARAMETERS;
	private static final Map<String, Algorithm> ALGORITHMS;

	static {
		PARAMETERS = Arrays.stream(Parameter.values())
				.collect(Collectors.toMap(Parameter::getParameterName, Function.identity()));
		ALGORITHMS = Arrays.stream(Algorithm.values())
				.collect(Collectors.toMap(Algorithm::getAlgorithmName, Function.identity()));
	}

	public static Algorithm getAlgorithm(String algorithmName) {
		return ALGORITHMS.get(algorithmName);
	}

	public static Parameter getParameter(String parameterName) {
		return PARAMETERS.get(parameterName);
	}

	public static List<Parameter> getAllParametersOfAlgorithm(String algorithmName) {
		List<Parameter> result = new ArrayList<>();

		Algorithm algo = ALGORITHMS.get(algorithmName);
		if (algo != null) {
			result.addAll(algo.getAlgorithmParameters());
			result.addAll(algo.getProcessParameters());
		}

		return result;
	}

	public static List<Parameter> getAlgorithmParametersOfAlgorithm(String algorithmName) {
		List<Parameter> result = new ArrayList<>();

		Algorithm algo = ALGORITHMS.get(algorithmName);
		if (algo != null) {
			result.addAll(algo.getAlgorithmParameters());
		}

		return result;
	}

	public static List<Parameter> getProcessParametersOfAlgorithm(String algorithmName) {
		List<Parameter> result = new ArrayList<>();

		Algorithm algo = ALGORITHMS.get(algorithmName);
		if (algo != null) {
			result.addAll(algo.getProcessParameters());
		}

		return result;
	}

}
