/**
 ********************************************************************************
 * Copyright (c) 2018-2020 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.model.provider

import java.util.Collection
import java.util.Collections
import java.util.List
import org.eclipse.app4mc.amalthea.model.AmaltheaPackage
import org.eclipse.app4mc.amalthea.model.AmaltheaServices
import org.eclipse.app4mc.amalthea.model.CallArgument
import org.eclipse.app4mc.amalthea.model.Component
import org.eclipse.app4mc.amalthea.model.ComponentInstance
import org.eclipse.app4mc.amalthea.model.ComponentPort
import org.eclipse.app4mc.amalthea.model.Composite
import org.eclipse.app4mc.amalthea.model.DataDependency
import org.eclipse.app4mc.amalthea.model.DirectionType
import org.eclipse.app4mc.amalthea.model.EnumMode
import org.eclipse.app4mc.amalthea.model.HwAccessPath
import org.eclipse.app4mc.amalthea.model.HwFeatureCategory
import org.eclipse.app4mc.amalthea.model.ISystem
import org.eclipse.app4mc.amalthea.model.InterfaceKind
import org.eclipse.app4mc.amalthea.model.ModeLabel
import org.eclipse.app4mc.amalthea.model.ModeLabelAccess
import org.eclipse.app4mc.amalthea.model.ModeValue
import org.eclipse.app4mc.amalthea.model.NumericMode
import org.eclipse.app4mc.amalthea.model.Process
import org.eclipse.app4mc.amalthea.model.QualifiedPort
import org.eclipse.app4mc.amalthea.model.Runnable
import org.eclipse.app4mc.amalthea.model.RunnableCall
import org.eclipse.app4mc.amalthea.model.RunnableParameter
import org.eclipse.app4mc.amalthea.model.util.SoftwareUtil
import org.eclipse.emf.common.util.BasicEList
import org.eclipse.emf.common.util.UniqueEList
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.edit.provider.ItemPropertyDescriptor

class CustomPropertyDescriptorService {

	/*****************************************************************************
	 * 						NeedEntry Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getNeedEntryValuesForKey(Object object) {
		val choiceOfValues = new UniqueEList<Object>();

		// get HW feature categories
		val type = AmaltheaPackage.eINSTANCE.getHwFeatureCategory()
		val objects = ItemPropertyDescriptor.getReachableObjectsOfType(object as EObject, type)
		
		// empty entry
		choiceOfValues.add(null)

		// entries: names of feature categories
		choiceOfValues.addAll(
			objects
			.map[obj | (obj as HwFeatureCategory).name]
			.filterNull
			.sort);

		return choiceOfValues
	}

	/*****************************************************************************
	 * 						CallArgument Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getCallArgumentValuesForParameter(Object object) {
		if (object instanceof CallArgument) {
			val choiceOfValues = new BasicEList<RunnableParameter>();
			
			// empty entry
			choiceOfValues.add(null)
			
			// parameters of called runnable
			val params = object.containingCall?.runnable?.parameters
			choiceOfValues.addAll(params ?: emptyList )
			
			return choiceOfValues
		}
		return Collections.EMPTY_LIST
	}


	/*****************************************************************************
	 * 						DataDependency Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getDataDependencyValuesForParameters(Object object) {
		if (object instanceof DataDependency) {
			val choiceOfValues = new BasicEList<RunnableParameter>();
			
			// Parameters (in, inout) of containing runnable
			val runnable = AmaltheaServices.getContainerOfType(object, Runnable)
			if (runnable !== null) {
				choiceOfValues.addAll(
					runnable.parameters
					.filter[e | e.direction == DirectionType::IN || e.direction == DirectionType::INOUT]
				)
			}
			return choiceOfValues
		}
		return Collections.EMPTY_LIST
	}

	def static Collection<?> getDataDependencyValuesForCallArguments(Object object) {
		if (object instanceof DataDependency) {
			val choiceOfValues = new BasicEList<CallArgument>();
			
			// CallArguments (out, inout) of calls in the same graph
			val runnable = AmaltheaServices.getContainerOfType(object, Runnable)
			val process = AmaltheaServices.getContainerOfType(object, Process)
			val activityGraph = runnable?.activityGraph ?: process?.activityGraph
			
			if (activityGraph !== null) {
				choiceOfValues.addAll(
					SoftwareUtil.collectActivityGraphItems(activityGraph, null, [e | e instanceof RunnableCall])
					.map[e | (e as RunnableCall).arguments]
					.flatten
					.filter[e | e.parameter?.direction == DirectionType::OUT || e.parameter?.direction == DirectionType::INOUT]
				)
			}
			return choiceOfValues
		}
		return Collections.EMPTY_LIST
	}

	/*****************************************************************************
	 * 						ModeValue Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getValuesForModeValue(Object object) {
		if (object instanceof ModeValue) {
			val choiceOfValues = new BasicEList<String>();
			
			// Identify mode and possible values
			val mode = object.label?.mode
			if (mode instanceof NumericMode) return null // standard editor
			
			if (mode instanceof EnumMode) {
				choiceOfValues.addAll(mode.literals.map[e | e.name])
			}
			
			return choiceOfValues
		}
		return Collections.EMPTY_LIST
	}

	/*****************************************************************************
	 * 						ModeLabel Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getInitialValuesForModeLabel(Object object) {
		if (object instanceof ModeLabel) {
			val choiceOfValues = new BasicEList<String>();
			
			// Identify mode and possible values
			val mode = object.mode
			if (mode instanceof NumericMode) return null // standard editor
			
			if (mode instanceof EnumMode) {
				choiceOfValues.addAll(mode.literals.map[e | e.name])
			}
			
			return choiceOfValues
		}
		return Collections.EMPTY_LIST
	}

	/*****************************************************************************
	 * 						ModeLabelAccess Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getValuesForModeLabelAccess(Object object) {
		if (object instanceof ModeLabelAccess) {
			val choiceOfValues = new BasicEList<String>();
			
			// Identify mode and possible values
			val mode = object.data?.mode
			if (mode instanceof NumericMode) return null // standard editor
			
			if (mode instanceof EnumMode) {
				choiceOfValues.addAll(mode.literals.map[e | e.name])
			}
			
			return choiceOfValues
		}
		return Collections.EMPTY_LIST
	}

	/*****************************************************************************
	 * 						HwAccessPath Property Descriptors
	 *****************************************************************************/

	def static Collection<?> filterValuesForPathElements(Object object, Collection<?> choices) {
		if (object instanceof HwAccessPath) {

			// Get the source (processing unit) of the access path and remove it from choices
			val source = object.containingAccessElement?.source

			choices.remove(source)
		}
		return choices
	}

	/*****************************************************************************
	 * 						QualifiedPort Property Descriptors
	 *****************************************************************************/

	def static Collection<?> getValuesForComponentInstance(Object object) {
		val choices = new BasicEList<ComponentInstance>();

		if (object instanceof QualifiedPort) {

			// Get container (System or Composite)
			val container = AmaltheaServices.getContainerOfType(object, ISystem)

			if (container !== null) {
				choices.add(null)	// represents the containing component
				choices.addAll(container.componentInstances)				
			}
		}
		return choices
	}

	def static Collection<?> getValuesForComponentPort(Object object) {
		val choices = new BasicEList<ComponentPort>();

		if (object instanceof QualifiedPort) {
			val component = getComponent(object)
			if (component !== null) {
				choices.addAll(filterPortList(component.ports, object))
				choices.add(0, null)
			}
		}

		return choices
	}
	
	def private static Component getComponent(QualifiedPort qualPort) {
		if (qualPort.instance === null) {
			// Get container (Composite only, System has no ports)
			return AmaltheaServices.getContainerOfType(qualPort, Composite)
		} else {
			// Get type of instance (Component)
			return qualPort.instance.type
		}

	}

	def private static List<ComponentPort> filterPortList(List<ComponentPort> ports, QualifiedPort qualPort) {
		val isInnerPort = (qualPort.instance !== null)
		val provideList = #[InterfaceKind.PROVIDES, InterfaceKind.PROVIDES_REQUIRES, InterfaceKind._UNDEFINED_]
		val requireList = #[InterfaceKind.REQUIRES, InterfaceKind.PROVIDES_REQUIRES, InterfaceKind._UNDEFINED_]

		switch qualPort.eContainingFeature {
			case AmaltheaPackage.eINSTANCE.connector_SourcePort:
				if (isInnerPort) {
					return ports.filter[e | provideList.contains(e.kind)].toList
				} else {
					return ports.filter[e | requireList.contains(e.kind)].toList
				}
			case AmaltheaPackage.eINSTANCE.connector_TargetPort:
				if (isInnerPort) {
					return ports.filter[e | requireList.contains(e.kind)].toList
				} else {
					return ports.filter[e | provideList.contains(e.kind)].toList
				}
			case AmaltheaPackage.eINSTANCE.ISystem_GroundedPorts:
				if (isInnerPort) {
					return ports
				} else {
					return emptyList
				}
			default:
				return emptyList
		}
	}

}
