/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.util;

import com.google.auto.value.AutoValue;
import java.io.Serializable;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Objects;
import org.apache.beam.sdk.annotations.Experimental;
import org.apache.beam.sdk.util.AutoValue_HistogramData_LinearBuckets;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.math.DoubleMath;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Experimental
public class HistogramData
implements Serializable {
    private static final Logger LOG = LoggerFactory.getLogger(HistogramData.class);
    private final BucketType bucketType;
    private long[] buckets;
    private long numBoundedBucketRecords;
    private long numTopRecords;
    private long numBottomRecords;

    public HistogramData(BucketType bucketType) {
        this.bucketType = bucketType;
        this.buckets = new long[bucketType.getNumBuckets()];
        this.numBoundedBucketRecords = 0L;
        this.numTopRecords = 0L;
        this.numBottomRecords = 0L;
    }

    public BucketType getBucketType() {
        return this.bucketType;
    }

    public static HistogramData linear(double start, double width, int numBuckets) {
        return new HistogramData(LinearBuckets.of(start, width, numBuckets));
    }

    public void record(double ... values) {
        for (double value : values) {
            this.record(value);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void update(HistogramData other) {
        HistogramData histogramData = other;
        synchronized (histogramData) {
            if (!this.bucketType.equals(other.bucketType) || this.buckets.length != other.buckets.length) {
                LOG.warn("Failed to update HistogramData from another with a different buckets");
                return;
            }
            this.incTopBucketCount(other.numTopRecords);
            this.incBottomBucketCount(other.numBottomRecords);
            for (int i = 0; i < other.buckets.length; ++i) {
                this.incBucketCount(i, other.buckets[i]);
            }
        }
    }

    public synchronized void incBucketCount(int bucketIndex, long count) {
        int n = bucketIndex;
        this.buckets[n] = this.buckets[n] + count;
        this.numBoundedBucketRecords += count;
    }

    public synchronized void incTopBucketCount(long count) {
        this.numTopRecords += count;
    }

    public synchronized void incBottomBucketCount(long count) {
        this.numBottomRecords += count;
    }

    public synchronized void clear() {
        this.buckets = new long[this.bucketType.getNumBuckets()];
        this.numBoundedBucketRecords = 0L;
        this.numTopRecords = 0L;
        this.numBottomRecords = 0L;
    }

    public synchronized void record(double value) {
        double rangeTo = this.bucketType.getRangeTo();
        double rangeFrom = this.bucketType.getRangeFrom();
        if (value >= rangeTo) {
            ++this.numTopRecords;
        } else if (value < rangeFrom) {
            ++this.numBottomRecords;
        } else {
            int n = this.bucketType.getBucketIndex(value);
            this.buckets[n] = this.buckets[n] + 1L;
            ++this.numBoundedBucketRecords;
        }
    }

    public synchronized long getTotalCount() {
        return this.numBoundedBucketRecords + this.numTopRecords + this.numBottomRecords;
    }

    public synchronized String getPercentileString(String elemType, String unit) {
        return String.format("Total number of %s: %s, P99: %.0f %s, P90: %.0f %s, P50: %.0f %s", elemType, this.getTotalCount(), this.p99(), unit, this.p90(), unit, this.p50(), unit);
    }

    public synchronized long getCount(int bucketIndex) {
        return this.buckets[bucketIndex];
    }

    public synchronized long getTopBucketCount() {
        return this.numTopRecords;
    }

    public synchronized long getBottomBucketCount() {
        return this.numBottomRecords;
    }

    public double p99() {
        return this.getLinearInterpolation(0.99);
    }

    public double p90() {
        return this.getLinearInterpolation(0.9);
    }

    public double p50() {
        return this.getLinearInterpolation(0.5);
    }

    private synchronized double getLinearInterpolation(double percentile) {
        int index;
        long totalNumOfRecords = this.getTotalCount();
        if (totalNumOfRecords == 0L) {
            return Double.NaN;
        }
        double recordSum = this.numBottomRecords;
        if (recordSum / (double)totalNumOfRecords >= percentile) {
            return Double.NEGATIVE_INFINITY;
        }
        for (index = 0; index < this.bucketType.getNumBuckets() && !((recordSum += (double)this.buckets[index]) / (double)totalNumOfRecords >= percentile); ++index) {
        }
        if (index == this.bucketType.getNumBuckets()) {
            return Double.POSITIVE_INFINITY;
        }
        double fracPercentile = percentile - (recordSum - (double)this.buckets[index]) / (double)totalNumOfRecords;
        double bucketPercentile = (double)this.buckets[index] / (double)totalNumOfRecords;
        double fracBucketSize = fracPercentile * this.bucketType.getBucketSize(index) / bucketPercentile;
        return this.bucketType.getRangeFrom() + this.bucketType.getAccumulatedBucketSize(index) + fracBucketSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized boolean equals(@Nullable Object object) {
        if (object instanceof HistogramData) {
            HistogramData other;
            HistogramData histogramData = other = (HistogramData)object;
            synchronized (histogramData) {
                return Objects.equals(this.bucketType, other.bucketType) && this.numBoundedBucketRecords == other.numBoundedBucketRecords && this.numTopRecords == other.numTopRecords && this.numBottomRecords == other.numBottomRecords && Arrays.equals(this.buckets, other.buckets);
            }
        }
        return false;
    }

    public synchronized int hashCode() {
        return Objects.hash(this.bucketType, this.numBoundedBucketRecords, this.numBottomRecords, this.numTopRecords, Arrays.hashCode(this.buckets));
    }

    @AutoValue
    public static abstract class LinearBuckets
    implements BucketType {
        public abstract double getStart();

        public abstract double getWidth();

        @Override
        public abstract int getNumBuckets();

        public static LinearBuckets of(double start, double width, int numBuckets) {
            if (width <= 0.0) {
                throw new IllegalArgumentException(String.format("width should be greater than zero: %f", width));
            }
            if (numBuckets <= 0) {
                throw new IllegalArgumentException(String.format("numBuckets should be greater than zero: %d", numBuckets));
            }
            return new AutoValue_HistogramData_LinearBuckets(start, width, numBuckets);
        }

        @Override
        public int getBucketIndex(double value) {
            return DoubleMath.roundToInt((double)((value - this.getStart()) / this.getWidth()), (RoundingMode)RoundingMode.FLOOR);
        }

        @Override
        public double getBucketSize(int index) {
            return this.getWidth();
        }

        @Override
        public double getAccumulatedBucketSize(int endIndex) {
            return this.getWidth() * (double)endIndex;
        }

        @Override
        public double getRangeFrom() {
            return this.getStart();
        }

        @Override
        public double getRangeTo() {
            return this.getStart() + (double)this.getNumBuckets() * this.getWidth();
        }
    }

    public static interface BucketType
    extends Serializable {
        public double getRangeFrom();

        public double getRangeTo();

        public int getNumBuckets();

        public int getBucketIndex(double var1);

        public double getBucketSize(int var1);

        public double getAccumulatedBucketSize(int var1);
    }
}

