/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: DOMPrefixResolver.java,v $
 * Version:       $Name:  $ $Revision: 1.3 $
 * Last Modified: $Date: 2004/06/21 10:24:23 $
 *****************************************************************************/
package org.ten60.netkernel.xml.xda;

import org.apache.xml.utils.PrefixResolver;
import java.util.*;
import org.w3c.dom.*;

/**
 * A PrefixResolver that works with with namespaces and is cacheable
 * @author  pjr
 */
public class DOMPrefixResolver implements PrefixResolver {
    
    private HashMap mHashMap;
    private String mBaseIdentifier;
    private boolean mHasNamespaces;
    
    public static final String XML_PREFIX="xml";
    public static final String XML_PREFIX_NAMESPACE="http://www.w3.org/XML/1998/namespace";
    
    
    /** Creates a new instance of WritablePrefixResolver */
    public DOMPrefixResolver(Document aDocument)
    {	this();
	setNamespaceforPrefix(XML_PREFIX,XML_PREFIX_NAMESPACE);
	buildPrefixResolver(aDocument);
	mHasNamespaces=true;
    }
    
    /** Creates a new instance of DOMPrefixResolver */
    public DOMPrefixResolver()
    {	mHashMap=new HashMap(4);
	mHashMap.put("", new MapItem("", "", null)); //Set default namespace
	
    }
    /**Find all xmlns namespace declarations and build a prefix resolver for xpath evaluation*/
    private void buildPrefixResolver(Node node)
    {	DOMPrefixResolver wpr = this;
	if(node.getNodeType()==node.ELEMENT_NODE) {
	    NamedNodeMap nnm=node.getAttributes();
	    if(nnm!=null)
	    {	for(int i=0;i<nnm.getLength();i++)
		{	Node n=nnm.item(i);
			String name=n.getNodeName();
			//System.err.println(name+": "+n.getNamespaceURI());
			if (name.startsWith("xmlns:"))
			{	String prefix=name.replaceAll("xmlns:", "");
				wpr.setNamespaceforPrefix(prefix,n.getNodeValue(), n);
			}
			else if (name.startsWith("xmlns")) {	// default NS
			    wpr.setNamespaceforPrefix("",n.getNodeValue(), n);
			}
		}
	    }
	}
	// recurse to all elements in document
	for (Node n=node.getFirstChild(); n!=null; n=n.getNextSibling())
	{   if (n instanceof Element)
	    {   buildPrefixResolver(n);
	    }
	}
    }
    
    /** Return the base identifier.
     *
     * @return The base identifier from where relative URIs should be absolutized, or null
     * if the base ID is unknown.
     * <p>
     * CAVEAT: Note that the base URI in an XML document may vary with where
     * you are in the document, if part of the doc's contents were brought in
     * via an external entity reference or if mechanisms such as xml:base have
     * been used. Unless this PrefixResolver is bound to a specific portion of
     * the document, or has been kept up to date via some other mechanism, it
     * may not accurately reflect that context information.
     *
     */
    public String getBaseIdentifier() {
	return mBaseIdentifier;
    }
    
    /** Given a namespace, get the corrisponding prefix.  This assumes that
     * the PrefixResolver holds its own namespace context, or is a namespace
     * context itself.
     *
     * @param prefix The prefix to look up, which may be an empty string ("") for the default Namespace.
     *
     * @return The associated Namespace URI, or null if the prefix
     *         is undeclared in this context.
     *
     */
    public String getNamespaceForPrefix(String aPrefix) {
	MapItem mi=(MapItem)mHashMap.get(aPrefix);
	if(mi!=null) return mi.getURI();
	else return null;
    }
    
    /** Given a namespace, get the corresponding prefix, based on the context node.
     *
     * @param prefix The prefix to look up, which may be an empty string ("") for the default Namespace.
     * @param context The node context from which to look up the URI.
     *
     * @return The associated Namespace URI as a string, or null if the prefix
     *         is undeclared in this context.
     *
     */
    public String getNamespaceForPrefix(String prefix, org.w3c.dom.Node context)
    {	MapItem mi=(MapItem)mHashMap.get(prefix);
	if(mi!=null && context==mi.getContext()) return mi.getURI();
	else return null;
    }
    
    public boolean handlesNullPrefixes(){
	return false;
    }
    
    public void setBaseIdentifier(String aString)
    {	mBaseIdentifier=aString;
    }
    
    public void setNamespaceforPrefix(String aPrefix, String aUri)
    {	MapItem mi=new MapItem(aPrefix, aUri, null);
	mHashMap.put(aPrefix,mi);
    }
    
    public void setNamespaceforPrefix(String aPrefix, String aUri, org.w3c.dom.Node aContext)
    {	MapItem mi=new MapItem(aPrefix, aUri, aContext);
	mHashMap.put(aPrefix,mi);
    }
    
    public final boolean hasNamespaces()
    {   return mHasNamespaces;
    }
    
    /*
     * Get the Hashmap containing prefix namespace mappings. Key is prefix String, Object is
     * DOMPrefixResolver.MapItem which holds the prefix to namespace URI relationship.
     */
    public Map getPrefixMap()
    {	return mHashMap;
    }
    
    public class MapItem{
	
	private String mPrefix;
	private String mUri;
	private org.w3c.dom.Node mContext;
	
	public MapItem(String aPrefix, String aUri, org.w3c.dom.Node aContext) {
	    mPrefix=aPrefix;
	    mUri=aUri;
	    mContext=aContext;
	}
	
	public final String getPrefix(){ return mPrefix;}
	public final String getURI(){ return mUri;}
	public final org.w3c.dom.Node getContext(){ return mContext;}
    }
}