/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: JMSConnectionAspect.java,v $
 * Version:       $Name:  $ $Revision: 1.4 $
 * Last Modified: $Date: 2005/04/14 17:56:21 $
 *****************************************************************************/
package org.ten60.netkernel.jms;

import com.ten60.netkernel.urii.IURAspect;
import org.ten60.netkernel.xml.xda.*;
import org.ten60.netkernel.layer1.nkf.*;

import javax.naming.*;
import javax.jms.*;
import java.util.*;
/**
 * An aspect which represents an open connection to a set of queues and topics on a
 * JMS Message Broker
 * @author  tab
 */
public class JMSConnectionAspect implements IURAspect
{
	private IXDAReadOnly mConfiguration;
	private Context mContext;
	private QueueConnectionFactory mQueueConnectionFactory;
	private QueueConnection mQueueConnection;
	private TopicConnectionFactory mTopicConnectionFactory;
	private TopicConnection mTopicConnection;
	private List mNames;
	
	/** Creates a new instance of JMSConnectionAspect based on XML configuration */
	public JMSConnectionAspect(IXDAReadOnly aConfiguration) throws XPathLocationException, NamingException, JMSException
	{	mConfiguration = aConfiguration;
		initialise();
	}
	
	/** Internal method to initialise the connection to the JNDI server and the topic and queue connection factories
	 */
	private void initialise() throws XPathLocationException, NamingException, JMSException
	{	// initialise context
		Properties props = new Properties();
		for (IXDAReadOnlyIterator i=mConfiguration.readOnlyIterator("jndiContext/*"); i.hasNext(); )
		{	i.next();
			String propertyName = i.eval("name()").getStringValue();
			String propertyValue = i.getText(".", true);
			props.setProperty(propertyName,propertyValue);
		}
		mContext = new InitialContext(props);
		
		// initialise queue connection factory
		if (mConfiguration.isTrue("queueConnectionFactory"))
		{	mQueueConnectionFactory = (QueueConnectionFactory)mContext.lookup(mConfiguration.getText("queueConnectionFactory",true));
			mQueueConnection = mQueueConnectionFactory.createQueueConnection();
		}
		// initialise topic connection factory
		if (mConfiguration.isTrue("topicConnectionFactory"))
		{	mTopicConnectionFactory = (TopicConnectionFactory)mContext.lookup(mConfiguration.getText("topicConnectionFactory",true));
			mTopicConnection = mTopicConnectionFactory.createTopicConnection();
		}
	}

	/** Return the queue connection if one has been configured, otherwise null */
	public QueueConnection getQueueConnection() throws NKFException
	{	return mQueueConnection;
	}
	
	/** Return the topic connection if one has been configured, otherwise null */
	public TopicConnection getTopicConnection() throws NKFException
	{	return mTopicConnection;
	}
	
	/** Return the JNDI initial context configured */
	public Context getJNDIContext()
	{	return mContext;
	}
	
	/** Return the XML configuration for the given logical queue/topic name */
	public IXDAReadOnly getConfiguration(String aName) throws NKFException
	{	IXDAReadOnlyIterator result = null;
		try
		{	result = mConfiguration.readOnlyIterator("*[name='"+aName+"']");
			if (result.hasNext())
			{	result.next();
			}
			else
			{	result=null;
			}
		}
		catch (XPathLocationException e)
		{
		}
		if (result==null)
		{	throw new NKFException("Queue/Topic not registered",aName,null);
		}
		return result;
	}
	
	/** Return the list of logical queue/topic names registered in this configuration */
	public List getNames() throws XPathLocationException
	{	if (mNames==null)
		{	List result = new ArrayList(16);
			for (IXDAReadOnlyIterator i=mConfiguration.readOnlyIterator("queue|topic"); i.hasNext(); )
			{	i.next();
				String name=i.getText("name",true);
				result.add(name);
			}
			mNames = Collections.unmodifiableList(result);
		}
		return mNames;		
	}
	
	public String getURIForPhysicalDestination(String aDestination)
	{	String result=null;
		try
		{	IXDAReadOnlyIterator i=mConfiguration.readOnlyIterator("jndiContext/*[text()='"+aDestination+"']");
			if (i.hasNext())
			{	String key=i.eval("name()").getStringValue();
				int j=key.indexOf('.');
				String type=key.substring(0,j);
				String dest=key.substring(j+1);
				result="jms-"+type+":"+dest;
			}
		} catch (XPathLocationException e)
		{;}
		return result;
	}
	
	/** Close queue and topic connections when we are GCed */
	public void finalize()
	{	if (mQueueConnection!=null)
		{	try
			{	mQueueConnection.close();
			} catch (JMSException e) {;}
		}
		if (mTopicConnection!=null)
		{	try
			{	mTopicConnection.close();
			} catch (JMSException e) {;}
		}
	}
}