/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: JMSAccessor.java,v $
 * Version:       $Name:  $ $Revision: 1.5 $
 * Last Modified: $Date: 2005/04/14 17:56:21 $
 *****************************************************************************/
package org.ten60.netkernel.jms;

import com.ten60.netkernel.urii.*;
import com.ten60.netkernel.urii.aspect.*;
import org.ten60.netkernel.layer1.nkf.impl.NKFAccessorImpl;
import org.ten60.netkernel.layer1.nkf.*;
import org.ten60.netkernel.layer1.representation.*;
import org.ten60.netkernel.xml.xda.*;

import javax.naming.*;
import javax.jms.*;
import java.net.URI;
import java.io.*;
import java.util.*;

/**
 * Accessor to support sending to JMS queues and publishing to JMS topics
 * @author  tab
 */
public class JMSAccessor extends NKFAccessorImpl
{
	/** Creates a new instance of JMSAccessor */
	public JMSAccessor()
	{	super(4,true,INKFRequestReadOnly.RQT_SINK | INKFRequestReadOnly.RQT_SOURCE);
	}
	
	/** invoked from the kernel this method chooses which internal functionality to use
	 */
	public void processRequest(INKFConvenienceHelper context) throws Exception
	{	
		int requestType = context.getThisRequest().getRequestType();
		if (requestType==INKFRequestReadOnly.RQT_SINK)
		{	sink(context);
		}
		else if (requestType==INKFRequestReadOnly.RQT_SOURCE && "jms-send".equals(context.getThisRequest().getActiveType()))
		{	send(context);
		}
		else
		{	throw new NKFException("Unsupported Operation");
		}
	}

	/** Gets either the default /etc/JMSConfig.xml or a supplied config from the config argument
	 */
	private JMSConnectionAspect getConnection(INKFConvenienceHelper context) throws Exception
	{	String config="ffcpl:/etc/JMSConfig.xml";
		if (context.exists("this:param:config"))
		{	config = "this:param:config";
		}
		JMSConnectionAspect connection = (JMSConnectionAspect)context.sourceAspect(config,JMSConnectionAspect.class);			
		return connection;
	}
	
	/** Method to handle a sink request to a queue or topic URI
	 */
	private void sink(INKFConvenienceHelper aContext) throws Exception
	{	String name = aContext.getThisRequest().getActiveType();
		URI destination = URI.create(aContext.getThisRequest().getURI());
		if (name==null)
		{	name = destination.getSchemeSpecificPart();
		}
		String scheme = destination.getScheme();
		innerSend(scheme,name, INKFRequestReadOnly.URI_SYSTEM, aContext);
	}
	
	/** Method to handle an active:jms-send request
	 */
	private void send(INKFConvenienceHelper aContext) throws Exception
	{	JMSConnectionAspect connection = getConnection(aContext);
		String destinationString = aContext.getThisRequest().getArgument("destination");
		if (destinationString==null)
		{	throw new NKFException("Expected destination argument");
		}
		URI destination = URI.create(destinationString);
		String name = destination.getSchemeSpecificPart();
		String scheme = destination.getScheme();
		innerSend(scheme,name, "this:param:body", aContext);		
	}
	
	/** inner method to establish a JMS session and send/publish a message
	 */
	private void innerSend(String aScheme, String aDestination, String aBodyURI, INKFConvenienceHelper aContext) throws Exception
	{	JMSConnectionAspect connection = getConnection(aContext);
		IXDAReadOnly config = connection.getConfiguration(aDestination);
		
		if (aScheme.equals("jms-queue"))
		{	Queue queue = (Queue) connection.getJNDIContext().lookup(aDestination);
			QueueConnection qc = connection.getQueueConnection();
			if (qc==null) throw new Exception("No QueueConnectionFactory registered");
	        QueueSession queueSession = qc.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
			QueueSender queueSender = queueSession.createSender(queue);
			Message  message = createMessage(queueSession, aBodyURI, config, connection, aContext);
			queueSender.send(message);
		}
		if (aScheme.equals("jms-topic"))
		{	Topic topic = (Topic) connection.getJNDIContext().lookup(aDestination);
			TopicConnection tc = connection.getTopicConnection();
			if (tc==null) throw new Exception("No TopicConnectionFactory registered");
	        TopicSession topicSession = tc.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
			TopicPublisher topicPublisher = topicSession.createPublisher(topic);
			Message  message = createMessage(topicSession, aBodyURI, config, connection, aContext);
			topicPublisher.publish(message);
		}
	}
	
	/** inner method to create a JMS message based on the current request
	 */
	private Message createMessage(Session aSession, String aBodyURI, IXDAReadOnly aConfig, JMSConnectionAspect aConnection, INKFConvenienceHelper aContext) throws Exception
	{	Message message;
		
		// create body
		String messageType=aConfig.getText("messageType", true);
		if (messageType.equals("TextMessage"))
		{	IURRepresentation representation = aContext.source(aBodyURI,IAspectString.class);
			String text = ((IAspectString)representation.getAspect(IAspectString.class)).getString();
			TextMessage m = aSession.createTextMessage();	
			m.setText(text);
			m.setStringProperty("Content-Type", representation.getMeta().getMimeType());
			message=m;
		}
		else if (messageType.equals("BytesMessage"))
		{	IURRepresentation representation = aContext.source(aBodyURI,IAspectReadableBinaryStream.class);
			InputStream is = ((IAspectReadableBinaryStream)representation.getAspect(IAspectReadableBinaryStream.class)).getInputStream();
			BytesMessage m = aSession.createBytesMessage();
			pipe(is,m);
			m.setStringProperty("Content-Type", representation.getMeta().getMimeType());
			message=m;
		}
		else
		{	throw new NKFException("Unsupported message type",messageType,null);
		}
		
		//add properties
		if (aContext.exists("this:param:properties"))
		{	IAspectNVP properties = (IAspectNVP)aContext.sourceAspect("this:param:properties",IAspectNVP.class);
			for (Iterator i=properties.getNames().iterator(); i.hasNext(); )
			{	String propertyName = (String)i.next();
				String propertyValue = properties.getValue(propertyName);
				message.setStringProperty(propertyName, propertyValue);
			}
		}
		
		//set any headers
		if (aContext.exists("this:param:header"))
		{	IAspectNVP header = (IAspectNVP)aContext.sourceAspect("this:param:header",IAspectNVP.class);
			for (Iterator i=header.getNames().iterator(); i.hasNext(); )
			{	String hn = (String)i.next();
				String headerValue = header.getValue(hn);
				if (hn.equals("JMSCorrelationID"))
				{	message.setJMSCorrelationID(headerValue);
				}
				else if (hn.equals("JMSDeliveryMode"))
				{	int mode;
					if (headerValue.equals("persistent")) mode=DeliveryMode.PERSISTENT;
					else if (headerValue.equals("non-persistent")) mode=DeliveryMode.NON_PERSISTENT;
					else throw new NKFException("Unsupported JMSDeliveryMode","expected persistent or non-persistent",null);
					message.setJMSDeliveryMode(mode);
				}
				else if (hn.equals("JMSExpiration"))
				{	long expiration = Long.parseLong(headerValue);
					message.setJMSExpiration(expiration);
				}
				else if (hn.equals("JMSMessageID"))
				{	message.setJMSMessageID(headerValue);
				}
				else if (hn.equals("JMSPriority"))
				{	int priority = Integer.parseInt(headerValue);
					message.setJMSPriority(priority);
				}
				else if (hn.equals("JMSRedelivered"))
				{	boolean redelivered = Boolean.getBoolean(headerValue);
					message.setJMSRedelivered(redelivered);
				}
				else if (hn.equals("JMSTimestamp"))
				{	long timestamp = Long.parseLong(headerValue);
					message.setJMSTimestamp(timestamp);
				}
				else if (hn.equals("JMSType"))
				{	message.setJMSType(headerValue);
				}
				else if (hn.equals("JMSReplyTo"))
				{	Destination dest = (Destination)aConnection.getJNDIContext().lookup(headerValue);
					message.setJMSReplyTo(dest);
				}
				else
				{	throw new NKFException("Unsupported Header field",hn,null);
				}					
			}
		}
		
		return message;
	}
	
/** copy an input stream to a BytesMessage and close streams when finished
	 * @throws IOException if there are any problems
	 */
	public static void pipe(InputStream aInput, BytesMessage aOutput) throws IOException, JMSException
	{	byte b[] = new byte[256];
		int c;
		try
		{	while ( (c=aInput.read(b))>0 )
			{	aOutput.writeBytes(b,0,c);
			}
		}
		finally
		{	aInput.close();
		}
	}		
		
}