/*
 * put your module comment here
 * formatted with JxBeauty (c) johann.langhofer@nextra.at
 */


package  com.streamsicle.fluid;

import  java.net.*;
import  java.io.*;
import  java.text.*;
import  java.util.*;
//import  javax.servlet.ServletContext;
// Import log4j classes.
//import org.apache.log4j.Category;
import java.util.logging.*;

/**
 * This server will listen for TCP/IP client connections.
 * <P>
 * The InterCast server can handle "any" number of client
 * connections. Whenever a client connects a query will
 * be read before the client is handled by the superclasses.
 * The bandwidth usage is directly proportional to the
 * number of connected clients. Hence, two clients and a
 * media with a bitrate of 128kb/s requires a bandwidth
 * of 256kb/s.
 *
 * @author Lars Samuelsson
 * @author forked by John Watkinson
 */
public class Server
      implements Runnable, IMetaDataListener {
   private boolean alive;
   private ServerSocket clientConnection;
   private int port;
   private String ruleFile;
   private Streamer streamer;
   private Vector clientList = new Vector();
   private InteractiveStream in;
   private RedirectorStream urlin;
   private byte[] currentSongMetaData;
   private ClientValidator cv;
   private PlayTimer timer;
   private Properties props;
   //private ServletContext ctx;
   private URI mRoot;
   //private static Category log = Category.getInstance(Server.class);
   private static Logger log=java.util.logging.Logger.getAnonymousLogger();
   /**
    * Creates an InterCast server
    */
   public Server (URI aRoot, Properties properties) {
      props = properties;
      mRoot=aRoot;
      alive = true;
   }

   public void reconfigureProperties(Properties aprops)
   {	props=aprops;
   }
   
   public void initialize () throws StreamingEngineException {
      in = new InteractiveStream(props);
      String externalStream = props.getProperty("external.stream.url");
      Streamer streamer = null;
      if (externalStream != null) {
        // Server wants to stream from another server
        urlin = new RedirectorStream(props);
        streamer = new Streamer(urlin, this);
        urlin.configure();
      } else
        streamer = new Streamer(in, this);
      // Configure input stream
      try {
        in.configure();
      } catch (StreamingEngineException e) {
        throw e;
      }
      // configure the server
      configure();
      // Initialize ClientValidator
      cv = new ClientValidator(ruleFile);
      // Initialize and start play timer
      timer = new PlayTimer();
      timer.start();
      // Start streaming
      streamer.start();
   }

   /**
    * Configures an InterCast server and starts the
    * server thread.
    * <P>
    * The properties that are read from the configuration
    * are:
    * <P>
    * <UL>
    * <LI> <B> streasmicle.port </B><BR>
    *      The port on which the Server will
    *      listen for connections
    * <LI> <B> streamsicle.rules </B><BR>
    *      The name of the file containing the
    *      connection rules for clients
    * <LI> <B> streamsicle.history </B><BR>
    *      The name of the class to use as the play history implementation.
    *      Currently the only implementation is
    *      com.streamsicle.MemoryPlayHistory (which is also the default).
    * </UL>
    *
    * @param conf The configuration from which this
    *             server can read parameters
    */
   public void configure () {
      try {
         URI root=URI.create("file:/");
		 URI path=root.relativize(URI.create("file:"+props.getProperty("streamsicle.rules")));
		 ruleFile=mRoot.resolve(path).getPath();
		 
		 if (ruleFile == null)
            throw  new Exception("No connection rule file specified.");         /* This is now read in ClientResponse so it doesn't need to
          * be passed around.  also, audio/mpeg is assumed if empty
          mimeType = props.getProperty("streamsicle.mime");
          if (mimeType == null)
          throw new Exception("No mime type specified.");
          */

         String portString = props.getProperty("streamsicle.port");
         if (portString == null) {
            throw  new Exception("No port number specified.");
         }
         else {
            port = Integer.parseInt(portString);
         }
         clientConnection = new ServerSocket(port);
         Thread engine = new Thread(this, "Streamsicle Daemon");
		 engine.setPriority(Thread.NORM_PRIORITY+(Thread.MAX_PRIORITY-Thread.NORM_PRIORITY)/2);
         engine.start();
         // instatiate play history object
         String playHistoryClass = props.getProperty("streamsicle.history");
         Object playHistoryObj;
         try {
            playHistoryObj = Class.forName(playHistoryClass).newInstance();
         } catch (Exception e) {
            playHistoryObj = null;
         }
         if (playHistoryObj == null || !(playHistoryObj instanceof com.streamsicle.IPlayHistory)) {
            System.err.println("Problem creating play history, using MemoryPlayHistory");
            playHistoryObj = new com.streamsicle.MemoryPlayHistory();
         }
         in.setPlayHistory((com.streamsicle.IPlayHistory)playHistoryObj);
      } catch (Exception e) {
         //log.fatal("Server configuration failed", e);
		 log.severe("Server configuration failed"+ e.getMessage());
      }
   }

   public Vector getClientList () {
      return  clientList;
   }
   
   public synchronized int getClients()
   {	return clientList.size();
   }

   public int getPort () {
      return  port;
   }

   public synchronized void connected (Client c) {
      clientList.addElement(c);

      // send the meta data since this client would have missed when the
      // meta data was sent out at the beginning of the song
      c.setMetaData(currentSongMetaData);
   }

   public synchronized void disconnected (Client c) {
      clientList.removeElement(c);
   }

   public synchronized void setPacket (byte[] packet) {
      for (int i = 0; i < clientList.size(); i++) {
         ((Client)clientList.elementAt(i)).setPacket(packet);
      }
   }

   public synchronized void setMetaData (byte[] metaData) {
      currentSongMetaData = metaData;
      for (int i = 0; i < clientList.size(); i++) {
         ((Client)clientList.elementAt(i)).setMetaData(metaData);
      }
   }

   public InteractiveStream getStream () {
      return  in;
   }

   public boolean verifyClientByAddress(InetAddress address) {
    return cv.isValid(address);
   }

   public boolean welcomeClient (Client client) throws IOException {
      Socket socket = client.getSocket();
      if( !cv.isValid(socket.getInetAddress()) ) {
         return false;
      }
      int maxNumClients = Integer.parseInt(props.getProperty("streamsicle.maxclients"));
      if ( (clientList.size() >= maxNumClients) && (maxNumClients > 0) ) {
         return false;
      }

      ClientQuery query = new ClientQuery(socket.getInputStream());
      // log.info( "ClientQuery: " + query);
      client.setSupportsMetaData(false);
      int minutes = query.getTimer();
      //log.debug("Minutes: " + minutes);
	  log.finer("Minutes: " + minutes);
      if (minutes != -1) {
         long deadline = System.currentTimeMillis() + minutes * 60 * 1000;
         timer.addEvent(client, deadline);
      }
      String metaDataHeader = query.getProperty( "Icy-MetaData" );
      if( metaDataHeader != null && metaDataHeader.trim().equals( "1" ) ) {
         client.setSupportsMetaData(true);
      }
      ClientResponse response = new ClientResponse( query, props );
      OutputStream out = socket.getOutputStream();
      client.setOutputStream(out);
      response.respond( out );
      String clientName = query.getProperty("User-Agent");
      if( clientName == null ) {
         clientName = "Unknown";
      }
      log.info(cv + " - " + clientName);
      return true;
   }

   public void shutdown () {
	   System.out.println("server.shutdown()");
      // Shut down timer
      timer.shutdown();
      // Close the connection listening socket
      try {
         clientConnection.close();
      } catch (IOException e) {
         e.printStackTrace();
      }
      synchronized (this) {
         for (int i = 0; i < clientList.size(); i++) {
            Client c = (Client)clientList.elementAt(i);
            c.shutdown();
         }
         // Empty the client list.
         clientList = new Vector(0);
         alive = false;
      }
   }

   /**
    * Parallel execution starts here.
    *
    * This thread will accept incoming connections on port
    * given through the config file.
    */
   public void run () {
      log.info("Accepting connections on port " + port);
      while (alive) {
         try {
            // accept and do as little processing as possible
            Socket socket = clientConnection.accept();
			Client client = new Client(this, socket, ruleFile);
            if (welcomeClient(client)) {
               connected(client);
               socket.setTcpNoDelay(true);
               client.start();
            }
            else {
               log.warning("Client was denied.");
               socket.close();
            }
         } catch (IOException e) {
         // accept next client
         }
      }
   }
}
