package com.streamsicle.fluid;

import java.net.*;
import java.io.*;
import java.util.*;
import org.ten60.orchextra.*;

/**
 * Welcomes clients by reading a query and sending a response.
 * <P>
 * Many clients do some kinds of negotiating
 * with the server when they connect. Sadly I must say that
 * there therefore is a need for a class like this one.
 * One would have hoped that streaming protocols would have
 * been kept clean from negotiation procedures like
 * these. A pure client should only listen for frames
 * as these contain all the necessary information in
 * streamed media.
 *
 * @author Lars Samuelsson
 * @author forked by John Watkinson
 */
public class Client implements Runnable, IMetaDataListener {
   private Server server;
   private Socket socket;
   private String ruleFile;
   private OutputStream out;
   private InetAddress address;
   private boolean newPacket;
   private byte[] packet;
   private byte[] outPacket;
   Vector packets;

   private byte[] metaData;
   private int bytesSinceMetaData = 0;
   private boolean supportsMetaData;
   private boolean alive;

   
    /**
     * Creates a client handler for the specified mime type.
     *
     * @param ruleFile The name of the file containing the
     *                 connection rules for clients
     */
    public Client (Server server,
		   Socket socket,
		   String ruleFile ) {
   	this.server = server;
   	this.socket = socket;
   	this.ruleFile = ruleFile;
      alive = true;
   	address = socket.getInetAddress();
      packets = new Vector();
    }

    public void setSupportsMetaData(boolean supported) {
      supportsMetaData = supported;
    }

    public Socket getSocket() {
      return this.socket;
    }

    public void setOutputStream(OutputStream out) {
      this.out = out;
    }

    public void start () {
   	new Thread(this, "Client: " + address).start();
    }

    public synchronized void setPacket (byte[] packet) {
      packets.addElement(packet);
	   notify();
    }

    public void shutdown () {
      alive = false;
    }

    public void run () {
       try {
          while (alive) {
             synchronized (this) {
                try {
                   while (packets.size() == 0) {
                      wait();
                   }
                } catch (InterruptedException e) {}
                outPacket = (byte[])packets.elementAt(0);
                packets.removeElementAt(0);
             }
             sendData();
          }
       } catch (IOException e) {
          OrchextraAccessor.log(OrchextraAccessor.INFO,this, "Client disconnected.");
       } catch (Exception e) {
          OrchextraAccessor.log(OrchextraAccessor.WARNING, this, "We used to not catch this little number right here: "+e);
       } finally {
          try {
             socket.close();
          } catch (IOException enough) {
             // Oh, forget it.
			 OrchextraAccessor.log(OrchextraAccessor.WARNING, this, "There was an IO exception closing the thread, its probably not important but: " + enough);
          }
       }
       server.disconnected(this);
    }

    private void sendData() throws IOException {
       int packetSize = outPacket.length;

       if( supportsMetaData &&
             packetSize + bytesSinceMetaData > META_DATA_INTERVAL ) {
          int bytesLeft = META_DATA_INTERVAL - bytesSinceMetaData;
          out.write( outPacket, 0 /* offset */, bytesLeft /* length */ );
          int frameLeft = packetSize - bytesLeft;
          byte[] meta = getMetaData();
          if( meta != null ) {
             out.write( meta );
          } else {
             // if no change in meta data, then  send null byte
             out.write( '\0' );
          }
          while (frameLeft > META_DATA_INTERVAL) {
             out.write(outPacket, bytesLeft, META_DATA_INTERVAL);
             bytesLeft += META_DATA_INTERVAL;
             frameLeft = packetSize - bytesLeft;
             meta = getMetaData();
             if( meta != null ) {
                out.write( meta );
             } else {
                // if no change in meta data, then  send null byte
                out.write( '\0' );
             }
          }
          out.write( outPacket, bytesLeft /* offset */, frameLeft /* length */ );
          out.flush();
          bytesSinceMetaData = frameLeft;
       } else {
          out.write( outPacket );
          out.flush();
          bytesSinceMetaData += packetSize;
       }
    }

    public synchronized void setMetaData( byte[] metaData ) {
       if( supportsMetaData ) {
          this.metaData = metaData;
       }
    }

    protected synchronized byte[] getMetaData () {
      byte[] meta = metaData;
      // Clear it so it is only sent once
      metaData = null;
      return meta;
    }

    public InetAddress getAddress () {
	   return address;
    }

    public String toString() {
      return "" + address;
    }
}
