package com.streamsicle;

import java.io.*;
import java.util.*;
import com.streamsicle.util.Constants;

/**
 * Play history implementation that stores the information in memory.  Data
 * will be lost between server restarts.
 */
public class MemoryPlayHistory implements IPlayHistory {
   private static final int DEFAULT_HISTORY_SIZE = 20;

   private int m_historySize;
   private Vector m_playHistory;

   public MemoryPlayHistory() {
      this( DEFAULT_HISTORY_SIZE );
   }

   public MemoryPlayHistory( int playHistorySize ) {
      m_historySize = playHistorySize;
      m_playHistory = new Vector();
   }

   public void addSong( MP3File song ) {
      if( song != null ) {
	      synchronized( this ) {
	         m_playHistory.insertElementAt(song, 0);
	         // if should be sufficient here, but do while just in case
	         while( m_playHistory.size() > m_historySize ) {
	            m_playHistory.removeElementAt(m_playHistory.size()-1);
	         }
	      }
      } else {
        System.err.println( "Finished loading songs." );
      }
   }


   /**
    * Returns the play history.  The implementation may only return the most
    * recent songs (the exact number is up to the implementation).  Most
    * recent songs will be first in the list.
    */
   public Vector getPlayHistory() {
      return getPlayHistory( m_historySize );
   }

   /**
    * Returns the play history.  Will return a list with num entries, unless
    * the implementation does not have that many entries internally.
    */
   public Vector getPlayHistory( int num ) {
      Vector listClone = (Vector) m_playHistory.clone();

      while( listClone.size() > num ) {
	      listClone.removeElementAt(listClone.size()-1);
      }

      return listClone;
   }

   public void clearPlayHistory() {
       synchronized(this) {
           if (m_playHistory != null) {
               m_playHistory.clear();
           }
       }
   }

   /**
    * Returns true if the implementation persists the play history.  If an
    * implementation persists the play history then it will be consistent
    * accross server shutdowns and restarts.
    */
   public boolean isPersistent() {
      return false;
   }
}
