/******************************************************************************
 * (c) Copyright 2002,2005, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: DependencyCostCache.java,v $
 * Version:       $Name:  $ $Revision: 1.13 $
 * Last Modified: $Date: 2005/06/01 17:00:39 $
 *****************************************************************************/
package org.ten60.netkernel.cache;

import com.ten60.netkernel.cache.*;
import com.ten60.netkernel.urrequest.*;
import com.ten60.netkernel.urii.*;
import com.ten60.netkernel.util.SysLogger;
import com.ten60.netkernel.util.XMLUtils;
import com.ten60.netkernel.container.*;
import com.ten60.netkernel.module.*;

import org.ten60.netkernel.layer1.meta.*;
import org.ten60.netkernel.layer1.representation.*;

import java.net.*;
import java.io.*;
import java.util.*;
import org.w3c.dom.*;

/**
 * A Cache that uses a cost algorithm based on creation cost, LRU and usage count
 * it has a prune cycle wich is adaptive to available memory
 * @author  tab
 */
public class DependencyCostCache implements ICachelet
{
	private Map mCache;
	private Container mContainer;
	private ModuleDefinition mModule;
	private HouseKeeper mHK;
	private ICachelet mBackingCache;
	
	private int mPutsSinceLastReap;
	private long mLastValueCompute;
	private boolean mReaping;
	
	private int mCACHESIZE=400;
	private int mPRUNESIZE=80;
	private int mMEMORYTHRESHOLD=66;
	private int mPUTSTHRESHOLD=60;
	private int mVALUEPERIOD=5000;
	
	/** Creates a new instance of HitCountCache */
	public DependencyCostCache()
	{	mCache=new HashMap(2*mCACHESIZE);
	}
	
	public void init(Container aContainer, ModuleDefinition aModule)
	{	mContainer=aContainer;
		mModule=aModule;
		try{
			URL configURL=aModule.getResource("/etc/CacheDependencyCostConfig.xml");
			configURL.openConnection();
			InputStream configIS=configURL.openStream();
			Document config=XMLUtils.parse(new InputStreamReader(configIS));
			Element e=XMLUtils.getFirstChildElement(config.getDocumentElement());
			while(e!=null)
			{	String name=e.getTagName();
				if (name.equals("cacheSize"))
				{	mCACHESIZE=getValue(e);
				}
				else if (name.equals("pruneSize"))
				{	mPRUNESIZE=getValue(e);
				}
				else if (name.equals("memoryThreshold"))
				{	mMEMORYTHRESHOLD=getValue(e);
				}
				else if (name.equals("putsThreshold"))
				{	mPUTSTHRESHOLD=getValue(e);
				}
				else if (name.equals("valuePeriod"))
				{	mVALUEPERIOD=getValue(e);
				}
				else if (name.equals("backingCache"))
				{	String cacheClass = XMLUtils.getText(e);
					if (cacheClass.length()!=0)
					{	Class c = Class.forName(cacheClass);
						mBackingCache = (ICachelet)c.newInstance();
						mBackingCache.init(aContainer, aModule);
					}
				}
				e=XMLUtils.getNextSiblingElement(e);
			}
			if(SysLogger.shouldLog(SysLogger.CACHE, this))
			{	SysLogger.log(SysLogger.CACHE, this, "DependencyCostCache in "+mModule.getURI()+" configured: CACHESIZE="+mCACHESIZE+" PRUNESIZE="+mPRUNESIZE+" MEMORYTHRESHOLD="+mMEMORYTHRESHOLD+" PUTSTHRESHOLD="+mPUTSTHRESHOLD);
			}
		}
		catch(Exception e)
		{	SysLogger.log(SysLogger.WARNING, this, "Could not find ffcpl:/etc/ConfigDependencyCostCache.xml in module "+mModule.getURI()+". Using defaults");
		}
	}
	
	private int getValue(Element e)
	{	NodeList nl=e.getChildNodes();
		int result=-1;
		for(int i=0;i<nl.getLength();i++)
		{	Node n=nl.item(i);
			if (n.getNodeType()==Node.TEXT_NODE)
			{	result=Integer.parseInt(n.getNodeValue().trim());
				break;
			}
		}
		return result;
	}
	
	public IURRepresentation get(URRequest aRequest)
	{	IURRepresentation result=null;

		int type = aRequest.getType();
		if (type==URRequest.RQT_SOURCE || type==URRequest.RQT_TRANSREPRESENT)
		{	CacheKey key = new CacheKey(aRequest,true);
			CachedResource cr=null;
			synchronized(mCache)
			{	cr= (CachedResource)mCache.get(key);
			}
			if (cr!=null)
			{	if (cr.getResource().getMeta().isExpired())
				{	// cached result has expired
					result=Cache.EXPIRED_RESOURCE;
				}
				else
				{	cr.increment();
					result=cr.getResource();
				}
			}
		}
		// see if our backing cache has the result
		if (result==null && mBackingCache!=null)
		{	result = mBackingCache.get(aRequest);
		}
		return result;
	}
	
	public void put(URResult aResult)
	{	boolean put=false;
		
		while(shouldReap())
		{	reap(aResult.getRequest().getSession());
			mReaping=false;
		}

		switch (aResult.getRequest().getType())
		{	case URRequest.RQT_SOURCE:
			case URRequest.RQT_TRANSREPRESENT:
			{	IURRepresentation resource = aResult.getResource();
				IURMeta meta = resource.getMeta();
				if (!meta.isIntermediate() && !meta.isExpired() && !hasRequestExpired(aResult.getRequest()))
				{	CacheKey key = new CacheKey(aResult.getRequest(),meta.isContextSensitive());
					if (key!=null)
					{	CachedResource cr = new CachedResource(resource, key);
						cr.updateCacheIndex(System.currentTimeMillis());
						synchronized(mCache)
						{	mCache.put(key,cr);
							mPutsSinceLastReap++;
						}
						put=true;
					}
				}
				break;
			}
			case URRequest.RQT_SINK:
			case URRequest.RQT_DELETE:
			{	// invalidate cache entry
				CacheKey key = new CacheKey(aResult.getRequest(),true);
				Object removed;
				synchronized(mCache)
				{	removed = mCache.remove(key);
				}
				if (removed==null && mBackingCache!=null)
				{	mBackingCache.put(aResult);
				}
				break;
			}
			default:
				break;
		}
		
	}
	
	private static boolean hasRequestExpired(URRequest aRequest)
	{	return aRequest.getArgs().size()>0;
	}
	
	private synchronized boolean shouldReap()
	{	return mReaping?(false):(mPutsSinceLastReap>mPUTSTHRESHOLD && (mCache.size()>mCACHESIZE || getMemoryUse()>mMEMORYTHRESHOLD));
	}
	
	private int getMemoryUse()
	{	if(mHK==null)
		{	mHK=(HouseKeeper)mContainer.getComponent(HouseKeeper.URI);
		}
		long peak=mHK.getBaselineMemory();
		long max=mHK.getMaxMemory();
		long used=(100*peak)/max;
		return (int)used;
	}
	
	private void reap(IRequestorSession aSession)
	{	SysLogger.log(SysLogger.CACHE, this, "DependencyCostCache Prune size="+mCache.size());
		List removed = (mBackingCache!=null)?new ArrayList(mPRUNESIZE):null;
		//long t0 = System.currentTimeMillis();
		
		//check for expiry and build list of not expired
		ArrayList ts;
		long now = System.currentTimeMillis();
		
		synchronized(mCache)
		{	ts= new ArrayList(mCache.values());
		}
		if (now-mLastValueCompute>mVALUEPERIOD)
		{	for (Iterator i =ts.iterator(); i.hasNext(); )
			{	CachedResource cr=(CachedResource)i.next();
				cr.updateCacheIndex(now);
			}
			mLastValueCompute=now;
		}
		
		//long t1 = System.currentTimeMillis();
		
		// sort
		Collections.sort(ts, Collections.reverseOrder());
		
		//long t2 = System.currentTimeMillis();
			
		//prune
		boolean log = SysLogger.shouldLog(SysLogger.CACHE, this);
		int pruned=0;
		int cachesize=mCache.size();
		synchronized(mCache)
		{	for (Iterator i =ts.iterator(); i.hasNext() && (cachesize>mCACHESIZE || pruned<mPRUNESIZE); )
			{	CachedResource cr=(CachedResource)i.next();
				CacheKey ck=cr.getKey();
				mCache.remove(ck);
				if (removed!=null)
				{	removed.add(cr);
				}
				pruned++;
				cachesize--;
			}
		}
		//long t3 = System.currentTimeMillis();
			
		synchronized(mCache)
		{	//System.out.println("pruned="+pruned+" puts="+mPutsSinceLastReap+" size="+mCache.size());
			mPutsSinceLastReap=0;
		}

		// push anything to the backing cache
		if (mBackingCache!=null)
		{	for (Iterator i=removed.iterator(); i.hasNext(); )
			{	CachedResource cr = (CachedResource)i.next();
				CacheKey ck=cr.getKey();
				URRequest req = new URRequest(ck.getURI(), null, aSession,null, URRequest.RQT_SOURCE, null, null, null);
				req.setCurrentContext(ck.getModule(),ck.getSuper());
				URResult result = new URResult(req, cr.getResource());
				mBackingCache.put(result);
			}
		}
		
		//System.out.println((t1-t0)+" "+(t2-t1)+" "+" "+(t3-t2)+" "+(System.currentTimeMillis()-t3));
	}
	
	public void write(java.io.Writer aWriter) throws java.io.IOException
	{	synchronized(mCache)
		{	long now = System.currentTimeMillis();
			for (Iterator i = mCache.entrySet().iterator(); i.hasNext(); )
			{	aWriter.write("<item>");
				Map.Entry entry = (Map.Entry)i.next();
				CacheKey key = (CacheKey)entry.getKey();
				CachedResource cr=(CachedResource)entry.getValue();
				aWriter.write("<key>");
				aWriter.write(XMLUtils.escape(key.getURI().toString()));
				aWriter.write("</key>");
				aWriter.write("<module>");
				aWriter.write(XMLUtils.escape(key.getModule().toString()));
				aWriter.write("</module>");
				aWriter.write("<age>");
				aWriter.write(Long.toString(now-cr.lastTouched()));
				aWriter.write("</age>");
				aWriter.write("<cost>");
				aWriter.write(Integer.toString(cr.getResource().getMeta().getCreationCost()));
				aWriter.write("</cost>");
				aWriter.write("<hits>");
				aWriter.write(Integer.toString(cr.getCount()));
				aWriter.write("</hits>");
				aWriter.write("<index>");
				aWriter.write(Long.toString(cr.updateCacheIndex(now)));
				aWriter.write("</index>");
				aWriter.write("</item>");
			}
		}
	}
	
	public ICachelet getBackingCache()
	{	return mBackingCache;
	}
	
	static final class CachedResource implements Comparable, Comparator
	{	private int mCount=1;
		private IURRepresentation mResource;
		private long mTouched;
		private CacheKey mKey;
		private int mStaticIndex;
		
		public CachedResource(IURRepresentation aResource, CacheKey aKey)
		{	mResource=aResource;
			mKey=aKey;
			touch();
		}
		
		public CachedResource(CacheKey aKey)
		{	this(null, aKey);
		}
		
		public int getCount()
		{	return mCount;
		}
		
		private void touch()
		{	mTouched=System.currentTimeMillis();
		}
		
		public void increment()
		{	mCount++;
			touch();
		}
		
		public void reset()
		{	mCount=0;
			touch();
		}
		
		public IURRepresentation getResource()
		{	return mResource;
		}
		
		public boolean isRemoved()
		{	return mResource==null;
		}
		
		public CacheKey getKey()
		{	return mKey;
		}
		
		public long lastTouched()
		{	return mTouched;
		}

		public int getCacheIndex()
		{	return mStaticIndex;
		}

		public int updateCacheIndex(long aNow)
		{	IURMeta meta = mResource.getMeta();
			long index = (aNow-mTouched)/((meta.getCreationCost()>>6)+mCount);
			mStaticIndex = (index>Integer.MAX_VALUE || meta.isExpired())?Integer.MAX_VALUE:(int)index;
			return mStaticIndex;
		}
		
		public int compare(Object o1, Object o2)
		{	CachedResource cr1=(CachedResource)o1;
			CachedResource cr2=(CachedResource)o2;
			return (int)(cr1.mStaticIndex-cr2.mStaticIndex);
		}
		
		public int compareTo(Object obj)
		{	return compare(this, obj);
		}		
	}
}