# frozen_string_literal: true

require 'date'
require 'time'

module GitlabQuality
  module TestTooling
    module Report
      # Base class for specific health problems reporting.
      # Uses the API to create GitLab issues for any passed test coming from JSON test reports.
      # We expect the test reports to come from a new RSpec process where we retried failing specs.
      #
      # - Takes the JSON test reports like rspec-*.json
      # - Takes a project where flaky test issues should be created
      # - For every passed test in the report:
      #   - Find issue by test hash or create a new issue if no issue was found
      #   - Add a Failures/Flakiness/Slowness/... report in the "<Failures/Flakiness/Slowness/...> reports" note
      class HealthProblemReporter < ReportAsIssue
        include Concerns::GroupAndCategoryLabels
        include Concerns::IssueReports

        BASE_SEARCH_LABELS    = ['test'].freeze
        FOUND_IN_MR_LABEL     = '~"found:in MR"'
        FOUND_IN_MASTER_LABEL = '~"found:master"'

        def initialize(
          input_files: [],
          dry_run: false,
          issue_update_enabled: true,
          gcs_enabled: false,
          gcs_project_id: nil,
          gcs_bucket: nil,
          gcs_credentials: nil,
          **kwargs)
          super(input_files: input_files, dry_run: dry_run, **kwargs)

          @dry_run               = dry_run
          @issue_update_enabled  = issue_update_enabled
          @gcs_enabled           = gcs_enabled
          @gcs_project_id        = gcs_project_id
          @gcs_bucket            = gcs_bucket
          @gcs_credentials       = gcs_credentials
        end

        def most_recent_report_date_for_issue(issue_iid:)
          reports_note = existing_reports_note(issue_iid: issue_iid)
          return unless reports_note

          most_recent_report_from_reports_note(reports_note)&.report_date
        end

        private

        attr_reader :dry_run, :issue_update_enabled, :gcs_enabled, :gcs_project_id, :gcs_bucket, :gcs_credentials

        def problem_type
          'unhealthy'
        end

        def test_is_applicable?(_test)
          false
        end

        def identity_labels
          []
        end

        def new_issue_labels(_test)
          []
        end

        def search_labels
          BASE_SEARCH_LABELS
        end

        def report_section_header
          ''
        end

        def reports_extra_content(_test)
          ''
        end

        def item_extra_content(_test)
          found_label
        end

        def most_recent_report_from_reports_note(reports_note)
          @most_recent_report_from_reports_note ||= report_lines(reports_note&.body.to_s).first
        end

        def run!
          Runtime::Logger.info "issue creation/update is disabled." unless issue_update_enabled
          Runtime::Logger.info "push to GCS is disabled." unless gcs_enabled
          return unless test_reporting_enabled?

          Runtime::Logger.info "Reporting tests in `#{files.join(',')}`."
          TestResults::Builder.new(file_glob: files, token: token, project: project).test_results_per_file do |test_results|
            Runtime::Logger.info "=> Processing #{test_results.count} tests in #{test_results.path}"

            process_test_results(test_results)
          end
        end

        def process_test_results(test_results)
          applicable_tests = test_results.select { |test| test_is_applicable?(test) }
          return if applicable_tests.empty?

          process_issues_for_tests(applicable_tests) if issue_update_enabled

          if gcs_enabled
            tests_data            = build_tests_data(applicable_tests)
            test_results_filename = File.basename(test_results.path)

            push_test_to_gcs(tests_data, test_results_filename)
          end

          Runtime::Logger.info " => Reported #{applicable_tests.size} #{problem_type} tests."
        end

        def process_issues_for_tests(tests)
          tests.each do |test|
            Runtime::Logger.info " => Processing issues for #{problem_type} test '#{test.name}'..."
            issues = existing_test_health_issues_for_test(test)
            create_or_update_test_health_issues(issues, test)
          end
        end

        def existing_test_health_issues_for_test(test)
          find_issues_by_hash(test_hash(test), state: 'opened', labels: search_labels)
        end

        def build_tests_data(tests)
          tests.map do |test|
            Runtime::Logger.info " => Building data for #{problem_type} test '#{test.name}'..."
            issues = existing_test_health_issues_for_test(test)
            test_datum(test, issues)
          end
        end

        def create_or_update_test_health_issues(issues, test)
          if issues.empty?
            issues << create_issue(test)
          else
            # Keep issues description up-to-date
            update_issues(issues, test)
          end

          update_reports(issues, test)
          collect_issues(test, issues)
        end

        def push_test_to_gcs(tests_data, test_results_filename)
          Runtime::Logger.info "will push the test data to GCS"

          GcsTools.gcs_client(project_id: gcs_project_id, credentials: gcs_credentials, dry_run: dry_run).put_object(
            gcs_bucket,
            gcs_metrics_file_name(test_results_filename),
            tests_data.to_json,
            force: true,
            content_type: 'application/json'
          )

          Runtime::Logger.info "Successfully pushed the #{gcs_metrics_file_name(test_results_filename)} file to the GCS bucket."
        end

        def gcs_metrics_file_name(test_results_filename)
          today = Time.now.to_date.iso8601

          "#{today}-#{test_results_filename}"
        end

        def update_reports(issues, test)
          issues.each do |issue|
            Runtime::Logger.info "   => Reporting #{problem_type} test to existing issue: #{issue.web_url}"
            add_report_to_issue(issue: issue, test: test, related_issues: (issues - [issue]))
          end
        end

        def add_report_to_issue(issue:, test:, related_issues:)
          current_reports_note = existing_reports_note(issue_iid: issue.iid)

          new_reports_list = new_reports_list(current_reports_note: current_reports_note, test: test)
          note_body        = append_quick_actions_to_note(
            new_reports_list: new_reports_list,
            related_issues: related_issues,
            options: {
              test: test
            }
          )

          if current_reports_note
            gitlab.edit_issue_note(
              issue_iid: issue.iid,
              note_id: current_reports_note.id,
              note: note_body
            )
          else
            gitlab.create_issue_note(iid: issue.iid, note: note_body)
          end
        end

        def new_reports_list(current_reports_note:, test:)
          increment_reports(
            current_reports_content: current_reports_note&.body.to_s,
            test: test,
            reports_section_header: report_section_header,
            item_extra_content: item_extra_content(test),
            reports_extra_content: reports_extra_content(test)
          )
        end

        def append_quick_actions_to_note(new_reports_list:, related_issues:, options: {})
          report = new_reports_list

          quick_actions = [
            health_problem_status_label_quick_action(new_reports_list, options: options),
            identity_labels_quick_action,
            relate_issues_quick_actions(related_issues)
          ]

          quick_actions.unshift(report).join("\n")
        end

        def existing_reports_note(issue_iid:)
          gitlab.find_issue_notes(iid: issue_iid).find do |note|
            note.body.start_with?(report_section_header)
          end
        end

        # We report a test if we have issue update and/or push to GCS enabled.
        def test_reporting_enabled?
          issue_update_enabled || gcs_enabled
        end

        # rubocop:disable Metrics/AbcSize
        def test_datum(test, issues)
          feature_category = test.feature_category.to_s.strip.empty? ? nil : test.feature_category
          product_group    = test.product_group.to_s.strip.empty? ? nil : test.product_group

          if !product_group && feature_category
            labels_inference = GitlabQuality::TestTooling::LabelsInference
            product_group = labels_inference.new.product_group_from_feature_category(feature_category)
          end

          {
            feature_category: feature_category,
            created_at: Time.now.utc.iso8601,
            description: test.name,
            filename: test.file,
            gitlab_project_id: Runtime::Env.ci_project_id,
            product_group: product_group,
            id: test.example_id,
            hash: test_hash(test),
            issue_url: issues.first&.web_url,
            job_id: Runtime::Env.ci_job_id,
            job_web_url: test.ci_job_url,
            job_status: Runtime::Env.ci_job_status,
            pipeline_id: Runtime::Env.ci_pipeline_id,
            pipeline_ref: Runtime::Env.ci_commit_ref_name,
            pipeline_web_url: Runtime::Env.ci_pipeline_url,
            stacktrace: test.full_stacktrace,
            test_level: test.level
          }
        end
        # rubocop:enable Metrics/AbcSize

        def found_label
          if ENV.key?('CI_MERGE_REQUEST_IID')
            FOUND_IN_MR_LABEL
          else
            FOUND_IN_MASTER_LABEL
          end
        end

        # Defined in subclasses
        def health_problem_status_label_quick_action(*)
          ''
        end

        def identity_labels_quick_action
          return if identity_labels.empty?

          label_names_to_label_quick_action(identity_labels)
        end

        def relate_issues_quick_actions(issues)
          issues.map do |issue|
            "/relate #{issue.web_url}"
          end.join("\n")
        end
      end
    end
  end
end
