/*
 *  crypt.c
 *
 *  $Id$
 *
 *  one-way crypting
 *
 *  This file is part of the OpenLink Software Virtuoso Open-Source (VOS)
 *  project.
 *
 *  Copyright (C) 1998-2015 OpenLink Software
 *
 *  This project is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; only version 2 of the License, dated June 1991.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 */

#include "Dk.h"
#include <assert.h>


static int
mod95 (register int val)
{
  /* The mathematical MOD does not match the computer MOD  */

  /*
   * Yes, what I do here may look strange, but it gets the job done,
   * and portably at that.
   */

  while (val >= 9500)
    val -= 9500;
  while (val >= 950)
    val -= 950;
  while (val >= 95)
    val -= 95;
  while (val < 0)
    val += 95;
  return (val);
}


static void
ue_crypt (register char *bptr, register int len)
/* register char *bptr;  buffer of characters to be encrypted */
/* register int len;     number of characters in the buffer */
{
  register int cc;		/* current character being considered */

  static long key = 0;		/* 29 bit encryption key */
  static int salt = 0;		/* salt to spice up key with */

  if (!bptr)
    {				/* is there anything here to encrypt? */
      key = len;		/* set the new key */
      salt = len;		/* set the new salt */
      return;
    }
  while (len--)
    {				/* for every character in the buffer */

      cc = *bptr;		/* get a character out of the buffer */

      /* only encipher printable characters */
      /* was: if ((cc >= ' ') && (cc <= '~')) */
      if (isprint (cc))
	{

/**  If the upper bit (bit 29) is set, feed it back into the key.  This
    assures us that the starting key affects the entire message.  **/

	  key &= 0x1FFFFFFFL;	/* strip off overflow */
	  if (key & 0x10000000L)
	    {
	      key ^= 0x0040A001L;	/* feedback */
	    }

/**  Down-bias the character, perform a Beaufort encryption, and
    up-bias the character again.  We want key to be positive
    so that the left shift here will be more portable and the
    mod95() faster   **/

	  cc = mod95 ((int) (key % 95) - (cc - ' ')) + ' ';

/**  the salt will spice up the key a little bit, helping to obscure
    any patterns in the clear text, particularly when all the
    characters (or long sequences of them) are the same.  We do
    not want the salt to go negative, or it will affect the key
    too radically.  It is always a good idea to chop off cyclics
    to prime values.  **/

	  if (++salt >= 20857)
	    {			/* prime modulus */
	      salt = 0;
	    }

/**  our autokey (a special case of the running key) is being
    generated by a weighted checksum of clear text, cipher
    text, and salt.   **/

	  key = key + key + cc + *bptr + salt;
	}
      *bptr++ = cc;		/* put character back into buffer */
    }
  return;
}


char *
xx_encrypt (char *keystr, char *str)
{
  assert (NULL != keystr);
  ue_crypt (NULL, 0);
  ue_crypt (keystr, (int) strlen (keystr));
  assert (NULL != keystr);
  ue_crypt (str, (int) strlen (str));
  return str;
}


#ifdef TEST_CRYPT
int
main (int argc, char *argv[])
{
  char str[1024];
  char keystr[1024];

  if (argc == 3)
    {
      strcpy_ck (keystr, argv[1]);
      strcpy_ck (str, argv[2]);
    }
  else
    {
      printf ("Usage: ec2crypt <key> <string>\n");
      exit (0);
    }

  printf ("key: %s\n", keystr);
  printf ("string: %s\n", str);
  printf ("keylen: %d\n", strlen (keystr));
  printf ("stringlen: %d\n", strlen (str));
  printf ("encrypted: %s\n", xx_encrypt (keystr, str));

  return (0);
}
#endif
