/***************************************************************************
 *
 * knetworkmanager-vpn_bus.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Timo Hoenig <thoenig@suse.de>, <thoenig@nouse.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/
#include <malloc.h>

#include "knetworkmanager-vpn.h"
#include "knetworkmanager-vpn_dbus.h"

KNetworkManager* VPNDBus::_ctx = NULL;

void
VPNDBus::updateVPNActivationStage (const char* name, NMVPNActStage stage)
{
	VPNConnection* vpnConnection = _ctx->getVPN ()->getVPNConnection (QString::fromUtf8(name));
	if (vpnConnection) {
		vpnConnection->setActivationStage (stage);
	}
}

void
VPNDBus::updateVPNConnectionCallback (DBusPendingCall* pcall, void* /*data*/)
{
	DBusMessage*  reply        = NULL;
	const char*   name         = NULL;
	const char*   user         = NULL;
	const char*   service_name = NULL;
	dbus_uint32_t stage;

	reply = dbus_pending_call_steal_reply (pcall);
	if (!reply) {
		goto out;
	}

	if (dbus_message_get_type (reply) == DBUS_MESSAGE_TYPE_ERROR) {
		DBusError error;

		dbus_error_init (&error);
		dbus_set_error_from_message (&error, reply);
		printf ("Error: %s, %s\n", error.name, error.message);
		dbus_error_free (&error);
		goto unref_out;
	}

	if (dbus_message_get_args (reply, NULL, DBUS_TYPE_STRING, &name,
						DBUS_TYPE_STRING, &user,
						DBUS_TYPE_STRING, &service_name,
						DBUS_TYPE_UINT32, &stage, DBUS_TYPE_INVALID)) {
		VPNConnection* vpnConnection = _ctx->getVPN ()->getVPNConnection (QString::fromUtf8(name));
		if (vpnConnection == NULL) {
			/* TODO: Add one we have not found?  Does not sound reasonable as we
			 *       will also fail at providing vpnData.
			 */
		} else {
			vpnConnection->setName            (QString::fromUtf8(name));
			vpnConnection->setServiceName     (QString::fromUtf8(service_name));
			vpnConnection->setUser            (QString::fromUtf8(user));
			vpnConnection->setActivationStage ((NMVPNActStage) stage);
			vpnConnection->setConfirmedByNM   (true);
		}
	}

unref_out:
	dbus_message_unref (reply);
out:
	dbus_pending_call_unref (pcall);

	return;
}

/* NM asks us to update a given VPN connection */

void
VPNDBus::updateVPNConnection (const char* name)
{
	DBusMessage*     msg   = NULL;
	DBusConnection*  con   = _ctx->getDBus ()->getConnection ();
	DBusPendingCall* pcall = NULL;

	if (!con|| !name) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH_VPN, NM_DBUS_INTERFACE_VPN, "getVPNConnectionProperties");
	if (msg) {
		dbus_message_append_args (msg, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
		dbus_connection_send_with_reply (con, msg, &pcall, -1);
		if (pcall) {
			dbus_pending_call_set_notify (pcall, VPNDBus::updateVPNConnectionCallback, NULL, NULL);
		}
		dbus_message_unref (msg);
	}

	return;
}

/* we tell NM to update a given VPN connection */

void
VPNDBus::updateVPNConnection (VPNConnection* vpnConnection)
{
	DBusMessage*    msg = NULL;
	DBusConnection* con = _ctx->getDBus ()->getConnection ();

	if (!con || !vpnConnection) {
		return;
	}
	
	msg = dbus_message_new_signal (NMI_DBUS_PATH, NMI_DBUS_INTERFACE, "VPNConnectionUpdate");
	if  (msg) {
		char* name = strdup(vpnConnection->getName ().utf8());

		dbus_message_append_args (msg, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
		dbus_connection_send (con, msg, NULL);
		dbus_message_unref (msg);

		free(name);
	}

	return;
}

void
VPNDBus::activateVPNConnection (VPNConnection* vpnConnection)
{
	DBusMessage*    msg = NULL;
	DBusMessageIter iter;
	DBusMessageIter iter_array;
	DBusConnection* con = _ctx->getDBus ()->getConnection ();

	if (!con || !vpnConnection) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH_VPN, NM_DBUS_INTERFACE_VPN, "activateVPNConnection");
	if (msg) {
		char* name = strdup(vpnConnection->getName ().utf8());
		QStringList passwords = vpnConnection->getPasswords ();

		dbus_message_iter_init_append (msg, &iter);
		dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &name);
		dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING_AS_STRING, &iter_array);
		for (QStringList::Iterator i = passwords.begin (); i != passwords.end (); ++i) {
			char* password = strdup((*i).utf8());
			dbus_message_iter_append_basic (&iter_array, DBUS_TYPE_STRING, &password);
			free(password);
		}
		dbus_message_iter_close_container (&iter, &iter_array);
		dbus_connection_send (con, msg, NULL);
		dbus_message_unref (msg);
		free(name);
	}

	return;
}

void
VPNDBus::disconnectVPNConnection (void)
{
	DBusMessage*    msg = NULL;
	DBusConnection* con = _ctx->getDBus ()->getConnection ();

	if (!con) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH_VPN, NM_DBUS_INTERFACE_VPN, "deactivateVPNConnection");
	if (msg) {
		dbus_connection_send (con, msg, NULL);
		dbus_message_unref (msg);
	}

	return;
}

void
VPNDBus::showVPNConnectionFailure (const char* member, const char* vpn_name, const char* err_msg)
{
	VPNConnection* vpnConnection = _ctx->getVPN ()->getVPNConnection (QString::fromUtf8(vpn_name));
	if (vpnConnection) {
		QString help1(member);
		QString help2(err_msg);
		vpnConnection->setVPNConnectionFailure(help1, help2);
	}

}

void
VPNDBus::push (KNetworkManager* ctx)
{
	_ctx = ctx;
}
