/*
 * This file is part of the KDE project.
 *
 * Copyright (C) 2008 Dirk Mueller <mueller@kde.org>
 * Copyright (C) 2008 Urs Wolfer <uwolfer @ kde.org>
 * Copyright (C) 2008 Michael Howell <mhowell123@gmail.com>
 * Copyright (C) 2009,2010 Dawit Alemayehu <adawit@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

// Own
#include "kwebpage.h"
#include "kwebwallet.h"

// Local
#include "kwebpluginfactory.h"

// KDE
#include <kaction.h>
#include <kfiledialog.h>
#include <kprotocolmanager.h>
#include <kjobuidelegate.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kstandardshortcut.h>
#include <kurl.h>
#include <kdebug.h>
#include <kmimetypetrader.h>
#include <klocalizedstring.h>
#include <ktemporaryfile.h>
#include <kio/accessmanager.h>
#include <kio/job.h>
#include <kio/jobuidelegate.h>
#include <kio/renamedialog.h>
#include <kio/scheduler.h>
#include <kparts/browseropenorsavequestion.h>

// Qt
#include <QtCore/QPointer>
#include <QtCore/QFileInfo>
#include <QtCore/QCoreApplication>
#include <QtWebKit/QWebFrame>
#include <QtNetwork/QNetworkReply>


#define QL1S(x)  QLatin1String(x)
#define QL1C(x)  QLatin1Char(x)

static bool isMimeTypeAssociatedWithSelf(const QString& mimeType, const KService::Ptr &offer)
{
    Q_UNUSED(mimeType);
    
    if (!offer)
        return false;
    
    kDebug(800) << offer->desktopEntryName();

    const QString& appName = QCoreApplication::applicationName();

    if (appName == offer->desktopEntryName() || offer->exec().trimmed().startsWith(appName))
        return true;

    // konqueror exception since it uses kfmclient to open html content...
    if (appName == QL1S("konqueror") && offer->exec().trimmed().startsWith(QL1S("kfmclient")))
        return true;

    return false;
}

static void extractMimeType(const QNetworkReply* reply, QString& mimeType)
{
    mimeType.clear();
    const KIO::MetaData& metaData = reply->attribute(static_cast<QNetworkRequest::Attribute>(KIO::AccessManager::MetaData)).toMap();
    if (metaData.contains(QL1S("content-type")))
        mimeType = metaData.value(QL1S("content-type"));

    if (!mimeType.isEmpty())
        return;

    if (!reply->hasRawHeader("Content-Type"))
        return;

    const QString value (QL1S(reply->rawHeader("Content-Type").simplified().constData()));
    const int index = value.indexOf(QL1C(';'));
    mimeType = ((index == -1) ? value : value.left(index));
}

static KUrl promptUser (QWidget *parent, const KUrl& url, const QString& suggestedName)
{
    KUrl destUrl;
    int result = KIO::R_OVERWRITE;
    const QString fileName ((suggestedName.isEmpty() ? url.fileName() : suggestedName));

    do {
        // convert filename to URL using fromPath to avoid trouble with ':' in filenames (#184202)
        destUrl = KFileDialog::getSaveFileName(KUrl::fromPath(fileName), QString(), parent);

        if (destUrl.isLocalFile()) {
            QFileInfo finfo (destUrl.toLocalFile());
            if (finfo.exists()) {
                QDateTime now = QDateTime::currentDateTime();
                KIO::RenameDialog dlg (parent, i18n("Overwrite File?"), url, destUrl,
                                       KIO::RenameDialog_Mode(KIO::M_OVERWRITE | KIO::M_SKIP),
                                       -1, finfo.size(),
                                       now.toTime_t(), finfo.created().toTime_t(),
                                       now.toTime_t(), finfo.lastModified().toTime_t());
                result = dlg.exec();
            }
        }
    } while (result == KIO::R_CANCEL && destUrl.isValid());

    return destUrl;
}

static bool downloadResource (const KUrl& srcUrl, const QString& suggestedName = QString(),
                              QWidget* parent = 0, const KIO::MetaData& metaData = KIO::MetaData())
{
    const KUrl& destUrl = promptUser(parent, srcUrl, suggestedName);

    if (!destUrl.isValid())
        return false;
    
    KIO::Job *job = KIO::file_copy(srcUrl, destUrl, -1, KIO::Overwrite);
    
    if (!metaData.isEmpty())
        job->setMetaData(metaData);

    job->addMetaData(QL1S("MaxCacheSize"), QL1S("0")); // Don't store in http cache.
    job->addMetaData(QL1S("cache"), QL1S("cache")); // Use entry from cache if available.
    job->uiDelegate()->setAutoErrorHandlingEnabled(true);
    return true;
}

class KWebPage::KWebPagePrivate
{
public:
    KWebPagePrivate() : inPrivateBrowsingMode(false) {}
    void _k_copyResultToTempFile(KJob * job)
    {
        if ( job->error() )
            job->uiDelegate()->showErrorMessage();
        else // Same as KRun::foundMimeType but with a different URL
            (void)KRun::runUrl(static_cast<KIO::FileCopyJob *>(job)->destUrl(), mimeType, window);
    }

    QPointer<QWidget> window;
    QString mimeType;
    QPointer<KWebWallet> wallet;
    bool inPrivateBrowsingMode;
};


KWebPage::KWebPage(QObject *parent, Integration flags)
         :QWebPage(parent), d(new KWebPagePrivate)
{ 
    // KDE KParts integration for <embed> tag...
    if (!flags || (flags & KPartsIntegration))
        setPluginFactory(new KWebPluginFactory(this));

    WId windowId = 0;
    QWidget *widget = qobject_cast<QWidget*>(parent);
    if (widget && widget->window())
        windowId = widget->window()->winId();

    // KDE IO (KIO) integration...
    if (!flags || (flags & KIOIntegration)) {
        KIO::Integration::AccessManager *manager = new KIO::Integration::AccessManager(this);
        // Disable QtWebKit's internal cache to avoid duplication with the one in KIO...
        manager->setCache(0);
        manager->setCookieJarWindowId(windowId);
        setNetworkAccessManager(manager);
    }

    // KWallet integration...
    if (!flags || (flags & KWalletIntegration))
        setWallet(new KWebWallet(0, windowId));

    action(Back)->setIcon(KIcon("go-previous"));
    action(Forward)->setIcon(KIcon("go-next"));
    action(Reload)->setIcon(KIcon("view-refresh"));
    action(Stop)->setIcon(KIcon("process-stop"));
    action(Cut)->setIcon(KIcon("edit-cut"));
    action(Copy)->setIcon(KIcon("edit-copy"));
    action(Paste)->setIcon(KIcon("edit-paste"));
    action(Undo)->setIcon(KIcon("edit-undo"));
    action(Redo)->setIcon(KIcon("edit-redo"));
    action(InspectElement)->setIcon(KIcon("view-process-all"));
    action(OpenLinkInNewWindow)->setIcon(KIcon("window-new"));
    action(OpenFrameInNewWindow)->setIcon(KIcon("window-new"));
    action(OpenImageInNewWindow)->setIcon(KIcon("window-new"));
    action(CopyLinkToClipboard)->setIcon(KIcon("edit-copy"));
    action(CopyImageToClipboard)->setIcon(KIcon("edit-copy"));
    action(ToggleBold)->setIcon(KIcon("format-text-bold"));
    action(ToggleItalic)->setIcon(KIcon("format-text-italic"));
    action(ToggleUnderline)->setIcon(KIcon("format-text-underline"));
    action(DownloadLinkToDisk)->setIcon(KIcon("document-save"));
    action(DownloadImageToDisk)->setIcon(KIcon("document-save"));

    settings()->setWebGraphic(QWebSettings::MissingPluginGraphic, KIcon("preferences-plugin").pixmap(32, 32));
    settings()->setWebGraphic(QWebSettings::MissingImageGraphic, KIcon("image-missing").pixmap(32, 32));
    settings()->setWebGraphic(QWebSettings::DefaultFrameIconGraphic, KIcon("applications-internet").pixmap(32, 32));

    action(Back)->setShortcut(KStandardShortcut::back().primary());
    action(Forward)->setShortcut(KStandardShortcut::forward().primary());
    action(Reload)->setShortcut(KStandardShortcut::reload().primary());
    action(Stop)->setShortcut(QKeySequence(Qt::Key_Escape));
    action(Cut)->setShortcut(KStandardShortcut::cut().primary());
    action(Copy)->setShortcut(KStandardShortcut::copy().primary());
    action(Paste)->setShortcut(KStandardShortcut::paste().primary());
    action(Undo)->setShortcut(KStandardShortcut::undo().primary());
    action(Redo)->setShortcut(KStandardShortcut::redo().primary());
    action(SelectAll)->setShortcut(KStandardShortcut::selectAll().primary());
}

KWebPage::~KWebPage()
{
    delete d;
}

bool KWebPage::isExternalContentAllowed() const
{
    KIO::AccessManager *manager = qobject_cast<KIO::AccessManager*>(networkAccessManager());
    if (manager)
        return manager->isExternalContentAllowed();
    return true;
}

KWebWallet *KWebPage::wallet() const
{
    return d->wallet;
}

void KWebPage::setAllowExternalContent(bool allow)
{
    KIO::AccessManager *manager = qobject_cast<KIO::AccessManager*>(networkAccessManager());
    if (manager)
      manager->setExternalContentAllowed(allow);
}

void KWebPage::setWallet(KWebWallet* wallet)
{
    // Delete the current wallet if this object is its parent...
    if (d->wallet && this == d->wallet->parent())
        delete d->wallet;

    d->wallet = wallet;

    if (d->wallet)
        d->wallet->setParent(this);
}

void KWebPage::downloadRequest(const QNetworkRequest &request)
{
    downloadResource(request.url(), QString(), view(),
                     request.attribute(static_cast<QNetworkRequest::Attribute>(KIO::AccessManager::MetaData)).toMap());
}

void KWebPage::downloadUrl(const KUrl &url)
{
    downloadResource(url, QString(), view());
}

static bool isReplyStatusOk(const QNetworkReply* reply)
{
    if (!reply || reply->error() != QNetworkReply::NoError)
        return false;

    bool ok = false;
    const int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt(&ok);

    if (!ok || statusCode < 200 || statusCode > 299)
        return false;

    return true;
}

void KWebPage::downloadResponse(QNetworkReply *reply)
{
    Q_ASSERT(reply);
    
    if (!reply)
        return;

    // Put the job on hold...
    KIO::Integration::AccessManager::putReplyOnHold(reply);

    // Reply url...
    const KUrl requestUrl (reply->request().url());

    // Get the top level window...
    QWidget* topLevelWindow = view() ? view()->window() : 0;
    
    // Get suggested file name...
    const KIO::MetaData& metaData = reply->attribute(static_cast<QNetworkRequest::Attribute>(KIO::AccessManager::MetaData)).toMap();
    const QString suggestedFileName = metaData.value(QL1S("content-disposition-filename"));
    const QString contentDispositionType = metaData.value(QL1S("content-disposition-type"));
   
    // Get the mime-type...
    QString mimeType;
    extractMimeType(reply, mimeType);
    // Convert executable text files to plain text...
    if (KParts::BrowserRun::isTextExecutable(mimeType))
        mimeType = QL1S("text/plain");

    //kDebug(800) << "Content-disposition:" << contentDispositionType << suggestedFileName;
    //kDebug(800) << "Got unsupported content of type:" << mimeType << "URL:" << requestUrl;
    //kDebug(800) << "Error code:" << reply->error() << reply->errorString();    
    if (isReplyStatusOk(reply)) {
        KParts::BrowserOpenOrSaveQuestion::Result result;      
        KParts::BrowserOpenOrSaveQuestion dlg(topLevelWindow, requestUrl, mimeType);
        dlg.setSuggestedFileName(suggestedFileName);
        dlg.setFeatures(KParts::BrowserOpenOrSaveQuestion::ServiceSelection);
        result = dlg.askOpenOrSave();

        switch (result) {
        case KParts::BrowserOpenOrSaveQuestion::Open:
            // Handle Post operations that return content...
            if (reply->operation() == QNetworkAccessManager::PostOperation) {
                d->mimeType = mimeType;
                d->window = topLevelWindow;
                QFileInfo finfo (suggestedFileName.isEmpty() ? requestUrl.fileName() : suggestedFileName);
                KTemporaryFile tempFile;
                tempFile.setSuffix(QL1C('.') + finfo.suffix());
                tempFile.setAutoRemove(false);
                tempFile.open();
                KUrl destUrl;
                destUrl.setPath(tempFile.fileName());
                KIO::Job *job = KIO::file_copy(requestUrl, destUrl, 0600, KIO::Overwrite);
                job->ui()->setWindow(topLevelWindow);
                connect(job, SIGNAL(result(KJob *)),
                        this, SLOT(_k_copyResultToTempFile(KJob*)));
                return;
            }

            // Ask before running any executables...
            if (KParts::BrowserRun::allowExecution(mimeType, requestUrl)) {
                KService::Ptr offer = dlg.selectedService();
                // HACK: The check below is necessary to break an infinite
                // recursion that occurs whenever this function is called as a result
                // of receiving content that can be rendered by the app using this engine.
                // For example a text/html header that containing a content-disposition
                // header is received by the app using this class.
                if (isMimeTypeAssociatedWithSelf(mimeType, offer)) {
                    QNetworkRequest req (reply->request());
                    req.setRawHeader("x-kdewebkit-ignore-disposition", "true");
                    currentFrame()->load(req);
                    return;
                }

                if (offer) {
                    KUrl::List list;
                    list.append(requestUrl);
                    //kDebug(800) << "Suggested file name:" << suggestedFileName;
                    if (offer->categories().contains(QL1S("KDE"), Qt::CaseInsensitive)) {
                        KIO::Scheduler::publishSlaveOnHold();
                        KRun::run(*offer, list, topLevelWindow , false, suggestedFileName);
                        return;
                    }
                    // For non KDE applications, we launch and kill the slave-on-hold...
                    KRun::run(*offer, list, topLevelWindow , false, suggestedFileName);
                }
            }
            break;
        case KParts::BrowserOpenOrSaveQuestion::Save:
            // Do not attempt to download directories and local files...
            if (mimeType == QL1S("inode/directory") || requestUrl.isLocalFile())
                break;

            downloadResource(requestUrl, suggestedFileName, topLevelWindow);
            return;
        case KParts::BrowserOpenOrSaveQuestion::Cancel:
        default:
            break;
        }        
    } else {
        KService::Ptr offer = KMimeTypeTrader::self()->preferredService(mimeType);
        if (isMimeTypeAssociatedWithSelf(mimeType, offer)) {
            QNetworkRequest req (reply->request());
            req.setRawHeader("x-kdewebkit-ignore-disposition", "true");
            currentFrame()->load(req);
            return;
        }        
    }
    
    // Remove any ioslave that was put on hold...
    KIO::Scheduler::removeSlaveOnHold();
}

QString KWebPage::sessionMetaData(const QString &key) const
{
    QString value;

    KIO::Integration::AccessManager *manager = qobject_cast<KIO::Integration::AccessManager *>(networkAccessManager());
    if (manager)
        value = manager->sessionMetaData().value(key);

    return value;
}

QString KWebPage::requestMetaData(const QString &key) const
{
    QString value;

    KIO::Integration::AccessManager *manager = qobject_cast<KIO::Integration::AccessManager *>(networkAccessManager());
    if (manager)
        value = manager->requestMetaData().value(key);

    return value;
}

void KWebPage::setSessionMetaData(const QString &key, const QString &value)
{
    KIO::Integration::AccessManager *manager = qobject_cast<KIO::Integration::AccessManager *>(networkAccessManager());
    if (manager)
        manager->sessionMetaData()[key] = value;
}

void KWebPage::setRequestMetaData(const QString &key, const QString &value)
{
    KIO::Integration::AccessManager *manager = qobject_cast<KIO::Integration::AccessManager *>(networkAccessManager());
    if (manager)
        manager->requestMetaData()[key] = value;
}

void KWebPage::removeSessionMetaData(const QString &key)
{
    KIO::Integration::AccessManager *manager = qobject_cast<KIO::Integration::AccessManager *>(networkAccessManager());
    if (manager)
        manager->sessionMetaData().remove(key);
}

void KWebPage::removeRequestMetaData(const QString &key)
{
    KIO::Integration::AccessManager *manager = qobject_cast<KIO::Integration::AccessManager *>(networkAccessManager());
    if (manager)
        manager->requestMetaData().remove(key);
}

QString KWebPage::userAgentForUrl(const QUrl& _url) const
{
    const KUrl url(_url);
    const QString userAgent = KProtocolManager::userAgentForHost((url.isLocalFile() ? QL1S("localhost") : url.host()));

    if (userAgent == KProtocolManager::defaultUserAgent())
        return QWebPage::userAgentForUrl(_url);

    return userAgent;
}

static void setDisableCookieJarStorage(QNetworkAccessManager* manager, bool status)
{
    if (manager) {
        KIO::Integration::CookieJar *cookieJar = manager ? qobject_cast<KIO::Integration::CookieJar*>(manager->cookieJar()) : 0;
        if (cookieJar) {
            //kDebug(800) << "Store cookies ?" << !status;
            cookieJar->setDisableCookieStorage(status);
        }
    }
}

bool KWebPage::acceptNavigationRequest(QWebFrame *frame, const QNetworkRequest &request, NavigationType type)
{
    kDebug(800) << "url:" << request.url() << ", type:" << type << ", frame:" << frame;

    if (frame && d->wallet && type == QWebPage::NavigationTypeFormSubmitted)
        d->wallet->saveFormData(frame);

    // Make sure nothing is cached when private browsing mode is enabled...
    if (settings()->testAttribute(QWebSettings::PrivateBrowsingEnabled)) {
        if (!d->inPrivateBrowsingMode) {
            setDisableCookieJarStorage(networkAccessManager(), true);
            setSessionMetaData(QL1S("no-cache"), QL1S("true"));
            d->inPrivateBrowsingMode = true;
        }
    } else  {
        if (d->inPrivateBrowsingMode) {
            setDisableCookieJarStorage(networkAccessManager(), false);
            removeSessionMetaData(QL1S("no-cache"));
            d->inPrivateBrowsingMode = false;
        }
    }

    /*
      If the navigation request is from the main frame, set the cross-domain
      meta-data value to the current url for proper integration with KCookieJar...
    */
    if (frame == mainFrame() && type != QWebPage::NavigationTypeReload)
        setSessionMetaData(QL1S("cross-domain"), request.url().toString());

    return QWebPage::acceptNavigationRequest(frame, request, type);
}

#include "kwebpage.moc"
