+++
title = "Set Python environmental variables under Windows"
weight = 100
+++

#### Setting Python environmental variables under Windows

With Python 3.10 in Conda or Pyenv platforms, the BornAgain GUI might crash
when a Python script is imported. This problem is not observed with
Python 3.11 or newer. Therefore, we recommend upgrading to a newer Python version `>= 3.11`.

If Python 3.10 is the preferred version, the Python-related environmental variables must be
set _manually_ before running the BornAgain GUI to avoid the crash.

To obtain the proper environmental variables, paste the following Python script
into a file, say `ba_hint.py`:

```python
import os, sys

SEP = '-' * 30
py_exe = sys.executable
py_root = os.path.realpath(os.path.dirname(py_exe))
py_lib = os.path.realpath(os.path.dirname(os.__file__))

print(SEP)
print("* Python root: '{}'\n"
      "* Python exectuable: '{}'\n"
      "* Python lib dir: '{}'"
      .format(py_root, py_exe, py_lib))
print(SEP)

ba_msg = "* BornAgain %s installed in the current Python environment."
try:
    import bornagain as ba
    print(ba_msg % "is")
except Exception as ex:
    print(ba_msg % "is _not_")
    print(SEP)
    raise ex
print(SEP)

print("""\
* Open a CMD.EXE shell and run the following commands before executing
  BornAgain GUI in the _same_ shell environment:
""")

print(f"""\
set py_path={py_root:}
set PATH=%py_path%;%PATH%
set PYTHONPATH=%py_path%\Lib;%py_path%\DLLs;%py_path%\Lib\site-packages;%PYTHONPATH%
""")
print(SEP)
```

Open a `CMD.EXE` shell and run the script `ba_hint.py` with the preferred Python interpreter:

```cmd
python C:\path\to\ba_hint.py
```

The script finds the root path where the Python interpreter is installed and checks if
BornAgain is installed as a Python package. If BornAgain is not installed,
an error message will be displayed.

If BornAgain is installed in the Python environment, the scripts provides
the correct environmental variables which must be set before executing
the BornAgain GUI; for instance,

```
------------------------------
* Python root: 'C:\conda\envs\py310'
* Python exectuable: 'C:\conda\envs\py310\python.exe'
* Python lib dir: 'C:\conda\envs\py310\Lib'
------------------------------
* BornAgain is installed in the current Python environment.
------------------------------
* Open a CMD.EXE shell and run the following commands before executing
  BornAgain GUI in the _same_ shell environment:

set py_path=C:\conda\envs\py310
set PATH=%py_path%;%PATH%
set PYTHONPATH=%py_path%\Lib;%py_path%\DLLs;%py_path%\Lib\site-packages;%PYTHONPATH%

------------------------------
```

Notice that the paths might differ on each machine.

With the above instructions, BornAgain GUI can be executed via
the following commands in a `CMD.EXE` shell; for instance:

```cmd
set py_path=C:\conda\envs\py310
set PATH=%py_path%;%PATH%
set PYTHONPATH=%py_path%\Lib;%py_path%\DLLs;%py_path%\Lib\site-packages;%PYTHONPATH%

C:\path\to\BornAgain\bin\bornagain.exe
```
