/* wptGPGPrefsDlg.cpp - WinPT GnuPG Preferences
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"
#include "wptNLS.h"
#include "wptTypes.h"
#include "wptErrors.h"
#include "wptGPG.h"
#include "wptRegistry.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptW32API.h"
#include "wptVersion.h"

static struct {
    const char * name;
    int id;
} opts[] = {
    {"HomeDir",    IDC_GPGPREFS_HOMEDIR},
    {"gpgProgram", IDC_GPGPREFS_EXEDIR},
    {"OptFile",    IDC_GPGPREFS_OPTFILE},
    {0}
};


BOOL CALLBACK
gpgprefs_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    char exedir[1024], homedir[1024], optfile[1024];
    char locale_dir[1024];
    char * p = NULL, t[512];
    const char * s;
    int have_optfile = 0;
    gpg_optfile_t opt = NULL;
    gpg_option_t e;
    UINT n;
    
    switch ( msg ) {
    case WM_INITDIALOG:
#ifndef LANG_DE
        SetWindowText( dlg, _("GnuPG Preferences") );
        SetDlgItemText( dlg, IDC_GPGPREFS_HOMEINF,
                       _("GnuPG home directory  (where both keyrings are located)") );
        SetDlgItemText( dlg, IDC_GPGPREFS_OPTINF,
                       _("GnuPG config file (default: use gpg.conf)") );
        SetDlgItemText( dlg, IDC_GPGPREFS_EXEINF,
                       _("GnuPG exe file location (full path with added gpg.exe)") );
        SetDlgItemText( dlg, IDC_GPGPREFS_LOCALINF,
                       _("Locale directory (to access the translation files)") );
#endif
	for( n=0; (s = opts[n].name); n++ ) {
	    p = get_reg_entry_gpg( s );
	    if( p ) {
		SetDlgItemText( dlg, opts[n].id, p );
		free_if_alloc( p );
	    }
	}
        
        if( (p = get_reg_entry_mo( )) ) {
            SetDlgItemText( dlg, IDC_GPGPREFS_LOCALE, p );
            free_if_alloc( p );
        }
	p = get_gnupg_cfgfile( );
	if( p ) {
	    parse_gpg_options( p, &opt );
	    free_if_alloc( p );
	    if( opt && find_option( opt, "force-v3-sigs" ) )
		CheckDlgButton( dlg, IDC_GPGPREFS_V3SIGS, BST_CHECKED );
	    if( opt && (e=find_option( opt, "comment" )) )
		SetDlgItemText( dlg, IDC_GPGPREFS_COMMENT, e->val );
	    if( opt && (e=find_option( opt, "encrypt-to" )) )
		SetDlgItemText( dlg, IDC_GPGPREFS_ENCTO, e->val );	    
	}
	center_window( dlg );
        SetForegroundWindow( dlg );
        return TRUE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch ( LOWORD( wparam ) ) {
        case IDC_GPGPREFS_SAVE:
            if( !GetDlgItemText( dlg, IDC_GPGPREFS_HOMEDIR, homedir, sizeof homedir -1 ) ) {
                msg_box( dlg, _("Please enter the GnuPG home directory."), _("Preferences"), MB_ERR );
                return FALSE;
            }
            if( dir_exist_check( homedir ) ) {
		_snprintf( t, sizeof t - 1, "%s: %s", homedir, winpt_strerror(WPTERR_DIR_OPEN) );
                msg_box( dlg, t, _("Preferences"), MB_ERR );
                return FALSE;
            }
            
            if( GetDlgItemText( dlg, IDC_GPGPREFS_OPTFILE, optfile, sizeof optfile - 1 ) ) {
                if ( file_exist_check( optfile ) ) {
                    n = msg_box(dlg, _("Could not find GPG config file.\n"
                                       "Do you want to create a config file?"),
                                _("Preferences"), MB_INFO|MB_YESNO );
                    if( n == IDNO ) {
                        SetDlgItemText( dlg, IDC_GPGPREFS_OPTFILE, "" );
                        return FALSE;
                    }
                    else if( n == IDYES ) {
                        FILE * fp = fopen( optfile, "wb" );
                        if( fp )
                            fclose( fp );
                        if( file_exist_check( optfile ) ) {
			    log_box( _("Preferences"), MB_ERR, "%s: %s", optfile, winpt_strerror( WPTERR_FILE_CREAT ) );
                            return FALSE;
                        }
                        have_optfile = 1;
                    }
                }
                else
                    have_optfile = 1;
                if( have_optfile ) {
                    if( set_reg_entry_gpg( "OptFile", optfile ) ) {
                        msg_box( dlg, _("Could not save 'OptFile' in the registry."), _("Preferences"), MB_ERR );
                        return FALSE;
                    }
                }
            }
            else {
                char * t = get_gnupg_cfgfile( );
                if( t && !file_exist_check(t) )
                    set_reg_entry_gpg( "OptFile", t );
                else
                    set_reg_entry_gpg( "OptFile", "" );
		free_if_alloc( t );
            }
            if ( set_reg_entry_gpg( "HomeDir", homedir ) ) {			
                msg_box( dlg, _("Could not save 'HomeDir' in the registry."), _("Preferences"), MB_ERR );
                return FALSE;
            }
            if( !GetDlgItemText( dlg, IDC_GPGPREFS_EXEDIR, exedir, sizeof exedir -1 ) ) {
                msg_box( dlg, _("Please enter where GPG.exe is located."), _("Preferences"), MB_ERR );
                return FALSE;
            }
            if( file_exist_check( exedir ) ) {
                msg_box( dlg, _("Could not find the GPG program in this directory."), _("Preferences"), MB_ERR );
                return FALSE;
            }
            if( set_reg_entry_gpg("gpgProgram", exedir ) ) {
                msg_box( dlg, _("Could not save 'gpgProgram' in the registry"), _("Preferences"), MB_ERR );
                return FALSE;
            }
            if( GetDlgItemText( dlg, IDC_GPGPREFS_LOCALE, locale_dir, sizeof (locale_dir) -1 ) ) {
                if (dir_exist_check (locale_dir)) {
		    log_box( _("Preferences"), MB_ERR, "%s: %s", locale_dir, winpt_strerror (WPTERR_DIR_OPEN));
                    return FALSE;
                }
                set_reg_entry_mo (locale_dir);
                set_gettext_file ("winpt", locale_dir);
            }
	    else
		set_reg_entry_mo ("");

	    p = get_gnupg_cfgfile ();
	    if( !p ) {
		msg_box( dlg, _("Could not get GPG config file"), _("Preferences"), MB_ERR );
		EndDialog( dlg, FALSE );
	    }
	    parse_gpg_options( p, &opt );

	    if( IsDlgButtonChecked( dlg, IDC_GPGPREFS_V3SIGS ) )
		modify_entry( opt, ENTRY_SINGLE, "force-v3-sigs", NULL );
	    else
		delete_option( opt, "force-v3-sigs" );

            if( (n = GetDlgItemText(dlg, IDC_GPGPREFS_COMMENT, t, sizeof t - 1 ) ))
		modify_entry( opt, ENTRY_MULTI, "comment", t );
	    else if( n == 0 )
		modify_entry( opt, ENTRY_MULTI, "comment", "\"\"" );
	    else
		delete_option( opt, "comment" );

            if( (n=GetDlgItemText( dlg, IDC_GPGPREFS_ENCTO, t, sizeof t -1 ) ) )
		modify_entry( opt, ENTRY_MULTI, "encrypt-to", t );
            else
		delete_option( opt, "encrypt-to" );

	    commit_gpg_options( p, opt );
	    release_gpg_options( opt );
            EndDialog( dlg, TRUE );
            return TRUE;
            
	case IDC_GPGPREFS_HOMEDLG:
	    const char * home;

	    home = get_folder_dlg( dlg, _("Choose GPG home directory"), NULL );
	    if( home )
		SetDlgItemText( dlg, IDC_GPGPREFS_HOMEDIR, home );
	    break;

        case IDC_GPGREFS_EXEDLG:
            const char * bin;
            bin = get_filename_dlg( dlg, FILE_OPEN, _("Choose GPG binary"), _("Executable Files (*.exe)\0*.exe\0\0"), NULL );
            if( bin )
                SetDlgItemText( dlg, IDC_GPGPREFS_EXEDIR, bin );
            return TRUE;
            
        case IDC_GPGPREFS_OPTDLG:
            const char * opt;
            opt = get_filename_dlg( dlg, FILE_OPEN, _("Choose GPG config file"), NULL, NULL );
            if( opt )
                SetDlgItemText( dlg, IDC_GPGPREFS_OPTFILE, opt );
            return TRUE;
        }
        break;
    }
    
    return FALSE;
} /* gpgprefs_dlg_proc */
