/* wptGPGParser.cpp - GPG config file parser
 *	Copyright (C) 2002, 2003, 2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <stdio.h>
#include <string.h>
#include <windows.h>

#include "wptTypes.h"
#include "wptW32API.h"
#include "wptGPG.h"
#include "wptErrors.h"

static void
add_opaque_option (gpg_optfile_t opt, const char * line, int used)
{
    gpg_option_t e, t;

    e = new gpg_option_s;
    if (!e)
        BUG (0);
    memset (e, 0, sizeof *e);
    e->used = used > 0? 1 : 0;
    e->name = m_strdup (line);
    e->type = ENTRY_OPAQUE;
    if (!opt->list)
	opt->list = e;
    else {	
	for (t = opt->list; t->next; t = t->next)
	    ;
	t->next = e;
    }
} /* add_opaque_option */


static void
add_single_option (gpg_optfile_t opt, const char * name, int used)
{
    gpg_option_t e, t;

    e = new gpg_option_s;
    if (!e)
	BUG (0);
    memset (e, 0, sizeof *e);
    e->used = used > 0? 1 : 0;
    e->name = m_strdup (name);
    e->type = ENTRY_SINGLE;
    if (!opt->list)
	opt->list = e;
    else {
	for (t = opt->list; t->next; t = t->next)
	    ;
	t->next = e;
    }
} /* add_single_option */


static int
add_group_option( gpg_optfile_t opt, const char *name, int used )
{
    gpg_option_t e, t;

    e = new gpg_option_s;    
    if( !e )
        BUG( NULL );
    memset( e, 0, sizeof *e );
    e->used = used > 0? 1 : 0;
    e->name = m_strdup( name );
    e->type = ENTRY_GROUP;    
    if( !opt->list )
	opt->list = e;
    else {
	for( t = opt->list; t->next; t = t->next )
	    ;
	t->next = e;
    }
    return 0;
} /* add_group_option */


static size_t
count_whitespaces( char *line )
{
    size_t i = 0, ncount = 0;

    if( !strchr( line, ' ' ) )
	return 0;
    for( i = 0; i < strlen( line ); i++ ) {
	if( line[i] == ' ' )
	    ncount++;
    }

    return ncount;
} /* count_whitespaces */


static int
add_multi_option( gpg_optfile_t opt, char *line, int used )
{
    gpg_option_t e ,t;
    char *p;
    int state = 0;

    e = new gpg_option_s;    
    if( !e )
        BUG( NULL );
    memset( e, 0, sizeof *e );
    if( count_whitespaces( line ) == 1 ) {
	while( (p = strsep( &line, " " )) && state != 2 ) {
	    switch( state ) {
	    case 0: e->name = m_strdup( p ); break;
	    case 1: e->val = m_strdup( p );  break;
	    }
	    state++;
	}
    }
    else {
	p = strchr( line, ' ' );
	state = p - line;
	e->name = new char[state + 1];
	if( !e->name )
	    BUG( NULL );
	memcpy( e->name, line, state );
	e->name[state] = '\0';
	strncpy( e->name, line, state );
	e->val = m_strdup( line + state + 1 );
    }
    e->used = used;
    e->type = ENTRY_MULTI;
    if( !opt->list )
	opt->list = e;
    else {
	for( t=opt->list; t->next; t = t->next )
	    ;
	t->next = e;
    }
    return 0;
} /* add_multi_option */


static gpg_group_t
new_group (gpg_group_t * grp, const char * name)
{
    gpg_group_t g, t;

    g = new gpg_group_s;    
    if( !g )
	BUG( NULL );
    memset( g, 0, sizeof *g );
    g->used = 1;
    g->name = m_strdup( name );
    if( !*grp )
	*grp = g;
    else {
	for( t = *grp; t->next; t = t->next )
	    ;
	t->next = g;
    }
    return g;
} /* new_group */


static const char*
_add_group( gpg_optfile_t opt, const char *line )
{
    char *p, *buf, *buf2, *name = NULL;
    gpg_group_t g;
    gpg_member_t m = NULL, t;

    p = strchr( line, '=' );
    if( p == NULL || strncmp( line, "group ", 6 ) )
        return NULL;
    buf = m_strdup( line + (p - line + 1) );
    if( !buf )
	BUG( NULL );    
    line += 6;
    p = strchr( line, '=' );
    if( p ) {
        size_t pos = p - line;
	name = new char[pos + 1];
	if( !name )
	    BUG( NULL );
	memset( name, 0, pos+1 );
        strncpy( name, line, pos );
    }
    g = new_group( &opt->grp, name );
    buf2 = buf;
    while( (p = strsep( &buf2, " " )) && *p ) {
	m = new gpg_member_s;        
        if( !m )
            BUG( NULL );
	memset( m, 0, sizeof *m );
        m->used = 1;
        m->name = m_strdup( p );
	if( !g->list )
	    g->list = m;
        else {
	    for( t=g->list; t->next; t = t->next )
		;
	    t->next = m;
	}
    }
    free_if_alloc( buf );
    if( !m )
        return NULL;
    return g->name;
} /* _add_group */

void
release_group( gpg_group_t grp )
{
    gpg_member_t m, m2;
    
    m = grp->list;
    while( m ) {
        m2 = m->next;
        free_if_alloc( m->name );
        free_if_alloc( m );
        m = m2;
    }
} /* release_group */


void
release_option (gpg_option_t opt)
{
    if (!opt)
	return;
    free_if_alloc( opt->name );
    free_if_alloc( opt->val );
    free_if_alloc( opt );
}


void
release_gpg_options( gpg_optfile_t opt )
{
    gpg_option_t e, e2;
    gpg_group_t g, g2;

    e = opt->list;
    while( e ) {
        e2 = e->next;
        release_option (e);
        e = e2;
    }
    g = opt->grp;
    while( g ) {
        g2 = g->next;
        release_group( g );
        g = g2;
    }
    free_if_alloc( opt );
} /* release_gpg_options */


int
commit_gpg_options (const char * file, gpg_optfile_t opt)
{
    FILE *inp;
    gpg_option_t e;
    gpg_group_t g;
    gpg_member_t m;
    int rc = 0;

    inp = fopen (file, "w+b");
    if( !inp )
        return -1;
    for( e = opt->list; e; e = e->next ) {
        if( !e->used )
            continue;
        switch( e->type ) {
        case ENTRY_OPAQUE:
            fprintf( inp, "%s", e->name );
            break;
            
        case ENTRY_MULTI:
            fprintf( inp, "%s %s\n", e->name, e->val );
            break;

        case ENTRY_SINGLE:
            fprintf( inp, "%s\n", e->name );
            break;

        case ENTRY_GROUP:
            g = find_group( opt, e->name );
            if( g && g->used ) {
                fprintf( inp, "group %s=", g->name );
                for( m = g->list; m; m = m->next ) {
                    if( m->used )
                        fprintf( inp, "%s ", m->name );
                }
                fprintf( inp, "\n" );
            }
            break;
        }
    }
    fclose (inp);
    return 0;
} /* commit_gpg_options */


int
parse_gpg_options( const char *file, gpg_optfile_t *r_opt )
{
    FILE *inp;
    char buf[1024], *p;
    gpg_optfile_t opt = NULL;

    inp = fopen( file, "rb" );
    if( inp == NULL ) {
	inp = fopen( file, "wb" );
	if( inp == NULL )
	    return -1;
    }
    opt = new gpg_optfile_s;
    if( !opt )
	BUG( NULL );
    memset( opt, 0, sizeof *opt );
    while( !feof( inp ) ) {
        p = fgets( buf, sizeof buf -1, inp );
        if( !p )
            break;
        if( *p == '#' || *p == '\r' || *p == '\n' || *p == '\t' ) {
            add_opaque_option( opt, p, 1 );
            continue;
        }
	if( strstr( p, "\r\n" ) )
	    p[strlen( p ) - 2] = '\0';
	else if( strstr( p, "\n" ) )
	    p[strlen( p ) - 1] = '\0';
        if( !strchr( p, ' ' ) )
            add_single_option( opt, p, 1 );
        else if( !strncmp( p, "group", 5 ) ) {
            const char *s = _add_group( opt, p );
            if( s )
                add_group_option( opt, s, 1 );
        }
        else
            add_multi_option( opt, p, 1 );
    }
    fclose( inp );
    if( r_opt )
        *r_opt = opt;
    return 0;
} /* parse_gpg_options */


gpg_option_t
find_option (gpg_optfile_t opt, const char * str)
{
    gpg_option_t e;

    for (e = opt->list; e; e = e->next) {
        if (e->type == ENTRY_OPAQUE)
            continue;
        if (!stricmp (e->name, str))
            break;
    }

    return e;
} /* find_option */


gpg_group_t
find_group( gpg_optfile_t opt, const char *str )
{
    gpg_group_t g;

    for( g = opt->grp; g; g = g->next ) {
        if( !stricmp( g->name, str ) )
            break;
    }

    return g;
} /* find_group */


gpg_member_t
find_member( gpg_optfile_t opt, const char *grp, const char *str )
{
    gpg_group_t g = find_group( opt, grp );
    gpg_member_t m = NULL;
    if( g ) {
        for( m = g->list; m; m = m->next ) {
            if( !stricmp( m->name, str ) )
                break;
        }
    }

    return m;
} /* find_member */


int
delete_group( gpg_optfile_t opt, const char *str )
{
    gpg_group_t g = find_group( opt, str );
    if( g ) {
        g->used = 0;
        return 0;
    }

    return -1;
} /* delete_group */


int
delete_member( gpg_optfile_t opt, const char *grp, const char *str )
{
    gpg_member_t m = find_member( opt, grp, str );
    if( m ) {
        m->used = 0;
        return 0;
    }

    return -1;
} /* delete_member */

    
int
delete_option (gpg_optfile_t opt, const char * str)
{
    gpg_option_t e = find_option (opt, str);
    if (e) {
        e->used = 0;
        return 0;
    }

    return -1;
} /* delete_option */


int
add_entry( gpg_optfile_t opt, int type, const char *name, const char *val )
{
    gpg_option_t e, t;

    e = new gpg_option_s;    
    if( !e )
	BUG( NULL );
    memset( e, 0, sizeof *e );
    e->used = 1;
    e->type = type;
    e->name = m_strdup( name );
    e->val = val? m_strdup( val ) : NULL;
    if( !opt->list )
	opt->list = e;
    else {
	for( t = opt->list; t->next; t = t->next )
	    ;
	t->next = e;
    }
    return 0;
} /* add_entry */


int
modify_entry( gpg_optfile_t opt, int type, const char *name, const char *val )
{
    gpg_option_t e;
    int rc = 0;

    e = find_option( opt, name );
    if( !e )
	rc = add_entry( opt, type, name, val );
    else if( type != ENTRY_SINGLE ) {
	free_if_alloc( e->val );
	e->val = m_strdup( val );
    }

    return rc;
} /* modify_entry */

    
int
add_member( gpg_optfile_t opt, const char *grp, const char *str )
{
    gpg_group_t g = find_group( opt, grp );
    gpg_member_t m, t;
    if( g ) {
	m = new gpg_member_s;        
        if( !m )
            BUG( NULL );
	memset( m, 0, sizeof *m );
        m->used = 1;
        m->name = m_strdup( str );
	if( !m->name )
	    BUG( NULL );
        if( !g->list )
	    g->list = m;
	else {
	    for( t = g->list; t->next; t = t->next )
		;
	    t->next = m;
	}
        return 0;
    }
    return -1;
} /* add_member */


int
add_group( gpg_optfile_t opt, const char *str )
{
    gpg_group_t g, t;

    g = new gpg_group_s;    
    if( !g )
	BUG( NULL );
    memset( g, 0, sizeof *g );
    g->used = 1;
    g->name = m_strdup( str );
    
    if( !opt->grp )
	opt->grp = g;
    else {
	for( t = opt->grp; t->next; t = t->next )
	    ;
	t->next = g;
    }
    add_entry( opt, ENTRY_GROUP, str, NULL );

    return 0;
} /* add_group */
 

#if 0
void
walk_group( const char *name, gpg_member_t mbr )
{
    gpg_member_t m;

    printf( "name=%s\n", name );
    for( m = mbr; m; m = m->next )
        printf( "%s\n", m->name );
}


int
main( int argc, char **argv )
{
    int rc;
    gpg_optfile_t opt;
    gpg_group_t g;
    gpg_option_t e;

    rc = parse_gpg_options( "c:\\gnupg\\gpg.conf", &opt );
    if( rc )
        printf( "parse_option: error\n" );
    else {
#if 0
        /*delete_option( opt, "encrypt-to" );*/
        delete_member( opt, "bar", "richard" );
        add_member( opt, "bar", "leo" );
        add_group( opt, "foobar" );
        delete_group( opt, "foobar" );
        add_entry( opt, ENTRY_SINGLE, "use-agent", NULL );        
#endif
        for( e = opt->list; e; e = e->next ) {
            if( e->type != 1 )
                printf( "%d: %s %s\n", e->used, e->name, e->val );
        }	
	/*commit_gpg_options( "c:\\gnupg\\gpg2.conf", opt );*/	
        for( g = opt->grp; g; g = g->next )
            walk_group( g->name, g->list );        
        release_option( opt );
    }
    
    return 0;
}
#endif