/* wptWipeFile.cpp - Secure file removal
 *	Copyright (C) 2001-2005 Timo Schulz
 *	Copyright (C) 2000 Matt Gauthier
 *
 * WinPT software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 **************************************************************************
 * This code based on the sunlink.c file from the SRM project, but        *
 * it was heavily modified to work with W32 and with added GCRYPT         *
 * support for gathering random bytes.                                    *
 *                                                                        *
 * The original code was placed under the GNU Lesser Public License,      *
 * even so I decide to put this file under the GNU General Public License.*
 **************************************************************************
 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include <direct.h>

#include "openpgp.h"
#include "wptW32API.h"
#include "wptErrors.h"
#include "wptTypes.h"


typedef unsigned __int64 DDWORD;

typedef struct {
    HANDLE fd;
    DDWORD filesize;
    DDWORD offset;
    BYTE  *buffer;
    DWORD  buffsize;
    const char *name;
    int    n_passes;
} wipe_context_s;


void (*progress_cb) (void *, DDWORD, DDWORD);
static void *progress_cb_value = NULL;

void (*unlink_cb)(void *, const char *, int, unsigned, unsigned) = NULL;
static void *unlink_cb_value = NULL;

static void 
overwrite (wipe_context_s * ctx)
{
    DDWORD blocks = 0, mod = 0;
    DWORD nwritten = 0;
    LONG size_high = 0;

    blocks = ctx->filesize / ctx->buffsize;
    mod = ctx->filesize % ctx->buffsize;
    SetFilePointer (ctx->fd, 0, &size_high, FILE_BEGIN);
    while (blocks--) {
	if (!WriteFile (ctx->fd, ctx->buffer, ctx->buffsize, &nwritten, NULL))
	    break;
	ctx->offset += nwritten;
	if (unlink_cb)
	    unlink_cb (unlink_cb_value, ctx->name, 0, (unsigned)ctx->offset, 
		       (unsigned)ctx->filesize*ctx->n_passes);
    }
    if (mod) {
	WriteFile (ctx->fd, ctx->buffer, (DWORD)mod, &nwritten, NULL);
	ctx->offset += nwritten;
	if (unlink_cb)
	    unlink_cb (unlink_cb_value, ctx->name, 0, (unsigned)ctx->offset, 
		       (unsigned)ctx->filesize*ctx->n_passes);
    }
    FlushFileBuffers (ctx->fd);
    SetFilePointer (ctx->fd, 0, &size_high, FILE_BEGIN);
} /* overwrite */


static void
randomize_buffer (byte * buf, size_t bufsize, int level)
{
    const int blocksize = 512;
    int blocks = bufsize / blocksize;
    int mod = bufsize % blocksize;
    
    while (blocks--) {
	gpg_randomize (buf, blocksize, level);
	buf += blocksize;
    }
    if (mod)
	gpg_randomize (buf, mod, level);
} /* randomize_buffer */


static void 
overwrite_random (int npasses, wipe_context_s * ctx) 
{  	
    int i;
    
    for (i = 0; i < npasses; i++) {
	randomize_buffer (ctx->buffer, ctx->buffsize, 0);
	overwrite (ctx);
    }
} /* overwrite_random */


static void 
overwrite_byte (int byte, wipe_context_s * ctx) 
{
    memset (ctx->buffer, byte, ctx->buffsize);
    overwrite (ctx);
} /* overwrite_byte */


static void
overwrite_bytes (int byte1, int byte2, int byte3, wipe_context_s * ctx)
{
    DWORD i;

    memset (ctx->buffer, byte1, ctx->buffsize);
    for (i = 1; i < ctx->buffsize; i += 3) {
	ctx->buffer[i] = byte2;
	ctx->buffer[i+1] = byte3;
    }
    overwrite (ctx);
} /* overwrite_bytes */


/**
 * For the case the file is not a regular file (this is true for
 * devices or directories) this function tries to rename the file
 * to random pattern and then it will be delete (without random!). 
 **/
int 
rename_unlink (const char * path)
{
    struct stat statbuf;
    char * new_name = NULL, * p = NULL, c;  
    int i = 0, rc = 0;
    int is_dir = 0;

    if (GetFileAttributes (path) & FILE_ATTRIBUTE_DIRECTORY)
	is_dir = 1;
    
    new_name = new char[strlen (path)+15];
    if (!new_name)
	BUG (0);
    
    strcpy (new_name, path);
    p = strrchr (new_name, '\\');
    if (p != NULL) {
	p++;
	*p = '\0';
    } 
    else
	p = new_name;
    do {
	while (i < 14) {
	    c = gpg_random_char (1);
	    *p = c; 
	    p++; 
	    i++;
	}
	*p = '\0';
    } while (stat (new_name, &statbuf) == 0);
    
    if (rename (path, new_name) == -1) {
	rc = WPTERR_FILE_READ;
	goto leave;
    }
    if (is_dir && RemoveDirectory (new_name) == FALSE)
	rc = WPTERR_FILE_REMOVE;
    else if (!DeleteFile (new_name))
	rc = WPTERR_FILE_REMOVE;
    
leave:
    free_if_alloc (new_name);
    return rc;
} /* rename_unlink */


static __int64
GetFileSize64 (const char * path)
{
    FILE *fp = fopen (path, "r");
    if (fp) {
	struct _stati64 statbuf;
	_fstati64 (fileno (fp), &statbuf);
	fclose (fp);
	return statbuf.st_size;
    }
    return -1;
} /* GetFileSize64 */


static int
_secure_unlink (const char * path, int mode, HANDLE *r_fd)
{
    wipe_context_s ctx;
    LONG size_high = 0;
    
    if (GetFileAttributes (path) & FILE_ATTRIBUTE_DIRECTORY)
	return rename_unlink (path);
    
    memset (&ctx, 0, sizeof (ctx));
    ctx.name = path;
    ctx.buffsize = 16384;
    ctx.buffer = new byte[ctx.buffsize];
    if (!ctx.buffer)
	BUG (0);
    
    ctx.filesize = GetFileSize64 (path);
    if (!ctx.filesize) {
	free_if_alloc (ctx.buffer);
	unlink (path);
	return 0;
    }

    ctx.fd = CreateFile (path, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, 
			 OPEN_ALWAYS, 0, NULL);
    if (ctx.fd == INVALID_HANDLE_VALUE) {
	free_if_alloc (ctx.buffer);
	return WPTERR_FILE_OPEN;
    }
    else if (r_fd)
	*r_fd = ctx.fd;
    
    gpg_quick_random_gen (1);
    if (unlink_cb)
	unlink_cb (unlink_cb_value, ctx.name, 0, 0, 0);

    switch (mode) {
    case WIPE_MODE_FAST:
	ctx.n_passes = 1;
	overwrite_random (1, &ctx);
	break;

    case WIPE_MODE_SIMPLE:
	ctx.n_passes = 2;
	overwrite_random (2, &ctx);
	break;
        
    case WIPE_MODE_DOD:
	ctx.n_passes = 5;
	overwrite_random (1, &ctx);
	overwrite_byte ((~1) & 0xFF, &ctx);
	overwrite_random (1, &ctx);
	overwrite_byte ((~4) & 0xFF, &ctx);
	overwrite_random (1, &ctx);
	break;
        
    case WIPE_MODE_GUTMANN:
	ctx.n_passes = 39;
	overwrite_random (4, &ctx);
	overwrite_byte( 0x55, &ctx );
	overwrite_byte ( 0xAA, &ctx );
	overwrite_bytes( 0x92, 0x49, 0x24, &ctx );
	overwrite_bytes( 0x49, 0x24, 0x92, &ctx );
	overwrite_bytes( 0x24, 0x92, 0x49, &ctx );
	overwrite_byte( 0x00, &ctx );
	overwrite_byte( 0x11, &ctx );
	overwrite_byte( 0x22, &ctx );
	overwrite_byte( 0x33, &ctx );
	overwrite_byte( 0x44, &ctx );
	overwrite_byte( 0x55, &ctx );
	overwrite_byte( 0x66, &ctx );
	overwrite_byte( 0x77, &ctx );
	overwrite_byte( 0x88, &ctx );
	overwrite_byte( 0x99, &ctx );
	overwrite_byte( 0xAA, &ctx );
	overwrite_byte( 0xBB, &ctx );
	overwrite_byte( 0xCC, &ctx );
	overwrite_byte( 0xDD, &ctx );
	overwrite_byte( 0xEE, &ctx );
	overwrite_byte( 0xFF, &ctx );
	overwrite_bytes( 0x92, 0x49, 0x24, &ctx );
	overwrite_bytes( 0x49, 0x24, 0x92, &ctx );
	overwrite_bytes( 0x24, 0x92, 0x49, &ctx );
	overwrite_bytes( 0x6D, 0xB6, 0xDB, &ctx );
	overwrite_bytes( 0xB6, 0xDB, 0x6D, &ctx );
	overwrite_bytes( 0xDB, 0x6D, 0xB6, &ctx );
	overwrite_random( 4, &ctx );
	break;
    }
        
    /* Set file length to zero so allocated clusters cannot be trailed */	
    SetFilePointer (ctx.fd, 0, &size_high, FILE_BEGIN);
    SetEndOfFile (ctx.fd);
    CloseHandle (ctx.fd);
    
    memset (ctx.buffer, 0, ctx.buffsize); /* burn the last evidence */	
    free_if_alloc (ctx.buffer);

    return rename_unlink (path);
}

int 
secure_unlink (const char *path, int mode)
{
    return _secure_unlink (path, mode, NULL);
} /* secure_unlink */



void
secure_unlink_set_cb (void (*cb)(void *, const char *, int, unsigned, unsigned), 
		      void *cb_value)
{
    unlink_cb = cb;
    unlink_cb_value = cb_value;
} /* secure_unlink_set_cb */


/* Windows 98 - Q188074 */
#define REGISTRY_FILESYSTEM "System\\CurrentControlSet\\Control\\FileSystem"
#define REGISTRY_LOWDISKSPACE "DisableLowDiskSpaceBroadcast"


/* disables the annoying warning Windows 98 displays when disk space is low */
static void
handle_lowdiskspace_notify( const char * drive, int disable )
{    
    OSVERSIONINFO ov;
    HKEY key;
    DWORD n;

    memset( &ov, 0, sizeof ov );
    ov.dwOSVersionInfoSize = sizeof ov;
    GetVersionEx( &ov );
    if( ov.dwPlatformId == VER_PLATFORM_WIN32_NT )
	return;

    if( disable ) {
	unsigned newi = (1 << (toupper((unsigned)drive) - (unsigned)'A'));
	if( RegOpenKey( HKEY_LOCAL_MACHINE, REGISTRY_FILESYSTEM, &key ) ) {
	    n = sizeof newi;
	    RegSetValue( key, REGISTRY_LOWDISKSPACE, REG_DWORD, (LPCTSTR)newi, n );
	    RegCloseKey( key );
        }
    }
    else {
	if( RegOpenKey( HKEY_LOCAL_MACHINE, REGISTRY_FILESYSTEM, &key ) ) {
	    RegDeleteKey( key, REGISTRY_LOWDISKSPACE );
	    RegCloseKey( key );
	}
    }
} /* handle_lowdiskspace_notify */


int
wipe_freespace (const char * drive, HANDLE *r_fd,
	        void (*cb)(void *, DDWORD, DDWORD), void * cb_value)
{
    ULARGE_INTEGER caller, total, frees;
    LONG hpart=0;
    HANDLE fd;
    int disktyp = GetDriveType (drive);
    int rc;
    char * file;

    if (disktyp != DRIVE_FIXED && disktyp != DRIVE_REMOVABLE)
	return -1;
    if (!GetDiskFreeSpaceEx (drive, &caller, &total, &frees))
	return -1;
    handle_lowdiskspace_notify (drive, 1);

    /* disk is full */
    if (!frees.LowPart)
	return 0;
    file = new char[strlen (drive)+8];
    if (!file)
	BUG (0);
    sprintf (file, "%stemp_winpt.tmp", drive);
    fd = CreateFile (file,
		    GENERIC_READ|GENERIC_WRITE,
		    FILE_SHARE_READ|FILE_SHARE_WRITE,
		    NULL, CREATE_ALWAYS, 0, NULL);
    if (fd == INVALID_HANDLE_VALUE) {
	free_if_alloc (file);
	return WPTERR_FILE_OPEN;
    }
    hpart = frees.HighPart;
    SetFilePointer (fd, frees.LowPart, &hpart, FILE_BEGIN);
    SetEndOfFile (fd);
    CloseHandle (fd);

    if (cb && cb_value) {
	progress_cb = cb;
	progress_cb_value = cb_value;
    }
    rc = _secure_unlink (file, WIPE_MODE_FAST, r_fd);

    handle_lowdiskspace_notify (drive, 0);
    free_if_alloc (file);
    return rc;
} /* wipe_freespace */
