/* wptJPG.cpp : JPG picture class
 *	Copyright (C) 2001 Dr.Yovav Gad <Sources@SuperMain.com>
 *	Copyright (C) 2005 Timo Schulz
 *
 * This file is part of PTD.
 *
 * PTD is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * PTD is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PTD; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

/*-----------------------------------------------------------------------------
 * Picture (Implementations) Version 1.00
 *
 * Routinges for showing JPG pictur files
 *
 * Author: Dr. Yovav Gad, EMail: Sources@SuperMain.com ,Web: www.SuperMain.com
 * 
 * This version uses a stripped down version of Picture.cpp and Picture.h.
 */
#include <windows.h>
#include <ocidl.h>
#include <olectl.h>

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>

#include "wptJPG.h"

#define HIMETRIC_INCH 2540
#define ERROR_TITLE "CJPG Error" /* Error Title (Related To This Class)...*/


CJPG::CJPG ()
{
    m_IPicture = NULL;
    m_Height = 0;
    m_Weight = 0;
    m_Width = 0;
}


CJPG::~CJPG ()
{
    if (m_IPicture != NULL)
	FreePictureData (); 
}



/* Free The Allocated Memory That Holdes The IPicture Interface Data
   And Clear Picture Information. */
void CJPG::FreePictureData()
{
    if (m_IPicture != NULL) {
	m_IPicture->Release();
	m_IPicture = NULL;
	m_Height = 0;
	m_Weight = 0;
	m_Width = 0;	
    }
}


/* Open a JPG File And Load It Into IPicture (Interface) */
BOOL CJPG::Load(LPCSTR sFilePathName)
{
    BOOL bResult = FALSE;
    FILE * f;
    int nSize = 0;

    if (m_IPicture != NULL) 
	FreePictureData ();

    f = fopen (sFilePathName, "rb");
    if (f) {
	struct stat st;
	fstat (fileno (f), &st);
	nSize = st.st_size;
	BYTE* pBuffer = new BYTE[nSize];
	if (fread(pBuffer, 1, nSize, f) > 0) {
	    if (LoadPictureData (pBuffer, nSize))
		bResult = TRUE;
	}
	fclose (f);
	delete [] pBuffer;
    }
    else { /* Open Failed... */
	MessageBox (NULL, strerror (errno), ERROR_TITLE, MB_OK | MB_ICONSTOP);
	bResult = FALSE;	
    }

    m_Weight = nSize; /* Update Picture Size Info... */

    if(m_IPicture != NULL) { /* Do Not Try To Read From Memory That Is Not Exist... */
	m_IPicture->get_Height (&m_Height);
	m_IPicture->get_Width (&m_Width);
	/* Calculate Its Size On a "Standard" (96 DPI) Device Context */
	m_Height = MulDiv (m_Height, 96, HIMETRIC_INCH);
	m_Width  = MulDiv (m_Width,  96, HIMETRIC_INCH);	
    }
    else { /* Picture Data Is Not a Known Picture Type */
	m_Height = 0;
	m_Width = 0;
	bResult = FALSE;	
    }
    return (bResult);
}



/* Read The Picture Data From a Source (File / Resource)
   And Load It Into The Current IPicture Object In Use */
BOOL CJPG::LoadPictureData (BYTE *pBuffer, int nSize)

{
    BOOL bResult = FALSE;
    HGLOBAL hGlobal;
    void* pData;
    IStream* pStream = NULL;
    
    hGlobal = GlobalAlloc (GMEM_MOVEABLE, nSize);
    if (hGlobal == NULL) {
	MessageBox (NULL, "Can not allocate enough memory", ERROR_TITLE, MB_OK | MB_ICONSTOP);
	return (FALSE);
    }

    pData = GlobalLock (hGlobal);
    memcpy (pData, pBuffer, nSize);
    GlobalUnlock (hGlobal);

    if (CreateStreamOnHGlobal (hGlobal, TRUE, &pStream) == S_OK) {
	HRESULT hr;
	if ((hr = OleLoadPicture (pStream, nSize, FALSE, IID_IPicture, (LPVOID *)&m_IPicture)) == E_NOINTERFACE) {
	    MessageBox (NULL, "IPicture interface is not supported\t", ERROR_TITLE, MB_OK | MB_ICONSTOP);
	    return(FALSE);
	}
	else { // S_OK
	    pStream->Release ();
	    pStream = NULL;
	    bResult = TRUE;
	}	
    }

    GlobalFree (hGlobal);
    return (bResult);
}


/* Draw The Loaded Picture Direct To The Client DC */
BOOL CJPG::Show (HDC pDC, POINT *LeftTop, POINT *WidthHeight, int MagnifyX, int MagnifyY)

{
    if (pDC == NULL || m_IPicture == NULL) 
	return FALSE;
    
    long Width  = 0;
    long Height = 0;
    m_IPicture->get_Width (&Width);
    m_IPicture->get_Height (&Height);
	
    if (MagnifyX == NULL) 
	MagnifyX = 0;
    if (MagnifyY == NULL) 
	MagnifyY = 0;
    MagnifyX = int(MulDiv (Width, GetDeviceCaps(pDC, LOGPIXELSX), HIMETRIC_INCH) * MagnifyX);
    MagnifyY = int(MulDiv (Height, GetDeviceCaps(pDC, LOGPIXELSY), HIMETRIC_INCH) * MagnifyY);
    
    RECT DrawRect;
    DrawRect.left = LeftTop->x;
    DrawRect.top = LeftTop->y;
    DrawRect.right = MagnifyX;
    DrawRect.bottom = MagnifyY;

    HRESULT hrP = NULL;

    hrP = m_IPicture->Render (pDC,
                      LeftTop->x,               // Left
                      LeftTop->y,               // Top
                      WidthHeight->x +MagnifyX, // Width
                      WidthHeight->y +MagnifyY, // Height
                      0,
                      Height,
                      Width,
                      -Height,
                      &DrawRect);

    if (SUCCEEDED (hrP)) 
	return (TRUE);

    MessageBox (NULL, "Can not allocate enough memory", ERROR_TITLE, MB_OK | MB_ICONSTOP);
    return (FALSE);
}


/* Get The Original Picture Pixel Size (Ignor What Current DC Is Using)
   Pointer To a Device Context Is Needed For Pixel Calculation, */
BOOL CJPG::UpdateSizeOnDC(HDC pDC)

{	
    if(pDC == NULL || m_IPicture == NULL) { 
	m_Height = 0; 
	m_Width = 0; 
	return (FALSE); 
    }

    m_IPicture->get_Height (&m_Height);
    m_IPicture->get_Width (&m_Width);

    // Get Current DPI - Dot Per Inch
    int CurrentDPI_X = GetDeviceCaps (pDC, LOGPIXELSX);
    int CurrentDPI_Y = GetDeviceCaps (pDC, LOGPIXELSY);

    m_Height = MulDiv (m_Height, CurrentDPI_Y, HIMETRIC_INCH);
    m_Width  = MulDiv (m_Width,  CurrentDPI_X, HIMETRIC_INCH);

    return (TRUE);
}


int
PTD_jpg_show (HWND hwnd, POINT *p, LPCSTR name)
{
    CJPG jpg;
    HDC hdc;
    POINT p2;
    BOOL rc;

    rc = jpg.Load (name);
    if (!rc)
	return -1;
    hdc = GetWindowDC (hwnd);
    rc = jpg.UpdateSizeOnDC (hdc);
    if (!rc) {
	ReleaseDC (hwnd, hdc);
	return -2;
    }

    p2.x = jpg.m_Width;
    p2.y = jpg.m_Height;
    rc = jpg.Show (hdc, p, &p2, 0, 0);

    ReleaseDC (hwnd, hdc);
    jpg.FreePictureData ();
    return rc;
}
