/* decrypt.c -  decrypt functions
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <windows.h>

#include "util.h"
#include "context.h"
#include "ops.h"

struct decrypt_result_s {
    gpgme_sig_t sig;
    int no_data;
    int no_seckey;
    int no_passphrase;
    int idea_cipher;
    int okay;
    int sig_okay;
    int failed;
    int bad_mdc;
    int bad_armor;
    int bad_passphrase;
    int key_algo;
    int file_start;
    int file_done;
    char keyid[16+1];
    char cardno[32+1];
    void * last_pw_handle;
    char * userid_hint;
    char * passphrase_info;
    char * file_name;    
};


void
gpgme_decrypt_get_sig_ctx (gpgme_ctx_t ctx, gpgme_sig_t * r_sig)
{
    if (!ctx)
	return;
    if (r_sig)
	*r_sig = NULL;
    if (ctx && ctx->result.decrypt->sig) {
	_gpgme_sig_ref (ctx->result.decrypt->sig);
	if (r_sig)
	    *r_sig = ctx->result.decrypt->sig;
    }
} /* gpgme_decrypt_get_sig_ctx */


void
gpgme_decrypt_get_status( gpgme_ctx_t ctx, char * keyid, 
			  gpgme_op_flags_t * r_flags )
{
    gpgme_op_flags_t flags = 0;
    _decrypt_result_t res;    

    if( !keyid )
	return;
    if( ctx->result_type != RESULT_TYPE_DECRYPT )
	return;
    if( !ctx->result.decrypt->okay
	&& ctx->result.decrypt->no_seckey ) {
	strcpy( keyid, ctx->result.decrypt->keyid );
	flags |= GPGME_OPFLAG_NOSECKEY;	
    }
    res = ctx->result.decrypt;	
    if (res->bad_armor)    
	flags |= GPGME_OPFLAG_BADARMOR;
    if (res->bad_mdc)
	flags |= GPGME_OPFLAG_BADMDC;
    if (r_flags)
	*r_flags = flags;
} /* gpgme_decrypt_get_status */


void
_gpgme_release_decrypt_result( _decrypt_result_t res )
{
    if( res ) {
        safe_free( res->passphrase_info );
        safe_free( res->userid_hint );
	gpgme_sig_release( res->sig );
        safe_free( res );
    }
} /* _gpgme_release_decrypt_result */


static gpgme_error_t
create_result_struct (gpgme_ctx_t ctx)
{
    assert( !ctx->result.decrypt );
    ctx->result.decrypt = calloc( 1, sizeof * ctx->result.decrypt );
    if( !ctx->result.decrypt )
        return mk_error( Out_Of_Core );
    ctx->result_type = RESULT_TYPE_DECRYPT;
    return 0;
} /* create_result_struct */


static void
list_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char * args )
{
    char keyid[32] = {0};

    if (ctx->out_of_core)
	return;
    if (code == STATUS_ENC_TO) {
	keyid[0] = (char)atol (args+17);
	strncpy (keyid+1, args, 16);
	gpgme_recipients_add_name (ctx->enc_to, keyid);
    }
} /* list_status_handler */


static gpgme_sig_t
add_signature( gpgme_ctx_t ctx )
{
    gpgme_sig_t sig, s;
    gpgme_error_t err;
    
    err = gpgme_sig_new( &sig );
    if( err ) {
	ctx->out_of_core = 1;
	return NULL;
    }
    if( ctx->result.decrypt->file_name ) {
	sig->file_name = strdup( ctx->result.decrypt->file_name );
	if( !sig->file_name ) {
	    ctx->out_of_core = 1;
	    safe_free( sig );
	    return NULL;
	}
    }
    if( !ctx->result.decrypt->sig )
	ctx->result.decrypt->sig = sig;    
    else {
	for( s = ctx->result.decrypt->sig; s->next; s=s->next )
	    ;
	s->next = sig;
    }
    return sig;
} /* add_signature */


static void
decrypt_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char *args )
{   
    static gpgme_sig_t sig = NULL;
    static char keyid[16+1];
    char *p = NULL, fpr[40+1];
    int i = 0, j=0;
    char ch = 0;
    
    if (ctx->out_of_core)
        return;
        
    if (ctx->result_type == RESULT_TYPE_NONE) {
        if (create_result_struct (ctx)) {
            ctx->out_of_core = 1;
            return;
        }
    }

    assert (ctx->result_type == RESULT_TYPE_DECRYPT);

    if (code == STATUS_GOODSIG || code == STATUS_REVKEYSIG 
	|| code == STATUS_EXPKEYSIG || code == STATUS_BADSIG 
	|| code == STATUS_ERRSIG) {
        sig = add_signature (ctx);
        if (!sig || ctx->out_of_core)
            return;
    }

    _gpgme_nodata_status_handler (code, args, &ctx->result.decrypt->no_data);
    _gpgme_pass_status_handler (code, args, &ctx->result.decrypt->bad_passphrase,
				&ctx->result.decrypt->no_data, 
				&ctx->result.decrypt->passphrase_info);
    if (sig)
	_gpgme_sigtrust_status_handler (code, args, &sig->trust);

    switch( code ) {
    case STATUS_EOF:
        break;
        
    case STATUS_USERID_HINT:
        safe_free( ctx->result.decrypt->userid_hint );
        p = ctx->result.decrypt->userid_hint = strdup( args );
	if (!p) {
            ctx->out_of_core = 1;
	    return;
	}
        break;
    
    case STATUS_DECRYPTION_OKAY:
        ctx->result.decrypt->okay = 1;
        break;
        
    case STATUS_DECRYPTION_FAILED:
        ctx->result.decrypt->failed = 1;
        break;

    case STATUS_RSA_OR_IDEA:
	ctx->result.decrypt->idea_cipher = 1;
	break;
        
    case STATUS_SIG_ID:
        DEBUG0( "Plaintext was signed!\n" );
        break;
        
    case STATUS_NO_SECKEY:
        ctx->result.decrypt->no_seckey++;
	strncpy( ctx->result.decrypt->keyid, args, 16 );
        break;
        
    case STATUS_NO_PUBKEY:
        sig->sigstat = GPGME_SIG_STAT_NOKEY;
        break;
        
    case STATUS_VALIDSIG:
        p = fpr;
        for( i = 0; i < DIM(fpr) && args[i] && args[i] != ' ' ; i++ )
            *p++ = args[i];
        *p = 0;
        /* skip the formatted date */
        while ( args[i] && args[i] == ' ')
            i++;
        while ( args[i] && args[i] != ' ')
            i++;
        /* and get the timestamp */
	sig->created = strtoul( args + i, NULL, 10 );
	ctx->result.decrypt->sig_okay = 1;
        break;
        
    case STATUS_GOODSIG:
    case STATUS_BADSIG:
	if( code == STATUS_GOODSIG )
	    sig->sigstat = GPGME_SIG_STAT_GOOD;
	else
	    sig->sigstat = GPGME_SIG_STAT_BAD;
	strncpy( sig->id, args, 16 );
        sig->user_id = p = calloc (1, strlen (args + 16) + 2);
	if (!p) 
	{
	    ctx->out_of_core = 1;
	    return;
	}
	strcpy (p, args + 16);
        break;

    case STATUS_EXPKEYSIG:
    case STATUS_REVKEYSIG:
	if( code == STATUS_EXPKEYSIG )
	    sig->sigstat = GPGME_SIG_STAT_E_GOOD;
	else if( code == STATUS_REVKEYSIG )
	    sig->sigstat = GPGME_SIG_STAT_R_GOOD;
	for( i=0, p=sig->id; args[i] && args[i] != ' '; i++ )
	    *p++ = args[i];
	*p = '\0';
	p = sig->user_id = calloc( 1, strlen( args+i ) + 2 );
	if( !p ) {
	    ctx->out_of_core = 1;
	    return;
	}
	j=0;
	while( args[i] )
	    p[j++] = args[i++];
	p[j++] = '\0';
	break;
        
    case STATUS_ERRSIG:
	sig->sigstat = GPGME_SIG_STAT_ERROR;        
        break;

    case STATUS_BADMDC:
	ctx->result.decrypt->bad_mdc = 1;
	break;

    case STATUS_BADARMOR:
	ctx->result.decrypt->bad_armor = 1;
	break;

    case STATUS_FILE_START:
	if (*args == '3') {
	    safe_free (ctx->result.decrypt->file_name);
	    p = ctx->result.decrypt->file_name = strdup (args+2);
	    if( !p ) {
		ctx->out_of_core = 1;
		return;
	    }
	}
	ctx->result.decrypt->file_start++;
	if (ctx->cb.interactiv)
	    ctx->cb.interactiv (ctx->cb.interactiv_value, code, NULL, args+2);
	break;

    case STATUS_FILE_DONE:
	ctx->result.decrypt->file_done++;
	if( ctx->cb.interactiv )
	    ctx->cb.interactiv( ctx->cb.interactiv_value, code, NULL, NULL );
	break;

    case STATUS_CARDCTRL:
	ch = args[i++];
	if (ch == '4') {
	    ctx->result.decrypt->no_seckey = -1;
	    break;
	}
	if (ch != '3')
	    break;
	i++;
	p = ctx->result.decrypt->cardno;
	for (; i-1 < DIM (ctx->result.decrypt->cardno) && args[i]; i++)
	    *p++ = args[i];
	*p = 0;
	break;
        
    default:
        break; /* ignore all other codes */
    }
} /* decrypt_status_handler */


static const char *
decrypt_command_handler (void * opaque, gpg_status_code_t code, const char * key)
{
    gpgme_ctx_t c = opaque;
    
    if (!code) {
        /* We have been called for cleanup */
        if (c->cb.passphrase) {
            c->cb.passphrase (c->cb.passphrase_value, NULL, 
                              &c->result.decrypt->last_pw_handle);
        }
        return NULL;
    }
    
    if (!key || !c->cb.passphrase)
        return NULL;

    if (c->result_type == RESULT_TYPE_NONE) {
        if (create_result_struct (c)) {
            c->out_of_core = 1;
            return NULL;
        }
    }
    
    if( code == STATUS_GET_HIDDEN 
	&& (!strcmp( key, "passphrase.enter" ) 
	 || !strcmp( key, "passphrase.pin.ask" )) ) {
        const char * userid_hint = c->result.decrypt->userid_hint;
        const char * passphrase_info = c->result.decrypt->passphrase_info;
	const char * cardno = c->result.decrypt->cardno;
        int bad_passphrase = c->result.decrypt->bad_passphrase;
	int is_card=0;
        char * buf;
        const char *s;
        
        c->result.decrypt->bad_passphrase = 0;
	is_card = !strcmp (key, "passphrase.pin.ask");
        if (!userid_hint)
            userid_hint = "[User ID hint missing]";
        if (!passphrase_info)
            passphrase_info = "[passphrase info missing]";
        buf = malloc (20 + strlen (userid_hint)
                        + strlen (passphrase_info) + 3);
        if( !buf ) {
            c->out_of_core = 1;
            return NULL;
        }
        sprintf (buf, "%s\n%s\n%s", bad_passphrase? "TRY_AGAIN":"ENTER_PASSPHRASE",
                 userid_hint, passphrase_info);
        s = c->cb.passphrase (c->cb.passphrase_value, is_card? cardno : buf,
			      &c->result.decrypt->last_pw_handle);
        safe_free (buf);
        return s;
    }
    else if( (code == STATUS_GET_BOOL 
	 && !strcmp( key, "openfile.overwrite.okay" ))
	 || (code == STATUS_GET_LINE && !strcmp( key, "openfile.askoutname" )) )
	 if (c->cb.interactiv)
	    return c->cb.interactiv( c->cb.interactiv_value, code, key, NULL );

    return NULL;
} /* decrypt_command_handler */


static gpgme_error_t
list_keys_start( gpgme_ctx_t ctx, gpgme_data_t ciph, const char * file,
		 gpgme_recipients_t * r_keys )
{
    gpgme_error_t rc;
    gpgme_recipients_t keys;
    FILE * fp;
    const char * s;
    char * p;    
    
    if (!r_keys)
	return mk_error (Invalid_Value);
    if (ciph && file || !ciph && !file)
	return mk_error (Invalid_Mode);

    *r_keys = NULL;
    fail_on_pending_request( ctx );
    ctx->pending = 1;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )	
	return rc;

    if( ciph ) {
	p = _gpgme_data_get_as_string( ciph );
	if( !p )
	    return mk_error( Out_Of_Core );

	s = _gpgme_get_tmpfile( 0 );
	fp = fopen( s, "wb" );
	if( !fp ) {
	    safe_free( p );
	    return mk_error( File_Error );
	}
	fwrite (p, 1, strlen (p), fp);
	fclose (fp);
	safe_free (p);
    }
    else
	s = file;

    rc = gpgme_recipients_new( &keys );
    if( rc )
	return rc;
    ctx->enc_to = keys;

    _gpgme_gpg_set_status_handler( ctx->gpg, list_status_handler, ctx );
    _gpgme_gpg_add_arg( ctx->gpg, "--list-only" );
    _gpgme_gpg_add_arg( ctx->gpg, s );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
	ctx->pending = 0;
	_gpgme_gpg_release( &ctx->gpg );
	gpgme_recipients_release( keys ); keys = NULL;
    }
    *r_keys = keys;
    return rc;
} /* list_keys_start */


/* It either works with a file or a data object but not with both! */
gpgme_error_t
gpgme_op_list_keys( gpgme_data_t ciph, const char * file, 
		    gpgme_recipients_t * r_rset )
{    
    gpgme_ctx_t ctx = NULL;
    gpgme_error_t err;

    if( !r_rset )
	return mk_error (Invalid_Value);
    *r_rset = NULL;
    err = gpgme_new( &ctx );
    if( !err )
	err = list_keys_start( ctx, ciph, file, r_rset );
    if( !err ) {
	gpgme_wait( ctx, 1 );
	ctx->pending = 0;
    }
    gpgme_release( ctx );
    return err;
} /* gpgme_op_list_keys */


static gpgme_error_t
file_decrypt_start( gpgme_ctx_t ctx, const char ** input, size_t nfiles,
		    const char * output )
{
    gpgme_error_t rc;
    
    if( !input )
	return mk_error( Invalid_Value );

    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        return rc;
        
    _gpgme_gpg_set_status_handler( ctx->gpg, decrypt_status_handler, ctx );
    if( !ctx->use_pass_fd )
        _gpgme_gpg_set_command_handler( ctx->gpg, decrypt_command_handler, ctx );
    else {
        rc = _gpgme_add_passphrase( ctx );
        if( rc ) {
            _gpgme_gpg_release( &ctx->gpg );
            return rc;
        }
    }
    
    if( !ctx->cb.interactiv )
	_gpgme_gpg_add_arg( ctx->gpg, "--yes" );
    if( nfiles > 1 || !output )
	_gpgme_gpg_add_arg( ctx->gpg, "--no-mangle-dos-filenames" );
    if( ctx->pipemode || nfiles > 1 )
	_gpgme_gpg_add_arg( ctx->gpg, "--decrypt-files" );
    else
	_gpgme_gpg_add_arg( ctx->gpg, "--decrypt" );
    
    /* cannot use --output with --decrypt-files */
    if( nfiles == 1 && !ctx->pipemode && output ) {
        _gpgme_gpg_add_arg( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg( ctx->gpg, output );
    }

    while( nfiles-- )
	_gpgme_gpg_add_arg( ctx->gpg, *input++ );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }	
    
    return rc;
} /* file_decrypt_start */


static gpgme_error_t
decrypt_start( gpgme_ctx_t ctx, gpgme_data_t ciph, gpgme_data_t plain )
{
    int rc = 0;
    
    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    _gpgme_release_result( ctx );
    ctx->out_of_core = 0;
    
    /* create a process object */
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;
    
    _gpgme_gpg_set_status_handler( ctx->gpg, decrypt_status_handler, ctx );
    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    if( ctx->cb.passphrase ) {
        rc = _gpgme_gpg_set_command_handler( ctx->gpg, decrypt_command_handler, ctx );
        if ( rc )
            goto leave;
    }
    else if( ctx->passphrase_value ) {
        rc = _gpgme_add_passphrase( ctx );
        if( rc )
            goto leave;
    }
    
    /* build the commandline */
    _gpgme_gpg_add_arg( ctx->gpg, "--decrypt" );
    /* Check the supplied data */
    if( !ciph || gpgme_data_get_type( ciph ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    _gpgme_data_set_mode( ciph, GPGME_DATA_MODE_OUT );
    if( gpgme_data_get_type( plain ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( Invalid_Value );
        goto leave;
    }
    _gpgme_data_set_mode( plain, GPGME_DATA_MODE_IN );
    
    /* Tell the gpg object about the data */
    _gpgme_gpg_add_arg ( ctx->gpg, "--output" );
    _gpgme_gpg_add_arg ( ctx->gpg, "-" );
    _gpgme_gpg_add_data( ctx->gpg, plain, 1 );
    _gpgme_gpg_add_data( ctx->gpg, ciph, 0 );
    
    /* and kick off the process */
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }
    return rc;
} /* decrypt_start */


static gpgme_error_t
get_decrypt_result( gpgme_ctx_t ctx )
{
    gpgme_error_t err;
    struct decrypt_result_s * res;

    assert( ctx->result.decrypt );
    res = ctx->result.decrypt;
    if( ctx->result_type != RESULT_TYPE_DECRYPT )
	err = mk_error( General_Error );
    else if (res->okay || res->sig_okay)
	err = 0;
    else if( ctx->out_of_core )
        err = mk_error( Out_Of_Core );
    else if( res->no_passphrase )
	err = mk_error( No_Passphrase );
    else if( res->bad_passphrase )
	err = mk_error( Bad_Passphrase );
    else if( res->no_seckey)
	err = mk_error( No_Seckey );
    else if( res->idea_cipher )
	err = mk_error( Cipher_IDEA );
    else if( res->failed || (res->file_start !=  res->file_done) )
	err = mk_error( Decryption_Failed );
    else if( !res->okay || !res->no_data )
	err = mk_error( No_Data );
    else if( gpgme_get_process_rc( ctx ) )
	err = mk_error( Internal_GPG_Problem );
    return err;
} /* get_decrypt_result */


/**
 * gpgme_op_decrypt:
 * @c: The context
 * @ciph: ciphertext input
 * @plain: plaintext output
 * 
 * This function decrypts @in to @out.
 * Other parameters are take from the context @c.
 * The function does wait for the result.
 * 
 * Return value:  0 on success or an errorcode. 
 **/
gpgme_error_t
gpgme_op_decrypt( gpgme_ctx_t ctx, gpgme_data_t ciph, gpgme_data_t plain )
				  
{
    gpgme_error_t err;

    err = decrypt_start( ctx, ciph, plain );
    if( !err ) {
        gpgme_wait( ctx, 1 );
	ctx->pending = 0;
	err = get_decrypt_result( ctx );
    }
    return err;
} /* gpgme_op_decrypt */


gpgme_error_t
gpgme_op_file_decrypt( gpgme_ctx_t ctx, const char * ciph, const char * plain )
{
    gpgme_error_t err;
    const char * files[1];

    files[0] = ciph;

    err = file_decrypt_start( ctx, files, 1, plain );
    if( !err ) {
	gpgme_wait( ctx, 1 );
	err = get_decrypt_result( ctx );
	ctx->pending = 0;
    }
    return err;
} /* gpgme_op_file_decrypt */


gpgme_error_t
gpgme_op_files_decrypt( gpgme_ctx_t ctx, const char ** files, size_t nfiles )
{
    gpgme_error_t err;

    err = file_decrypt_start( ctx, files, nfiles, NULL );
    if( !err ) {
	gpgme_wait( ctx, 1 );
	err = get_decrypt_result( ctx );
	ctx->pending = 0;
    }
    return err;
}


gpgme_error_t
gpgme_op_clip_decrypt( gpgme_ctx_t ctx )
{
    gpgme_error_t err;
    gpgme_data_t ciph = NULL;
    gpgme_data_t plain = NULL;
    
    err = gpgme_data_new_from_clipboard (&ciph);
    if( !err )            
	err = gpgme_data_new( &plain );
    if( !err )
	err = gpgme_op_decrypt( ctx, ciph, plain );    

    gpgme_data_release_and_set_clipboard( plain );
    gpgme_data_release( ciph );	

    return err;
} /* gpgme_op_clip_decrypt */
