/* context.h 
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2002, 2003, 2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifndef CONTEXT_H
#define CONTEXT_H

#include "gpgme.h"
#include "types.h"
#include "rungpg.h"  /* for _gpg_object_t */
#include "common-status.h"

typedef enum {
    RESULT_TYPE_NONE = 0,
    RESULT_TYPE_VERIFY,
    RESULT_TYPE_DECRYPT,
    RESULT_TYPE_SIGN,
    RESULT_TYPE_SIGN_ENCRYPT,
    RESULT_TYPE_IMPORT,
    RESULT_TYPE_ENCRYPT,
    RESULT_TYPE_SYMENC,
    RESULT_TYPE_EDITKEY,
} result_type_t;


struct key_queue_item_s {
    struct key_queue_item_s *next;
    gpgme_key_t key;
};


/* Currently we need it at several places, so we put the definition 
 * into this header file 
 */
struct gpgme_context_s {
    int initialized;
    int pending;   /* a gpg request is still pending */

    /* at some points we need to allocate memory but we are not
     * able to handle a malloc problem at that point, so we set this
     * flag to indicate this condition */
    int out_of_core;   
    int cancel;    /* cancel operation request */

    _gpg_object_t gpg; /* the running gpg process */

    unsigned int use_throwkeyid:1; /* anonymous packets (without keyids) */
    unsigned int use_armor:1;
    unsigned int use_textmode:1;
    unsigned int use_file:1;
    unsigned int use_tmpfiles:1;
    unsigned int use_pass_fd:1;
    unsigned int use_logging:1;
    unsigned int interactive:1;
    unsigned int pipemode:1;
    
    int keylist_mode;
    int list_opts;
    
    unsigned int force_trust:1; /* force to --always-trust */
    unsigned int force_opt:1; /* some special options */
    unsigned int force_mdc:1;

    int problem;
    gpgme_wipe_t wipe_fnc;
    unsigned int with_secret_key:1; /* for --export */

    int signers_size;  /* size of the following array */
    gpgme_key_t * signers;

    result_type_t result_type;
    union {
        _verify_result_t verify;
        _decrypt_result_t decrypt;
        _sign_result_t sign;
	_import_result_t import;
	_sign_encrypt_result_t sign_enc;
	_encrypt_result_t encrypt;
	_symenc_result_t symenc;
	_editkey_result_t editk;
    } result;	

    gpgme_data_t notation;    /* last signature notation */
    gpgme_data_t logging;

    gpgme_key_t tmp_key;       /* used by keylist.c */
    int tmp_i;
    char * tmp_id;
    char tmp_keyid[16+1];

    void * tmp_res;

    volatile int key_cond;  /* something new is available */
    struct key_queue_item_s * key_queue;
    struct trust_queue_item_s * trust_queue;

    const char * passphrase_value;

    struct {
	gpgme_passphrase_cb_t passphrase;
	void * passphrase_value;

	gpgme_progress_cb_t progress;
	void * progress_value;
	int progress_value_int;
	int progess_tmp;

	gpgme_interactive_cb_t interactiv;
	void * interactiv_value;
    } cb;

    /* interface for --card-edit and --edit-key */    
    void * edit_opaque;
    int edit_cmd;

    gpgme_card_t card;

    void * key_lookup;

    gpgme_data_t help_data_1;

    int cipher_algo;
    struct {
	int digest_algo;
	int mode;
	unsigned used:1;
    } s2k;

    gpgme_recipients_t enc_to;

    char * locusr;
    
    char * keygen_fpr;

    char * comment;

    char * homedir;

    int no_compress;
    unsigned long proc_rc;
};


struct gpgme_data_s {
    size_t len;
    const char *data;
    gpgme_data_type_t type;
    gpgme_data_mode_t mode;

    int (*read_cb)( void *, char *, size_t, size_t *);
    void *read_cb_value;
    int read_cb_eof;

    size_t readpos;
    size_t writepos;
    size_t private_len;
    char *private_buffer;
};

struct user_id_flag_s {
    unsigned int revoked:1;
    unsigned int invalid:1;
    unsigned int primary:1;
    unsigned mdc:1;
    unsigned no_ks_modify:1;
};

struct user_id_s {
    struct user_id_s *next;
    struct user_id_flag_s flags;
    gpgme_validity_t validity;
    unsigned long created;
    char * hash;
    const char * name_part;    /* all 3 point into strings behind name */
    const char * email_part;   /* or to read-only strings */
    const char * comment_part;
    char name[1];
};

struct gpgme_recipients_s {
    struct user_id_s * list;
    int checked;   /* wether the recipients are all valid */
    int count;
};

struct user_id_info_s {
    struct user_id_info_s *next;
    struct user_id_flag_s flags;
    int validity;
    char * name;
    char * prefs;    
    int idx;
};

struct gpgme_user_id_info_s {
    struct user_id_info_s * list;
};


struct keycache_s {
    struct keycache_s * next;
    int flags;
    gpgme_key_t key;
    gpgme_key_t pubpart;
};

struct gpgme_keycache_s {    
    struct keycache_s * item;
    struct keycache_s * tmp;
    int secret;
    int pos;
    void (*cb) (void *, const char *, int, unsigned, unsigned);
    void * cb_value;
    int cb_value2;
};

struct gpgme_signature_s {
    struct gpgme_signature_s * next;
    int ref_count;
    gpgme_key_t key;
    gpgme_sigstat_t sigstat;
    ulong created;
    ulong expired;
    gpgme_pk_cipher_t key_algo;
    gpgme_md_t md_algo;    
    int sigclass;
    int trust;
    char id[41];
    char * notation;
    char * user_id;
    char * policy_url;
    char * file_name;
    char * key_server;
    struct {
	unsigned expired:1;
    } flags;
};

struct gpgme_editcard_s {
    union {
	struct {
	    const char * surname;
	    const char * givenname;
	    const char * keyurl;
	    const char * login;
	    const char * lang;
	    char         sex;
	} edit;
	struct {
	    const char * pinold;
	    const char * pinnew;
	} pin;
    } u;
    struct {
	char * name;
	char * email;
	char * comment;
	char * expdate;
	const char * passwd;
	int flags;
    } keygen;
    const char * apin;
    const char * upin;
    int type;
    int cnt;
    int cancel;
    const char *(*card_cb) (int code, void * opaque);
    void * cb_value;
};


struct gpgme_card_s {
    char * aid;
    char * version;
    char ver[6];
    char * vendor;
    char * serial;
    char * surname;
    char * givenname;
    char * lang;
    char sex;
    char * url;
    char * login;
    int force_pin;
    char * maxpinlen[3];
    int sig_count;
    char * fpr[3];
};

struct gpgme_editkey_s {
    int key_pair;
    int type;

    union {
	struct {
	    const char * trust_val;
	} trust;
	struct {
	    int sig_class;
	    int sig_type;
	    struct {
		const char * val;
		const char * depth;
		const char * regexp;
	    } trust;
	    const char * exp_date;
	    const char * passwd;
	} sign;
	struct {
	    char * name;
	    char * email;
	    char * comment;
	    int use_comment;
	    const char * passwd;
	} adduid;
	struct {
	    const char * uid;
	    const char * passwd;
	} addrev;
	struct {
	    const char * jpg;
	    const char * passwd;
	} addphoto;
	struct {	
	    int id;
	} deluid;
	struct {
	    int id;
	} delkey;
	struct {
	    int algo;
	    int size;
	    int valid;
	    const char * passwd;
	} addkey;
	struct {
	    const char * old_passwd;
	    const char * new_passwd;
	    int send_old;
	    int allow_empty;
	} passwd;
	struct {
	    const char * uid;
	    const char * reason_id;
	    char * reason_text;
	    int reason_sent;
	    const char * passwd;
	    unsigned int pgp:1;
	} revoke;
	struct {
	    int id;
	    const char * passwd;
	} primary;
	struct {
	    int id;
	    int days;
	    const char * date;
	    const char * passwd;
	} expire;
	struct {
	    int id;
	    const char * passwd;
	} revsig;
	struct {
	    int id;
	    int reason;
	    const char * passwd;
	} revkey;
	struct {
	    const char * new_prefs;
	    const char * passwd;
	    int uid_idx;
	    int id;
	} pref;
	struct {
	    int uid;
	    int signo;
	    int currno;
	} delsig;
	struct {
	    const char * passwd;
	    const char * url;
	} keyserv;
    } u;
};

struct gpgme_rev_cert_s {
    long timestamp;
    char keyid[16+1];
    char *reason;
    unsigned char code;
};

#define fail_on_pending_request(c)                            \
          do {                                                \
                if (!(c))         return GPGME_Invalid_Value; \
                if ((c)->pending) return GPGME_Busy;          \
             } while (0)

#define wait_on_request_or_fail(c)                            \
          do {                                                \
                if (!(c))          return GPGME_Invalid_Value;\
                if (!(c)->pending) return GPGME_No_Request;   \
                gpgme_wait ((c), 1);                          \
             } while (0)



#endif /* CONTEXT_H */
