/* kbnode.c -  keyblock node utility functions
 * Copyright (C) 1998, 1999, 2000, 2001 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <malloc.h>
#include <sys/types.h>

#include "openpgp.h"

#define is_deleted_kbnode(a)  ((a)->private_flag & 1)
#define is_cloned_kbnode(a)   ((a)->private_flag & 2)


static gpg_kbnode_t
alloc_node( void )
{
    gpg_kbnode_t n;
    
    n = malloc( sizeof *n );
    n->next = NULL;
    n->pkt = NULL;
    n->flag = 0;
    n->private_flag=0;
    n->recno = 0;
    return n;
}

static void
free_node( gpg_kbnode_t n )
{
    if( n ) {      
	free( n );
    }
}



gpg_kbnode_t
gpg_new_kbnode( PACKET *pkt )
{
    gpg_kbnode_t n = alloc_node();
    n->pkt = pkt;
    return n;
}


gpg_kbnode_t
gpg_clone_kbnode( gpg_kbnode_t node )
{
    gpg_kbnode_t n = alloc_node();

    n->pkt = node->pkt;
    n->private_flag = node->private_flag | 2; /* mark cloned */
    return n;
}


void
gpg_release_kbnode( gpg_kbnode_t n )
{
    gpg_kbnode_t n2;

    while( n ) {
	n2 = n->next;
	if( !is_cloned_kbnode(n) ) {
	    gpg_free_packet( n->pkt );
	    free( n->pkt );
	}
	free_node( n );
	n = n2;
    }
}


/****************
 * Delete NODE.
 * Note: This only works with walk_kbnode!!
 */
void
gpg_delete_kbnode( gpg_kbnode_t node )
{
    node->private_flag |= 1;
}



/****************
 * Append NODE to ROOT.  ROOT must exist!
 */
void
gpg_add_kbnode( gpg_kbnode_t root, gpg_kbnode_t node )
{
    gpg_kbnode_t n1;

    for(n1=root; n1->next; n1 = n1->next)
	;
    n1->next = node;
}


/****************
 * Find the previous node (if PKTTYPE = 0) or the previous node
 * with pkttype PKTTYPE in the list starting with ROOT of NODE.
 */
gpg_kbnode_t
gpg_find_prev_kbnode( gpg_kbnode_t root, gpg_kbnode_t node, int pkttype )
{
    gpg_kbnode_t n1;

    for (n1=NULL; root && root != node; root = root->next ) {
        if (!pkttype ||root->pkt->pkttype == pkttype)
            n1 = root;
    }
    return n1;
}

/****************
 * Ditto, but find the next packet.  The behaviour is trivial if
 * PKTTYPE is 0 but if it is specified, the next node with a packet
 * of this type is returned.  The function has some knowledge about
 * the valid ordering of packets: e.g. if the next signature packet
 * is requested, the function will not return one if it encounters
 * a user-id.
 */
gpg_kbnode_t
gpg_find_next_kbnode( gpg_kbnode_t node, int pkttype )
{
    for( node=node->next ; node; node = node->next ) {
	if( !pkttype )
	    return node;
	else if( pkttype == PKT_USER_ID 
		 && (	node->pkt->pkttype == PKT_PUBLIC_KEY
		     || node->pkt->pkttype == PKT_SECRET_KEY ) )
	    return NULL;
	else if( pkttype == PKT_SIGNATURE
		 && (	node->pkt->pkttype == PKT_USER_ID
		     || node->pkt->pkttype == PKT_PUBLIC_KEY
		     || node->pkt->pkttype == PKT_SECRET_KEY ) )
	    return NULL;
	else if( node->pkt->pkttype == pkttype )
	    return node;
    }
    return NULL;
}


gpg_kbnode_t
gpg_find_kbnode( gpg_kbnode_t node, int pkttype )
{
    for( ; node; node = node->next ) {
	if( node->pkt->pkttype == pkttype )
	    return node;
    }
    return NULL;
}



/****************
 * Walk through a list of kbnodes. This function returns
 * the next kbnode for each call; before using the function the first
 * time, the caller must set CONTEXT to NULL (This has simply the effect
 * to start with ROOT).
 */
gpg_kbnode_t
gpg_walk_kbnode( gpg_kbnode_t root, gpg_kbnode_t *context, int all )
{
    gpg_kbnode_t n;

    do {
	if( !*context ) {
	    *context = root;
	    n = root;
	}
	else {
	    n = (*context)->next;
	    *context = n;
	}
    } while( !all && n && is_deleted_kbnode(n) );

    return n;
}

void
gpg_clear_kbnode_flags( gpg_kbnode_t n )
{
    for( ; n; n = n->next ) {
	n->flag = 0;
    }
}





