/* valaexpression.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <valaccode.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaExpressionPrivate ValaExpressionPrivate;
#define _vala_code_node_unref0(var) ((var == NULL) ? NULL : (var = (vala_code_node_unref (var), NULL)))
#define _gee_collection_object_unref0(var) ((var == NULL) ? NULL : (var = (gee_collection_object_unref (var), NULL)))
#define _vala_ccode_node_unref0(var) ((var == NULL) ? NULL : (var = (vala_ccode_node_unref (var), NULL)))

#define VALA_TYPE_BLOCK (vala_block_get_type ())
#define VALA_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BLOCK, ValaBlock))
#define VALA_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BLOCK, ValaBlockClass))
#define VALA_IS_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BLOCK))
#define VALA_IS_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BLOCK))
#define VALA_BLOCK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BLOCK, ValaBlockClass))

typedef struct _ValaBlock ValaBlock;
typedef struct _ValaBlockClass ValaBlockClass;

#define VALA_TYPE_STATEMENT (vala_statement_get_type ())
#define VALA_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STATEMENT, ValaStatement))
#define VALA_IS_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STATEMENT))
#define VALA_STATEMENT_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_STATEMENT, ValaStatementIface))

typedef struct _ValaStatement ValaStatement;
typedef struct _ValaStatementIface ValaStatementIface;

struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

struct _ValaExpression {
	ValaCodeNode parent_instance;
	ValaExpressionPrivate * priv;
	GeeArrayList* temp_vars;
};

struct _ValaExpressionClass {
	ValaCodeNodeClass parent_class;
	gboolean (*is_constant) (ValaExpression* self);
	gboolean (*is_pure) (ValaExpression* self);
	gboolean (*is_non_null) (ValaExpression* self);
};

struct _ValaExpressionPrivate {
	ValaDataType* _value_type;
	ValaDataType* _formal_value_type;
	ValaDataType* _target_type;
	ValaDataType* _formal_target_type;
	ValaSymbol* _symbol_reference;
	gboolean _lvalue;
	GeeList* array_sizes;
	ValaCCodeExpression* _delegate_target;
};

struct _ValaStatementIface {
	GTypeInterface parent_iface;
};


static gpointer vala_expression_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
#define VALA_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_EXPRESSION, ValaExpressionPrivate))
enum  {
	VALA_EXPRESSION_DUMMY_PROPERTY
};
gboolean vala_expression_is_constant (ValaExpression* self);
static gboolean vala_expression_real_is_constant (ValaExpression* self);
gboolean vala_expression_is_pure (ValaExpression* self);
static gboolean vala_expression_real_is_pure (ValaExpression* self);
gboolean vala_expression_is_non_null (ValaExpression* self);
static gboolean vala_expression_real_is_non_null (ValaExpression* self);
void vala_expression_append_array_size (ValaExpression* self, ValaCCodeExpression* size);
GeeList* vala_expression_get_array_sizes (ValaExpression* self);
GType vala_block_get_type (void);
GType vala_statement_get_type (void);
void vala_block_insert_before (ValaBlock* self, ValaStatement* stmt, ValaStatement* new_stmt);
ValaStatement* vala_expression_get_parent_statement (ValaExpression* self);
void vala_expression_insert_statement (ValaExpression* self, ValaBlock* block, ValaStatement* stmt);
ValaCodeNode* vala_code_node_construct (GType object_type);
ValaExpression* vala_expression_construct (GType object_type);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_formal_value_type (ValaExpression* self);
void vala_expression_set_formal_value_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_target_type (ValaExpression* self);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_formal_target_type (ValaExpression* self);
void vala_expression_set_formal_target_type (ValaExpression* self, ValaDataType* value);
ValaSymbol* vala_expression_get_symbol_reference (ValaExpression* self);
void vala_expression_set_symbol_reference (ValaExpression* self, ValaSymbol* value);
gboolean vala_expression_get_lvalue (ValaExpression* self);
void vala_expression_set_lvalue (ValaExpression* self, gboolean value);
ValaCCodeExpression* vala_expression_get_delegate_target (ValaExpression* self);
void vala_expression_set_delegate_target (ValaExpression* self, ValaCCodeExpression* value);
ValaCodeNode* vala_code_node_get_parent_node (ValaCodeNode* self);
static void vala_expression_finalize (ValaCodeNode* obj);



static gboolean vala_expression_real_is_constant (ValaExpression* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = FALSE;
	return result;
}


gboolean vala_expression_is_constant (ValaExpression* self) {
	return VALA_EXPRESSION_GET_CLASS (self)->is_constant (self);
}


static gboolean vala_expression_real_is_pure (ValaExpression* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_critical ("Type `%s' does not implement abstract method `vala_expression_is_pure'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return FALSE;
}


gboolean vala_expression_is_pure (ValaExpression* self) {
	return VALA_EXPRESSION_GET_CLASS (self)->is_pure (self);
}


static gboolean vala_expression_real_is_non_null (ValaExpression* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = FALSE;
	return result;
}


gboolean vala_expression_is_non_null (ValaExpression* self) {
	return VALA_EXPRESSION_GET_CLASS (self)->is_non_null (self);
}


void vala_expression_append_array_size (ValaExpression* self, ValaCCodeExpression* size) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (size != NULL);
	gee_collection_add ((GeeCollection*) self->priv->array_sizes, size);
}


GeeList* vala_expression_get_array_sizes (ValaExpression* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, self->priv->array_sizes);
	return result;
}


void vala_expression_insert_statement (ValaExpression* self, ValaBlock* block, ValaStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (block != NULL);
	g_return_if_fail (stmt != NULL);
	vala_block_insert_before (block, vala_expression_get_parent_statement (self), stmt);
}


ValaExpression* vala_expression_construct (GType object_type) {
	ValaExpression* self;
	self = (ValaExpression*) vala_code_node_construct (object_type);
	return self;
}


ValaDataType* vala_expression_get_value_type (ValaExpression* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_value_type;
	return result;
}


static gpointer _vala_code_node_ref0 (gpointer self) {
	return self ? vala_code_node_ref (self) : NULL;
}


void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_value_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_value_type), _tmp0_);
}


ValaDataType* vala_expression_get_formal_value_type (ValaExpression* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_formal_value_type;
	return result;
}


void vala_expression_set_formal_value_type (ValaExpression* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_formal_value_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_formal_value_type), _tmp0_);
}


ValaDataType* vala_expression_get_target_type (ValaExpression* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_target_type;
	return result;
}


void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_target_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_target_type), _tmp0_);
}


ValaDataType* vala_expression_get_formal_target_type (ValaExpression* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_formal_target_type;
	return result;
}


void vala_expression_set_formal_target_type (ValaExpression* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_formal_target_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_formal_target_type), _tmp0_);
}


ValaSymbol* vala_expression_get_symbol_reference (ValaExpression* self) {
	ValaSymbol* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_symbol_reference;
	return result;
}


void vala_expression_set_symbol_reference (ValaExpression* self, ValaSymbol* value) {
	g_return_if_fail (self != NULL);
	self->priv->_symbol_reference = value;
}


gboolean vala_expression_get_lvalue (ValaExpression* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_lvalue;
	return result;
}


void vala_expression_set_lvalue (ValaExpression* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_lvalue = value;
}


ValaCCodeExpression* vala_expression_get_delegate_target (ValaExpression* self) {
	ValaCCodeExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_delegate_target;
	return result;
}


static gpointer _vala_ccode_node_ref0 (gpointer self) {
	return self ? vala_ccode_node_ref (self) : NULL;
}


void vala_expression_set_delegate_target (ValaExpression* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_delegate_target = (_tmp0_ = _vala_ccode_node_ref0 (value), _vala_ccode_node_unref0 (self->priv->_delegate_target), _tmp0_);
}


ValaStatement* vala_expression_get_parent_statement (ValaExpression* self) {
	ValaStatement* result;
	ValaCodeNode* _tmp0_;
	ValaExpression* expr;
	ValaCodeNode* _tmp1_;
	ValaStatement* stmt;
	ValaCodeNode* _tmp2_;
	ValaLocalVariable* local;
	g_return_val_if_fail (self != NULL, NULL);
	expr = _vala_code_node_ref0 ((_tmp0_ = vala_code_node_get_parent_node ((ValaCodeNode*) self), VALA_IS_EXPRESSION (_tmp0_) ? ((ValaExpression*) _tmp0_) : NULL));
	stmt = _vala_code_node_ref0 ((_tmp1_ = vala_code_node_get_parent_node ((ValaCodeNode*) self), VALA_IS_STATEMENT (_tmp1_) ? ((ValaStatement*) _tmp1_) : NULL));
	local = _vala_code_node_ref0 ((_tmp2_ = vala_code_node_get_parent_node ((ValaCodeNode*) self), VALA_IS_LOCAL_VARIABLE (_tmp2_) ? ((ValaLocalVariable*) _tmp2_) : NULL));
	if (stmt != NULL) {
		result = VALA_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self));
		_vala_code_node_unref0 (expr);
		_vala_code_node_unref0 (stmt);
		_vala_code_node_unref0 (local);
		return result;
	} else {
		if (expr != NULL) {
			result = vala_expression_get_parent_statement (expr);
			_vala_code_node_unref0 (expr);
			_vala_code_node_unref0 (stmt);
			_vala_code_node_unref0 (local);
			return result;
		} else {
			if (local != NULL) {
				result = VALA_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) local));
				_vala_code_node_unref0 (expr);
				_vala_code_node_unref0 (stmt);
				_vala_code_node_unref0 (local);
				return result;
			} else {
				result = NULL;
				_vala_code_node_unref0 (expr);
				_vala_code_node_unref0 (stmt);
				_vala_code_node_unref0 (local);
				return result;
			}
		}
	}
	_vala_code_node_unref0 (expr);
	_vala_code_node_unref0 (stmt);
	_vala_code_node_unref0 (local);
}


static void vala_expression_class_init (ValaExpressionClass * klass) {
	vala_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaExpressionPrivate));
	VALA_EXPRESSION_CLASS (klass)->is_constant = vala_expression_real_is_constant;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_expression_real_is_pure;
	VALA_EXPRESSION_CLASS (klass)->is_non_null = vala_expression_real_is_non_null;
}


static void vala_expression_instance_init (ValaExpression * self) {
	self->priv = VALA_EXPRESSION_GET_PRIVATE (self);
	self->temp_vars = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->array_sizes = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_expression_finalize (ValaCodeNode* obj) {
	ValaExpression * self;
	self = VALA_EXPRESSION (obj);
	_vala_code_node_unref0 (self->priv->_value_type);
	_vala_code_node_unref0 (self->priv->_formal_value_type);
	_vala_code_node_unref0 (self->priv->_target_type);
	_vala_code_node_unref0 (self->priv->_formal_target_type);
	_gee_collection_object_unref0 (self->temp_vars);
	_gee_collection_object_unref0 (self->priv->array_sizes);
	_vala_ccode_node_unref0 (self->priv->_delegate_target);
	VALA_CODE_NODE_CLASS (vala_expression_parent_class)->finalize (obj);
}


GType vala_expression_get_type (void) {
	static GType vala_expression_type_id = 0;
	if (vala_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaExpression), 0, (GInstanceInitFunc) vala_expression_instance_init, NULL };
		vala_expression_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaExpression", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_expression_type_id;
}




