/* valagidlparser.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gidlparser.h>
#include <gidlmodule.h>
#include <stdio.h>
#include <gidlnode.h>
#include <gmetadata.h>
#include <float.h>
#include <math.h>


#define VALA_TYPE_GIDL_PARSER (vala_gidl_parser_get_type ())
#define VALA_GIDL_PARSER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GIDL_PARSER, ValaGIdlParser))
#define VALA_GIDL_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GIDL_PARSER, ValaGIdlParserClass))
#define VALA_IS_GIDL_PARSER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GIDL_PARSER))
#define VALA_IS_GIDL_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GIDL_PARSER))
#define VALA_GIDL_PARSER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GIDL_PARSER, ValaGIdlParserClass))

typedef struct _ValaGIdlParser ValaGIdlParser;
typedef struct _ValaGIdlParserClass ValaGIdlParserClass;
typedef struct _ValaGIdlParserPrivate ValaGIdlParserPrivate;

/**
 * Code visitor parsing all GIDL files.
 */
struct _ValaGIdlParser {
	ValaCodeVisitor parent_instance;
	ValaGIdlParserPrivate * priv;
};

struct _ValaGIdlParserClass {
	ValaCodeVisitorClass parent_class;
};

struct _ValaGIdlParserPrivate {
	ValaCodeContext* context;
	ValaSourceFile* current_source_file;
	ValaSourceReference* current_source_reference;
	ValaNamespace* current_namespace;
	ValaTypeSymbol* current_data_type;
	GeeMap* codenode_attributes_map;
	GeeMap* codenode_attributes_patterns;
	GeeSet* current_type_symbol_set;
	GeeMap* cname_type_map;
};



GType vala_gidl_parser_get_type (void);
#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context);
static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaTypeSymbol* t);
static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_gidl_parser_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_gidl_parser_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static void _g_list_free_g_idl_module_free (GList* self);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode, int* result_length1);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns);
static void _g_list_free_vala_code_node_unref (GList* self);
static ValaDelegate* vala_gidl_parser_parse_delegate (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_union (ValaGIdlParser* self, GIdlNodeUnion* un_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module);
static ValaTypeSymbol* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_interface);
static ValaDataType* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param, ValaParameterDirection* direction);
static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static ValaDataType* vala_gidl_parser_parse_type_string (ValaGIdlParser* self, const char* n);
static ValaDataType* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node, ValaParameterDirection* direction);
static ValaMethod* vala_gidl_parser_parse_virtual (ValaGIdlParser* self, GIdlNodeVFunc* v, GIdlNodeFunction* func, gboolean is_interface);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gboolean vala_gidl_parser_is_simple_type (ValaGIdlParser* self, const char* type_name);
static gboolean vala_gidl_parser_param_is_exception (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_create_method (ValaGIdlParser* self, const char* name, const char* symbol, GIdlNodeParam* res, GList* parameters, gboolean is_constructor, gboolean is_interface);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
ValaGIdlParser* vala_gidl_parser_new (void);
ValaGIdlParser* vala_gidl_parser_construct (GType object_type);
ValaGIdlParser* vala_gidl_parser_new (void);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_finalize (ValaCodeVisitor* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context) {
	GeeMap* _tmp0_;
	ValaCodeContext* _tmp2_;
	ValaCodeContext* _tmp1_;
	GeeMap* _tmp3_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp0_ = NULL;
	self->priv->cname_type_map = (_tmp0_ = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, VALA_TYPE_TYPESYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->cname_type_map == NULL) ? NULL : (self->priv->cname_type_map = (gee_collection_object_unref (self->priv->cname_type_map), NULL)), _tmp0_);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->context = (_tmp2_ = (_tmp1_ = context, (_tmp1_ == NULL) ? NULL : vala_code_context_ref (_tmp1_)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp2_);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
	_tmp3_ = NULL;
	self->priv->cname_type_map = (_tmp3_ = NULL, (self->priv->cname_type_map == NULL) ? NULL : (self->priv->cname_type_map = (gee_collection_object_unref (self->priv->cname_type_map), NULL)), _tmp3_);
}


static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (ns != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
}


static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (cl != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) cl);
}


static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (st != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) st);
}


static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (iface != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) iface);
}


static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (en != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) en);
}


static void vala_gidl_parser_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (ed != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) ed);
}


static void vala_gidl_parser_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (d != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) d);
}


static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaTypeSymbol* t) {
	char* _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (t != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = !gee_map_contains (self->priv->cname_type_map, _tmp0_ = vala_typesymbol_get_cname (t, FALSE)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		char* _tmp2_;
		_tmp2_ = NULL;
		gee_map_set (self->priv->cname_type_map, _tmp2_ = vala_typesymbol_get_cname (t, FALSE), t);
		_tmp2_ = (g_free (_tmp2_), NULL);
	}
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (source_file != NULL);
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gi")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void _g_list_free_g_idl_module_free (GList* self) {
	g_list_foreach (self, (GFunc) g_idl_module_free, NULL);
	g_list_free (self);
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file) {
	GError * _inner_error_;
	char* _tmp0_;
	char* _tmp1_;
	char* metadata_filename;
	ValaSourceFile* _tmp3_;
	ValaSourceFile* _tmp2_;
	GeeMap* _tmp4_;
	GeeMap* _tmp5_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
	_inner_error_ = NULL;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	metadata_filename = (_tmp1_ = g_strdup_printf ("%s.metadata", _tmp0_ = g_strndup (vala_source_file_get_filename (source_file), (gsize) (strlen (vala_source_file_get_filename (source_file)) - strlen (".gi")))), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	self->priv->current_source_file = (_tmp3_ = (_tmp2_ = source_file, (_tmp2_ == NULL) ? NULL : vala_source_file_ref (_tmp2_)), (self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)), _tmp3_);
	_tmp4_ = NULL;
	self->priv->codenode_attributes_map = (_tmp4_ = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal), (self->priv->codenode_attributes_map == NULL) ? NULL : (self->priv->codenode_attributes_map = (gee_collection_object_unref (self->priv->codenode_attributes_map), NULL)), _tmp4_);
	_tmp5_ = NULL;
	self->priv->codenode_attributes_patterns = (_tmp5_ = (GeeMap*) gee_hash_map_new (G_TYPE_POINTER, NULL, NULL, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_hash, (GEqualFunc) g_pattern_spec_equal, g_direct_equal), (self->priv->codenode_attributes_patterns == NULL) ? NULL : (self->priv->codenode_attributes_patterns = (gee_collection_object_unref (self->priv->codenode_attributes_patterns), NULL)), _tmp5_);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			char* _tmp8_;
			gboolean _tmp7_;
			char* _tmp6_;
			metadata = NULL;
			_tmp8_ = NULL;
			_tmp6_ = NULL;
			_tmp7_ = g_file_get_contents (metadata_filename, &_tmp6_, NULL, &_inner_error_);
			metadata = (_tmp8_ = _tmp6_, metadata = (g_free (metadata), NULL), _tmp8_);
			_tmp7_;
			if (_inner_error_ != NULL) {
				metadata = (g_free (metadata), NULL);
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				goto __finally0;
			}
			{
				char** _tmp9_;
				char** line_collection;
				int line_collection_length1;
				int line_it;
				_tmp9_ = NULL;
				line_collection = _tmp9_ = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = _vala_array_length (_tmp9_);
				for (line_it = 0; line_it < _vala_array_length (_tmp9_); line_it = line_it + 1) {
					const char* _tmp12_;
					char* line;
					_tmp12_ = NULL;
					line = (_tmp12_ = line_collection[line_it], (_tmp12_ == NULL) ? NULL : g_strdup (_tmp12_));
					{
						char** _tmp11_;
						gint tokens_size;
						gint tokens_length1;
						char** _tmp10_;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							/* ignore comment lines*/
							line = (g_free (line), NULL);
							continue;
						}
						_tmp11_ = NULL;
						_tmp10_ = NULL;
						tokens = (_tmp11_ = _tmp10_ = g_strsplit (line, " ", 2), tokens_length1 = _vala_array_length (_tmp10_), tokens_size = tokens_length1, _tmp11_);
						if (NULL == tokens[0]) {
							line = (g_free (line), NULL);
							tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
							continue;
						}
						if (NULL != g_utf8_strchr (tokens[0], (glong) (-1), (gunichar) '*')) {
							GPatternSpec* pattern;
							pattern = g_pattern_spec_new (tokens[0]);
							gee_map_set (self->priv->codenode_attributes_patterns, pattern, tokens[0]);
						}
						gee_map_set (self->priv->codenode_attributes_map, tokens[0], tokens[1]);
						line = (g_free (line), NULL);
						tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
					}
				}
				line_collection = (_vala_array_free (line_collection, line_collection_length1, (GDestroyNotify) g_free), NULL);
			}
			metadata = (g_free (metadata), NULL);
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				char* _tmp13_;
				_tmp13_ = NULL;
				vala_report_error (NULL, _tmp13_ = g_strdup_printf ("Unable to read metadata file: %s", e->message));
				_tmp13_ = (g_free (_tmp13_), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally0:
		if (_inner_error_ != NULL) {
			metadata_filename = (g_free (metadata_filename), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return;
		}
	}
	{
		GList* modules;
		ValaSourceReference* _tmp14_;
		modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_MARKUP_ERROR) {
				goto __catch1_g_markup_error;
			}
			goto __finally1;
		}
		_tmp14_ = NULL;
		self->priv->current_source_reference = (_tmp14_ = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL) ? NULL : (self->priv->current_source_reference = (vala_source_reference_unref (self->priv->current_source_reference), NULL)), _tmp14_);
		{
			GList* module_collection;
			GList* module_it;
			module_collection = modules;
			for (module_it = module_collection; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = (GIdlModule*) module_it->data;
				{
					ValaNamespace* ns;
					ns = vala_gidl_parser_parse_module (self, module);
					if (ns != NULL) {
						vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
					}
					(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
				}
			}
		}
		(modules == NULL) ? NULL : (modules = (_g_list_free_g_idl_module_free (modules), NULL));
	}
	goto __finally1;
	__catch1_g_markup_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			fprintf (stdout, "error parsing GIDL file: %s\n", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally1:
	if (_inner_error_ != NULL) {
		metadata_filename = (g_free (metadata_filename), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	metadata_filename = (g_free (metadata_filename), NULL);
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns) {
	char** _tmp1_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp0_;
	char** attributes;
	const char* _tmp18_;
	char* _tmp19_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	g_return_val_if_fail (ns != NULL, NULL);
	_tmp1_ = NULL;
	attributes = (_tmp1_ = vala_gidl_parser_get_attributes (self, type_name, &_tmp0_), attributes_length1 = _tmp0_, attributes_size = attributes_length1, _tmp1_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp5_;
				char* attr;
				_tmp5_ = NULL;
				attr = (_tmp5_ = attr_collection[attr_it], (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_));
				{
					char** _tmp3_;
					gint nv_size;
					gint nv_length1;
					char** _tmp2_;
					char** nv;
					_tmp3_ = NULL;
					_tmp2_ = NULL;
					nv = (_tmp3_ = _tmp2_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp2_), nv_size = nv_length1, _tmp3_);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp4_;
						_tmp4_ = NULL;
						return (_tmp4_ = vala_gidl_parser_eval (self, nv[1]), attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp4_);
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (g_str_has_prefix (type_name, vala_symbol_get_name ((ValaSymbol*) ns))) {
		const char* _tmp6_;
		char* _tmp7_;
		_tmp6_ = NULL;
		_tmp7_ = NULL;
		return (_tmp7_ = (_tmp6_ = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) ns), -1)), (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp7_);
	} else {
		gboolean _tmp8_;
		_tmp8_ = FALSE;
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) ns), "GLib") == 0) {
			_tmp8_ = g_str_has_prefix (type_name, "G");
		} else {
			_tmp8_ = FALSE;
		}
		if (_tmp8_) {
			const char* _tmp9_;
			char* _tmp10_;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			return (_tmp10_ = (_tmp9_ = g_utf8_offset_to_pointer (type_name, (glong) 1), (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp10_);
		} else {
			char* best_match;
			best_match = NULL;
			{
				GeeList* _tmp11_;
				GeeIterator* _tmp12_;
				GeeIterator* _cprefix_it;
				_tmp11_ = NULL;
				_tmp12_ = NULL;
				_cprefix_it = (_tmp12_ = gee_iterable_iterator ((GeeIterable*) (_tmp11_ = vala_namespace_get_cprefixes (ns))), (_tmp11_ == NULL) ? NULL : (_tmp11_ = (gee_collection_object_unref (_tmp11_), NULL)), _tmp12_);
				while (gee_iterator_next (_cprefix_it)) {
					char* cprefix;
					cprefix = (char*) gee_iterator_get (_cprefix_it);
					if (g_str_has_prefix (type_name, cprefix)) {
						gboolean _tmp13_;
						_tmp13_ = FALSE;
						if (best_match == NULL) {
							_tmp13_ = TRUE;
						} else {
							_tmp13_ = g_utf8_strlen (cprefix, -1) > g_utf8_strlen (best_match, -1);
						}
						if (_tmp13_) {
							char* _tmp15_;
							const char* _tmp14_;
							_tmp15_ = NULL;
							_tmp14_ = NULL;
							best_match = (_tmp15_ = (_tmp14_ = cprefix, (_tmp14_ == NULL) ? NULL : g_strdup (_tmp14_)), best_match = (g_free (best_match), NULL), _tmp15_);
						}
					}
					cprefix = (g_free (cprefix), NULL);
				}
				(_cprefix_it == NULL) ? NULL : (_cprefix_it = (gee_collection_object_unref (_cprefix_it), NULL));
			}
			if (best_match != NULL) {
				const char* _tmp16_;
				char* _tmp17_;
				_tmp16_ = NULL;
				_tmp17_ = NULL;
				return (_tmp17_ = (_tmp16_ = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (best_match, -1)), (_tmp16_ == NULL) ? NULL : g_strdup (_tmp16_)), best_match = (g_free (best_match), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp17_);
			}
			best_match = (g_free (best_match), NULL);
		}
	}
	_tmp18_ = NULL;
	_tmp19_ = NULL;
	return (_tmp19_ = (_tmp18_ = type_name, (_tmp18_ == NULL) ? NULL : g_strdup (_tmp18_)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp19_);
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns) {
	char* _tmp1_;
	char* _tmp0_;
	gboolean _tmp2_;
	const char* _tmp8_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (const_name != NULL, NULL);
	g_return_val_if_fail (ns != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	if ((_tmp2_ = g_str_has_prefix (const_name, _tmp1_ = g_strconcat (_tmp0_ = g_utf8_strup (vala_symbol_get_name ((ValaSymbol*) ns), -1), "_", NULL)), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_)) {
		const char* _tmp3_;
		_tmp3_ = NULL;
		return (_tmp3_ = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) ns), -1) + 1), (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_));
	} else {
		gboolean _tmp5_;
		_tmp5_ = FALSE;
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) ns), "GLib") == 0) {
			_tmp5_ = g_str_has_prefix (const_name, "G_");
		} else {
			_tmp5_ = FALSE;
		}
		if (_tmp5_) {
			const char* _tmp6_;
			_tmp6_ = NULL;
			return (_tmp6_ = g_utf8_offset_to_pointer (const_name, (glong) 2), (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_));
		}
	}
	_tmp8_ = NULL;
	return (_tmp8_ = const_name, (_tmp8_ == NULL) ? NULL : g_strdup (_tmp8_));
}


static void _g_list_free_vala_code_node_unref (GList* self) {
	g_list_foreach (self, (GFunc) vala_code_node_unref, NULL);
	g_list_free (self);
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module) {
	ValaSymbol* sym;
	ValaNamespace* ns;
	ValaNamespace* _tmp5_;
	ValaNamespace* _tmp4_;
	char** _tmp7_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp6_;
	char** attributes;
	ValaNamespace* _tmp25_;
	ValaNamespace* _tmp27_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (self->priv->context)), module->name);
	ns = NULL;
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* _tmp1_;
		ValaNamespace* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		ns = (_tmp1_ = (_tmp0_ = VALA_NAMESPACE (sym), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp1_);
		if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
			GList* _tmp2_;
			_tmp2_ = NULL;
			((ValaCodeNode*) ns)->attributes = (_tmp2_ = NULL, (((ValaCodeNode*) ns)->attributes == NULL) ? NULL : (((ValaCodeNode*) ns)->attributes = (_g_list_free_vala_code_node_unref (((ValaCodeNode*) ns)->attributes), NULL)), _tmp2_);
			vala_code_node_set_source_reference ((ValaCodeNode*) ns, self->priv->current_source_reference);
		}
	} else {
		ValaNamespace* _tmp3_;
		_tmp3_ = NULL;
		ns = (_tmp3_ = vala_namespace_new (module->name, self->priv->current_source_reference), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp3_);
	}
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	self->priv->current_namespace = (_tmp5_ = (_tmp4_ = ns, (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_)), (self->priv->current_namespace == NULL) ? NULL : (self->priv->current_namespace = (vala_code_node_unref (self->priv->current_namespace), NULL)), _tmp5_);
	_tmp7_ = NULL;
	attributes = (_tmp7_ = vala_gidl_parser_get_attributes (self, vala_symbol_get_name ((ValaSymbol*) ns), &_tmp6_), attributes_length1 = _tmp6_, attributes_size = attributes_length1, _tmp7_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp17_;
				char* attr;
				_tmp17_ = NULL;
				attr = (_tmp17_ = attr_collection[attr_it], (_tmp17_ == NULL) ? NULL : g_strdup (_tmp17_));
				{
					char** _tmp9_;
					gint nv_size;
					gint nv_length1;
					char** _tmp8_;
					char** nv;
					_tmp9_ = NULL;
					_tmp8_ = NULL;
					nv = (_tmp9_ = _tmp8_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp8_), nv_size = nv_length1, _tmp9_);
					if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
						char* _tmp10_;
						_tmp10_ = NULL;
						vala_namespace_set_cheader_filename (ns, _tmp10_ = vala_gidl_parser_eval (self, nv[1]));
						_tmp10_ = (g_free (_tmp10_), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "cprefix") == 0) {
							char** _tmp14_;
							gint cprefixes_size;
							gint cprefixes_length1;
							char** _tmp12_;
							char* _tmp11_;
							char** _tmp13_;
							char** cprefixes;
							_tmp14_ = NULL;
							_tmp12_ = NULL;
							_tmp11_ = NULL;
							_tmp13_ = NULL;
							cprefixes = (_tmp14_ = (_tmp13_ = _tmp12_ = g_strsplit (_tmp11_ = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp11_ = (g_free (_tmp11_), NULL), _tmp13_), cprefixes_length1 = _vala_array_length (_tmp12_), cprefixes_size = cprefixes_length1, _tmp14_);
							{
								char** name_collection;
								int name_collection_length1;
								int name_it;
								name_collection = cprefixes;
								name_collection_length1 = cprefixes_length1;
								for (name_it = 0; name_it < cprefixes_length1; name_it = name_it + 1) {
									const char* _tmp15_;
									char* name;
									_tmp15_ = NULL;
									name = (_tmp15_ = name_collection[name_it], (_tmp15_ == NULL) ? NULL : g_strdup (_tmp15_));
									{
										vala_namespace_add_cprefix (ns, name);
										name = (g_free (name), NULL);
									}
								}
							}
							cprefixes = (_vala_array_free (cprefixes, cprefixes_length1, (GDestroyNotify) g_free), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "lower_case_cprefix") == 0) {
								char* _tmp16_;
								_tmp16_ = NULL;
								vala_namespace_set_lower_case_cprefix (ns, _tmp16_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp16_ = (g_free (_tmp16_), NULL);
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	{
		GList* node_collection;
		GList* node_it;
		node_collection = module->entries;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = (GIdlNode*) node_it->data;
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaDelegate* cb;
					char* _tmp18_;
					cb = vala_gidl_parser_parse_delegate (self, (GIdlNodeFunction*) node);
					if (cb == NULL) {
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						continue;
					}
					_tmp18_ = NULL;
					vala_symbol_set_name ((ValaSymbol*) cb, _tmp18_ = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name ((ValaSymbol*) cb), ns));
					_tmp18_ = (g_free (_tmp18_), NULL);
					vala_namespace_add_delegate (ns, cb);
					vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cb);
					(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						vala_gidl_parser_parse_struct (self, (GIdlNodeStruct*) node, ns, module);
					} else {
						if (node->type == G_IDL_NODE_UNION) {
							vala_gidl_parser_parse_union (self, (GIdlNodeUnion*) node, ns, module);
						} else {
							if (node->type == G_IDL_NODE_BOXED) {
								vala_gidl_parser_parse_boxed (self, (GIdlNodeBoxed*) node, ns, module);
							} else {
								if (node->type == G_IDL_NODE_ENUM) {
									ValaTypeSymbol* en;
									char* _tmp19_;
									en = vala_gidl_parser_parse_enum (self, (GIdlNodeEnum*) node);
									if (en == NULL) {
										(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
										continue;
									}
									_tmp19_ = NULL;
									vala_symbol_set_name ((ValaSymbol*) en, _tmp19_ = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name ((ValaSymbol*) en), ns));
									_tmp19_ = (g_free (_tmp19_), NULL);
									if (VALA_IS_ERROR_DOMAIN (en)) {
										ValaTypeSymbol* _tmp20_;
										_tmp20_ = NULL;
										vala_namespace_add_error_domain (ns, (_tmp20_ = en, VALA_IS_ERROR_DOMAIN (_tmp20_) ? ((ValaErrorDomain*) _tmp20_) : NULL));
									} else {
										ValaTypeSymbol* _tmp21_;
										_tmp21_ = NULL;
										vala_namespace_add_enum (ns, (_tmp21_ = en, VALA_IS_ENUM (_tmp21_) ? ((ValaEnum*) _tmp21_) : NULL));
									}
									vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) en);
									(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
								} else {
									if (node->type == G_IDL_NODE_FLAGS) {
										ValaTypeSymbol* _tmp22_;
										ValaEnum* en;
										char* _tmp23_;
										_tmp22_ = NULL;
										en = (_tmp22_ = vala_gidl_parser_parse_enum (self, (GIdlNodeEnum*) node), VALA_IS_ENUM (_tmp22_) ? ((ValaEnum*) _tmp22_) : NULL);
										if (en == NULL) {
											(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
											continue;
										}
										_tmp23_ = NULL;
										vala_symbol_set_name ((ValaSymbol*) en, _tmp23_ = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name ((ValaSymbol*) en), ns));
										_tmp23_ = (g_free (_tmp23_), NULL);
										vala_enum_set_is_flags (en, TRUE);
										vala_namespace_add_enum (ns, en);
										vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) en);
										(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
									} else {
										if (node->type == G_IDL_NODE_OBJECT) {
											vala_gidl_parser_parse_object (self, (GIdlNodeInterface*) node, ns, module);
										} else {
											if (node->type == G_IDL_NODE_INTERFACE) {
												vala_gidl_parser_parse_interface (self, (GIdlNodeInterface*) node, ns, module);
											} else {
												if (node->type == G_IDL_NODE_CONSTANT) {
													ValaConstant* c;
													c = vala_gidl_parser_parse_constant (self, (GIdlNodeConstant*) node);
													if (c != NULL) {
														char* _tmp24_;
														_tmp24_ = NULL;
														vala_symbol_set_name ((ValaSymbol*) c, _tmp24_ = vala_gidl_parser_fix_const_name (self, vala_symbol_get_name ((ValaSymbol*) c), ns));
														_tmp24_ = (g_free (_tmp24_), NULL);
														vala_namespace_add_constant (ns, c);
														vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) c);
													}
													(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
												} else {
													if (node->type == G_IDL_NODE_FUNCTION) {
														ValaMethod* m;
														m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) node, FALSE);
														if (m != NULL) {
															vala_method_set_binding (m, MEMBER_BINDING_STATIC);
															vala_namespace_add_method (ns, m);
															vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) m);
														}
														(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	_tmp25_ = NULL;
	self->priv->current_namespace = (_tmp25_ = NULL, (self->priv->current_namespace == NULL) ? NULL : (self->priv->current_namespace = (vala_code_node_unref (self->priv->current_namespace), NULL)), _tmp25_);
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* _tmp26_;
		_tmp26_ = NULL;
		return (_tmp26_ = NULL, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp26_);
	}
	_tmp27_ = NULL;
	return (_tmp27_ = ns, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp27_);
}


static ValaDelegate* vala_gidl_parser_parse_delegate (ValaGIdlParser* self, GIdlNodeFunction* f_node) {
	GIdlNode* node;
	ValaDataType* return_type;
	ValaDelegate* cb;
	gboolean check_has_target;
	char** _tmp1_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp0_;
	char** attributes;
	guint remaining_params;
	ValaDelegate* _tmp38_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (f_node != NULL, NULL);
	node = (GIdlNode*) f_node;
	return_type = vala_gidl_parser_parse_param (self, f_node->result, NULL);
	cb = vala_delegate_new (node->name, return_type, self->priv->current_source_reference);
	vala_symbol_set_access ((ValaSymbol*) cb, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	check_has_target = TRUE;
	_tmp1_ = NULL;
	attributes = (_tmp1_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp0_), attributes_length1 = _tmp0_, attributes_size = attributes_length1, _tmp1_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp14_;
				char* attr;
				_tmp14_ = NULL;
				attr = (_tmp14_ = attr_collection[attr_it], (_tmp14_ == NULL) ? NULL : g_strdup (_tmp14_));
				{
					char** _tmp3_;
					gint nv_size;
					gint nv_length1;
					char** _tmp2_;
					char** nv;
					_tmp3_ = NULL;
					_tmp2_ = NULL;
					nv = (_tmp3_ = _tmp2_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp2_), nv_size = nv_length1, _tmp3_);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp4_;
						gboolean _tmp5_;
						_tmp4_ = NULL;
						if ((_tmp5_ = _vala_strcmp0 (_tmp4_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp4_ = (g_free (_tmp4_), NULL), _tmp5_)) {
							ValaDelegate* _tmp6_;
							_tmp6_ = NULL;
							return (_tmp6_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp6_);
						}
					} else {
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp7_;
							_tmp7_ = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cb, _tmp7_ = vala_gidl_parser_eval (self, nv[1]));
							_tmp7_ = (g_free (_tmp7_), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "has_target") == 0) {
								char* _tmp8_;
								gboolean _tmp9_;
								_tmp8_ = NULL;
								if ((_tmp9_ = _vala_strcmp0 (_tmp8_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_)) {
									check_has_target = FALSE;
								} else {
									char* _tmp10_;
									gboolean _tmp11_;
									_tmp10_ = NULL;
									if ((_tmp11_ = _vala_strcmp0 (_tmp10_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp10_ = (g_free (_tmp10_), NULL), _tmp11_)) {
										vala_delegate_set_has_target (cb, TRUE);
									}
								}
							} else {
								if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
									char* _tmp12_;
									gboolean _tmp13_;
									_tmp12_ = NULL;
									if ((_tmp13_ = _vala_strcmp0 (_tmp12_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp12_ = (g_free (_tmp12_), NULL), _tmp13_)) {
										vala_data_type_set_value_owned (return_type, TRUE);
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	remaining_params = g_list_length (f_node->parameters);
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = (GIdlNodeParam*) param_it->data;
			{
				GIdlNode* param_node;
				gboolean _tmp15_;
				gboolean _tmp16_;
				param_node = (GIdlNode*) param;
				_tmp15_ = FALSE;
				_tmp16_ = FALSE;
				if (check_has_target) {
					_tmp16_ = remaining_params == 1;
				} else {
					_tmp16_ = FALSE;
				}
				if (_tmp16_) {
					gboolean _tmp17_;
					_tmp17_ = FALSE;
					if (_vala_strcmp0 (param_node->name, "user_data") == 0) {
						_tmp17_ = TRUE;
					} else {
						_tmp17_ = _vala_strcmp0 (param_node->name, "data") == 0;
					}
					_tmp15_ = _tmp17_;
				} else {
					_tmp15_ = FALSE;
				}
				if (_tmp15_) {
					/* hide user_data parameter for instance delegates*/
					vala_delegate_set_has_target (cb, TRUE);
				} else {
					const char* _tmp18_;
					char* param_name;
					ValaParameterDirection direction;
					ValaDataType* param_type;
					ValaFormalParameter* p;
					gboolean hide_param;
					gboolean show_param;
					char** _tmp23_;
					gint _tmp22_;
					char* _tmp21_;
					gboolean _tmp37_;
					_tmp18_ = NULL;
					param_name = (_tmp18_ = param_node->name, (_tmp18_ == NULL) ? NULL : g_strdup (_tmp18_));
					if (_vala_strcmp0 (param_name, "string") == 0) {
						char* _tmp19_;
						/* avoid conflict with string type*/
						_tmp19_ = NULL;
						param_name = (_tmp19_ = g_strdup ("str"), param_name = (g_free (param_name), NULL), _tmp19_);
					} else {
						if (_vala_strcmp0 (param_name, "self") == 0) {
							char* _tmp20_;
							/* avoid conflict with delegate target*/
							_tmp20_ = NULL;
							param_name = (_tmp20_ = g_strdup ("_self"), param_name = (g_free (param_name), NULL), _tmp20_);
						}
					}
					direction = 0;
					param_type = vala_gidl_parser_parse_param (self, param, &direction);
					p = vala_formal_parameter_new (param_name, param_type, NULL);
					vala_formal_parameter_set_direction (p, direction);
					hide_param = FALSE;
					show_param = FALSE;
					_tmp23_ = NULL;
					_tmp21_ = NULL;
					attributes = (_tmp23_ = vala_gidl_parser_get_attributes (self, _tmp21_ = g_strdup_printf ("%s.%s", node->name, param_node->name), &_tmp22_), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), attributes_length1 = _tmp22_, attributes_size = attributes_length1, _tmp23_);
					_tmp21_ = (g_free (_tmp21_), NULL);
					if (attributes != NULL) {
						{
							char** attr_collection;
							int attr_collection_length1;
							int attr_it;
							attr_collection = attributes;
							attr_collection_length1 = attributes_length1;
							for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
								const char* _tmp36_;
								char* attr;
								_tmp36_ = NULL;
								attr = (_tmp36_ = attr_collection[attr_it], (_tmp36_ == NULL) ? NULL : g_strdup (_tmp36_));
								{
									char** _tmp25_;
									gint nv_size;
									gint nv_length1;
									char** _tmp24_;
									char** nv;
									_tmp25_ = NULL;
									_tmp24_ = NULL;
									nv = (_tmp25_ = _tmp24_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp24_), nv_size = nv_length1, _tmp25_);
									if (_vala_strcmp0 (nv[0], "hidden") == 0) {
										char* _tmp26_;
										gboolean _tmp27_;
										_tmp26_ = NULL;
										if ((_tmp27_ = _vala_strcmp0 (_tmp26_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp26_ = (g_free (_tmp26_), NULL), _tmp27_)) {
											hide_param = TRUE;
										} else {
											char* _tmp28_;
											gboolean _tmp29_;
											_tmp28_ = NULL;
											if ((_tmp29_ = _vala_strcmp0 (_tmp28_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp28_ = (g_free (_tmp28_), NULL), _tmp29_)) {
												show_param = TRUE;
											}
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_out") == 0) {
											char* _tmp30_;
											gboolean _tmp31_;
											_tmp30_ = NULL;
											if ((_tmp31_ = _vala_strcmp0 (_tmp30_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp30_ = (g_free (_tmp30_), NULL), _tmp31_)) {
												vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_OUT);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
												char* _tmp32_;
												gboolean _tmp33_;
												_tmp32_ = NULL;
												if ((_tmp33_ = _vala_strcmp0 (_tmp32_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp32_ = (g_free (_tmp32_), NULL), _tmp33_)) {
													vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_REF);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "takes_ownership") == 0) {
													char* _tmp34_;
													gboolean _tmp35_;
													_tmp34_ = NULL;
													if ((_tmp35_ = _vala_strcmp0 (_tmp34_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp34_ = (g_free (_tmp34_), NULL), _tmp35_)) {
														vala_data_type_set_value_owned (param_type, TRUE);
													}
												}
											}
										}
									}
									attr = (g_free (attr), NULL);
									nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
								}
							}
						}
					}
					_tmp37_ = FALSE;
					if (show_param) {
						_tmp37_ = TRUE;
					} else {
						_tmp37_ = !hide_param;
					}
					if (_tmp37_) {
						vala_delegate_add_parameter (cb, p);
					}
					param_name = (g_free (param_name), NULL);
					(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
					(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
				}
				remaining_params--;
			}
		}
	}
	_tmp38_ = NULL;
	return (_tmp38_ = cb, (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp38_);
}


static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname) {
	char** _tmp1_;
	gint st_attributes_size;
	gint st_attributes_length1;
	gint _tmp0_;
	char** st_attributes;
	gboolean _tmp8_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (cname != NULL, FALSE);
	_tmp1_ = NULL;
	st_attributes = (_tmp1_ = vala_gidl_parser_get_attributes (self, cname, &_tmp0_), st_attributes_length1 = _tmp0_, st_attributes_size = st_attributes_length1, _tmp1_);
	if (st_attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = st_attributes;
			attr_collection_length1 = st_attributes_length1;
			for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp7_;
				char* attr;
				_tmp7_ = NULL;
				attr = (_tmp7_ = attr_collection[attr_it], (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
				{
					char** _tmp3_;
					gint nv_size;
					gint nv_length1;
					char** _tmp2_;
					char** nv;
					gboolean _tmp4_;
					_tmp3_ = NULL;
					_tmp2_ = NULL;
					nv = (_tmp3_ = _tmp2_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp2_), nv_size = nv_length1, _tmp3_);
					_tmp4_ = FALSE;
					if (_vala_strcmp0 (nv[0], "is_value_type") == 0) {
						char* _tmp5_;
						_tmp5_ = NULL;
						_tmp4_ = _vala_strcmp0 (_tmp5_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0;
						_tmp5_ = (g_free (_tmp5_), NULL);
					} else {
						_tmp4_ = FALSE;
					}
					if (_tmp4_) {
						gboolean _tmp6_;
						return (_tmp6_ = FALSE, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp6_);
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	return (_tmp8_ = TRUE, st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp8_);
}


static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st_node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	node = (GIdlNode*) st_node;
	if (st_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0_;
		ValaStruct* st;
		ValaTypeSymbol* _tmp15_;
		ValaTypeSymbol* _tmp14_;
		ValaTypeSymbol* _tmp16_;
		_tmp0_ = NULL;
		st = (_tmp0_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL);
		if (st == NULL) {
			ValaStruct* _tmp1_;
			char** _tmp3_;
			gint st_attributes_size;
			gint st_attributes_length1;
			gint _tmp2_;
			char** st_attributes;
			_tmp1_ = NULL;
			st = (_tmp1_ = vala_struct_new (name, self->priv->current_source_reference), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp1_);
			vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp3_ = NULL;
			st_attributes = (_tmp3_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp2_), st_attributes_length1 = _tmp2_, st_attributes_size = st_attributes_length1, _tmp3_);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = st_attributes;
					attr_collection_length1 = st_attributes_length1;
					for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp13_;
						char* attr;
						_tmp13_ = NULL;
						attr = (_tmp13_ = attr_collection[attr_it], (_tmp13_ == NULL) ? NULL : g_strdup (_tmp13_));
						{
							char** _tmp5_;
							gint nv_size;
							gint nv_length1;
							char** _tmp4_;
							char** nv;
							_tmp5_ = NULL;
							_tmp4_ = NULL;
							nv = (_tmp5_ = _tmp4_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp4_), nv_size = nv_length1, _tmp5_);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp6_;
								_tmp6_ = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) st, _tmp6_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp6_ = (g_free (_tmp6_), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp7_;
									gboolean _tmp8_;
									_tmp7_ = NULL;
									if ((_tmp8_ = _vala_strcmp0 (_tmp7_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp7_ = (g_free (_tmp7_), NULL), _tmp8_)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
										(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
										name = (g_free (name), NULL);
										return;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "simple_type") == 0) {
										char* _tmp9_;
										gboolean _tmp10_;
										_tmp9_ = NULL;
										if ((_tmp10_ = _vala_strcmp0 (_tmp9_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp9_ = (g_free (_tmp9_), NULL), _tmp10_)) {
											vala_struct_set_simple_type (st, TRUE);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "use_const") == 0) {
											char* _tmp11_;
											gboolean _tmp12_;
											_tmp11_ = NULL;
											if ((_tmp12_ = _vala_strcmp0 (_tmp11_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp11_ = (g_free (_tmp11_), NULL), _tmp12_)) {
												vala_struct_set_use_const (st, FALSE);
											}
										}
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) st);
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp15_ = NULL;
		_tmp14_ = NULL;
		self->priv->current_data_type = (_tmp15_ = (_tmp14_ = (ValaTypeSymbol*) st, (_tmp14_ == NULL) ? NULL : vala_code_node_ref (_tmp14_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp15_);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		_tmp16_ = NULL;
		self->priv->current_data_type = (_tmp16_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp16_);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	} else {
		ValaSymbol* _tmp17_;
		ValaClass* cl;
		ValaTypeSymbol* _tmp36_;
		ValaTypeSymbol* _tmp35_;
		gboolean ref_function_void;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypeSymbol* _tmp47_;
		_tmp17_ = NULL;
		cl = (_tmp17_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp17_) ? ((ValaClass*) _tmp17_) : NULL);
		if (cl == NULL) {
			char* base_class;
			ValaClass* _tmp18_;
			char** _tmp20_;
			gint cl_attributes_size;
			gint cl_attributes_length1;
			gint _tmp19_;
			char** cl_attributes;
			base_class = NULL;
			_tmp18_ = NULL;
			cl = (_tmp18_ = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp18_);
			vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			vala_class_set_is_compact (cl, TRUE);
			_tmp20_ = NULL;
			cl_attributes = (_tmp20_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp19_), cl_attributes_length1 = _tmp19_, cl_attributes_size = cl_attributes_length1, _tmp20_);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = cl_attributes;
					attr_collection_length1 = cl_attributes_length1;
					for (attr_it = 0; attr_it < cl_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp34_;
						char* attr;
						_tmp34_ = NULL;
						attr = (_tmp34_ = attr_collection[attr_it], (_tmp34_ == NULL) ? NULL : g_strdup (_tmp34_));
						{
							char** _tmp22_;
							gint nv_size;
							gint nv_length1;
							char** _tmp21_;
							char** nv;
							_tmp22_ = NULL;
							_tmp21_ = NULL;
							nv = (_tmp22_ = _tmp21_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp21_), nv_size = nv_length1, _tmp22_);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp23_;
								_tmp23_ = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp23_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp23_ = (g_free (_tmp23_), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "base_class") == 0) {
									char* _tmp24_;
									_tmp24_ = NULL;
									base_class = (_tmp24_ = vala_gidl_parser_eval (self, nv[1]), base_class = (g_free (base_class), NULL), _tmp24_);
								} else {
									if (_vala_strcmp0 (nv[0], "hidden") == 0) {
										char* _tmp25_;
										gboolean _tmp26_;
										_tmp25_ = NULL;
										if ((_tmp26_ = _vala_strcmp0 (_tmp25_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp25_ = (g_free (_tmp25_), NULL), _tmp26_)) {
											attr = (g_free (attr), NULL);
											nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
											base_class = (g_free (base_class), NULL);
											cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
											(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
											name = (g_free (name), NULL);
											return;
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_immutable") == 0) {
											char* _tmp27_;
											gboolean _tmp28_;
											_tmp27_ = NULL;
											if ((_tmp28_ = _vala_strcmp0 (_tmp27_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp27_ = (g_free (_tmp27_), NULL), _tmp28_)) {
												vala_class_set_is_immutable (cl, TRUE);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "is_fundamental") == 0) {
												char* _tmp29_;
												gboolean _tmp30_;
												_tmp29_ = NULL;
												if ((_tmp30_ = _vala_strcmp0 (_tmp29_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp29_ = (g_free (_tmp29_), NULL), _tmp30_)) {
													vala_class_set_is_compact (cl, FALSE);
												}
											} else {
												gboolean _tmp31_;
												_tmp31_ = FALSE;
												if (_vala_strcmp0 (nv[0], "abstract") == 0) {
													_tmp31_ = base_class != NULL;
												} else {
													_tmp31_ = FALSE;
												}
												if (_tmp31_) {
													char* _tmp32_;
													gboolean _tmp33_;
													_tmp32_ = NULL;
													if ((_tmp33_ = _vala_strcmp0 (_tmp32_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp32_ = (g_free (_tmp32_), NULL), _tmp33_)) {
														vala_class_set_is_abstract (cl, TRUE);
													}
												}
											}
										}
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
			if (base_class != NULL) {
				ValaDataType* parent;
				parent = vala_gidl_parser_parse_type_string (self, base_class);
				vala_class_add_base_type (cl, parent);
				(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
			}
			base_class = (g_free (base_class), NULL);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp36_ = NULL;
		_tmp35_ = NULL;
		self->priv->current_data_type = (_tmp36_ = (_tmp35_ = (ValaTypeSymbol*) cl, (_tmp35_ == NULL) ? NULL : vala_code_node_ref (_tmp35_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp36_);
		ref_function_void = FALSE;
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp38_;
							const char* _tmp37_;
							ValaDataType* _tmp39_;
							_tmp38_ = NULL;
							_tmp37_ = NULL;
							ref_function = (_tmp38_ = (_tmp37_ = ((GIdlNodeFunction*) member)->symbol, (_tmp37_ == NULL) ? NULL : g_strdup (_tmp37_)), ref_function = (g_free (ref_function), NULL), _tmp38_);
							_tmp39_ = NULL;
							ref_function_void = VALA_IS_VOID_TYPE (_tmp39_ = vala_gidl_parser_parse_type (self, ((GIdlNodeFunction*) member)->result->type, NULL));
							(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_code_node_unref (_tmp39_), NULL));
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp41_;
								const char* _tmp40_;
								_tmp41_ = NULL;
								_tmp40_ = NULL;
								unref_function = (_tmp41_ = (_tmp40_ = ((GIdlNodeFunction*) member)->symbol, (_tmp40_ == NULL) ? NULL : g_strdup (_tmp40_)), unref_function = (g_free (unref_function), NULL), _tmp41_);
							} else {
								gboolean _tmp42_;
								_tmp42_ = FALSE;
								if (_vala_strcmp0 (member->name, "free") == 0) {
									_tmp42_ = TRUE;
								} else {
									_tmp42_ = _vala_strcmp0 (member->name, "destroy") == 0;
								}
								if (_tmp42_) {
									char* _tmp44_;
									const char* _tmp43_;
									_tmp44_ = NULL;
									_tmp43_ = NULL;
									free_function = (_tmp44_ = (_tmp43_ = ((GIdlNodeFunction*) member)->symbol, (_tmp43_ == NULL) ? NULL : g_strdup (_tmp43_)), free_function = (g_free (free_function), NULL), _tmp44_);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp46_;
										const char* _tmp45_;
										_tmp46_ = NULL;
										_tmp45_ = NULL;
										copy_function = (_tmp46_ = (_tmp45_ = ((GIdlNodeFunction*) member)->symbol, (_tmp45_ == NULL) ? NULL : g_strdup (_tmp45_)), copy_function = (g_free (copy_function), NULL), _tmp46_);
									}
									m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
			vala_class_set_ref_function_void (cl, ref_function_void);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp47_ = NULL;
		self->priv->current_data_type = (_tmp47_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp47_);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static void vala_gidl_parser_parse_union (ValaGIdlParser* self, GIdlNodeUnion* un_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (self != NULL);
	g_return_if_fail (un_node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	node = (GIdlNode*) un_node;
	if (un_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0_;
		ValaStruct* st;
		ValaTypeSymbol* _tmp11_;
		ValaTypeSymbol* _tmp10_;
		ValaTypeSymbol* _tmp12_;
		_tmp0_ = NULL;
		st = (_tmp0_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL);
		if (st == NULL) {
			ValaStruct* _tmp1_;
			char** _tmp3_;
			gint st_attributes_size;
			gint st_attributes_length1;
			gint _tmp2_;
			char** st_attributes;
			_tmp1_ = NULL;
			st = (_tmp1_ = vala_struct_new (name, self->priv->current_source_reference), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp1_);
			vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp3_ = NULL;
			st_attributes = (_tmp3_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp2_), st_attributes_length1 = _tmp2_, st_attributes_size = st_attributes_length1, _tmp3_);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = st_attributes;
					attr_collection_length1 = st_attributes_length1;
					for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp9_;
						char* attr;
						_tmp9_ = NULL;
						attr = (_tmp9_ = attr_collection[attr_it], (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
						{
							char** _tmp5_;
							gint nv_size;
							gint nv_length1;
							char** _tmp4_;
							char** nv;
							_tmp5_ = NULL;
							_tmp4_ = NULL;
							nv = (_tmp5_ = _tmp4_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp4_), nv_size = nv_length1, _tmp5_);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp6_;
								_tmp6_ = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) st, _tmp6_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp6_ = (g_free (_tmp6_), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp7_;
									gboolean _tmp8_;
									_tmp7_ = NULL;
									if ((_tmp8_ = _vala_strcmp0 (_tmp7_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp7_ = (g_free (_tmp7_), NULL), _tmp8_)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
										(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
										name = (g_free (name), NULL);
										return;
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) st);
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp11_ = NULL;
		_tmp10_ = NULL;
		self->priv->current_data_type = (_tmp11_ = (_tmp10_ = (ValaTypeSymbol*) st, (_tmp10_ == NULL) ? NULL : vala_code_node_ref (_tmp10_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp11_);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = un_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		_tmp12_ = NULL;
		self->priv->current_data_type = (_tmp12_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp12_);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	} else {
		ValaSymbol* _tmp13_;
		ValaClass* cl;
		ValaTypeSymbol* _tmp24_;
		ValaTypeSymbol* _tmp23_;
		gboolean ref_function_void;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypeSymbol* _tmp35_;
		_tmp13_ = NULL;
		cl = (_tmp13_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp13_) ? ((ValaClass*) _tmp13_) : NULL);
		if (cl == NULL) {
			ValaClass* _tmp14_;
			char** _tmp16_;
			gint cl_attributes_size;
			gint cl_attributes_length1;
			gint _tmp15_;
			char** cl_attributes;
			_tmp14_ = NULL;
			cl = (_tmp14_ = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp14_);
			vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			vala_class_set_is_compact (cl, TRUE);
			_tmp16_ = NULL;
			cl_attributes = (_tmp16_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp15_), cl_attributes_length1 = _tmp15_, cl_attributes_size = cl_attributes_length1, _tmp16_);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = cl_attributes;
					attr_collection_length1 = cl_attributes_length1;
					for (attr_it = 0; attr_it < cl_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp22_;
						char* attr;
						_tmp22_ = NULL;
						attr = (_tmp22_ = attr_collection[attr_it], (_tmp22_ == NULL) ? NULL : g_strdup (_tmp22_));
						{
							char** _tmp18_;
							gint nv_size;
							gint nv_length1;
							char** _tmp17_;
							char** nv;
							_tmp18_ = NULL;
							_tmp17_ = NULL;
							nv = (_tmp18_ = _tmp17_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp17_), nv_size = nv_length1, _tmp18_);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp19_;
								_tmp19_ = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp19_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp19_ = (g_free (_tmp19_), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp20_;
									gboolean _tmp21_;
									_tmp20_ = NULL;
									if ((_tmp21_ = _vala_strcmp0 (_tmp20_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp20_ = (g_free (_tmp20_), NULL), _tmp21_)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
										(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
										name = (g_free (name), NULL);
										return;
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp24_ = NULL;
		_tmp23_ = NULL;
		self->priv->current_data_type = (_tmp24_ = (_tmp23_ = (ValaTypeSymbol*) cl, (_tmp23_ == NULL) ? NULL : vala_code_node_ref (_tmp23_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp24_);
		ref_function_void = FALSE;
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = un_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp26_;
							const char* _tmp25_;
							ValaDataType* _tmp27_;
							_tmp26_ = NULL;
							_tmp25_ = NULL;
							ref_function = (_tmp26_ = (_tmp25_ = ((GIdlNodeFunction*) member)->symbol, (_tmp25_ == NULL) ? NULL : g_strdup (_tmp25_)), ref_function = (g_free (ref_function), NULL), _tmp26_);
							_tmp27_ = NULL;
							ref_function_void = VALA_IS_VOID_TYPE (_tmp27_ = vala_gidl_parser_parse_type (self, ((GIdlNodeFunction*) member)->result->type, NULL));
							(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_code_node_unref (_tmp27_), NULL));
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp29_;
								const char* _tmp28_;
								_tmp29_ = NULL;
								_tmp28_ = NULL;
								unref_function = (_tmp29_ = (_tmp28_ = ((GIdlNodeFunction*) member)->symbol, (_tmp28_ == NULL) ? NULL : g_strdup (_tmp28_)), unref_function = (g_free (unref_function), NULL), _tmp29_);
							} else {
								gboolean _tmp30_;
								_tmp30_ = FALSE;
								if (_vala_strcmp0 (member->name, "free") == 0) {
									_tmp30_ = TRUE;
								} else {
									_tmp30_ = _vala_strcmp0 (member->name, "destroy") == 0;
								}
								if (_tmp30_) {
									char* _tmp32_;
									const char* _tmp31_;
									_tmp32_ = NULL;
									_tmp31_ = NULL;
									free_function = (_tmp32_ = (_tmp31_ = ((GIdlNodeFunction*) member)->symbol, (_tmp31_ == NULL) ? NULL : g_strdup (_tmp31_)), free_function = (g_free (free_function), NULL), _tmp32_);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp34_;
										const char* _tmp33_;
										_tmp34_ = NULL;
										_tmp33_ = NULL;
										copy_function = (_tmp34_ = (_tmp33_ = ((GIdlNodeFunction*) member)->symbol, (_tmp33_ == NULL) ? NULL : g_strdup (_tmp33_)), copy_function = (g_free (copy_function), NULL), _tmp34_);
									}
									m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
			vala_class_set_ref_function_void (cl, ref_function_void);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp35_ = NULL;
		self->priv->current_data_type = (_tmp35_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp35_);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	char** _tmp1_;
	gint node_attributes_size;
	gint node_attributes_length1;
	gint _tmp0_;
	char** node_attributes;
	g_return_if_fail (self != NULL);
	g_return_if_fail (boxed_node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	node = (GIdlNode*) boxed_node;
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	_tmp1_ = NULL;
	node_attributes = (_tmp1_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp0_), node_attributes_length1 = _tmp0_, node_attributes_size = node_attributes_length1, _tmp1_);
	if (node_attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = node_attributes;
			attr_collection_length1 = node_attributes_length1;
			for (attr_it = 0; attr_it < node_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp4_;
				char* attr;
				_tmp4_ = NULL;
				attr = (_tmp4_ = attr_collection[attr_it], (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_));
				{
					char** _tmp3_;
					gint nv_size;
					gint nv_length1;
					char** _tmp2_;
					char** nv;
					_tmp3_ = NULL;
					_tmp2_ = NULL;
					nv = (_tmp3_ = _tmp2_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp2_), nv_size = nv_length1, _tmp3_);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						name = (g_free (name), NULL);
						node_attributes = (_vala_array_free (node_attributes, node_attributes_length1, (GDestroyNotify) g_free), NULL);
						return;
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp5_;
		ValaStruct* st;
		ValaTypeSymbol* _tmp17_;
		ValaTypeSymbol* _tmp16_;
		ValaTypeSymbol* _tmp18_;
		_tmp5_ = NULL;
		st = (_tmp5_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_STRUCT (_tmp5_) ? ((ValaStruct*) _tmp5_) : NULL);
		if (st == NULL) {
			ValaStruct* _tmp6_;
			char** _tmp8_;
			gint st_attributes_size;
			gint st_attributes_length1;
			gint _tmp7_;
			char** st_attributes;
			char* _tmp15_;
			_tmp6_ = NULL;
			st = (_tmp6_ = vala_struct_new (name, self->priv->current_source_reference), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp6_);
			vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp8_ = NULL;
			st_attributes = (_tmp8_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp7_), st_attributes_length1 = _tmp7_, st_attributes_size = st_attributes_length1, _tmp8_);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = st_attributes;
					attr_collection_length1 = st_attributes_length1;
					for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp14_;
						char* attr;
						_tmp14_ = NULL;
						attr = (_tmp14_ = attr_collection[attr_it], (_tmp14_ == NULL) ? NULL : g_strdup (_tmp14_));
						{
							char** _tmp10_;
							gint nv_size;
							gint nv_length1;
							char** _tmp9_;
							char** nv;
							_tmp10_ = NULL;
							_tmp9_ = NULL;
							nv = (_tmp10_ = _tmp9_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp9_), nv_size = nv_length1, _tmp10_);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp11_;
								_tmp11_ = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) st, _tmp11_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp11_ = (g_free (_tmp11_), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "use_const") == 0) {
									char* _tmp12_;
									gboolean _tmp13_;
									_tmp12_ = NULL;
									if ((_tmp13_ = _vala_strcmp0 (_tmp12_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp12_ = (g_free (_tmp12_), NULL), _tmp13_)) {
										vala_struct_set_use_const (st, FALSE);
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			_tmp15_ = NULL;
			vala_struct_set_type_id (st, _tmp15_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) st, "TYPE_"));
			_tmp15_ = (g_free (_tmp15_), NULL);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) st);
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp17_ = NULL;
		_tmp16_ = NULL;
		self->priv->current_data_type = (_tmp17_ = (_tmp16_ = (ValaTypeSymbol*) st, (_tmp16_ == NULL) ? NULL : vala_code_node_ref (_tmp16_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp17_);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		_tmp18_ = NULL;
		self->priv->current_data_type = (_tmp18_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp18_);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	} else {
		ValaSymbol* _tmp19_;
		ValaClass* cl;
		ValaTypeSymbol* _tmp31_;
		ValaTypeSymbol* _tmp30_;
		gboolean ref_function_void;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypeSymbol* _tmp42_;
		_tmp19_ = NULL;
		cl = (_tmp19_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp19_) ? ((ValaClass*) _tmp19_) : NULL);
		if (cl == NULL) {
			ValaClass* _tmp20_;
			char** _tmp22_;
			gint cl_attributes_size;
			gint cl_attributes_length1;
			gint _tmp21_;
			char** cl_attributes;
			char* _tmp29_;
			_tmp20_ = NULL;
			cl = (_tmp20_ = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp20_);
			vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			vala_class_set_is_compact (cl, TRUE);
			_tmp22_ = NULL;
			cl_attributes = (_tmp22_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp21_), cl_attributes_length1 = _tmp21_, cl_attributes_size = cl_attributes_length1, _tmp22_);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = cl_attributes;
					attr_collection_length1 = cl_attributes_length1;
					for (attr_it = 0; attr_it < cl_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp28_;
						char* attr;
						_tmp28_ = NULL;
						attr = (_tmp28_ = attr_collection[attr_it], (_tmp28_ == NULL) ? NULL : g_strdup (_tmp28_));
						{
							char** _tmp24_;
							gint nv_size;
							gint nv_length1;
							char** _tmp23_;
							char** nv;
							_tmp24_ = NULL;
							_tmp23_ = NULL;
							nv = (_tmp24_ = _tmp23_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp23_), nv_size = nv_length1, _tmp24_);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp25_;
								_tmp25_ = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp25_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp25_ = (g_free (_tmp25_), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "is_immutable") == 0) {
									char* _tmp26_;
									gboolean _tmp27_;
									_tmp26_ = NULL;
									if ((_tmp27_ = _vala_strcmp0 (_tmp26_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp26_ = (g_free (_tmp26_), NULL), _tmp27_)) {
										vala_class_set_is_immutable (cl, TRUE);
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			_tmp29_ = NULL;
			vala_class_set_type_id (cl, _tmp29_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) cl, "TYPE_"));
			_tmp29_ = (g_free (_tmp29_), NULL);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp31_ = NULL;
		_tmp30_ = NULL;
		self->priv->current_data_type = (_tmp31_ = (_tmp30_ = (ValaTypeSymbol*) cl, (_tmp30_ == NULL) ? NULL : vala_code_node_ref (_tmp30_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp31_);
		ref_function_void = FALSE;
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp33_;
							const char* _tmp32_;
							ValaDataType* _tmp34_;
							_tmp33_ = NULL;
							_tmp32_ = NULL;
							ref_function = (_tmp33_ = (_tmp32_ = ((GIdlNodeFunction*) member)->symbol, (_tmp32_ == NULL) ? NULL : g_strdup (_tmp32_)), ref_function = (g_free (ref_function), NULL), _tmp33_);
							_tmp34_ = NULL;
							ref_function_void = VALA_IS_VOID_TYPE (_tmp34_ = vala_gidl_parser_parse_type (self, ((GIdlNodeFunction*) member)->result->type, NULL));
							(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_code_node_unref (_tmp34_), NULL));
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp36_;
								const char* _tmp35_;
								_tmp36_ = NULL;
								_tmp35_ = NULL;
								unref_function = (_tmp36_ = (_tmp35_ = ((GIdlNodeFunction*) member)->symbol, (_tmp35_ == NULL) ? NULL : g_strdup (_tmp35_)), unref_function = (g_free (unref_function), NULL), _tmp36_);
							} else {
								gboolean _tmp37_;
								_tmp37_ = FALSE;
								if (_vala_strcmp0 (member->name, "free") == 0) {
									_tmp37_ = TRUE;
								} else {
									_tmp37_ = _vala_strcmp0 (member->name, "destroy") == 0;
								}
								if (_tmp37_) {
									char* _tmp39_;
									const char* _tmp38_;
									_tmp39_ = NULL;
									_tmp38_ = NULL;
									free_function = (_tmp39_ = (_tmp38_ = ((GIdlNodeFunction*) member)->symbol, (_tmp38_ == NULL) ? NULL : g_strdup (_tmp38_)), free_function = (g_free (free_function), NULL), _tmp39_);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp41_;
										const char* _tmp40_;
										_tmp41_ = NULL;
										_tmp40_ = NULL;
										copy_function = (_tmp41_ = (_tmp40_ = ((GIdlNodeFunction*) member)->symbol, (_tmp40_ == NULL) ? NULL : g_strdup (_tmp40_)), copy_function = (g_free (copy_function), NULL), _tmp41_);
									}
									m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
			vala_class_set_ref_function_void (cl, ref_function_void);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp42_ = NULL;
		self->priv->current_data_type = (_tmp42_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp42_);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
	node_attributes = (_vala_array_free (node_attributes, node_attributes_length1, (GDestroyNotify) g_free), NULL);
}


static ValaTypeSymbol* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node) {
	GIdlNode* node;
	ValaEnum* en;
	gboolean _tmp0_;
	char* common_prefix;
	gboolean is_errordomain;
	GeeArrayList* cheader_filenames;
	char** _tmp11_;
	gint en_attributes_size;
	gint en_attributes_length1;
	gint _tmp10_;
	char** en_attributes;
	ValaTypeSymbol* _tmp30_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (en_node != NULL, NULL);
	node = (GIdlNode*) en_node;
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp0_ = FALSE;
	if (en_node->gtype_name != NULL) {
		_tmp0_ = _vala_strcmp0 (en_node->gtype_name, "") != 0;
	} else {
		_tmp0_ = FALSE;
	}
	vala_enum_set_has_type_id (en, _tmp0_);
	common_prefix = NULL;
	{
		GList* value_collection;
		GList* value_it;
		value_collection = en_node->values;
		for (value_it = value_collection; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = (GIdlNode*) value_it->data;
			{
				if (common_prefix == NULL) {
					char* _tmp2_;
					const char* _tmp1_;
					_tmp2_ = NULL;
					_tmp1_ = NULL;
					common_prefix = (_tmp2_ = (_tmp1_ = value->name, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), common_prefix = (g_free (common_prefix), NULL), _tmp2_);
					while (TRUE) {
						gboolean _tmp3_;
						char* _tmp4_;
						_tmp3_ = FALSE;
						if (g_utf8_strlen (common_prefix, -1) > 0) {
							_tmp3_ = !g_str_has_suffix (common_prefix, "_");
						} else {
							_tmp3_ = FALSE;
						}
						if (!_tmp3_) {
							break;
						}
						/* FIXME: could easily be made faster*/
						_tmp4_ = NULL;
						common_prefix = (_tmp4_ = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp4_);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* _tmp5_;
						_tmp5_ = NULL;
						common_prefix = (_tmp5_ = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp5_);
					}
				}
				while (TRUE) {
					gboolean _tmp6_;
					char* _tmp9_;
					_tmp6_ = FALSE;
					if (g_utf8_strlen (common_prefix, -1) > 0) {
						gboolean _tmp7_;
						_tmp7_ = FALSE;
						if (!g_str_has_suffix (common_prefix, "_")) {
							_tmp7_ = TRUE;
						} else {
							gboolean _tmp8_;
							_tmp8_ = FALSE;
							if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (value->name, strlen (common_prefix))))) {
								_tmp8_ = (g_utf8_strlen (value->name, -1) - g_utf8_strlen (common_prefix, -1)) <= 1;
							} else {
								_tmp8_ = FALSE;
							}
							_tmp7_ = _tmp8_;
						}
						_tmp6_ = _tmp7_;
					} else {
						_tmp6_ = FALSE;
					}
					if (!_tmp6_) {
						break;
					}
					/* enum values may not consist solely of digits*/
					_tmp9_ = NULL;
					common_prefix = (_tmp9_ = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp9_);
				}
			}
		}
	}
	is_errordomain = FALSE;
	cheader_filenames = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	_tmp11_ = NULL;
	en_attributes = (_tmp11_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp10_), en_attributes_length1 = _tmp10_, en_attributes_size = en_attributes_length1, _tmp11_);
	if (en_attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = en_attributes;
			attr_collection_length1 = en_attributes_length1;
			for (attr_it = 0; attr_it < en_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp25_;
				char* attr;
				_tmp25_ = NULL;
				attr = (_tmp25_ = attr_collection[attr_it], (_tmp25_ == NULL) ? NULL : g_strdup (_tmp25_));
				{
					char** _tmp13_;
					gint nv_size;
					gint nv_length1;
					char** _tmp12_;
					char** nv;
					_tmp13_ = NULL;
					_tmp12_ = NULL;
					nv = (_tmp13_ = _tmp12_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp12_), nv_size = nv_length1, _tmp13_);
					if (_vala_strcmp0 (nv[0], "common_prefix") == 0) {
						char* _tmp14_;
						_tmp14_ = NULL;
						common_prefix = (_tmp14_ = vala_gidl_parser_eval (self, nv[1]), common_prefix = (g_free (common_prefix), NULL), _tmp14_);
					} else {
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp15_;
							char* _tmp16_;
							_tmp15_ = NULL;
							gee_collection_add ((GeeCollection*) cheader_filenames, _tmp15_ = vala_gidl_parser_eval (self, nv[1]));
							_tmp15_ = (g_free (_tmp15_), NULL);
							_tmp16_ = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) en, _tmp16_ = vala_gidl_parser_eval (self, nv[1]));
							_tmp16_ = (g_free (_tmp16_), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "hidden") == 0) {
								char* _tmp17_;
								gboolean _tmp18_;
								_tmp17_ = NULL;
								if ((_tmp18_ = _vala_strcmp0 (_tmp17_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp17_ = (g_free (_tmp17_), NULL), _tmp18_)) {
									ValaTypeSymbol* _tmp19_;
									_tmp19_ = NULL;
									return (_tmp19_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL)), common_prefix = (g_free (common_prefix), NULL), (cheader_filenames == NULL) ? NULL : (cheader_filenames = (gee_collection_object_unref (cheader_filenames), NULL)), en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp19_);
								}
							} else {
								if (_vala_strcmp0 (nv[0], "rename_to") == 0) {
									char* _tmp20_;
									_tmp20_ = NULL;
									vala_symbol_set_name ((ValaSymbol*) en, _tmp20_ = vala_gidl_parser_eval (self, nv[1]));
									_tmp20_ = (g_free (_tmp20_), NULL);
								} else {
									if (_vala_strcmp0 (nv[0], "errordomain") == 0) {
										char* _tmp21_;
										gboolean _tmp22_;
										_tmp21_ = NULL;
										if ((_tmp22_ = _vala_strcmp0 (_tmp21_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp21_ = (g_free (_tmp21_), NULL), _tmp22_)) {
											is_errordomain = TRUE;
										}
									} else {
										if (_vala_strcmp0 (nv[0], "to_string") == 0) {
											ValaUnresolvedType* return_type;
											ValaUnresolvedSymbol* _tmp23_;
											ValaMethod* m;
											char* _tmp24_;
											return_type = vala_unresolved_type_new ();
											_tmp23_ = NULL;
											vala_unresolved_type_set_unresolved_symbol (return_type, _tmp23_ = vala_unresolved_symbol_new (NULL, "string", NULL));
											(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_code_node_unref (_tmp23_), NULL));
											vala_data_type_set_value_owned ((ValaDataType*) return_type, FALSE);
											m = vala_method_new ("to_string", (ValaDataType*) return_type, self->priv->current_source_reference);
											vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
											_tmp24_ = NULL;
											vala_method_set_cname (m, _tmp24_ = vala_gidl_parser_eval (self, nv[1]));
											_tmp24_ = (g_free (_tmp24_), NULL);
											vala_enum_add_method (en, m);
											(return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL));
											(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* value2_collection;
		GList* value2_it;
		value2_collection = en_node->values;
		for (value2_it = value2_collection; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = (GIdlNode*) value2_it->data;
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)), NULL);
				vala_enum_add_value (en, ev);
				(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
			}
		}
	}
	if (is_errordomain) {
		ValaErrorDomain* ed;
		ValaTypeSymbol* _tmp29_;
		ed = vala_error_domain_new (vala_symbol_get_name ((ValaSymbol*) en), self->priv->current_source_reference);
		vala_symbol_set_access ((ValaSymbol*) ed, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		vala_error_domain_set_cprefix (ed, common_prefix);
		{
			GeeIterator* _filename_it;
			_filename_it = gee_iterable_iterator ((GeeIterable*) cheader_filenames);
			while (gee_iterator_next (_filename_it)) {
				char* filename;
				filename = (char*) gee_iterator_get (_filename_it);
				vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) ed, filename);
				filename = (g_free (filename), NULL);
			}
			(_filename_it == NULL) ? NULL : (_filename_it = (gee_collection_object_unref (_filename_it), NULL));
		}
		{
			GeeList* _tmp26_;
			GeeIterator* _tmp27_;
			GeeIterator* _ev_it;
			_tmp26_ = NULL;
			_tmp27_ = NULL;
			_ev_it = (_tmp27_ = gee_iterable_iterator ((GeeIterable*) (_tmp26_ = vala_enum_get_values (en))), (_tmp26_ == NULL) ? NULL : (_tmp26_ = (gee_collection_object_unref (_tmp26_), NULL)), _tmp27_);
			while (gee_iterator_next (_ev_it)) {
				ValaEnumValue* ev;
				ValaErrorCode* _tmp28_;
				ev = (ValaEnumValue*) gee_iterator_get (_ev_it);
				_tmp28_ = NULL;
				vala_error_domain_add_code (ed, _tmp28_ = vala_error_code_new (vala_symbol_get_name ((ValaSymbol*) ev), NULL));
				(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_code_node_unref (_tmp28_), NULL));
				(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
			}
			(_ev_it == NULL) ? NULL : (_ev_it = (gee_collection_object_unref (_ev_it), NULL));
		}
		_tmp29_ = NULL;
		return (_tmp29_ = (ValaTypeSymbol*) ed, (en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL)), common_prefix = (g_free (common_prefix), NULL), (cheader_filenames == NULL) ? NULL : (cheader_filenames = (gee_collection_object_unref (cheader_filenames), NULL)), en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp29_);
	}
	_tmp30_ = NULL;
	return (_tmp30_ = (ValaTypeSymbol*) en, common_prefix = (g_free (common_prefix), NULL), (cheader_filenames == NULL) ? NULL : (cheader_filenames = (gee_collection_object_unref (cheader_filenames), NULL)), en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp30_);
}


static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	char* base_class;
	ValaSymbol* _tmp0_;
	ValaClass* cl;
	ValaTypeSymbol* _tmp19_;
	ValaTypeSymbol* _tmp18_;
	GeeSet* _tmp20_;
	GeeHashMap* current_type_func_map;
	GeeHashMap* current_type_vfunc_map;
	ValaTypeSymbol* _tmp27_;
	GeeSet* _tmp28_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	name = vala_gidl_parser_fix_type_name (self, ((GIdlNode*) node)->name, ns);
	base_class = NULL;
	_tmp0_ = NULL;
	cl = (_tmp0_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL);
	if (cl == NULL) {
		ValaClass* _tmp1_;
		char** _tmp3_;
		gint attributes_size;
		gint attributes_length1;
		gint _tmp2_;
		char** attributes;
		_tmp1_ = NULL;
		cl = (_tmp1_ = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp1_);
		vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp3_ = NULL;
		attributes = (_tmp3_ = vala_gidl_parser_get_attributes (self, node->gtype_name, &_tmp2_), attributes_length1 = _tmp2_, attributes_size = attributes_length1, _tmp3_);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_collection_length1;
				int attr_it;
				attr_collection = attributes;
				attr_collection_length1 = attributes_length1;
				for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
					const char* _tmp13_;
					char* attr;
					_tmp13_ = NULL;
					attr = (_tmp13_ = attr_collection[attr_it], (_tmp13_ == NULL) ? NULL : g_strdup (_tmp13_));
					{
						char** _tmp5_;
						gint nv_size;
						gint nv_length1;
						char** _tmp4_;
						char** nv;
						_tmp5_ = NULL;
						_tmp4_ = NULL;
						nv = (_tmp5_ = _tmp4_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp4_), nv_size = nv_length1, _tmp5_);
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp6_;
							_tmp6_ = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp6_ = vala_gidl_parser_eval (self, nv[1]));
							_tmp6_ = (g_free (_tmp6_), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "base_class") == 0) {
								char* _tmp7_;
								_tmp7_ = NULL;
								base_class = (_tmp7_ = vala_gidl_parser_eval (self, nv[1]), base_class = (g_free (base_class), NULL), _tmp7_);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp8_;
									gboolean _tmp9_;
									_tmp8_ = NULL;
									if ((_tmp9_ = _vala_strcmp0 (_tmp8_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL);
										name = (g_free (name), NULL);
										base_class = (g_free (base_class), NULL);
										(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
										return;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "type_check_function") == 0) {
										char* _tmp10_;
										_tmp10_ = NULL;
										vala_class_set_type_check_function (cl, _tmp10_ = vala_gidl_parser_eval (self, nv[1]));
										_tmp10_ = (g_free (_tmp10_), NULL);
									} else {
										if (_vala_strcmp0 (nv[0], "abstract") == 0) {
											char* _tmp11_;
											gboolean _tmp12_;
											_tmp11_ = NULL;
											if ((_tmp12_ = _vala_strcmp0 (_tmp11_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp11_ = (g_free (_tmp11_), NULL), _tmp12_)) {
												vala_class_set_is_abstract (cl, TRUE);
											}
										}
									}
								}
							}
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
					}
				}
			}
		}
		vala_namespace_add_class (ns, cl);
		vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
		attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL);
	}
	if (base_class != NULL) {
		ValaDataType* parent;
		parent = vala_gidl_parser_parse_type_string (self, base_class);
		vala_class_add_base_type (cl, parent);
		(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
	} else {
		if (node->parent != NULL) {
			ValaDataType* parent;
			parent = vala_gidl_parser_parse_type_string (self, node->parent);
			vala_class_add_base_type (cl, parent);
			(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
		} else {
			ValaUnresolvedSymbol* _tmp14_;
			ValaUnresolvedSymbol* _tmp15_;
			ValaUnresolvedSymbol* gobject_symbol;
			ValaUnresolvedType* _tmp16_;
			_tmp14_ = NULL;
			_tmp15_ = NULL;
			gobject_symbol = (_tmp15_ = vala_unresolved_symbol_new (_tmp14_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "Object", NULL), (_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_code_node_unref (_tmp14_), NULL)), _tmp15_);
			_tmp16_ = NULL;
			vala_class_add_base_type (cl, (ValaDataType*) (_tmp16_ = vala_unresolved_type_new_from_symbol (gobject_symbol, NULL)));
			(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_code_node_unref (_tmp16_), NULL));
			(gobject_symbol == NULL) ? NULL : (gobject_symbol = (vala_code_node_unref (gobject_symbol), NULL));
		}
	}
	{
		GList* iface_name_collection;
		GList* iface_name_it;
		iface_name_collection = node->interfaces;
		for (iface_name_it = iface_name_collection; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			const char* _tmp17_;
			char* iface_name;
			_tmp17_ = NULL;
			iface_name = (_tmp17_ = (const char*) iface_name_it->data, (_tmp17_ == NULL) ? NULL : g_strdup (_tmp17_));
			{
				ValaDataType* iface;
				iface = vala_gidl_parser_parse_type_string (self, iface_name);
				vala_class_add_base_type (cl, iface);
				iface_name = (g_free (iface_name), NULL);
				(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			}
		}
	}
	_tmp19_ = NULL;
	_tmp18_ = NULL;
	self->priv->current_data_type = (_tmp19_ = (_tmp18_ = (ValaTypeSymbol*) cl, (_tmp18_ == NULL) ? NULL : vala_code_node_ref (_tmp18_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp19_);
	_tmp20_ = NULL;
	self->priv->current_type_symbol_set = (_tmp20_ = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->priv->current_type_symbol_set == NULL) ? NULL : (self->priv->current_type_symbol_set = (gee_collection_object_unref (self->priv->current_type_symbol_set), NULL)), _tmp20_);
	current_type_func_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_POINTER, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	current_type_vfunc_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gee_map_set ((GeeMap*) current_type_func_map, member->name, (GIdlNodeFunction*) member);
				}
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set ((GeeMap*) current_type_vfunc_map, member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					/* Ignore if vfunc (handled below) */
					if (!gee_map_contains ((GeeMap*) current_type_vfunc_map, member->name)) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_class_add_method (cl, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					}
				} else {
					if (member->type == G_IDL_NODE_VFUNC) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_virtual (self, (GIdlNodeVFunc*) member, (GIdlNodeFunction*) gee_map_get ((GeeMap*) current_type_func_map, member->name), FALSE);
						if (m != NULL) {
							vala_class_add_method (cl, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_PROPERTY) {
							ValaProperty* prop;
							prop = vala_gidl_parser_parse_property (self, (GIdlNodeProperty*) member);
							if (prop != NULL) {
								vala_class_add_property (cl, prop);
							}
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						} else {
							if (member->type == G_IDL_NODE_SIGNAL) {
								ValaSignal* sig;
								sig = vala_gidl_parser_parse_signal (self, (GIdlNodeSignal*) member);
								if (sig != NULL) {
									vala_class_add_signal (cl, sig);
								}
								(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
							}
						}
					}
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (!gee_collection_contains ((GeeCollection*) self->priv->current_type_symbol_set, member->name)) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
		}
	}
	{
		GeeList* _tmp21_;
		GeeIterator* _tmp22_;
		GeeIterator* _prop_it;
		_tmp21_ = NULL;
		_tmp22_ = NULL;
		_prop_it = (_tmp22_ = gee_iterable_iterator ((GeeIterable*) (_tmp21_ = vala_object_type_symbol_get_properties ((ValaObjectTypeSymbol*) cl))), (_tmp21_ == NULL) ? NULL : (_tmp21_ = (gee_collection_object_unref (_tmp21_), NULL)), _tmp22_);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			char* getter;
			gboolean _tmp23_;
			char* setter;
			gboolean _tmp24_;
			gboolean _tmp25_;
			gboolean _tmp26_;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			getter = g_strdup_printf ("get_%s", vala_symbol_get_name ((ValaSymbol*) prop));
			_tmp23_ = FALSE;
			if (vala_property_get_get_accessor (prop) != NULL) {
				_tmp23_ = !gee_collection_contains ((GeeCollection*) self->priv->current_type_symbol_set, getter);
			} else {
				_tmp23_ = FALSE;
			}
			if (_tmp23_) {
				vala_property_set_no_accessor_method (prop, TRUE);
			}
			setter = g_strdup_printf ("set_%s", vala_symbol_get_name ((ValaSymbol*) prop));
			_tmp24_ = FALSE;
			_tmp25_ = FALSE;
			if (vala_property_get_set_accessor (prop) != NULL) {
				_tmp25_ = vala_property_accessor_get_writable (vala_property_get_set_accessor (prop));
			} else {
				_tmp25_ = FALSE;
			}
			if (_tmp25_) {
				_tmp24_ = !gee_collection_contains ((GeeCollection*) self->priv->current_type_symbol_set, setter);
			} else {
				_tmp24_ = FALSE;
			}
			if (_tmp24_) {
				vala_property_set_no_accessor_method (prop, TRUE);
			}
			_tmp26_ = FALSE;
			if (vala_property_get_no_accessor_method (prop)) {
				_tmp26_ = vala_property_get_get_accessor (prop) != NULL;
			} else {
				_tmp26_ = FALSE;
			}
			if (_tmp26_) {
				vala_data_type_set_value_owned (vala_property_accessor_get_value_type (vala_property_get_get_accessor (prop)), TRUE);
			}
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			getter = (g_free (getter), NULL);
			setter = (g_free (setter), NULL);
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	_tmp27_ = NULL;
	self->priv->current_data_type = (_tmp27_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp27_);
	_tmp28_ = NULL;
	self->priv->current_type_symbol_set = (_tmp28_ = NULL, (self->priv->current_type_symbol_set == NULL) ? NULL : (self->priv->current_type_symbol_set = (gee_collection_object_unref (self->priv->current_type_symbol_set), NULL)), _tmp28_);
	name = (g_free (name), NULL);
	base_class = (g_free (base_class), NULL);
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(current_type_func_map == NULL) ? NULL : (current_type_func_map = (gee_collection_object_unref (current_type_func_map), NULL));
	(current_type_vfunc_map == NULL) ? NULL : (current_type_vfunc_map = (gee_collection_object_unref (current_type_vfunc_map), NULL));
}


static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	ValaSymbol* _tmp0_;
	ValaInterface* iface;
	ValaTypeSymbol* _tmp11_;
	ValaTypeSymbol* _tmp10_;
	GeeHashMap* current_type_func_map;
	GeeHashMap* current_type_vfunc_map;
	ValaTypeSymbol* _tmp12_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	name = vala_gidl_parser_fix_type_name (self, node->gtype_name, ns);
	_tmp0_ = NULL;
	iface = (_tmp0_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_INTERFACE (_tmp0_) ? ((ValaInterface*) _tmp0_) : NULL);
	if (iface == NULL) {
		ValaInterface* _tmp1_;
		char** _tmp3_;
		gint attributes_size;
		gint attributes_length1;
		gint _tmp2_;
		char** attributes;
		_tmp1_ = NULL;
		iface = (_tmp1_ = vala_interface_new (name, self->priv->current_source_reference), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp1_);
		vala_symbol_set_access ((ValaSymbol*) iface, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp3_ = NULL;
		attributes = (_tmp3_ = vala_gidl_parser_get_attributes (self, node->gtype_name, &_tmp2_), attributes_length1 = _tmp2_, attributes_size = attributes_length1, _tmp3_);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_collection_length1;
				int attr_it;
				attr_collection = attributes;
				attr_collection_length1 = attributes_length1;
				for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
					const char* _tmp8_;
					char* attr;
					_tmp8_ = NULL;
					attr = (_tmp8_ = attr_collection[attr_it], (_tmp8_ == NULL) ? NULL : g_strdup (_tmp8_));
					{
						char** _tmp5_;
						gint nv_size;
						gint nv_length1;
						char** _tmp4_;
						char** nv;
						_tmp5_ = NULL;
						_tmp4_ = NULL;
						nv = (_tmp5_ = _tmp4_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp4_), nv_size = nv_length1, _tmp5_);
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp6_;
							_tmp6_ = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) iface, _tmp6_ = vala_gidl_parser_eval (self, nv[1]));
							_tmp6_ = (g_free (_tmp6_), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "lower_case_csuffix") == 0) {
								char* _tmp7_;
								_tmp7_ = NULL;
								vala_interface_set_lower_case_csuffix (iface, _tmp7_ = vala_gidl_parser_eval (self, nv[1]));
								_tmp7_ = (g_free (_tmp7_), NULL);
							}
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
					}
				}
			}
		}
		{
			GList* prereq_name_collection;
			GList* prereq_name_it;
			prereq_name_collection = node->prerequisites;
			for (prereq_name_it = prereq_name_collection; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
				const char* _tmp9_;
				char* prereq_name;
				_tmp9_ = NULL;
				prereq_name = (_tmp9_ = (const char*) prereq_name_it->data, (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
				{
					ValaDataType* prereq;
					prereq = vala_gidl_parser_parse_type_string (self, prereq_name);
					vala_interface_add_prerequisite (iface, prereq);
					prereq_name = (g_free (prereq_name), NULL);
					(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				}
			}
		}
		vala_namespace_add_interface (ns, iface);
		vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) iface);
		attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL);
	}
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	self->priv->current_data_type = (_tmp11_ = (_tmp10_ = (ValaTypeSymbol*) iface, (_tmp10_ == NULL) ? NULL : vala_code_node_ref (_tmp10_)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp11_);
	current_type_func_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_POINTER, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	current_type_vfunc_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gee_map_set ((GeeMap*) current_type_func_map, member->name, (GIdlNodeFunction*) member);
				}
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set ((GeeMap*) current_type_vfunc_map, member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					/* Ignore if vfunc (handled below) */
					if (!gee_map_contains ((GeeMap*) current_type_vfunc_map, member->name)) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, TRUE);
						if (m != NULL) {
							vala_interface_add_method (iface, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					}
				} else {
					if (member->type == G_IDL_NODE_VFUNC) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_virtual (self, (GIdlNodeVFunc*) member, (GIdlNodeFunction*) gee_map_get ((GeeMap*) current_type_func_map, member->name), TRUE);
						if (m != NULL) {
							vala_interface_add_method (iface, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_PROPERTY) {
							ValaProperty* prop;
							prop = vala_gidl_parser_parse_property (self, (GIdlNodeProperty*) member);
							if (prop != NULL) {
								vala_interface_add_property (iface, prop);
							}
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						} else {
							if (member->type == G_IDL_NODE_SIGNAL) {
								ValaSignal* sig;
								sig = vala_gidl_parser_parse_signal (self, (GIdlNodeSignal*) member);
								if (sig != NULL) {
									vala_interface_add_signal (iface, sig);
								}
								(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
							}
						}
					}
				}
			}
		}
	}
	_tmp12_ = NULL;
	self->priv->current_data_type = (_tmp12_ = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp12_);
	name = (g_free (name), NULL);
	(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
	(current_type_func_map == NULL) ? NULL : (current_type_func_map = (gee_collection_object_unref (current_type_func_map), NULL));
	(current_type_vfunc_map == NULL) ? NULL : (current_type_vfunc_map = (gee_collection_object_unref (current_type_vfunc_map), NULL));
}


static ValaDataType* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node, ValaParameterDirection* direction) {
	ValaParameterDirection dir;
	ValaUnresolvedType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_node != NULL, NULL);
	dir = VALA_PARAMETER_DIRECTION_IN;
	type = vala_unresolved_type_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			ValaVoidType* _tmp0_;
			ValaDataType* _tmp1_;
			ValaDataType* _tmp2_;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_tmp2_ = NULL;
			return (_tmp2_ = (_tmp1_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp0_ = vala_void_type_new (NULL)), NULL), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp1_), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2_);
		} else {
			ValaDataType* _tmp3_;
			_tmp3_ = NULL;
			return (_tmp3_ = (ValaDataType*) vala_void_type_new (NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp3_);
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			ValaUnresolvedSymbol* _tmp4_;
			_tmp4_ = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, _tmp4_ = vala_unresolved_symbol_new (NULL, "bool", NULL));
			(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				ValaUnresolvedSymbol* _tmp5_;
				_tmp5_ = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, _tmp5_ = vala_unresolved_symbol_new (NULL, "char", NULL));
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					ValaUnresolvedSymbol* _tmp6_;
					_tmp6_ = NULL;
					vala_unresolved_type_set_unresolved_symbol (type, _tmp6_ = vala_unresolved_symbol_new (NULL, "uchar", NULL));
					(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						ValaUnresolvedSymbol* _tmp7_;
						_tmp7_ = NULL;
						vala_unresolved_type_set_unresolved_symbol (type, _tmp7_ = vala_unresolved_symbol_new (NULL, "int16", NULL));
						(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							ValaUnresolvedSymbol* _tmp8_;
							_tmp8_ = NULL;
							vala_unresolved_type_set_unresolved_symbol (type, _tmp8_ = vala_unresolved_symbol_new (NULL, "uint16", NULL));
							(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								ValaUnresolvedSymbol* _tmp9_;
								_tmp9_ = NULL;
								vala_unresolved_type_set_unresolved_symbol (type, _tmp9_ = vala_unresolved_symbol_new (NULL, "int32", NULL));
								(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_code_node_unref (_tmp9_), NULL));
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									ValaUnresolvedSymbol* _tmp10_;
									_tmp10_ = NULL;
									vala_unresolved_type_set_unresolved_symbol (type, _tmp10_ = vala_unresolved_symbol_new (NULL, "uint32", NULL));
									(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										ValaUnresolvedSymbol* _tmp11_;
										_tmp11_ = NULL;
										vala_unresolved_type_set_unresolved_symbol (type, _tmp11_ = vala_unresolved_symbol_new (NULL, "int64", NULL));
										(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_code_node_unref (_tmp11_), NULL));
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											ValaUnresolvedSymbol* _tmp12_;
											_tmp12_ = NULL;
											vala_unresolved_type_set_unresolved_symbol (type, _tmp12_ = vala_unresolved_symbol_new (NULL, "uint64", NULL));
											(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_code_node_unref (_tmp12_), NULL));
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												ValaUnresolvedSymbol* _tmp13_;
												_tmp13_ = NULL;
												vala_unresolved_type_set_unresolved_symbol (type, _tmp13_ = vala_unresolved_symbol_new (NULL, "int", NULL));
												(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_code_node_unref (_tmp13_), NULL));
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													ValaUnresolvedSymbol* _tmp14_;
													_tmp14_ = NULL;
													vala_unresolved_type_set_unresolved_symbol (type, _tmp14_ = vala_unresolved_symbol_new (NULL, "uint", NULL));
													(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_code_node_unref (_tmp14_), NULL));
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														ValaUnresolvedSymbol* _tmp15_;
														_tmp15_ = NULL;
														vala_unresolved_type_set_unresolved_symbol (type, _tmp15_ = vala_unresolved_symbol_new (NULL, "long", NULL));
														(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_code_node_unref (_tmp15_), NULL));
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															ValaUnresolvedSymbol* _tmp16_;
															_tmp16_ = NULL;
															vala_unresolved_type_set_unresolved_symbol (type, _tmp16_ = vala_unresolved_symbol_new (NULL, "ulong", NULL));
															(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_code_node_unref (_tmp16_), NULL));
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																ValaUnresolvedSymbol* _tmp17_;
																_tmp17_ = NULL;
																vala_unresolved_type_set_unresolved_symbol (type, _tmp17_ = vala_unresolved_symbol_new (NULL, "ssize_t", NULL));
																(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_code_node_unref (_tmp17_), NULL));
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	ValaUnresolvedSymbol* _tmp18_;
																	_tmp18_ = NULL;
																	vala_unresolved_type_set_unresolved_symbol (type, _tmp18_ = vala_unresolved_symbol_new (NULL, "size_t", NULL));
																	(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_code_node_unref (_tmp18_), NULL));
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		ValaUnresolvedSymbol* _tmp19_;
																		_tmp19_ = NULL;
																		vala_unresolved_type_set_unresolved_symbol (type, _tmp19_ = vala_unresolved_symbol_new (NULL, "float", NULL));
																		(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_code_node_unref (_tmp19_), NULL));
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			ValaUnresolvedSymbol* _tmp20_;
																			_tmp20_ = NULL;
																			vala_unresolved_type_set_unresolved_symbol (type, _tmp20_ = vala_unresolved_symbol_new (NULL, "double", NULL));
																			(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_code_node_unref (_tmp20_), NULL));
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				ValaUnresolvedSymbol* _tmp21_;
																				_tmp21_ = NULL;
																				vala_unresolved_type_set_unresolved_symbol (type, _tmp21_ = vala_unresolved_symbol_new (NULL, "string", NULL));
																				(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_code_node_unref (_tmp21_), NULL));
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					ValaUnresolvedSymbol* _tmp22_;
																					_tmp22_ = NULL;
																					vala_unresolved_type_set_unresolved_symbol (type, _tmp22_ = vala_unresolved_symbol_new (NULL, "string", NULL));
																					(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_code_node_unref (_tmp22_), NULL));
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaDataType* element_type;
																						ValaUnresolvedType* _tmp25_;
																						ValaUnresolvedType* _tmp24_;
																						ValaDataType* _tmp23_;
																						ValaDataType* _tmp27_;
																						element_type = vala_gidl_parser_parse_type (self, type_node->parameter_type1, NULL);
																						_tmp25_ = NULL;
																						_tmp24_ = NULL;
																						_tmp23_ = NULL;
																						type = (_tmp25_ = (_tmp24_ = (_tmp23_ = element_type, VALA_IS_UNRESOLVED_TYPE (_tmp23_) ? ((ValaUnresolvedType*) _tmp23_) : NULL), (_tmp24_ == NULL) ? NULL : vala_code_node_ref (_tmp24_)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp25_);
																						if (type == NULL) {
																							ValaDataType* _tmp26_;
																							_tmp26_ = NULL;
																							return (_tmp26_ = element_type, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp26_);
																						}
																						_tmp27_ = NULL;
																						return (_tmp27_ = (ValaDataType*) vala_array_type_new (element_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) element_type)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp27_);
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							ValaUnresolvedSymbol* _tmp29_;
																							ValaUnresolvedSymbol* _tmp28_;
																							_tmp29_ = NULL;
																							_tmp28_ = NULL;
																							vala_unresolved_type_set_unresolved_symbol (type, _tmp29_ = vala_unresolved_symbol_new (_tmp28_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "List", NULL));
																							(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_code_node_unref (_tmp29_), NULL));
																							(_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_code_node_unref (_tmp28_), NULL));
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								ValaUnresolvedSymbol* _tmp31_;
																								ValaUnresolvedSymbol* _tmp30_;
																								_tmp31_ = NULL;
																								_tmp30_ = NULL;
																								vala_unresolved_type_set_unresolved_symbol (type, _tmp31_ = vala_unresolved_symbol_new (_tmp30_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "SList", NULL));
																								(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_code_node_unref (_tmp31_), NULL));
																								(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_code_node_unref (_tmp30_), NULL));
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									ValaUnresolvedSymbol* _tmp33_;
																									ValaUnresolvedSymbol* _tmp32_;
																									_tmp33_ = NULL;
																									_tmp32_ = NULL;
																									vala_unresolved_type_set_unresolved_symbol (type, _tmp33_ = vala_unresolved_symbol_new (_tmp32_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "HashTable", NULL));
																									(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_code_node_unref (_tmp33_), NULL));
																									(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_code_node_unref (_tmp32_), NULL));
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										ValaUnresolvedSymbol* _tmp35_;
																										ValaUnresolvedSymbol* _tmp34_;
																										_tmp35_ = NULL;
																										_tmp34_ = NULL;
																										vala_unresolved_type_set_unresolved_symbol (type, _tmp35_ = vala_unresolved_symbol_new (_tmp34_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "Error", NULL));
																										(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_code_node_unref (_tmp35_), NULL));
																										(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_code_node_unref (_tmp34_), NULL));
																									} else {
																										if (type_node->is_interface) {
																											const char* _tmp36_;
																											char* n;
																											gboolean _tmp40_;
																											_tmp36_ = NULL;
																											n = (_tmp36_ = type_node->interface, (_tmp36_ == NULL) ? NULL : g_strdup (_tmp36_));
																											if (_vala_strcmp0 (n, "") == 0) {
																												ValaDataType* _tmp37_;
																												_tmp37_ = NULL;
																												return (_tmp37_ = NULL, n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp37_);
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* _tmp39_;
																												const char* _tmp38_;
																												_tmp39_ = NULL;
																												_tmp38_ = NULL;
																												n = (_tmp39_ = (_tmp38_ = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (_tmp38_ == NULL) ? NULL : g_strdup (_tmp38_)), n = (g_free (n), NULL), _tmp39_);
																											}
																											_tmp40_ = FALSE;
																											if (type_node->is_pointer) {
																												gboolean _tmp41_;
																												_tmp41_ = FALSE;
																												if (_vala_strcmp0 (n, "gchar") == 0) {
																													_tmp41_ = TRUE;
																												} else {
																													_tmp41_ = _vala_strcmp0 (n, "char") == 0;
																												}
																												_tmp40_ = _tmp41_;
																											} else {
																												_tmp40_ = FALSE;
																											}
																											if (_tmp40_) {
																												ValaUnresolvedSymbol* _tmp42_;
																												_tmp42_ = NULL;
																												vala_unresolved_type_set_unresolved_symbol (type, _tmp42_ = vala_unresolved_symbol_new (NULL, "string", NULL));
																												(_tmp42_ == NULL) ? NULL : (_tmp42_ = (vala_code_node_unref (_tmp42_), NULL));
																												if (g_str_has_suffix (type_node->unparsed, "**")) {
																													dir = VALA_PARAMETER_DIRECTION_OUT;
																												}
																											} else {
																												if (_vala_strcmp0 (n, "gunichar") == 0) {
																													ValaUnresolvedSymbol* _tmp43_;
																													_tmp43_ = NULL;
																													vala_unresolved_type_set_unresolved_symbol (type, _tmp43_ = vala_unresolved_symbol_new (NULL, "unichar", NULL));
																													(_tmp43_ == NULL) ? NULL : (_tmp43_ = (vala_code_node_unref (_tmp43_), NULL));
																												} else {
																													if (_vala_strcmp0 (n, "gchar") == 0) {
																														ValaUnresolvedSymbol* _tmp44_;
																														_tmp44_ = NULL;
																														vala_unresolved_type_set_unresolved_symbol (type, _tmp44_ = vala_unresolved_symbol_new (NULL, "char", NULL));
																														(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_code_node_unref (_tmp44_), NULL));
																													} else {
																														gboolean _tmp45_;
																														_tmp45_ = FALSE;
																														if (_vala_strcmp0 (n, "guchar") == 0) {
																															_tmp45_ = TRUE;
																														} else {
																															_tmp45_ = _vala_strcmp0 (n, "guint8") == 0;
																														}
																														if (_tmp45_) {
																															ValaUnresolvedSymbol* _tmp46_;
																															_tmp46_ = NULL;
																															vala_unresolved_type_set_unresolved_symbol (type, _tmp46_ = vala_unresolved_symbol_new (NULL, "uchar", NULL));
																															(_tmp46_ == NULL) ? NULL : (_tmp46_ = (vala_code_node_unref (_tmp46_), NULL));
																															if (type_node->is_pointer) {
																																ValaDataType* _tmp47_;
																																_tmp47_ = NULL;
																																return (_tmp47_ = (ValaDataType*) vala_array_type_new ((ValaDataType*) type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp47_);
																															}
																														} else {
																															if (_vala_strcmp0 (n, "gushort") == 0) {
																																ValaUnresolvedSymbol* _tmp48_;
																																_tmp48_ = NULL;
																																vala_unresolved_type_set_unresolved_symbol (type, _tmp48_ = vala_unresolved_symbol_new (NULL, "ushort", NULL));
																																(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_code_node_unref (_tmp48_), NULL));
																															} else {
																																if (_vala_strcmp0 (n, "gshort") == 0) {
																																	ValaUnresolvedSymbol* _tmp49_;
																																	_tmp49_ = NULL;
																																	vala_unresolved_type_set_unresolved_symbol (type, _tmp49_ = vala_unresolved_symbol_new (NULL, "short", NULL));
																																	(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_code_node_unref (_tmp49_), NULL));
																																} else {
																																	gboolean _tmp50_;
																																	_tmp50_ = FALSE;
																																	if (_vala_strcmp0 (n, "gconstpointer") == 0) {
																																		_tmp50_ = TRUE;
																																	} else {
																																		_tmp50_ = _vala_strcmp0 (n, "void") == 0;
																																	}
																																	if (_tmp50_) {
																																		ValaVoidType* _tmp51_;
																																		ValaDataType* _tmp52_;
																																		ValaDataType* _tmp53_;
																																		_tmp51_ = NULL;
																																		_tmp52_ = NULL;
																																		_tmp53_ = NULL;
																																		return (_tmp53_ = (_tmp52_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp51_ = vala_void_type_new (NULL)), NULL), (_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_code_node_unref (_tmp51_), NULL)), _tmp52_), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp53_);
																																	} else {
																																		gboolean _tmp54_;
																																		_tmp54_ = FALSE;
																																		if (_vala_strcmp0 (n, "goffset") == 0) {
																																			_tmp54_ = TRUE;
																																		} else {
																																			_tmp54_ = _vala_strcmp0 (n, "off_t") == 0;
																																		}
																																		if (_tmp54_) {
																																			ValaUnresolvedSymbol* _tmp55_;
																																			_tmp55_ = NULL;
																																			vala_unresolved_type_set_unresolved_symbol (type, _tmp55_ = vala_unresolved_symbol_new (NULL, "int64", NULL));
																																			(_tmp55_ == NULL) ? NULL : (_tmp55_ = (vala_code_node_unref (_tmp55_), NULL));
																																		} else {
																																			if (_vala_strcmp0 (n, "value_array") == 0) {
																																				ValaUnresolvedSymbol* _tmp57_;
																																				ValaUnresolvedSymbol* _tmp56_;
																																				_tmp57_ = NULL;
																																				_tmp56_ = NULL;
																																				vala_unresolved_type_set_unresolved_symbol (type, _tmp57_ = vala_unresolved_symbol_new (_tmp56_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "ValueArray", NULL));
																																				(_tmp57_ == NULL) ? NULL : (_tmp57_ = (vala_code_node_unref (_tmp57_), NULL));
																																				(_tmp56_ == NULL) ? NULL : (_tmp56_ = (vala_code_node_unref (_tmp56_), NULL));
																																			} else {
																																				if (_vala_strcmp0 (n, "time_t") == 0) {
																																					ValaUnresolvedSymbol* _tmp58_;
																																					_tmp58_ = NULL;
																																					vala_unresolved_type_set_unresolved_symbol (type, _tmp58_ = vala_unresolved_symbol_new (NULL, "ulong", NULL));
																																					(_tmp58_ == NULL) ? NULL : (_tmp58_ = (vala_code_node_unref (_tmp58_), NULL));
																																				} else {
																																					if (_vala_strcmp0 (n, "socklen_t") == 0) {
																																						ValaUnresolvedSymbol* _tmp59_;
																																						_tmp59_ = NULL;
																																						vala_unresolved_type_set_unresolved_symbol (type, _tmp59_ = vala_unresolved_symbol_new (NULL, "uint32", NULL));
																																						(_tmp59_ == NULL) ? NULL : (_tmp59_ = (vala_code_node_unref (_tmp59_), NULL));
																																					} else {
																																						if (_vala_strcmp0 (n, "mode_t") == 0) {
																																							ValaUnresolvedSymbol* _tmp60_;
																																							_tmp60_ = NULL;
																																							vala_unresolved_type_set_unresolved_symbol (type, _tmp60_ = vala_unresolved_symbol_new (NULL, "uint", NULL));
																																							(_tmp60_ == NULL) ? NULL : (_tmp60_ = (vala_code_node_unref (_tmp60_), NULL));
																																						} else {
																																							gboolean _tmp61_;
																																							_tmp61_ = FALSE;
																																							if (_vala_strcmp0 (n, "gint") == 0) {
																																								_tmp61_ = TRUE;
																																							} else {
																																								_tmp61_ = _vala_strcmp0 (n, "pid_t") == 0;
																																							}
																																							if (_tmp61_) {
																																								ValaUnresolvedSymbol* _tmp62_;
																																								_tmp62_ = NULL;
																																								vala_unresolved_type_set_unresolved_symbol (type, _tmp62_ = vala_unresolved_symbol_new (NULL, "int", NULL));
																																								(_tmp62_ == NULL) ? NULL : (_tmp62_ = (vala_code_node_unref (_tmp62_), NULL));
																																							} else {
																																								gboolean _tmp63_;
																																								_tmp63_ = FALSE;
																																								if (_vala_strcmp0 (n, "unsigned") == 0) {
																																									_tmp63_ = TRUE;
																																								} else {
																																									_tmp63_ = _vala_strcmp0 (n, "unsigned-int") == 0;
																																								}
																																								if (_tmp63_) {
																																									ValaUnresolvedSymbol* _tmp64_;
																																									_tmp64_ = NULL;
																																									vala_unresolved_type_set_unresolved_symbol (type, _tmp64_ = vala_unresolved_symbol_new (NULL, "uint", NULL));
																																									(_tmp64_ == NULL) ? NULL : (_tmp64_ = (vala_code_node_unref (_tmp64_), NULL));
																																								} else {
																																									if (_vala_strcmp0 (n, "FILE") == 0) {
																																										ValaUnresolvedSymbol* _tmp66_;
																																										ValaUnresolvedSymbol* _tmp65_;
																																										_tmp66_ = NULL;
																																										_tmp65_ = NULL;
																																										vala_unresolved_type_set_unresolved_symbol (type, _tmp66_ = vala_unresolved_symbol_new (_tmp65_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "FileStream", NULL));
																																										(_tmp66_ == NULL) ? NULL : (_tmp66_ = (vala_code_node_unref (_tmp66_), NULL));
																																										(_tmp65_ == NULL) ? NULL : (_tmp65_ = (vala_code_node_unref (_tmp65_), NULL));
																																									} else {
																																										if (_vala_strcmp0 (n, "struct") == 0) {
																																											ValaVoidType* _tmp67_;
																																											ValaDataType* _tmp68_;
																																											ValaDataType* _tmp69_;
																																											_tmp67_ = NULL;
																																											_tmp68_ = NULL;
																																											_tmp69_ = NULL;
																																											return (_tmp69_ = (_tmp68_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp67_ = vala_void_type_new (NULL)), NULL), (_tmp67_ == NULL) ? NULL : (_tmp67_ = (vala_code_node_unref (_tmp67_), NULL)), _tmp68_), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp69_);
																																										} else {
																																											if (_vala_strcmp0 (n, "iconv_t") == 0) {
																																												ValaVoidType* _tmp70_;
																																												ValaDataType* _tmp71_;
																																												ValaDataType* _tmp72_;
																																												_tmp70_ = NULL;
																																												_tmp71_ = NULL;
																																												_tmp72_ = NULL;
																																												return (_tmp72_ = (_tmp71_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp70_ = vala_void_type_new (NULL)), NULL), (_tmp70_ == NULL) ? NULL : (_tmp70_ = (vala_code_node_unref (_tmp70_), NULL)), _tmp71_), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp72_);
																																											} else {
																																												if (_vala_strcmp0 (n, "GType") == 0) {
																																													ValaUnresolvedSymbol* _tmp74_;
																																													ValaUnresolvedSymbol* _tmp73_;
																																													_tmp74_ = NULL;
																																													_tmp73_ = NULL;
																																													vala_unresolved_type_set_unresolved_symbol (type, _tmp74_ = vala_unresolved_symbol_new (_tmp73_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), "Type", NULL));
																																													(_tmp74_ == NULL) ? NULL : (_tmp74_ = (vala_code_node_unref (_tmp74_), NULL));
																																													(_tmp73_ == NULL) ? NULL : (_tmp73_ = (vala_code_node_unref (_tmp73_), NULL));
																																													if (type_node->is_pointer) {
																																														ValaDataType* _tmp75_;
																																														_tmp75_ = NULL;
																																														return (_tmp75_ = (ValaDataType*) vala_array_type_new ((ValaDataType*) type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp75_);
																																													}
																																												} else {
																																													if (_vala_strcmp0 (n, "GStrv") == 0) {
																																														ValaUnresolvedSymbol* _tmp76_;
																																														ValaDataType* _tmp77_;
																																														_tmp76_ = NULL;
																																														vala_unresolved_type_set_unresolved_symbol (type, _tmp76_ = vala_unresolved_symbol_new (NULL, "string", NULL));
																																														(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_code_node_unref (_tmp76_), NULL));
																																														_tmp77_ = NULL;
																																														return (_tmp77_ = (ValaDataType*) vala_array_type_new ((ValaDataType*) type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp77_);
																																													} else {
																																														ValaDataType* named_type;
																																														ValaUnresolvedType* _tmp80_;
																																														ValaUnresolvedType* _tmp79_;
																																														ValaDataType* _tmp78_;
																																														named_type = vala_gidl_parser_parse_type_string (self, n);
																																														_tmp80_ = NULL;
																																														_tmp79_ = NULL;
																																														_tmp78_ = NULL;
																																														type = (_tmp80_ = (_tmp79_ = (_tmp78_ = named_type, VALA_IS_UNRESOLVED_TYPE (_tmp78_) ? ((ValaUnresolvedType*) _tmp78_) : NULL), (_tmp79_ == NULL) ? NULL : vala_code_node_ref (_tmp79_)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp80_);
																																														if (type == NULL) {
																																															ValaDataType* _tmp81_;
																																															_tmp81_ = NULL;
																																															return (_tmp81_ = named_type, n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp81_);
																																														}
																																														if (vala_gidl_parser_is_simple_type (self, n)) {
																																															if (type_node->is_pointer) {
																																																dir = VALA_PARAMETER_DIRECTION_OUT;
																																															}
																																														} else {
																																															if (g_str_has_suffix (type_node->unparsed, "**")) {
																																																dir = VALA_PARAMETER_DIRECTION_OUT;
																																															}
																																														}
																																														(named_type == NULL) ? NULL : (named_type = (vala_code_node_unref (named_type), NULL));
																																													}
																																												}
																																											}
																																										}
																																									}
																																								}
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											n = (g_free (n), NULL);
																										} else {
																											fprintf (stdout, "%d\n", (gint) type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	if ((&(*direction)) != NULL) {
		(*direction) = dir;
	}
	return (ValaDataType*) type;
}


static gboolean vala_gidl_parser_is_simple_type (ValaGIdlParser* self, const char* type_name) {
	ValaTypeSymbol* _tmp0_;
	ValaStruct* st;
	gboolean _tmp1_;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type_name != NULL, FALSE);
	_tmp0_ = NULL;
	st = (_tmp0_ = (ValaTypeSymbol*) gee_map_get (self->priv->cname_type_map, type_name), VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL);
	_tmp1_ = FALSE;
	if (st != NULL) {
		_tmp1_ = vala_struct_is_simple_type (st);
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		gboolean _tmp2_;
		return (_tmp2_ = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp2_);
	}
	return (_tmp3_ = FALSE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp3_);
}


static ValaDataType* vala_gidl_parser_parse_type_string (ValaGIdlParser* self, const char* n) {
	ValaUnresolvedType* type;
	ValaTypeSymbol* dt;
	char** _tmp7_;
	gint type_attributes_size;
	gint type_attributes_length1;
	gint _tmp6_;
	char** type_attributes;
	char* ns_name;
	ValaDataType* _tmp32_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (n != NULL, NULL);
	if (_vala_strcmp0 (n, "va_list") == 0) {
		ValaVoidType* _tmp0_;
		ValaDataType* _tmp1_;
		/* unsupported*/
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		return (_tmp1_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp0_ = vala_void_type_new (NULL)), NULL), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp1_);
	}
	type = vala_unresolved_type_new ();
	dt = (ValaTypeSymbol*) gee_map_get (self->priv->cname_type_map, n);
	if (dt != NULL) {
		ValaUnresolvedSymbol* parent_symbol;
		ValaUnresolvedSymbol* _tmp4_;
		ValaDataType* _tmp5_;
		parent_symbol = NULL;
		if (vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) dt)) != NULL) {
			ValaUnresolvedSymbol* _tmp3_;
			_tmp3_ = NULL;
			parent_symbol = (_tmp3_ = vala_unresolved_symbol_new (NULL, vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) dt)), NULL), (parent_symbol == NULL) ? NULL : (parent_symbol = (vala_code_node_unref (parent_symbol), NULL)), _tmp3_);
		}
		_tmp4_ = NULL;
		vala_unresolved_type_set_unresolved_symbol (type, _tmp4_ = vala_unresolved_symbol_new (parent_symbol, vala_symbol_get_name ((ValaSymbol*) dt), NULL));
		(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
		_tmp5_ = NULL;
		return (_tmp5_ = (ValaDataType*) type, (parent_symbol == NULL) ? NULL : (parent_symbol = (vala_code_node_unref (parent_symbol), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), _tmp5_);
	}
	_tmp7_ = NULL;
	type_attributes = (_tmp7_ = vala_gidl_parser_get_attributes (self, n, &_tmp6_), type_attributes_length1 = _tmp6_, type_attributes_size = type_attributes_length1, _tmp7_);
	ns_name = NULL;
	if (NULL != type_attributes) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = type_attributes;
			attr_collection_length1 = type_attributes_length1;
			for (attr_it = 0; attr_it < type_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp17_;
				char* attr;
				_tmp17_ = NULL;
				attr = (_tmp17_ = attr_collection[attr_it], (_tmp17_ == NULL) ? NULL : g_strdup (_tmp17_));
				{
					char** _tmp9_;
					gint nv_size;
					gint nv_length1;
					char** _tmp8_;
					char** nv;
					_tmp9_ = NULL;
					_tmp8_ = NULL;
					nv = (_tmp9_ = _tmp8_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp8_), nv_size = nv_length1, _tmp9_);
					if (_vala_strcmp0 (nv[0], "cprefix") == 0) {
						ValaUnresolvedSymbol* _tmp11_;
						char* _tmp10_;
						_tmp11_ = NULL;
						_tmp10_ = NULL;
						vala_unresolved_type_set_unresolved_symbol (type, _tmp11_ = vala_unresolved_symbol_new (NULL, g_utf8_offset_to_pointer (n, g_utf8_strlen (_tmp10_ = vala_gidl_parser_eval (self, nv[1]), -1)), NULL));
						(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_code_node_unref (_tmp11_), NULL));
						_tmp10_ = (g_free (_tmp10_), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "name") == 0) {
							ValaUnresolvedSymbol* _tmp13_;
							char* _tmp12_;
							_tmp13_ = NULL;
							_tmp12_ = NULL;
							vala_unresolved_type_set_unresolved_symbol (type, _tmp13_ = vala_unresolved_symbol_new (NULL, _tmp12_ = vala_gidl_parser_eval (self, nv[1]), NULL));
							(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_code_node_unref (_tmp13_), NULL));
							_tmp12_ = (g_free (_tmp12_), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "namespace") == 0) {
								char* _tmp14_;
								_tmp14_ = NULL;
								ns_name = (_tmp14_ = vala_gidl_parser_eval (self, nv[1]), ns_name = (g_free (ns_name), NULL), _tmp14_);
							} else {
								if (_vala_strcmp0 (nv[0], "rename_to") == 0) {
									ValaUnresolvedSymbol* _tmp16_;
									char* _tmp15_;
									_tmp16_ = NULL;
									_tmp15_ = NULL;
									vala_unresolved_type_set_unresolved_symbol (type, _tmp16_ = vala_unresolved_symbol_new (NULL, _tmp15_ = vala_gidl_parser_eval (self, nv[1]), NULL));
									(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_code_node_unref (_tmp16_), NULL));
									_tmp15_ = (g_free (_tmp15_), NULL);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (vala_unresolved_type_get_unresolved_symbol (type) != NULL) {
		ValaDataType* _tmp22_;
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) vala_unresolved_type_get_unresolved_symbol (type)), "pointer") == 0) {
			ValaVoidType* _tmp18_;
			ValaDataType* _tmp19_;
			ValaDataType* _tmp20_;
			_tmp18_ = NULL;
			_tmp19_ = NULL;
			_tmp20_ = NULL;
			return (_tmp20_ = (_tmp19_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp18_ = vala_void_type_new (NULL)), NULL), (_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_code_node_unref (_tmp18_), NULL)), _tmp19_), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, (GDestroyNotify) g_free), NULL), ns_name = (g_free (ns_name), NULL), _tmp20_);
		}
		if (ns_name != NULL) {
			ValaUnresolvedSymbol* _tmp21_;
			_tmp21_ = NULL;
			vala_unresolved_symbol_set_inner (vala_unresolved_type_get_unresolved_symbol (type), _tmp21_ = vala_unresolved_symbol_new (NULL, ns_name, NULL));
			(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_code_node_unref (_tmp21_), NULL));
		}
		_tmp22_ = NULL;
		return (_tmp22_ = (ValaDataType*) type, (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, (GDestroyNotify) g_free), NULL), ns_name = (g_free (ns_name), NULL), _tmp22_);
	}
	if (g_str_has_prefix (n, vala_symbol_get_name ((ValaSymbol*) self->priv->current_namespace))) {
		ValaUnresolvedSymbol* _tmp24_;
		ValaUnresolvedSymbol* _tmp23_;
		_tmp24_ = NULL;
		_tmp23_ = NULL;
		vala_unresolved_type_set_unresolved_symbol (type, _tmp24_ = vala_unresolved_symbol_new (_tmp23_ = vala_unresolved_symbol_new (NULL, vala_symbol_get_name ((ValaSymbol*) self->priv->current_namespace), NULL), g_utf8_offset_to_pointer (n, g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) self->priv->current_namespace), -1)), NULL));
		(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_code_node_unref (_tmp24_), NULL));
		(_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_code_node_unref (_tmp23_), NULL));
	} else {
		if (g_str_has_prefix (n, "G")) {
			ValaUnresolvedSymbol* _tmp26_;
			ValaUnresolvedSymbol* _tmp25_;
			_tmp26_ = NULL;
			_tmp25_ = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, _tmp26_ = vala_unresolved_symbol_new (_tmp25_ = vala_unresolved_symbol_new (NULL, "GLib", NULL), g_utf8_offset_to_pointer (n, (glong) 1), NULL));
			(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_code_node_unref (_tmp26_), NULL));
			(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_code_node_unref (_tmp25_), NULL));
		} else {
			char** _tmp28_;
			gint name_parts_size;
			gint name_parts_length1;
			char** _tmp27_;
			char** name_parts;
			_tmp28_ = NULL;
			_tmp27_ = NULL;
			name_parts = (_tmp28_ = _tmp27_ = g_strsplit (n, ".", 2), name_parts_length1 = _vala_array_length (_tmp27_), name_parts_size = name_parts_length1, _tmp28_);
			if (name_parts[1] == NULL) {
				ValaUnresolvedSymbol* _tmp29_;
				_tmp29_ = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, _tmp29_ = vala_unresolved_symbol_new (NULL, name_parts[0], NULL));
				(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_code_node_unref (_tmp29_), NULL));
			} else {
				ValaUnresolvedSymbol* _tmp31_;
				ValaUnresolvedSymbol* _tmp30_;
				_tmp31_ = NULL;
				_tmp30_ = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, _tmp31_ = vala_unresolved_symbol_new (_tmp30_ = vala_unresolved_symbol_new (NULL, name_parts[0], NULL), name_parts[1], NULL));
				(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_code_node_unref (_tmp31_), NULL));
				(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_code_node_unref (_tmp30_), NULL));
			}
			name_parts = (_vala_array_free (name_parts, name_parts_length1, (GDestroyNotify) g_free), NULL);
		}
	}
	_tmp32_ = NULL;
	return (_tmp32_ = (ValaDataType*) type, (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, (GDestroyNotify) g_free), NULL), ns_name = (g_free (ns_name), NULL), _tmp32_);
}


static ValaDataType* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param, ValaParameterDirection* direction) {
	ValaDataType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (param != NULL, NULL);
	type = vala_gidl_parser_parse_type (self, param->type, &(*direction));
	/* disable for now as null_ok not yet correctly set
	 type.non_null = !param.null_ok;*/
	return type;
}


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static ValaMethod* vala_gidl_parser_create_method (ValaGIdlParser* self, const char* name, const char* symbol, GIdlNodeParam* res, GList* parameters, gboolean is_constructor, gboolean is_interface) {
	ValaDataType* return_type;
	ValaMethod* m;
	gboolean _tmp1_;
	gboolean add_ellipsis;
	gboolean suppress_throws;
	char** _tmp21_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp20_;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaDataType* last_param_type;
	gboolean _tmp140_;
	ValaMethod* _tmp142_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (symbol != NULL, NULL);
	return_type = NULL;
	if (res != NULL) {
		ValaDataType* _tmp0_;
		_tmp0_ = NULL;
		return_type = (_tmp0_ = vala_gidl_parser_parse_param (self, res, NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp0_);
	}
	m = NULL;
	_tmp1_ = FALSE;
	if (!is_interface) {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if (is_constructor) {
			_tmp2_ = TRUE;
		} else {
			_tmp2_ = g_str_has_prefix (name, "new");
		}
		_tmp1_ = _tmp2_;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		ValaMethod* _tmp3_;
		gboolean _tmp4_;
		_tmp3_ = NULL;
		m = (_tmp3_ = (ValaMethod*) vala_creation_method_new (NULL, name, self->priv->current_source_reference), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp3_);
		vala_method_set_has_construct_function (m, FALSE);
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) m), "new") == 0) {
			vala_symbol_set_name ((ValaSymbol*) m, NULL);
		} else {
			if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) m), "new_")) {
				vala_symbol_set_name ((ValaSymbol*) m, g_utf8_offset_to_pointer (vala_symbol_get_name ((ValaSymbol*) m), g_utf8_strlen ("new_", -1)));
			}
		}
		_tmp4_ = FALSE;
		if (VALA_IS_CLASS (self->priv->current_data_type)) {
			_tmp4_ = res != NULL;
		} else {
			_tmp4_ = FALSE;
		}
		/* For classes, check whether a creation method return type equals to the
		 type of the class created. If the types do not match (e.g. in most
		 gtk widgets) add an attribute to the creation method indicating the used
		 return type.*/
		if (_tmp4_) {
			char* _tmp6_;
			char* _tmp5_;
			gboolean _tmp7_;
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			if ((_tmp7_ = _vala_strcmp0 (_tmp6_ = g_strdup_printf ("%s*", _tmp5_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), res->type->unparsed) != 0, _tmp6_ = (g_free (_tmp6_), NULL), _tmp5_ = (g_free (_tmp5_), NULL), _tmp7_)) {
				vala_creation_method_set_custom_return_type_cname (VALA_CREATION_METHOD (m), res->type->unparsed);
			}
		}
	} else {
		ValaMethod* _tmp8_;
		_tmp8_ = NULL;
		m = (_tmp8_ = vala_method_new (name, return_type, self->priv->current_source_reference), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp8_);
	}
	vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add ((GeeCollection*) self->priv->current_type_symbol_set, name);
	}
	if (self->priv->current_data_type != NULL) {
		char** _tmp13_;
		gint sig_attributes_size;
		gint sig_attributes_length1;
		gint _tmp11_;
		char* _tmp10_;
		char* _tmp9_;
		char** _tmp12_;
		char** sig_attributes;
		_tmp13_ = NULL;
		_tmp10_ = NULL;
		_tmp9_ = NULL;
		_tmp12_ = NULL;
		sig_attributes = (_tmp13_ = (_tmp12_ = vala_gidl_parser_get_attributes (self, _tmp10_ = g_strdup_printf ("%s::%s", _tmp9_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), name), &_tmp11_), _tmp10_ = (g_free (_tmp10_), NULL), _tmp9_ = (g_free (_tmp9_), NULL), _tmp12_), sig_attributes_length1 = _tmp11_, sig_attributes_size = sig_attributes_length1, _tmp13_);
		if (sig_attributes != NULL) {
			{
				char** attr_collection;
				int attr_collection_length1;
				int attr_it;
				attr_collection = sig_attributes;
				attr_collection_length1 = sig_attributes_length1;
				for (attr_it = 0; attr_it < sig_attributes_length1; attr_it = attr_it + 1) {
					const char* _tmp19_;
					char* attr;
					_tmp19_ = NULL;
					attr = (_tmp19_ = attr_collection[attr_it], (_tmp19_ == NULL) ? NULL : g_strdup (_tmp19_));
					{
						char** _tmp15_;
						gint nv_size;
						gint nv_length1;
						char** _tmp14_;
						char** nv;
						gboolean _tmp16_;
						_tmp15_ = NULL;
						_tmp14_ = NULL;
						nv = (_tmp15_ = _tmp14_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp14_), nv_size = nv_length1, _tmp15_);
						_tmp16_ = FALSE;
						if (_vala_strcmp0 (nv[0], "has_emitter") == 0) {
							char* _tmp17_;
							_tmp17_ = NULL;
							_tmp16_ = _vala_strcmp0 (_tmp17_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0;
							_tmp17_ = (g_free (_tmp17_), NULL);
						} else {
							_tmp16_ = FALSE;
						}
						if (_tmp16_) {
							ValaMethod* _tmp18_;
							_tmp18_ = NULL;
							return (_tmp18_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, (GDestroyNotify) g_free), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp18_);
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
					}
				}
			}
		}
		sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, (GDestroyNotify) g_free), NULL);
	}
	add_ellipsis = FALSE;
	suppress_throws = FALSE;
	_tmp21_ = NULL;
	attributes = (_tmp21_ = vala_gidl_parser_get_attributes (self, symbol, &_tmp20_), attributes_length1 = _tmp20_, attributes_size = attributes_length1, _tmp21_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp62_;
				char* attr;
				_tmp62_ = NULL;
				attr = (_tmp62_ = attr_collection[attr_it], (_tmp62_ == NULL) ? NULL : g_strdup (_tmp62_));
				{
					char** _tmp23_;
					gint nv_size;
					gint nv_length1;
					char** _tmp22_;
					char** nv;
					_tmp23_ = NULL;
					_tmp22_ = NULL;
					nv = (_tmp23_ = _tmp22_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp22_), nv_size = nv_length1, _tmp23_);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp24_;
						vala_method_set_cname (m, vala_symbol_get_name ((ValaSymbol*) m));
						_tmp24_ = NULL;
						vala_symbol_set_name ((ValaSymbol*) m, _tmp24_ = vala_gidl_parser_eval (self, nv[1]));
						_tmp24_ = (g_free (_tmp24_), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "hidden") == 0) {
							char* _tmp25_;
							gboolean _tmp26_;
							_tmp25_ = NULL;
							if ((_tmp26_ = _vala_strcmp0 (_tmp25_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp25_ = (g_free (_tmp25_), NULL), _tmp26_)) {
								ValaMethod* _tmp27_;
								_tmp27_ = NULL;
								return (_tmp27_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp27_);
							}
						} else {
							if (_vala_strcmp0 (nv[0], "ellipsis") == 0) {
								char* _tmp28_;
								gboolean _tmp29_;
								_tmp28_ = NULL;
								if ((_tmp29_ = _vala_strcmp0 (_tmp28_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp28_ = (g_free (_tmp28_), NULL), _tmp29_)) {
									add_ellipsis = TRUE;
								}
							} else {
								if (_vala_strcmp0 (nv[0], "printf_format") == 0) {
									char* _tmp30_;
									gboolean _tmp31_;
									_tmp30_ = NULL;
									if ((_tmp31_ = _vala_strcmp0 (_tmp30_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp30_ = (g_free (_tmp30_), NULL), _tmp31_)) {
										vala_method_set_printf_format (m, TRUE);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
										char* _tmp32_;
										gboolean _tmp33_;
										_tmp32_ = NULL;
										if ((_tmp33_ = _vala_strcmp0 (_tmp32_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp32_ = (g_free (_tmp32_), NULL), _tmp33_)) {
											vala_data_type_set_value_owned (return_type, TRUE);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "nullable") == 0) {
											char* _tmp34_;
											gboolean _tmp35_;
											_tmp34_ = NULL;
											if ((_tmp35_ = _vala_strcmp0 (_tmp34_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp34_ = (g_free (_tmp34_), NULL), _tmp35_)) {
												vala_data_type_set_nullable (return_type, TRUE);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "sentinel") == 0) {
												char* _tmp36_;
												_tmp36_ = NULL;
												vala_method_set_sentinel (m, _tmp36_ = vala_gidl_parser_eval (self, nv[1]));
												_tmp36_ = (g_free (_tmp36_), NULL);
											} else {
												if (_vala_strcmp0 (nv[0], "is_array") == 0) {
													char* _tmp37_;
													gboolean _tmp38_;
													_tmp37_ = NULL;
													if ((_tmp38_ = _vala_strcmp0 (_tmp37_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp37_ = (g_free (_tmp37_), NULL), _tmp38_)) {
														ValaDataType* _tmp39_;
														_tmp39_ = NULL;
														return_type = (_tmp39_ = (ValaDataType*) vala_array_type_new (return_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) return_type)), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp39_);
														vala_method_set_return_type (m, return_type);
													}
												} else {
													if (_vala_strcmp0 (nv[0], "throws") == 0) {
														char* _tmp40_;
														gboolean _tmp41_;
														_tmp40_ = NULL;
														if ((_tmp41_ = _vala_strcmp0 (_tmp40_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp40_ = (g_free (_tmp40_), NULL), _tmp41_)) {
															suppress_throws = TRUE;
														}
													} else {
														if (_vala_strcmp0 (nv[0], "no_array_length") == 0) {
															char* _tmp42_;
															gboolean _tmp43_;
															_tmp42_ = NULL;
															if ((_tmp43_ = _vala_strcmp0 (_tmp42_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp42_ = (g_free (_tmp42_), NULL), _tmp43_)) {
																vala_method_set_no_array_length (m, TRUE);
															}
														} else {
															if (_vala_strcmp0 (nv[0], "array_null_terminated") == 0) {
																char* _tmp44_;
																gboolean _tmp45_;
																_tmp44_ = NULL;
																if ((_tmp45_ = _vala_strcmp0 (_tmp44_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp44_ = (g_free (_tmp44_), NULL), _tmp45_)) {
																	vala_method_set_no_array_length (m, TRUE);
																	vala_method_set_array_null_terminated (m, TRUE);
																}
															} else {
																if (_vala_strcmp0 (nv[0], "type_name") == 0) {
																	char* _tmp46_;
																	ValaUnresolvedSymbol* _tmp47_;
																	ValaUnresolvedSymbol* sym;
																	_tmp46_ = NULL;
																	_tmp47_ = NULL;
																	sym = (_tmp47_ = vala_unresolved_symbol_new (NULL, _tmp46_ = vala_gidl_parser_eval (self, nv[1]), NULL), _tmp46_ = (g_free (_tmp46_), NULL), _tmp47_);
																	if (VALA_IS_UNRESOLVED_TYPE (return_type)) {
																		vala_unresolved_type_set_unresolved_symbol (VALA_UNRESOLVED_TYPE (return_type), sym);
																	} else {
																		ValaDataType* _tmp48_;
																		/* Overwrite old return_type, so "type_name" must be before any
																		 other return type modifying metadata*/
																		_tmp48_ = NULL;
																		vala_method_set_return_type (m, return_type = (_tmp48_ = (ValaDataType*) vala_unresolved_type_new_from_symbol (sym, vala_code_node_get_source_reference ((ValaCodeNode*) return_type)), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp48_));
																	}
																	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
																} else {
																	if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
																		char** _tmp52_;
																		gint type_args_size;
																		gint type_args_length1;
																		char** _tmp50_;
																		char* _tmp49_;
																		char** _tmp51_;
																		char** type_args;
																		_tmp52_ = NULL;
																		_tmp50_ = NULL;
																		_tmp49_ = NULL;
																		_tmp51_ = NULL;
																		type_args = (_tmp52_ = (_tmp51_ = _tmp50_ = g_strsplit (_tmp49_ = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp49_ = (g_free (_tmp49_), NULL), _tmp51_), type_args_length1 = _vala_array_length (_tmp50_), type_args_size = type_args_length1, _tmp52_);
																		{
																			char** type_arg_collection;
																			int type_arg_collection_length1;
																			int type_arg_it;
																			type_arg_collection = type_args;
																			type_arg_collection_length1 = type_args_length1;
																			for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
																				const char* _tmp55_;
																				char* type_arg;
																				_tmp55_ = NULL;
																				type_arg = (_tmp55_ = type_arg_collection[type_arg_it], (_tmp55_ == NULL) ? NULL : g_strdup (_tmp55_));
																				{
																					ValaUnresolvedSymbol* _tmp53_;
																					ValaUnresolvedType* _tmp54_;
																					ValaUnresolvedType* arg_type;
																					_tmp53_ = NULL;
																					_tmp54_ = NULL;
																					arg_type = (_tmp54_ = vala_unresolved_type_new_from_symbol (_tmp53_ = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp53_ == NULL) ? NULL : (_tmp53_ = (vala_code_node_unref (_tmp53_), NULL)), _tmp54_);
																					vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
																					vala_data_type_add_type_argument (return_type, (ValaDataType*) arg_type);
																					type_arg = (g_free (type_arg), NULL);
																					(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
																				}
																			}
																		}
																		type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
																	} else {
																		if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
																			char* _tmp56_;
																			_tmp56_ = NULL;
																			vala_member_add_cheader_filename ((ValaMember*) m, _tmp56_ = vala_gidl_parser_eval (self, nv[1]));
																			_tmp56_ = (g_free (_tmp56_), NULL);
																		} else {
																			if (_vala_strcmp0 (nv[0], "abstract") == 0) {
																				char* _tmp57_;
																				gboolean _tmp58_;
																				_tmp57_ = NULL;
																				if ((_tmp58_ = _vala_strcmp0 (_tmp57_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp57_ = (g_free (_tmp57_), NULL), _tmp58_)) {
																					vala_method_set_is_abstract (m, TRUE);
																				}
																			} else {
																				if (_vala_strcmp0 (nv[0], "virtual") == 0) {
																					char* _tmp59_;
																					gboolean _tmp60_;
																					_tmp59_ = NULL;
																					if ((_tmp60_ = _vala_strcmp0 (_tmp59_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp59_ = (g_free (_tmp59_), NULL), _tmp60_)) {
																						vala_method_set_is_virtual (m, TRUE);
																					}
																				} else {
																					if (_vala_strcmp0 (nv[0], "vfunc_name") == 0) {
																						char* _tmp61_;
																						_tmp61_ = NULL;
																						vala_method_set_vfunc_name (m, _tmp61_ = vala_gidl_parser_eval (self, nv[1]));
																						_tmp61_ = (g_free (_tmp61_), NULL);
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_method_set_cname (m, symbol);
	first = TRUE;
	last_param = NULL;
	last_param_type = NULL;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = (GIdlNodeParam*) param_it->data;
			{
				GIdlNode* param_node;
				gboolean _tmp75_;
				const char* _tmp77_;
				char* param_name;
				ValaParameterDirection direction;
				ValaDataType* param_type;
				ValaFormalParameter* p;
				gboolean hide_param;
				gboolean show_param;
				gboolean set_array_length_pos;
				double array_length_pos;
				gboolean set_delegate_target_pos;
				double delegate_target_pos;
				gboolean array_requested;
				char** _tmp82_;
				gint _tmp81_;
				char* _tmp80_;
				gboolean _tmp131_;
				gboolean _tmp135_;
				ValaFormalParameter* _tmp137_;
				ValaFormalParameter* _tmp136_;
				ValaDataType* _tmp139_;
				ValaDataType* _tmp138_;
				param_node = (GIdlNode*) param;
				if (first) {
					gboolean _tmp63_;
					gboolean _tmp64_;
					gboolean _tmp65_;
					first = FALSE;
					_tmp63_ = FALSE;
					_tmp64_ = FALSE;
					_tmp65_ = FALSE;
					if (!VALA_IS_CREATION_METHOD (m)) {
						_tmp65_ = self->priv->current_data_type != NULL;
					} else {
						_tmp65_ = FALSE;
					}
					if (_tmp65_) {
						_tmp64_ = param->type->is_interface;
					} else {
						_tmp64_ = FALSE;
					}
					if (_tmp64_) {
						gboolean _tmp66_;
						_tmp66_ = FALSE;
						if (_vala_strcmp0 (param_node->name, "self") == 0) {
							_tmp66_ = TRUE;
						} else {
							char* _tmp67_;
							_tmp67_ = NULL;
							_tmp66_ = g_str_has_suffix (param->type->interface, _tmp67_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE));
							_tmp67_ = (g_free (_tmp67_), NULL);
						}
						_tmp63_ = _tmp66_;
					} else {
						_tmp63_ = FALSE;
					}
					if (_tmp63_) {
						/* instance method*/
						continue;
					} else {
						gboolean _tmp68_;
						gboolean _tmp69_;
						gboolean _tmp70_;
						_tmp68_ = FALSE;
						_tmp69_ = FALSE;
						_tmp70_ = FALSE;
						if (!VALA_IS_CREATION_METHOD (m)) {
							_tmp70_ = self->priv->current_data_type != NULL;
						} else {
							_tmp70_ = FALSE;
						}
						if (_tmp70_) {
							_tmp69_ = param->type->is_interface;
						} else {
							_tmp69_ = FALSE;
						}
						if (_tmp69_) {
							gboolean _tmp71_;
							_tmp71_ = FALSE;
							if (_vala_strcmp0 (param_node->name, "klass") == 0) {
								_tmp71_ = TRUE;
							} else {
								char* _tmp73_;
								char* _tmp72_;
								_tmp73_ = NULL;
								_tmp72_ = NULL;
								_tmp71_ = g_str_has_suffix (param->type->interface, _tmp73_ = g_strdup_printf ("%sClass", _tmp72_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)));
								_tmp73_ = (g_free (_tmp73_), NULL);
								_tmp72_ = (g_free (_tmp72_), NULL);
							}
							_tmp68_ = _tmp71_;
						} else {
							_tmp68_ = FALSE;
						}
						if (_tmp68_) {
							/* class method*/
							vala_method_set_binding (m, MEMBER_BINDING_CLASS);
							if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) m), "class_")) {
								char* _tmp74_;
								_tmp74_ = NULL;
								vala_symbol_set_name ((ValaSymbol*) m, _tmp74_ = string_substring (vala_symbol_get_name ((ValaSymbol*) m), g_utf8_strlen ("class_", -1), g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) m), -1) - g_utf8_strlen ("class_", -1)));
								_tmp74_ = (g_free (_tmp74_), NULL);
							}
							continue;
						} else {
							/* static method*/
							vala_method_set_binding (m, MEMBER_BINDING_STATIC);
						}
					}
				}
				_tmp75_ = FALSE;
				if (suppress_throws == FALSE) {
					_tmp75_ = vala_gidl_parser_param_is_exception (self, param);
				} else {
					_tmp75_ = FALSE;
				}
				if (_tmp75_) {
					ValaDataType* _tmp76_;
					_tmp76_ = NULL;
					vala_code_node_add_error_type ((ValaCodeNode*) m, _tmp76_ = vala_gidl_parser_parse_type (self, param->type, NULL));
					(_tmp76_ == NULL) ? NULL : (_tmp76_ = (vala_code_node_unref (_tmp76_), NULL));
					continue;
				}
				_tmp77_ = NULL;
				param_name = (_tmp77_ = param_node->name, (_tmp77_ == NULL) ? NULL : g_strdup (_tmp77_));
				if (_vala_strcmp0 (param_name, "result") == 0) {
					char* _tmp78_;
					/* avoid conflict with generated result variable*/
					_tmp78_ = NULL;
					param_name = (_tmp78_ = g_strdup ("_result"), param_name = (g_free (param_name), NULL), _tmp78_);
				} else {
					if (_vala_strcmp0 (param_name, "string") == 0) {
						char* _tmp79_;
						/* avoid conflict with string type*/
						_tmp79_ = NULL;
						param_name = (_tmp79_ = g_strdup ("str"), param_name = (g_free (param_name), NULL), _tmp79_);
					}
				}
				direction = 0;
				param_type = vala_gidl_parser_parse_param (self, param, &direction);
				p = vala_formal_parameter_new (param_name, param_type, NULL);
				vala_formal_parameter_set_direction (p, direction);
				hide_param = FALSE;
				show_param = FALSE;
				set_array_length_pos = FALSE;
				array_length_pos = (double) 0;
				set_delegate_target_pos = FALSE;
				delegate_target_pos = (double) 0;
				array_requested = FALSE;
				_tmp82_ = NULL;
				_tmp80_ = NULL;
				attributes = (_tmp82_ = vala_gidl_parser_get_attributes (self, _tmp80_ = g_strdup_printf ("%s.%s", symbol, param_node->name), &_tmp81_), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), attributes_length1 = _tmp81_, attributes_size = attributes_length1, _tmp82_);
				_tmp80_ = (g_free (_tmp80_), NULL);
				if (attributes != NULL) {
					{
						char** attr_collection;
						int attr_collection_length1;
						int attr_it;
						attr_collection = attributes;
						attr_collection_length1 = attributes_length1;
						for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
							const char* _tmp130_;
							char* attr;
							_tmp130_ = NULL;
							attr = (_tmp130_ = attr_collection[attr_it], (_tmp130_ == NULL) ? NULL : g_strdup (_tmp130_));
							{
								char** _tmp84_;
								gint nv_size;
								gint nv_length1;
								char** _tmp83_;
								char** nv;
								_tmp84_ = NULL;
								_tmp83_ = NULL;
								nv = (_tmp84_ = _tmp83_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp83_), nv_size = nv_length1, _tmp84_);
								if (_vala_strcmp0 (nv[0], "is_array") == 0) {
									char* _tmp85_;
									gboolean _tmp86_;
									_tmp85_ = NULL;
									if ((_tmp86_ = _vala_strcmp0 (_tmp85_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp85_ = (g_free (_tmp85_), NULL), _tmp86_)) {
										ValaDataType* _tmp87_;
										_tmp87_ = NULL;
										param_type = (_tmp87_ = (ValaDataType*) vala_array_type_new (param_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) param_type)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp87_);
										vala_formal_parameter_set_parameter_type (p, param_type);
										vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_IN);
										array_requested = TRUE;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "is_out") == 0) {
										char* _tmp88_;
										gboolean _tmp89_;
										_tmp88_ = NULL;
										if ((_tmp89_ = _vala_strcmp0 (_tmp88_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp88_ = (g_free (_tmp88_), NULL), _tmp89_)) {
											gboolean _tmp90_;
											vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_OUT);
											_tmp90_ = FALSE;
											if (!array_requested) {
												_tmp90_ = VALA_IS_ARRAY_TYPE (param_type);
											} else {
												_tmp90_ = FALSE;
											}
											if (_tmp90_) {
												ValaArrayType* _tmp91_;
												ValaArrayType* array_type;
												ValaDataType* _tmp93_;
												ValaDataType* _tmp92_;
												_tmp91_ = NULL;
												array_type = (_tmp91_ = VALA_ARRAY_TYPE (param_type), (_tmp91_ == NULL) ? NULL : vala_code_node_ref (_tmp91_));
												_tmp93_ = NULL;
												_tmp92_ = NULL;
												param_type = (_tmp93_ = (_tmp92_ = vala_array_type_get_element_type (array_type), (_tmp92_ == NULL) ? NULL : vala_code_node_ref (_tmp92_)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp93_);
												vala_formal_parameter_set_parameter_type (p, param_type);
												(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
											}
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
											char* _tmp94_;
											gboolean _tmp95_;
											_tmp94_ = NULL;
											if ((_tmp95_ = _vala_strcmp0 (_tmp94_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp94_ = (g_free (_tmp94_), NULL), _tmp95_)) {
												gboolean _tmp96_;
												vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_REF);
												_tmp96_ = FALSE;
												if (!array_requested) {
													_tmp96_ = VALA_IS_ARRAY_TYPE (param_type);
												} else {
													_tmp96_ = FALSE;
												}
												if (_tmp96_) {
													ValaArrayType* _tmp97_;
													ValaArrayType* array_type;
													ValaDataType* _tmp99_;
													ValaDataType* _tmp98_;
													_tmp97_ = NULL;
													array_type = (_tmp97_ = VALA_ARRAY_TYPE (param_type), (_tmp97_ == NULL) ? NULL : vala_code_node_ref (_tmp97_));
													_tmp99_ = NULL;
													_tmp98_ = NULL;
													param_type = (_tmp99_ = (_tmp98_ = vala_array_type_get_element_type (array_type), (_tmp98_ == NULL) ? NULL : vala_code_node_ref (_tmp98_)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp99_);
													vala_formal_parameter_set_parameter_type (p, param_type);
													(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
												}
											}
										} else {
											if (_vala_strcmp0 (nv[0], "nullable") == 0) {
												char* _tmp100_;
												gboolean _tmp101_;
												_tmp100_ = NULL;
												if ((_tmp101_ = _vala_strcmp0 (_tmp100_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp100_ = (g_free (_tmp100_), NULL), _tmp101_)) {
													vala_data_type_set_nullable (param_type, TRUE);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
													char* _tmp102_;
													gboolean _tmp103_;
													_tmp102_ = NULL;
													if ((_tmp103_ = _vala_strcmp0 (_tmp102_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp102_ = (g_free (_tmp102_), NULL), _tmp103_)) {
														vala_data_type_set_value_owned (param_type, TRUE);
													}
												} else {
													if (_vala_strcmp0 (nv[0], "takes_ownership") == 0) {
														char* _tmp104_;
														gboolean _tmp105_;
														_tmp104_ = NULL;
														if ((_tmp105_ = _vala_strcmp0 (_tmp104_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp104_ = (g_free (_tmp104_), NULL), _tmp105_)) {
															vala_data_type_set_value_owned (param_type, TRUE);
														}
													} else {
														if (_vala_strcmp0 (nv[0], "value_owned") == 0) {
															char* _tmp106_;
															gboolean _tmp107_;
															_tmp106_ = NULL;
															if ((_tmp107_ = _vala_strcmp0 (_tmp106_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp106_ = (g_free (_tmp106_), NULL), _tmp107_)) {
																vala_data_type_set_value_owned (param_type, FALSE);
															} else {
																char* _tmp108_;
																gboolean _tmp109_;
																_tmp108_ = NULL;
																if ((_tmp109_ = _vala_strcmp0 (_tmp108_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp108_ = (g_free (_tmp108_), NULL), _tmp109_)) {
																	vala_data_type_set_value_owned (param_type, TRUE);
																}
															}
														} else {
															if (_vala_strcmp0 (nv[0], "hidden") == 0) {
																char* _tmp110_;
																gboolean _tmp111_;
																_tmp110_ = NULL;
																if ((_tmp111_ = _vala_strcmp0 (_tmp110_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp110_ = (g_free (_tmp110_), NULL), _tmp111_)) {
																	hide_param = TRUE;
																} else {
																	char* _tmp112_;
																	gboolean _tmp113_;
																	_tmp112_ = NULL;
																	if ((_tmp113_ = _vala_strcmp0 (_tmp112_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp112_ = (g_free (_tmp112_), NULL), _tmp113_)) {
																		show_param = TRUE;
																	}
																}
															} else {
																if (_vala_strcmp0 (nv[0], "no_array_length") == 0) {
																	char* _tmp114_;
																	gboolean _tmp115_;
																	_tmp114_ = NULL;
																	if ((_tmp115_ = _vala_strcmp0 (_tmp114_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp114_ = (g_free (_tmp114_), NULL), _tmp115_)) {
																		vala_formal_parameter_set_no_array_length (p, TRUE);
																	}
																} else {
																	if (_vala_strcmp0 (nv[0], "array_null_terminated") == 0) {
																		char* _tmp116_;
																		gboolean _tmp117_;
																		_tmp116_ = NULL;
																		if ((_tmp117_ = _vala_strcmp0 (_tmp116_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp116_ = (g_free (_tmp116_), NULL), _tmp117_)) {
																			vala_formal_parameter_set_no_array_length (p, TRUE);
																			vala_formal_parameter_set_array_null_terminated (p, TRUE);
																		}
																	} else {
																		if (_vala_strcmp0 (nv[0], "array_length_pos") == 0) {
																			char* _tmp118_;
																			set_array_length_pos = TRUE;
																			_tmp118_ = NULL;
																			array_length_pos = strtod (_tmp118_ = vala_gidl_parser_eval (self, nv[1]), NULL);
																			_tmp118_ = (g_free (_tmp118_), NULL);
																		} else {
																			if (_vala_strcmp0 (nv[0], "delegate_target_pos") == 0) {
																				char* _tmp119_;
																				set_delegate_target_pos = TRUE;
																				_tmp119_ = NULL;
																				delegate_target_pos = strtod (_tmp119_ = vala_gidl_parser_eval (self, nv[1]), NULL);
																				_tmp119_ = (g_free (_tmp119_), NULL);
																			} else {
																				if (_vala_strcmp0 (nv[0], "type_name") == 0) {
																					ValaUnresolvedSymbol* _tmp121_;
																					char* _tmp120_;
																					_tmp121_ = NULL;
																					_tmp120_ = NULL;
																					vala_unresolved_type_set_unresolved_symbol (VALA_UNRESOLVED_TYPE (param_type), _tmp121_ = vala_unresolved_symbol_new (NULL, _tmp120_ = vala_gidl_parser_eval (self, nv[1]), NULL));
																					(_tmp121_ == NULL) ? NULL : (_tmp121_ = (vala_code_node_unref (_tmp121_), NULL));
																					_tmp120_ = (g_free (_tmp120_), NULL);
																				} else {
																					if (_vala_strcmp0 (nv[0], "ctype") == 0) {
																						char* _tmp122_;
																						_tmp122_ = NULL;
																						vala_formal_parameter_set_ctype (p, _tmp122_ = vala_gidl_parser_eval (self, nv[1]));
																						_tmp122_ = (g_free (_tmp122_), NULL);
																					} else {
																						if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
																							char** _tmp126_;
																							gint type_args_size;
																							gint type_args_length1;
																							char** _tmp124_;
																							char* _tmp123_;
																							char** _tmp125_;
																							char** type_args;
																							_tmp126_ = NULL;
																							_tmp124_ = NULL;
																							_tmp123_ = NULL;
																							_tmp125_ = NULL;
																							type_args = (_tmp126_ = (_tmp125_ = _tmp124_ = g_strsplit (_tmp123_ = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp123_ = (g_free (_tmp123_), NULL), _tmp125_), type_args_length1 = _vala_array_length (_tmp124_), type_args_size = type_args_length1, _tmp126_);
																							{
																								char** type_arg_collection;
																								int type_arg_collection_length1;
																								int type_arg_it;
																								type_arg_collection = type_args;
																								type_arg_collection_length1 = type_args_length1;
																								for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
																									const char* _tmp129_;
																									char* type_arg;
																									_tmp129_ = NULL;
																									type_arg = (_tmp129_ = type_arg_collection[type_arg_it], (_tmp129_ == NULL) ? NULL : g_strdup (_tmp129_));
																									{
																										ValaUnresolvedSymbol* _tmp127_;
																										ValaUnresolvedType* _tmp128_;
																										ValaUnresolvedType* arg_type;
																										_tmp127_ = NULL;
																										_tmp128_ = NULL;
																										arg_type = (_tmp128_ = vala_unresolved_type_new_from_symbol (_tmp127_ = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp127_ == NULL) ? NULL : (_tmp127_ = (vala_code_node_unref (_tmp127_), NULL)), _tmp128_);
																										vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
																										vala_data_type_add_type_argument (param_type, (ValaDataType*) arg_type);
																										type_arg = (g_free (type_arg), NULL);
																										(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
																									}
																								}
																							}
																							type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
								attr = (g_free (attr), NULL);
								nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
							}
						}
					}
				}
				_tmp131_ = FALSE;
				if (last_param != NULL) {
					char* _tmp132_;
					_tmp132_ = NULL;
					_tmp131_ = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) p), _tmp132_ = g_strconcat ("n_", vala_symbol_get_name ((ValaSymbol*) last_param), NULL)) == 0;
					_tmp132_ = (g_free (_tmp132_), NULL);
				} else {
					_tmp131_ = FALSE;
				}
				if (_tmp131_) {
					if (!VALA_IS_ARRAY_TYPE (last_param_type)) {
						ValaDataType* _tmp133_;
						/* last_param is array, p is array length*/
						_tmp133_ = NULL;
						last_param_type = (_tmp133_ = (ValaDataType*) vala_array_type_new (last_param_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) last_param_type)), (last_param_type == NULL) ? NULL : (last_param_type = (vala_code_node_unref (last_param_type), NULL)), _tmp133_);
						vala_formal_parameter_set_parameter_type (last_param, last_param_type);
						vala_formal_parameter_set_direction (last_param, VALA_PARAMETER_DIRECTION_IN);
					}
					/* hide array length param*/
					hide_param = TRUE;
				} else {
					gboolean _tmp134_;
					_tmp134_ = FALSE;
					if (last_param != NULL) {
						_tmp134_ = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) p), "user_data") == 0;
					} else {
						_tmp134_ = FALSE;
					}
					if (_tmp134_) {
						/* last_param is delegate
						 hide deleate target param*/
						hide_param = TRUE;
					}
				}
				_tmp135_ = FALSE;
				if (show_param) {
					_tmp135_ = TRUE;
				} else {
					_tmp135_ = !hide_param;
				}
				if (_tmp135_) {
					vala_method_add_parameter (m, p);
					if (set_array_length_pos) {
						vala_formal_parameter_set_carray_length_parameter_position (p, array_length_pos);
					}
					if (set_delegate_target_pos) {
						vala_formal_parameter_set_cdelegate_target_parameter_position (p, delegate_target_pos);
					}
				}
				_tmp137_ = NULL;
				_tmp136_ = NULL;
				last_param = (_tmp137_ = (_tmp136_ = p, (_tmp136_ == NULL) ? NULL : vala_code_node_ref (_tmp136_)), (last_param == NULL) ? NULL : (last_param = (vala_code_node_unref (last_param), NULL)), _tmp137_);
				_tmp139_ = NULL;
				_tmp138_ = NULL;
				last_param_type = (_tmp139_ = (_tmp138_ = param_type, (_tmp138_ == NULL) ? NULL : vala_code_node_ref (_tmp138_)), (last_param_type == NULL) ? NULL : (last_param_type = (vala_code_node_unref (last_param_type), NULL)), _tmp139_);
				param_name = (g_free (param_name), NULL);
				(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
				(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		vala_method_set_binding (m, MEMBER_BINDING_STATIC);
	}
	_tmp140_ = FALSE;
	if (last_param != NULL) {
		_tmp140_ = g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) last_param), "first_");
	} else {
		_tmp140_ = FALSE;
	}
	if (_tmp140_) {
		vala_formal_parameter_set_ellipsis (last_param, TRUE);
	} else {
		if (add_ellipsis) {
			ValaFormalParameter* _tmp141_;
			_tmp141_ = NULL;
			vala_method_add_parameter (m, _tmp141_ = vala_formal_parameter_new_with_ellipsis (NULL));
			(_tmp141_ == NULL) ? NULL : (_tmp141_ = (vala_code_node_unref (_tmp141_), NULL));
		}
	}
	_tmp142_ = NULL;
	return (_tmp142_ = m, (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), (last_param == NULL) ? NULL : (last_param = (vala_code_node_unref (last_param), NULL)), (last_param_type == NULL) ? NULL : (last_param_type = (vala_code_node_unref (last_param_type), NULL)), _tmp142_);
}


static gboolean vala_gidl_parser_param_is_exception (ValaGIdlParser* self, GIdlNodeParam* param) {
	const char* _tmp1_;
	char* s;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (param != NULL, FALSE);
	if (!param->type->is_error) {
		return FALSE;
	}
	_tmp1_ = NULL;
	s = (_tmp1_ = g_strchomp (param->type->unparsed), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
	if (g_str_has_suffix (s, "**")) {
		gboolean _tmp2_;
		return (_tmp2_ = TRUE, s = (g_free (s), NULL), _tmp2_);
	}
	return (_tmp3_ = FALSE, s = (g_free (s), NULL), _tmp3_);
}


static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_interface) {
	GIdlNode* node;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (f != NULL, NULL);
	node = (GIdlNode*) f;
	if (f->deprecated) {
		return NULL;
	}
	return vala_gidl_parser_create_method (self, node->name, f->symbol, f->result, f->parameters, f->is_constructor, is_interface);
}


static ValaMethod* vala_gidl_parser_parse_virtual (ValaGIdlParser* self, GIdlNodeVFunc* v, GIdlNodeFunction* func, gboolean is_interface) {
	GIdlNode* node;
	char* _tmp0_;
	char* _tmp1_;
	char* symbol;
	GList* _tmp4_;
	ValaMethod* m;
	ValaMethod* _tmp7_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (v != NULL, NULL);
	node = (GIdlNode*) v;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	symbol = (_tmp1_ = g_strdup_printf ("%s%s", _tmp0_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self->priv->current_data_type), node->name), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	if (func != NULL) {
		char* _tmp3_;
		const char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		symbol = (_tmp3_ = (_tmp2_ = func->symbol, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), symbol = (g_free (symbol), NULL), _tmp3_);
	}
	_tmp4_ = NULL;
	if (func != NULL) {
		_tmp4_ = func->parameters;
	} else {
		_tmp4_ = v->parameters;
	}
	m = vala_gidl_parser_create_method (self, node->name, symbol, v->result, _tmp4_, FALSE, is_interface);
	if (m != NULL) {
		gboolean _tmp5_;
		gboolean _tmp6_;
		vala_method_set_binding (m, MEMBER_BINDING_INSTANCE);
		_tmp5_ = FALSE;
		if (vala_method_get_is_abstract (m)) {
			_tmp5_ = TRUE;
		} else {
			_tmp5_ = is_interface;
		}
		vala_method_set_is_virtual (m, !_tmp5_);
		_tmp6_ = FALSE;
		if (vala_method_get_is_abstract (m)) {
			_tmp6_ = TRUE;
		} else {
			_tmp6_ = is_interface;
		}
		vala_method_set_is_abstract (m, _tmp6_);
		if (func == NULL) {
			((ValaCodeNode*) m)->attributes = g_list_append (((ValaCodeNode*) m)->attributes, vala_attribute_new ("NoWrapper", NULL));
		}
	}
	_tmp7_ = NULL;
	return (_tmp7_ = m, symbol = (g_free (symbol), NULL), _tmp7_);
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name) {
	GString* str;
	const char* _tmp0_;
	char* i;
	const char* _tmp3_;
	char* _tmp4_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	str = g_string_new ("");
	_tmp0_ = NULL;
	i = (_tmp0_ = name, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2_;
		const char* _tmp1_;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		i = (_tmp2_ = (_tmp1_ = g_utf8_next_char (i), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), i = (g_free (i), NULL), _tmp2_);
	}
	_tmp3_ = NULL;
	_tmp4_ = NULL;
	return (_tmp4_ = (_tmp3_ = str->str, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), (str == NULL) ? NULL : (str = (g_string_free (str, TRUE), NULL)), i = (g_free (i), NULL), _tmp4_);
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node) {
	GIdlNode* node;
	gboolean _tmp1_;
	ValaDataType* _tmp3_;
	char* _tmp2_;
	ValaProperty* _tmp4_;
	ValaProperty* prop;
	char** _tmp13_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp11_;
	char* _tmp10_;
	char* _tmp9_;
	char** _tmp12_;
	char** attributes;
	ValaProperty* _tmp27_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (prop_node != NULL, NULL);
	node = (GIdlNode*) prop_node;
	if (prop_node->deprecated) {
		return NULL;
	}
	_tmp1_ = FALSE;
	if (!prop_node->readable) {
		_tmp1_ = !prop_node->writable;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp4_ = NULL;
	prop = (_tmp4_ = vala_property_new (_tmp2_ = vala_gidl_parser_fix_prop_name (self, node->name), _tmp3_ = vala_gidl_parser_parse_type (self, prop_node->type, NULL), NULL, NULL, self->priv->current_source_reference), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL)), _tmp2_ = (g_free (_tmp2_), NULL), _tmp4_);
	vala_symbol_set_access ((ValaSymbol*) prop, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_property_set_interface_only (prop, TRUE);
	if (prop_node->readable) {
		ValaPropertyAccessor* _tmp6_;
		ValaDataType* _tmp5_;
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		vala_property_set_get_accessor (prop, _tmp6_ = vala_property_accessor_new (TRUE, FALSE, FALSE, _tmp5_ = vala_data_type_copy (vala_property_get_property_type (prop)), NULL, NULL));
		(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
		(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* _tmp8_;
		ValaDataType* _tmp7_;
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		vala_property_set_set_accessor (prop, _tmp8_ = vala_property_accessor_new (FALSE, FALSE, FALSE, _tmp7_ = vala_data_type_copy (vala_property_get_property_type (prop)), NULL, NULL));
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
		if (prop_node->construct_only) {
			vala_property_accessor_set_construction (vala_property_get_set_accessor (prop), TRUE);
		} else {
			vala_property_accessor_set_writable (vala_property_get_set_accessor (prop), TRUE);
			vala_property_accessor_set_construction (vala_property_get_set_accessor (prop), prop_node->construct);
		}
	}
	_tmp13_ = NULL;
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	_tmp12_ = NULL;
	attributes = (_tmp13_ = (_tmp12_ = vala_gidl_parser_get_attributes (self, _tmp10_ = g_strdup_printf ("%s:%s", _tmp9_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), node->name), &_tmp11_), _tmp10_ = (g_free (_tmp10_), NULL), _tmp9_ = (g_free (_tmp9_), NULL), _tmp12_), attributes_length1 = _tmp11_, attributes_size = attributes_length1, _tmp13_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp26_;
				char* attr;
				_tmp26_ = NULL;
				attr = (_tmp26_ = attr_collection[attr_it], (_tmp26_ == NULL) ? NULL : g_strdup (_tmp26_));
				{
					char** _tmp15_;
					gint nv_size;
					gint nv_length1;
					char** _tmp14_;
					char** nv;
					_tmp15_ = NULL;
					_tmp14_ = NULL;
					nv = (_tmp15_ = _tmp14_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp14_), nv_size = nv_length1, _tmp15_);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp16_;
						gboolean _tmp17_;
						_tmp16_ = NULL;
						if ((_tmp17_ = _vala_strcmp0 (_tmp16_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp16_ = (g_free (_tmp16_), NULL), _tmp17_)) {
							ValaProperty* _tmp18_;
							_tmp18_ = NULL;
							return (_tmp18_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp18_);
						}
					} else {
						if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
							char** _tmp22_;
							gint type_args_size;
							gint type_args_length1;
							char** _tmp20_;
							char* _tmp19_;
							char** _tmp21_;
							char** type_args;
							_tmp22_ = NULL;
							_tmp20_ = NULL;
							_tmp19_ = NULL;
							_tmp21_ = NULL;
							type_args = (_tmp22_ = (_tmp21_ = _tmp20_ = g_strsplit (_tmp19_ = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp19_ = (g_free (_tmp19_), NULL), _tmp21_), type_args_length1 = _vala_array_length (_tmp20_), type_args_size = type_args_length1, _tmp22_);
							{
								char** type_arg_collection;
								int type_arg_collection_length1;
								int type_arg_it;
								type_arg_collection = type_args;
								type_arg_collection_length1 = type_args_length1;
								for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
									const char* _tmp25_;
									char* type_arg;
									_tmp25_ = NULL;
									type_arg = (_tmp25_ = type_arg_collection[type_arg_it], (_tmp25_ == NULL) ? NULL : g_strdup (_tmp25_));
									{
										ValaUnresolvedSymbol* _tmp23_;
										ValaUnresolvedType* _tmp24_;
										ValaUnresolvedType* arg_type;
										_tmp23_ = NULL;
										_tmp24_ = NULL;
										arg_type = (_tmp24_ = vala_unresolved_type_new_from_symbol (_tmp23_ = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_code_node_unref (_tmp23_), NULL)), _tmp24_);
										vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
										vala_data_type_add_type_argument (vala_property_get_property_type (prop), (ValaDataType*) arg_type);
										type_arg = (g_free (type_arg), NULL);
										(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
									}
								}
							}
							type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add ((GeeCollection*) self->priv->current_type_symbol_set, vala_symbol_get_name ((ValaSymbol*) prop));
	}
	_tmp27_ = NULL;
	return (_tmp27_ = prop, attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp27_);
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node) {
	GIdlNode* node;
	ValaDataType* type;
	ValaConstant* c;
	char** _tmp2_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp1_;
	char** attributes;
	ValaConstant* _tmp10_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (const_node != NULL, NULL);
	node = (GIdlNode*) const_node;
	type = vala_gidl_parser_parse_type (self, const_node->type, NULL);
	if (type == NULL) {
		ValaConstant* _tmp0_;
		_tmp0_ = NULL;
		return (_tmp0_ = NULL, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0_);
	}
	c = vala_constant_new (node->name, type, NULL, self->priv->current_source_reference);
	vala_symbol_set_external ((ValaSymbol*) c, TRUE);
	_tmp2_ = NULL;
	attributes = (_tmp2_ = vala_gidl_parser_get_attributes (self, node->name, &_tmp1_), attributes_length1 = _tmp1_, attributes_size = attributes_length1, _tmp2_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp9_;
				char* attr;
				_tmp9_ = NULL;
				attr = (_tmp9_ = attr_collection[attr_it], (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
				{
					char** _tmp4_;
					gint nv_size;
					gint nv_length1;
					char** _tmp3_;
					char** nv;
					_tmp4_ = NULL;
					_tmp3_ = NULL;
					nv = (_tmp4_ = _tmp3_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp3_), nv_size = nv_length1, _tmp4_);
					if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
						char* _tmp5_;
						_tmp5_ = NULL;
						vala_member_add_cheader_filename ((ValaMember*) c, _tmp5_ = vala_gidl_parser_eval (self, nv[1]));
						_tmp5_ = (g_free (_tmp5_), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "hidden") == 0) {
							char* _tmp6_;
							gboolean _tmp7_;
							_tmp6_ = NULL;
							if ((_tmp7_ = _vala_strcmp0 (_tmp6_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp6_ = (g_free (_tmp6_), NULL), _tmp7_)) {
								ValaConstant* _tmp8_;
								_tmp8_ = NULL;
								return (_tmp8_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp8_);
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_symbol_set_access ((ValaSymbol*) c, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp10_ = NULL;
	return (_tmp10_ = c, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp10_);
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node) {
	GIdlNode* node;
	gboolean unhidden;
	ValaDataType* type;
	char* cheader_filename;
	char* ctype;
	char** _tmp5_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp3_;
	char* _tmp2_;
	char* _tmp1_;
	char** _tmp4_;
	char** attributes;
	gboolean _tmp28_;
	const char* _tmp30_;
	char* field_name;
	ValaField* field;
	ValaField* _tmp32_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (field_node != NULL, NULL);
	node = (GIdlNode*) field_node;
	unhidden = FALSE;
	type = vala_gidl_parser_parse_type (self, field_node->type, NULL);
	if (type == NULL) {
		ValaField* _tmp0_;
		_tmp0_ = NULL;
		return (_tmp0_ = NULL, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0_);
	}
	cheader_filename = NULL;
	ctype = NULL;
	_tmp5_ = NULL;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp4_ = NULL;
	attributes = (_tmp5_ = (_tmp4_ = vala_gidl_parser_get_attributes (self, _tmp2_ = g_strdup_printf ("%s.%s", _tmp1_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), node->name), &_tmp3_), _tmp2_ = (g_free (_tmp2_), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp4_), attributes_length1 = _tmp3_, attributes_size = attributes_length1, _tmp5_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp27_;
				char* attr;
				_tmp27_ = NULL;
				attr = (_tmp27_ = attr_collection[attr_it], (_tmp27_ == NULL) ? NULL : g_strdup (_tmp27_));
				{
					char** _tmp7_;
					gint nv_size;
					gint nv_length1;
					char** _tmp6_;
					char** nv;
					_tmp7_ = NULL;
					_tmp6_ = NULL;
					nv = (_tmp7_ = _tmp6_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp6_), nv_size = nv_length1, _tmp7_);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp8_;
						gboolean _tmp9_;
						_tmp8_ = NULL;
						if ((_tmp9_ = _vala_strcmp0 (_tmp8_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_)) {
							ValaField* _tmp10_;
							_tmp10_ = NULL;
							return (_tmp10_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), cheader_filename = (g_free (cheader_filename), NULL), ctype = (g_free (ctype), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp10_);
						} else {
							unhidden = TRUE;
						}
					} else {
						if (_vala_strcmp0 (nv[0], "is_array") == 0) {
							char* _tmp11_;
							gboolean _tmp12_;
							_tmp11_ = NULL;
							if ((_tmp12_ = _vala_strcmp0 (_tmp11_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp11_ = (g_free (_tmp11_), NULL), _tmp12_)) {
								ValaDataType* _tmp13_;
								_tmp13_ = NULL;
								type = (_tmp13_ = (ValaDataType*) vala_array_type_new (type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp13_);
							}
						} else {
							if (_vala_strcmp0 (nv[0], "weak") == 0) {
								char* _tmp14_;
								gboolean _tmp15_;
								_tmp14_ = NULL;
								if ((_tmp15_ = _vala_strcmp0 (_tmp14_ = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp14_ = (g_free (_tmp14_), NULL), _tmp15_)) {
									vala_data_type_set_value_owned (type, TRUE);
								}
							} else {
								if (_vala_strcmp0 (nv[0], "type_name") == 0) {
									ValaUnresolvedSymbol* _tmp17_;
									char* _tmp16_;
									_tmp17_ = NULL;
									_tmp16_ = NULL;
									vala_unresolved_type_set_unresolved_symbol (VALA_UNRESOLVED_TYPE (type), _tmp17_ = vala_unresolved_symbol_new (NULL, _tmp16_ = vala_gidl_parser_eval (self, nv[1]), NULL));
									(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_code_node_unref (_tmp17_), NULL));
									_tmp16_ = (g_free (_tmp16_), NULL);
								} else {
									if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
										char** _tmp21_;
										gint type_args_size;
										gint type_args_length1;
										char** _tmp19_;
										char* _tmp18_;
										char** _tmp20_;
										char** type_args;
										_tmp21_ = NULL;
										_tmp19_ = NULL;
										_tmp18_ = NULL;
										_tmp20_ = NULL;
										type_args = (_tmp21_ = (_tmp20_ = _tmp19_ = g_strsplit (_tmp18_ = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp18_ = (g_free (_tmp18_), NULL), _tmp20_), type_args_length1 = _vala_array_length (_tmp19_), type_args_size = type_args_length1, _tmp21_);
										{
											char** type_arg_collection;
											int type_arg_collection_length1;
											int type_arg_it;
											type_arg_collection = type_args;
											type_arg_collection_length1 = type_args_length1;
											for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
												const char* _tmp24_;
												char* type_arg;
												_tmp24_ = NULL;
												type_arg = (_tmp24_ = type_arg_collection[type_arg_it], (_tmp24_ == NULL) ? NULL : g_strdup (_tmp24_));
												{
													ValaUnresolvedSymbol* _tmp22_;
													ValaUnresolvedType* _tmp23_;
													ValaUnresolvedType* arg_type;
													_tmp22_ = NULL;
													_tmp23_ = NULL;
													arg_type = (_tmp23_ = vala_unresolved_type_new_from_symbol (_tmp22_ = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_code_node_unref (_tmp22_), NULL)), _tmp23_);
													vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
													vala_data_type_add_type_argument (type, (ValaDataType*) arg_type);
													type_arg = (g_free (type_arg), NULL);
													(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
												}
											}
										}
										type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
									} else {
										if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
											char* _tmp25_;
											_tmp25_ = NULL;
											cheader_filename = (_tmp25_ = vala_gidl_parser_eval (self, nv[1]), cheader_filename = (g_free (cheader_filename), NULL), _tmp25_);
										} else {
											if (_vala_strcmp0 (nv[0], "ctype") == 0) {
												char* _tmp26_;
												_tmp26_ = NULL;
												ctype = (_tmp26_ = vala_gidl_parser_eval (self, nv[1]), ctype = (g_free (ctype), NULL), _tmp26_);
											}
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	_tmp28_ = FALSE;
	if (g_str_has_prefix (node->name, "_")) {
		_tmp28_ = !unhidden;
	} else {
		_tmp28_ = FALSE;
	}
	if (_tmp28_) {
		ValaField* _tmp29_;
		_tmp29_ = NULL;
		return (_tmp29_ = NULL, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), cheader_filename = (g_free (cheader_filename), NULL), ctype = (g_free (ctype), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp29_);
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add ((GeeCollection*) self->priv->current_type_symbol_set, node->name);
	}
	_tmp30_ = NULL;
	field_name = (_tmp30_ = node->name, (_tmp30_ == NULL) ? NULL : g_strdup (_tmp30_));
	if (_vala_strcmp0 (field_name, "string") == 0) {
		char* _tmp31_;
		/* avoid conflict with string type*/
		_tmp31_ = NULL;
		field_name = (_tmp31_ = g_strdup ("str"), field_name = (g_free (field_name), NULL), _tmp31_);
	}
	field = vala_field_new (field_name, type, NULL, self->priv->current_source_reference);
	vala_symbol_set_access ((ValaSymbol*) field, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (_vala_strcmp0 (field_name, node->name) != 0) {
		vala_field_set_cname (field, node->name);
	}
	if (ctype != NULL) {
		vala_field_set_ctype (field, ctype);
	}
	if (cheader_filename != NULL) {
		vala_member_add_cheader_filename ((ValaMember*) field, cheader_filename);
	}
	vala_field_set_no_array_length (field, TRUE);
	_tmp32_ = NULL;
	return (_tmp32_ = field, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), cheader_filename = (g_free (cheader_filename), NULL), ctype = (g_free (ctype), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), field_name = (g_free (field_name), NULL), _tmp32_);
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode, int* result_length1) {
	char* attributes;
	char** _tmp8_;
	char** _tmp9_;
	char** _tmp10_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (codenode != NULL, NULL);
	attributes = (char*) gee_map_get (self->priv->codenode_attributes_map, codenode);
	if (attributes == NULL) {
		gboolean dot_required;
		gboolean colon_required;
		GeeSet* pattern_specs;
		dot_required = NULL != g_utf8_strchr (codenode, (glong) (-1), (gunichar) '.');
		colon_required = NULL != g_utf8_strchr (codenode, (glong) (-1), (gunichar) ':');
		pattern_specs = gee_map_get_keys (self->priv->codenode_attributes_patterns);
		{
			GeeIterator* _pattern_it;
			_pattern_it = gee_iterable_iterator ((GeeIterable*) pattern_specs);
			while (gee_iterator_next (_pattern_it)) {
				GPatternSpec* pattern;
				char* pspec;
				gboolean _tmp0_;
				gboolean _tmp1_;
				pattern = gee_iterator_get (_pattern_it);
				pspec = (char*) gee_map_get (self->priv->codenode_attributes_patterns, pattern);
				_tmp0_ = FALSE;
				_tmp1_ = FALSE;
				if (dot_required) {
					_tmp1_ = NULL == g_utf8_strchr (pspec, (glong) (-1), (gunichar) '.');
				} else {
					_tmp1_ = FALSE;
				}
				if (_tmp1_) {
					_tmp0_ = TRUE;
				} else {
					gboolean _tmp2_;
					_tmp2_ = FALSE;
					if (colon_required) {
						_tmp2_ = NULL == g_utf8_strchr (pspec, (glong) (-1), (gunichar) ':');
					} else {
						_tmp2_ = FALSE;
					}
					_tmp0_ = _tmp2_;
				}
				if (_tmp0_) {
					pspec = (g_free (pspec), NULL);
					continue;
				}
				if (g_pattern_match_string (pattern, codenode)) {
					gint _tmp3_;
					char** _tmp4_;
					char** _tmp5_;
					_tmp4_ = NULL;
					_tmp5_ = NULL;
					return (_tmp5_ = (_tmp4_ = vala_gidl_parser_get_attributes (self, pspec, &_tmp3_), *result_length1 = _tmp3_, _tmp4_), pspec = (g_free (pspec), NULL), (_pattern_it == NULL) ? NULL : (_pattern_it = (gee_collection_object_unref (_pattern_it), NULL)), (pattern_specs == NULL) ? NULL : (pattern_specs = (gee_collection_object_unref (pattern_specs), NULL)), attributes = (g_free (attributes), NULL), _tmp5_);
				}
				pspec = (g_free (pspec), NULL);
			}
			(_pattern_it == NULL) ? NULL : (_pattern_it = (gee_collection_object_unref (_pattern_it), NULL));
		}
		(pattern_specs == NULL) ? NULL : (pattern_specs = (gee_collection_object_unref (pattern_specs), NULL));
	}
	if (attributes == NULL) {
		gpointer _tmp6_;
		char** _tmp7_;
		_tmp7_ = NULL;
		return (_tmp7_ = (_tmp6_ = NULL, *result_length1 = 0, _tmp6_), attributes = (g_free (attributes), NULL), _tmp7_);
	}
	_tmp8_ = NULL;
	_tmp9_ = NULL;
	_tmp10_ = NULL;
	return (_tmp10_ = (_tmp9_ = _tmp8_ = g_strsplit (attributes, " ", 0), *result_length1 = _vala_array_length (_tmp8_), _tmp9_), attributes = (g_free (attributes), NULL), _tmp10_);
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (s != NULL, NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, (glong) 1), (gsize) (strlen (s) - 2));
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node) {
	GIdlNode* node;
	gboolean _tmp0_;
	ValaDataType* _tmp3_;
	char* _tmp2_;
	ValaSignal* _tmp4_;
	ValaSignal* sig;
	char** _tmp9_;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp7_;
	char* _tmp6_;
	char* _tmp5_;
	char** _tmp8_;
	char** attributes;
	gboolean first;
	ValaSignal* _tmp41_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sig_node != NULL, NULL);
	node = (GIdlNode*) sig_node;
	_tmp0_ = FALSE;
	if (sig_node->deprecated) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sig_node->result == NULL;
	}
	if (_tmp0_) {
		return NULL;
	}
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp4_ = NULL;
	sig = (_tmp4_ = vala_signal_new (_tmp2_ = vala_gidl_parser_fix_prop_name (self, node->name), _tmp3_ = vala_gidl_parser_parse_param (self, sig_node->result, NULL), self->priv->current_source_reference), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL)), _tmp2_ = (g_free (_tmp2_), NULL), _tmp4_);
	vala_symbol_set_access ((ValaSymbol*) sig, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp9_ = NULL;
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	_tmp8_ = NULL;
	attributes = (_tmp9_ = (_tmp8_ = vala_gidl_parser_get_attributes (self, _tmp6_ = g_strdup_printf ("%s::%s", _tmp5_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), vala_symbol_get_name ((ValaSymbol*) sig)), &_tmp7_), _tmp6_ = (g_free (_tmp6_), NULL), _tmp5_ = (g_free (_tmp5_), NULL), _tmp8_), attributes_length1 = _tmp7_, attributes_size = attributes_length1, _tmp9_);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp18_;
				char* attr;
				_tmp18_ = NULL;
				attr = (_tmp18_ = attr_collection[attr_it], (_tmp18_ == NULL) ? NULL : g_strdup (_tmp18_));
				{
					char** _tmp11_;
					gint nv_size;
					gint nv_length1;
					char** _tmp10_;
					char** nv;
					_tmp11_ = NULL;
					_tmp10_ = NULL;
					nv = (_tmp11_ = _tmp10_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp10_), nv_size = nv_length1, _tmp11_);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp12_;
						vala_signal_set_cname (sig, vala_symbol_get_name ((ValaSymbol*) sig));
						_tmp12_ = NULL;
						vala_symbol_set_name ((ValaSymbol*) sig, _tmp12_ = vala_gidl_parser_eval (self, nv[1]));
						_tmp12_ = (g_free (_tmp12_), NULL);
					} else {
						gboolean _tmp13_;
						_tmp13_ = FALSE;
						if (_vala_strcmp0 (nv[0], "has_emitter") == 0) {
							char* _tmp14_;
							_tmp14_ = NULL;
							_tmp13_ = _vala_strcmp0 (_tmp14_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0;
							_tmp14_ = (g_free (_tmp14_), NULL);
						} else {
							_tmp13_ = FALSE;
						}
						if (_tmp13_) {
							vala_signal_set_has_emitter (sig, TRUE);
						} else {
							if (_vala_strcmp0 (nv[0], "hidden") == 0) {
								char* _tmp15_;
								gboolean _tmp16_;
								_tmp15_ = NULL;
								if ((_tmp16_ = _vala_strcmp0 (_tmp15_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp15_ = (g_free (_tmp15_), NULL), _tmp16_)) {
									ValaSignal* _tmp17_;
									_tmp17_ = NULL;
									return (_tmp17_ = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp17_);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_signal_set_is_virtual (sig, TRUE);
	first = TRUE;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = sig_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = (GIdlNodeParam*) param_it->data;
			{
				GIdlNode* param_node;
				ValaParameterDirection direction;
				ValaDataType* param_type;
				ValaFormalParameter* p;
				char** _tmp22_;
				gint _tmp21_;
				char* _tmp20_;
				char* _tmp19_;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = (GIdlNode*) param;
				direction = 0;
				param_type = vala_gidl_parser_parse_param (self, param, &direction);
				p = vala_formal_parameter_new (param_node->name, param_type, NULL);
				vala_formal_parameter_set_direction (p, direction);
				vala_signal_add_parameter (sig, p);
				_tmp22_ = NULL;
				_tmp20_ = NULL;
				_tmp19_ = NULL;
				attributes = (_tmp22_ = vala_gidl_parser_get_attributes (self, _tmp20_ = g_strdup_printf ("%s::%s.%s", _tmp19_ = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), vala_symbol_get_name ((ValaSymbol*) sig), param_node->name), &_tmp21_), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), attributes_length1 = _tmp21_, attributes_size = attributes_length1, _tmp22_);
				_tmp20_ = (g_free (_tmp20_), NULL);
				_tmp19_ = (g_free (_tmp19_), NULL);
				if (attributes != NULL) {
					char* ns_name;
					ns_name = NULL;
					{
						char** attr_collection;
						int attr_collection_length1;
						int attr_it;
						attr_collection = attributes;
						attr_collection_length1 = attributes_length1;
						for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
							const char* _tmp39_;
							char* attr;
							_tmp39_ = NULL;
							attr = (_tmp39_ = attr_collection[attr_it], (_tmp39_ == NULL) ? NULL : g_strdup (_tmp39_));
							{
								char** _tmp24_;
								gint nv_size;
								gint nv_length1;
								char** _tmp23_;
								char** nv;
								_tmp24_ = NULL;
								_tmp23_ = NULL;
								nv = (_tmp24_ = _tmp23_ = g_strsplit (attr, "=", 2), nv_length1 = _vala_array_length (_tmp23_), nv_size = nv_length1, _tmp24_);
								if (_vala_strcmp0 (nv[0], "is_array") == 0) {
									char* _tmp25_;
									gboolean _tmp26_;
									_tmp25_ = NULL;
									if ((_tmp26_ = _vala_strcmp0 (_tmp25_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp25_ = (g_free (_tmp25_), NULL), _tmp26_)) {
										ValaDataType* _tmp27_;
										_tmp27_ = NULL;
										param_type = (_tmp27_ = (ValaDataType*) vala_array_type_new (param_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) param_type)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp27_);
										vala_formal_parameter_set_parameter_type (p, param_type);
										vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_IN);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "is_out") == 0) {
										char* _tmp28_;
										gboolean _tmp29_;
										_tmp28_ = NULL;
										if ((_tmp29_ = _vala_strcmp0 (_tmp28_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp28_ = (g_free (_tmp28_), NULL), _tmp29_)) {
											vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_OUT);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
											char* _tmp30_;
											gboolean _tmp31_;
											_tmp30_ = NULL;
											if ((_tmp31_ = _vala_strcmp0 (_tmp30_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp30_ = (g_free (_tmp30_), NULL), _tmp31_)) {
												vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_REF);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "nullable") == 0) {
												char* _tmp32_;
												gboolean _tmp33_;
												_tmp32_ = NULL;
												if ((_tmp33_ = _vala_strcmp0 (_tmp32_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp32_ = (g_free (_tmp32_), NULL), _tmp33_)) {
													vala_data_type_set_nullable (param_type, TRUE);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
													char* _tmp34_;
													gboolean _tmp35_;
													_tmp34_ = NULL;
													if ((_tmp35_ = _vala_strcmp0 (_tmp34_ = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp34_ = (g_free (_tmp34_), NULL), _tmp35_)) {
														vala_data_type_set_value_owned (param_type, TRUE);
													}
												} else {
													if (_vala_strcmp0 (nv[0], "type_name") == 0) {
														ValaUnresolvedSymbol* _tmp37_;
														char* _tmp36_;
														_tmp37_ = NULL;
														_tmp36_ = NULL;
														vala_unresolved_type_set_unresolved_symbol (VALA_UNRESOLVED_TYPE (param_type), _tmp37_ = vala_unresolved_symbol_new (NULL, _tmp36_ = vala_gidl_parser_eval (self, nv[1]), NULL));
														(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_code_node_unref (_tmp37_), NULL));
														_tmp36_ = (g_free (_tmp36_), NULL);
													} else {
														if (_vala_strcmp0 (nv[0], "namespace_name") == 0) {
															char* _tmp38_;
															_tmp38_ = NULL;
															ns_name = (_tmp38_ = vala_gidl_parser_eval (self, nv[1]), ns_name = (g_free (ns_name), NULL), _tmp38_);
														}
													}
												}
											}
										}
									}
								}
								attr = (g_free (attr), NULL);
								nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
							}
						}
					}
					if (ns_name != NULL) {
						ValaUnresolvedSymbol* _tmp40_;
						_tmp40_ = NULL;
						vala_unresolved_symbol_set_inner (vala_unresolved_type_get_unresolved_symbol (VALA_UNRESOLVED_TYPE (param_type)), _tmp40_ = vala_unresolved_symbol_new (NULL, ns_name, NULL));
						(_tmp40_ == NULL) ? NULL : (_tmp40_ = (vala_code_node_unref (_tmp40_), NULL));
					}
					ns_name = (g_free (ns_name), NULL);
				}
				(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
				(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
			}
		}
	}
	_tmp41_ = NULL;
	return (_tmp41_ = sig, attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp41_);
}


/**
 * Code visitor parsing all GIDL files.
 */
ValaGIdlParser* vala_gidl_parser_construct (GType object_type) {
	ValaGIdlParser* self;
	self = (ValaGIdlParser*) g_type_create_instance (object_type);
	return self;
}


ValaGIdlParser* vala_gidl_parser_new (void) {
	return vala_gidl_parser_construct (VALA_TYPE_GIDL_PARSER);
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass) {
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gidl_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gidl_parser_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gidl_parser_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gidl_parser_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gidl_parser_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gidl_parser_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_gidl_parser_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_gidl_parser_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_instance_init (ValaGIdlParser * self) {
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_finalize (ValaCodeVisitor* obj) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL));
	(self->priv->current_source_reference == NULL) ? NULL : (self->priv->current_source_reference = (vala_source_reference_unref (self->priv->current_source_reference), NULL));
	(self->priv->current_namespace == NULL) ? NULL : (self->priv->current_namespace = (vala_code_node_unref (self->priv->current_namespace), NULL));
	(self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL));
	(self->priv->codenode_attributes_map == NULL) ? NULL : (self->priv->codenode_attributes_map = (gee_collection_object_unref (self->priv->codenode_attributes_map), NULL));
	(self->priv->codenode_attributes_patterns == NULL) ? NULL : (self->priv->codenode_attributes_patterns = (gee_collection_object_unref (self->priv->codenode_attributes_patterns), NULL));
	(self->priv->current_type_symbol_set == NULL) ? NULL : (self->priv->current_type_symbol_set = (gee_collection_object_unref (self->priv->current_type_symbol_set), NULL));
	(self->priv->cname_type_map == NULL) ? NULL : (self->priv->cname_type_map = (gee_collection_object_unref (self->priv->cname_type_map), NULL));
	VALA_CODE_VISITOR_CLASS (vala_gidl_parser_parent_class)->finalize (obj);
}


GType vala_gidl_parser_get_type (void) {
	static GType vala_gidl_parser_type_id = 0;
	if (vala_gidl_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_instance_init, NULL };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




