/* valainitializerlist.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaExpressionPrivate ValaExpressionPrivate;

#define VALA_TYPE_INITIALIZER_LIST (vala_initializer_list_get_type ())
#define VALA_INITIALIZER_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INITIALIZER_LIST, ValaInitializerList))
#define VALA_INITIALIZER_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListClass))
#define VALA_IS_INITIALIZER_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INITIALIZER_LIST))
#define VALA_IS_INITIALIZER_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INITIALIZER_LIST))
#define VALA_INITIALIZER_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListClass))

typedef struct _ValaInitializerList ValaInitializerList;
typedef struct _ValaInitializerListClass ValaInitializerListClass;
typedef struct _ValaInitializerListPrivate ValaInitializerListPrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ARRAY_TYPE (vala_array_type_get_type ())
#define VALA_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayType))
#define VALA_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))
#define VALA_IS_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_TYPE))
#define VALA_IS_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_TYPE))
#define VALA_ARRAY_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))

typedef struct _ValaArrayType ValaArrayType;
typedef struct _ValaArrayTypeClass ValaArrayTypeClass;

#define VALA_TYPE_ARRAY_CREATION_EXPRESSION (vala_array_creation_expression_get_type ())
#define VALA_ARRAY_CREATION_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpression))
#define VALA_ARRAY_CREATION_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpressionClass))
#define VALA_IS_ARRAY_CREATION_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_CREATION_EXPRESSION))
#define VALA_IS_ARRAY_CREATION_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_CREATION_EXPRESSION))
#define VALA_ARRAY_CREATION_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_CREATION_EXPRESSION, ValaArrayCreationExpressionClass))

typedef struct _ValaArrayCreationExpression ValaArrayCreationExpression;
typedef struct _ValaArrayCreationExpressionClass ValaArrayCreationExpressionClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_CONSTANT (vala_constant_get_type ())
#define VALA_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CONSTANT, ValaConstant))
#define VALA_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CONSTANT, ValaConstantClass))
#define VALA_IS_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CONSTANT))
#define VALA_IS_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CONSTANT))
#define VALA_CONSTANT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CONSTANT, ValaConstantClass))

typedef struct _ValaConstant ValaConstant;
typedef struct _ValaConstantClass ValaConstantClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_STRUCT (vala_struct_get_type ())
#define VALA_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STRUCT, ValaStruct))
#define VALA_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_STRUCT, ValaStructClass))
#define VALA_IS_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STRUCT))
#define VALA_IS_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_STRUCT))
#define VALA_STRUCT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_STRUCT, ValaStructClass))

typedef struct _ValaStruct ValaStruct;
typedef struct _ValaStructClass ValaStructClass;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_UNARY_EXPRESSION (vala_unary_expression_get_type ())
#define VALA_UNARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_UNARY_EXPRESSION, ValaUnaryExpression))
#define VALA_UNARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_UNARY_EXPRESSION, ValaUnaryExpressionClass))
#define VALA_IS_UNARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_UNARY_EXPRESSION))
#define VALA_IS_UNARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_UNARY_EXPRESSION))
#define VALA_UNARY_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_UNARY_EXPRESSION, ValaUnaryExpressionClass))

typedef struct _ValaUnaryExpression ValaUnaryExpression;
typedef struct _ValaUnaryExpressionClass ValaUnaryExpressionClass;

#define VALA_TYPE_UNARY_OPERATOR (vala_unary_operator_get_type ())

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Base class for all code nodes that might be used as an expression.
 */
struct _ValaExpression {
	ValaCodeNode parent_instance;
	ValaExpressionPrivate * priv;
	GeeArrayList* temp_vars;
};

struct _ValaExpressionClass {
	ValaCodeNodeClass parent_class;
	gboolean (*is_constant) (ValaExpression* self);
	gboolean (*is_pure) (ValaExpression* self);
	gboolean (*is_non_null) (ValaExpression* self);
};

/**
 * Represents an array or struct initializer list in the source code.
 */
struct _ValaInitializerList {
	ValaExpression parent_instance;
	ValaInitializerListPrivate * priv;
};

struct _ValaInitializerListClass {
	ValaExpressionClass parent_class;
};

struct _ValaInitializerListPrivate {
	GeeList* initializers;
};

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;

typedef enum  {
	VALA_UNARY_OPERATOR_NONE,
	VALA_UNARY_OPERATOR_PLUS,
	VALA_UNARY_OPERATOR_MINUS,
	VALA_UNARY_OPERATOR_LOGICAL_NEGATION,
	VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT,
	VALA_UNARY_OPERATOR_INCREMENT,
	VALA_UNARY_OPERATOR_DECREMENT,
	VALA_UNARY_OPERATOR_REF,
	VALA_UNARY_OPERATOR_OUT
} ValaUnaryOperator;



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_initializer_list_get_type (void);
#define VALA_INITIALIZER_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListPrivate))
enum  {
	VALA_INITIALIZER_LIST_DUMMY_PROPERTY
};
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_initializer_list_append (ValaInitializerList* self, ValaExpression* expr);
GeeList* vala_initializer_list_get_initializers (ValaInitializerList* self);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
ValaInitializerList* vala_initializer_list_new (ValaSourceReference* source_reference);
ValaInitializerList* vala_initializer_list_construct (GType object_type, ValaSourceReference* source_reference);
ValaInitializerList* vala_initializer_list_new (ValaSourceReference* source_reference);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_initializer_list_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_visitor_visit_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list);
static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
gboolean vala_expression_is_pure (ValaExpression* self);
static gboolean vala_initializer_list_real_is_pure (ValaExpression* base);
static void vala_initializer_list_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
ValaDataType* vala_expression_get_target_type (ValaExpression* self);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
GType vala_reference_type_get_type (void);
GType vala_array_type_get_type (void);
ValaCodeNode* vala_code_node_get_parent_node (ValaCodeNode* self);
GType vala_array_creation_expression_get_type (void);
GType vala_member_get_type (void);
GType vala_constant_get_type (void);
ValaDataType* vala_array_type_get_element_type (ValaArrayType* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
gint vala_array_type_get_rank (ValaArrayType* self);
ValaArrayCreationExpression* vala_array_creation_expression_new (ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference);
ValaArrayCreationExpression* vala_array_creation_expression_construct (GType object_type, ValaDataType* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
void vala_code_node_replace_expression (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
void vala_array_type_set_rank (ValaArrayType* self, gint value);
GType vala_typesymbol_get_type (void);
ValaTypeSymbol* vala_data_type_get_data_type (ValaDataType* self);
GType vala_struct_get_type (void);
GType vala_field_get_type (void);
GeeList* vala_struct_get_fields (ValaStruct* self);
char* vala_code_node_to_string (ValaCodeNode* self);
GType member_binding_get_type (void);
MemberBinding vala_field_get_binding (ValaField* self);
ValaDataType* vala_field_get_field_type (ValaField* self);
gboolean vala_data_type_get_value_owned (ValaDataType* self);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
GType vala_unary_expression_get_type (void);
GType vala_unary_operator_get_type (void);
ValaUnaryOperator vala_unary_expression_get_operator (ValaUnaryExpression* self);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value);
static gboolean vala_initializer_list_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
gint vala_initializer_list_get_size (ValaInitializerList* self);
static gpointer vala_initializer_list_parent_class = NULL;
static void vala_initializer_list_finalize (ValaCodeNode* obj);



/**
 * Appends the specified expression to this initializer 
 *
 * @param expr an expression
 */
void vala_initializer_list_append (ValaInitializerList* self, ValaExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_collection_add ((GeeCollection*) self->priv->initializers, expr);
	vala_code_node_set_parent_node ((ValaCodeNode*) expr, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the expression 
 *
 * @return expression list
 */
GeeList* vala_initializer_list_get_initializers (ValaInitializerList* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->initializers);
}


/**
 * Creates a new initializer 
 *
 * @param source_reference reference to source code
 * @return                 newly created initializer list
 */
ValaInitializerList* vala_initializer_list_construct (GType object_type, ValaSourceReference* source_reference) {
	ValaInitializerList* self;
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaInitializerList*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaInitializerList* vala_initializer_list_new (ValaSourceReference* source_reference) {
	return vala_initializer_list_construct (VALA_TYPE_INITIALIZER_LIST, source_reference);
}


static void vala_initializer_list_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializers);
		while (gee_iterator_next (_expr_it)) {
			ValaExpression* expr;
			expr = (ValaExpression*) gee_iterator_get (_expr_it);
			vala_code_node_accept ((ValaCodeNode*) expr, visitor);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
}


static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_initializer_list (visitor, self);
}


static gboolean vala_initializer_list_real_is_pure (ValaExpression* base) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	{
		GeeIterator* _initializer_it;
		_initializer_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializers);
		while (gee_iterator_next (_initializer_it)) {
			ValaExpression* initializer;
			initializer = (ValaExpression*) gee_iterator_get (_initializer_it);
			if (!vala_expression_is_pure (initializer)) {
				gboolean _tmp0_;
				return (_tmp0_ = FALSE, (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), (_initializer_it == NULL) ? NULL : (_initializer_it = (gee_collection_object_unref (_initializer_it), NULL)), _tmp0_);
			}
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
		}
		(_initializer_it == NULL) ? NULL : (_initializer_it = (gee_collection_object_unref (_initializer_it), NULL));
	}
	return TRUE;
}


static void vala_initializer_list_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->initializers); i++) {
			ValaExpression* _tmp0_;
			gboolean _tmp1_;
			_tmp0_ = NULL;
			if ((_tmp1_ = (_tmp0_ = (ValaExpression*) gee_list_get (self->priv->initializers, i)) == old_node, (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp1_)) {
				gee_list_set (self->priv->initializers, i, new_node);
			}
		}
	}
}


static gboolean vala_initializer_list_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaInitializerList * self;
	gboolean _error_;
	self = (ValaInitializerList*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (vala_expression_get_target_type ((ValaExpression*) self) == NULL) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "initializer list used for unknown type");
		return FALSE;
	} else {
		if (VALA_IS_ARRAY_TYPE (vala_expression_get_target_type ((ValaExpression*) self))) {
			ValaArrayType* _tmp2_;
			ValaArrayType* array_type;
			gboolean _tmp3_;
			gboolean _tmp4_;
			ValaDataType* inner_target_type;
			/* initializer is used as array initializer */
			_tmp2_ = NULL;
			array_type = (_tmp2_ = VALA_ARRAY_TYPE (vala_expression_get_target_type ((ValaExpression*) self)), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
			_tmp3_ = FALSE;
			_tmp4_ = FALSE;
			if (!VALA_IS_ARRAY_CREATION_EXPRESSION (vala_code_node_get_parent_node ((ValaCodeNode*) self))) {
				_tmp4_ = !VALA_IS_CONSTANT (vala_code_node_get_parent_node ((ValaCodeNode*) self));
			} else {
				_tmp4_ = FALSE;
			}
			if (_tmp4_) {
				_tmp3_ = !VALA_IS_INITIALIZER_LIST (vala_code_node_get_parent_node ((ValaCodeNode*) self));
			} else {
				_tmp3_ = FALSE;
			}
			if (_tmp3_) {
				ValaCodeNode* _tmp5_;
				ValaCodeNode* old_parent_node;
				ValaDataType* _tmp6_;
				ValaArrayCreationExpression* _tmp7_;
				ValaArrayCreationExpression* array_creation;
				gboolean _tmp8_;
				/* transform shorthand form
				     int[] array = { 42 };
				 into
				     int[] array = new int[] { 42 };*/
				_tmp5_ = NULL;
				old_parent_node = (_tmp5_ = vala_code_node_get_parent_node ((ValaCodeNode*) self), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
				_tmp6_ = NULL;
				_tmp7_ = NULL;
				array_creation = (_tmp7_ = vala_array_creation_expression_new (_tmp6_ = vala_data_type_copy (vala_array_type_get_element_type (array_type)), vala_array_type_get_rank (array_type), self, vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL)), _tmp7_);
				vala_expression_set_target_type ((ValaExpression*) array_creation, vala_expression_get_target_type ((ValaExpression*) self));
				vala_code_node_replace_expression (old_parent_node, (ValaExpression*) self, (ValaExpression*) array_creation);
				vala_code_node_set_checked ((ValaCodeNode*) self, FALSE);
				return (_tmp8_ = vala_code_node_check ((ValaCodeNode*) array_creation, analyzer), (old_parent_node == NULL) ? NULL : (old_parent_node = (vala_code_node_unref (old_parent_node), NULL)), (array_creation == NULL) ? NULL : (array_creation = (vala_code_node_unref (array_creation), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp8_);
			}
			inner_target_type = NULL;
			if (vala_array_type_get_rank (array_type) > 1) {
				ValaArrayType* inner_array_type;
				gint _tmp9_;
				ValaDataType* _tmp11_;
				ValaDataType* _tmp10_;
				/* allow initialization of multi-dimensional arrays*/
				inner_array_type = VALA_ARRAY_TYPE (vala_data_type_copy ((ValaDataType*) array_type));
				_tmp9_ = vala_array_type_get_rank (inner_array_type);
				vala_array_type_set_rank (inner_array_type, _tmp9_ - 1);
				_tmp9_;
				_tmp11_ = NULL;
				_tmp10_ = NULL;
				inner_target_type = (_tmp11_ = (_tmp10_ = (ValaDataType*) inner_array_type, (_tmp10_ == NULL) ? NULL : vala_code_node_ref (_tmp10_)), (inner_target_type == NULL) ? NULL : (inner_target_type = (vala_code_node_unref (inner_target_type), NULL)), _tmp11_);
				(inner_array_type == NULL) ? NULL : (inner_array_type = (vala_code_node_unref (inner_array_type), NULL));
			} else {
				ValaDataType* _tmp12_;
				_tmp12_ = NULL;
				inner_target_type = (_tmp12_ = vala_data_type_copy (vala_array_type_get_element_type (array_type)), (inner_target_type == NULL) ? NULL : (inner_target_type = (vala_code_node_unref (inner_target_type), NULL)), _tmp12_);
			}
			{
				GeeList* _tmp13_;
				GeeIterator* _tmp14_;
				GeeIterator* _e_it;
				_tmp13_ = NULL;
				_tmp14_ = NULL;
				_e_it = (_tmp14_ = gee_iterable_iterator ((GeeIterable*) (_tmp13_ = vala_initializer_list_get_initializers (self))), (_tmp13_ == NULL) ? NULL : (_tmp13_ = (gee_collection_object_unref (_tmp13_), NULL)), _tmp14_);
				while (gee_iterator_next (_e_it)) {
					ValaExpression* e;
					e = (ValaExpression*) gee_iterator_get (_e_it);
					vala_expression_set_target_type (e, inner_target_type);
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
				}
				(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
			}
			(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			(inner_target_type == NULL) ? NULL : (inner_target_type = (vala_code_node_unref (inner_target_type), NULL));
		} else {
			if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_expression_get_target_type ((ValaExpression*) self)))) {
				ValaStruct* _tmp15_;
				ValaStruct* st;
				GeeList* _tmp16_;
				GeeIterator* _tmp17_;
				GeeIterator* field_it;
				/* initializer is used as struct initializer */
				_tmp15_ = NULL;
				st = (_tmp15_ = VALA_STRUCT (vala_data_type_get_data_type (vala_expression_get_target_type ((ValaExpression*) self))), (_tmp15_ == NULL) ? NULL : vala_code_node_ref (_tmp15_));
				_tmp16_ = NULL;
				_tmp17_ = NULL;
				field_it = (_tmp17_ = gee_iterable_iterator ((GeeIterable*) (_tmp16_ = vala_struct_get_fields (st))), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (gee_collection_object_unref (_tmp16_), NULL)), _tmp17_);
				{
					GeeList* _tmp18_;
					GeeIterator* _tmp19_;
					GeeIterator* _e_it;
					_tmp18_ = NULL;
					_tmp19_ = NULL;
					_e_it = (_tmp19_ = gee_iterable_iterator ((GeeIterable*) (_tmp18_ = vala_initializer_list_get_initializers (self))), (_tmp18_ == NULL) ? NULL : (_tmp18_ = (gee_collection_object_unref (_tmp18_), NULL)), _tmp19_);
					while (gee_iterator_next (_e_it)) {
						ValaExpression* e;
						ValaField* field;
						ValaDataType* _tmp25_;
						e = (ValaExpression*) gee_iterator_get (_e_it);
						field = NULL;
						while (field == NULL) {
							ValaField* _tmp23_;
							if (!gee_iterator_next (field_it)) {
								char* _tmp21_;
								char* _tmp20_;
								gboolean _tmp22_;
								vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
								_tmp21_ = NULL;
								_tmp20_ = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), _tmp21_ = g_strdup_printf ("too many expressions in initializer list for `%s'", _tmp20_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type ((ValaExpression*) self))));
								_tmp21_ = (g_free (_tmp21_), NULL);
								_tmp20_ = (g_free (_tmp20_), NULL);
								return (_tmp22_ = FALSE, (e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL)), (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), (_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (field_it == NULL) ? NULL : (field_it = (gee_collection_object_unref (field_it), NULL)), _tmp22_);
							}
							_tmp23_ = NULL;
							field = (_tmp23_ = (ValaField*) gee_iterator_get (field_it), (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), _tmp23_);
							if (vala_field_get_binding (field) != MEMBER_BINDING_INSTANCE) {
								ValaField* _tmp24_;
								/* we only initialize instance fields*/
								_tmp24_ = NULL;
								field = (_tmp24_ = NULL, (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), _tmp24_);
							}
						}
						_tmp25_ = NULL;
						vala_expression_set_target_type (e, _tmp25_ = vala_data_type_copy (vala_field_get_field_type (field)));
						(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_code_node_unref (_tmp25_), NULL));
						if (!vala_data_type_get_value_owned (vala_expression_get_target_type ((ValaExpression*) self))) {
							vala_data_type_set_value_owned (vala_expression_get_target_type (e), FALSE);
						}
						(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
						(field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL));
					}
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
				}
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
				(field_it == NULL) ? NULL : (field_it = (gee_collection_object_unref (field_it), NULL));
			} else {
				char* _tmp27_;
				char* _tmp26_;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp27_ = NULL;
				_tmp26_ = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp27_ = g_strdup_printf ("initializer list used for `%s', which is neither array nor struct", _tmp26_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type ((ValaExpression*) self))));
				_tmp27_ = (g_free (_tmp27_), NULL);
				_tmp26_ = (g_free (_tmp26_), NULL);
				return FALSE;
			}
		}
	}
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializers);
		while (gee_iterator_next (_expr_it)) {
			ValaExpression* expr;
			expr = (ValaExpression*) gee_iterator_get (_expr_it);
			vala_code_node_check ((ValaCodeNode*) expr, analyzer);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
	_error_ = FALSE;
	{
		GeeList* _tmp29_;
		GeeIterator* _tmp30_;
		GeeIterator* _e_it;
		_tmp29_ = NULL;
		_tmp30_ = NULL;
		_e_it = (_tmp30_ = gee_iterable_iterator ((GeeIterable*) (_tmp29_ = vala_initializer_list_get_initializers (self))), (_tmp29_ == NULL) ? NULL : (_tmp29_ = (gee_collection_object_unref (_tmp29_), NULL)), _tmp30_);
		while (gee_iterator_next (_e_it)) {
			ValaExpression* e;
			ValaUnaryExpression* _tmp32_;
			ValaExpression* _tmp31_;
			ValaUnaryExpression* unary;
			gboolean _tmp33_;
			e = (ValaExpression*) gee_iterator_get (_e_it);
			if (vala_expression_get_value_type (e) == NULL) {
				_error_ = TRUE;
				(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
				continue;
			}
			_tmp32_ = NULL;
			_tmp31_ = NULL;
			unary = (_tmp32_ = (_tmp31_ = e, VALA_IS_UNARY_EXPRESSION (_tmp31_) ? ((ValaUnaryExpression*) _tmp31_) : NULL), (_tmp32_ == NULL) ? NULL : vala_code_node_ref (_tmp32_));
			_tmp33_ = FALSE;
			if (unary != NULL) {
				gboolean _tmp34_;
				_tmp34_ = FALSE;
				if (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_REF) {
					_tmp34_ = TRUE;
				} else {
					_tmp34_ = vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_OUT;
				}
				_tmp33_ = _tmp34_;
			} else {
				_tmp33_ = FALSE;
			}
			if (_tmp33_) {
			} else {
				if (!vala_data_type_compatible (vala_expression_get_value_type (e), vala_expression_get_target_type (e))) {
					char* _tmp37_;
					char* _tmp36_;
					char* _tmp35_;
					_error_ = TRUE;
					vala_code_node_set_error ((ValaCodeNode*) e, TRUE);
					_tmp37_ = NULL;
					_tmp36_ = NULL;
					_tmp35_ = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), _tmp37_ = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", _tmp35_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (e)), _tmp36_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (e))));
					_tmp37_ = (g_free (_tmp37_), NULL);
					_tmp36_ = (g_free (_tmp36_), NULL);
					_tmp35_ = (g_free (_tmp35_), NULL);
				}
			}
			(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
			(unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL));
		}
		(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
	}
	if (!_error_) {
		/* everything seems to be correct */
		vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_target_type ((ValaExpression*) self));
	}
	return !_error_;
}


gint vala_initializer_list_get_size (ValaInitializerList* self) {
	g_return_val_if_fail (self != NULL, 0);
	return gee_collection_get_size ((GeeCollection*) self->priv->initializers);
}


static void vala_initializer_list_class_init (ValaInitializerListClass * klass) {
	vala_initializer_list_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_initializer_list_finalize;
	g_type_class_add_private (klass, sizeof (ValaInitializerListPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_initializer_list_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_initializer_list_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_initializer_list_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_initializer_list_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_initializer_list_real_check;
}


static void vala_initializer_list_instance_init (ValaInitializerList * self) {
	self->priv = VALA_INITIALIZER_LIST_GET_PRIVATE (self);
	self->priv->initializers = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_initializer_list_finalize (ValaCodeNode* obj) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (obj);
	(self->priv->initializers == NULL) ? NULL : (self->priv->initializers = (gee_collection_object_unref (self->priv->initializers), NULL));
	VALA_CODE_NODE_CLASS (vala_initializer_list_parent_class)->finalize (obj);
}


GType vala_initializer_list_get_type (void) {
	static GType vala_initializer_list_type_id = 0;
	if (vala_initializer_list_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInitializerListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_initializer_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInitializerList), 0, (GInstanceInitFunc) vala_initializer_list_instance_init, NULL };
		vala_initializer_list_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaInitializerList", &g_define_type_info, 0);
	}
	return vala_initializer_list_type_id;
}




