/* valalocalvariable.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valamemberaccess.h>
#include <vala/valalambdaexpression.h>
#include <vala/valamethod.h>
#include <vala/valadelegatetype.h>
#include <vala/valadelegate.h>
#include <vala/valapointertype.h>
#include <vala/valasourcefile.h>
#include <vala/valascope.h>
#include <vala/valablock.h>




struct _ValaLocalVariablePrivate {
	gboolean _floating;
	ValaExpression* _initializer;
	ValaDataType* _variable_type;
};

#define VALA_LOCAL_VARIABLE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariablePrivate))
enum  {
	VALA_LOCAL_VARIABLE_DUMMY_PROPERTY
};
static void vala_local_variable_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_local_variable_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_local_variable_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_local_variable_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_local_variable_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_local_variable_parent_class = NULL;
static void vala_local_variable_finalize (ValaCodeNode* obj);



/**
 * Creates a new local variable.
 *
 * @param name   name of the variable
 * @param init   optional initializer expression
 * @param source reference to source code
 * @return       newly created variable declarator
 */
ValaLocalVariable* vala_local_variable_construct (GType object_type, ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaLocalVariable* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaLocalVariable*) vala_symbol_construct (object_type, name, source_reference);
	vala_local_variable_set_variable_type (self, variable_type);
	vala_local_variable_set_initializer (self, initializer);
	return self;
}


ValaLocalVariable* vala_local_variable_new (ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	return vala_local_variable_construct (VALA_TYPE_LOCAL_VARIABLE, variable_type, name, initializer, source_reference);
}


static void vala_local_variable_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_local_variable (visitor, self);
}


static void vala_local_variable_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_local_variable_get_initializer (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_local_variable_get_initializer (self), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_local_variable_get_initializer (self));
	}
	if (vala_local_variable_get_variable_type (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_local_variable_get_variable_type (self), visitor);
	}
}


static void vala_local_variable_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_local_variable_get_initializer (self) == old_node) {
		vala_local_variable_set_initializer (self, new_node);
	}
}


static void vala_local_variable_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_local_variable_get_variable_type (self) == old_type) {
		vala_local_variable_set_variable_type (self, new_type);
	}
}


static gboolean vala_local_variable_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaLocalVariable * self;
	ValaBlock* _tmp22;
	ValaSymbol* _tmp21;
	ValaBlock* block;
	gboolean _tmp23;
	self = (ValaLocalVariable*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (vala_local_variable_get_variable_type (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_local_variable_get_variable_type (self), analyzer);
	}
	if (vala_local_variable_get_initializer (self) != NULL) {
		vala_expression_set_target_type (vala_local_variable_get_initializer (self), vala_local_variable_get_variable_type (self));
		vala_code_node_check ((ValaCodeNode*) vala_local_variable_get_initializer (self), analyzer);
	}
	if (vala_local_variable_get_variable_type (self) == NULL) {
		ValaDataType* _tmp3;
		/* var type */
		if (vala_local_variable_get_initializer (self) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "var declaration not allowed without initializer");
			return FALSE;
		}
		if (vala_expression_get_value_type (vala_local_variable_get_initializer (self)) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "var declaration not allowed with non-typed initializer");
			return FALSE;
		}
		_tmp3 = NULL;
		vala_local_variable_set_variable_type (self, _tmp3 = vala_data_type_copy (vala_expression_get_value_type (vala_local_variable_get_initializer (self))));
		(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
		vala_data_type_set_value_owned (vala_local_variable_get_variable_type (self), TRUE);
		vala_data_type_set_floating_reference (vala_local_variable_get_variable_type (self), FALSE);
		vala_expression_set_target_type (vala_local_variable_get_initializer (self), vala_local_variable_get_variable_type (self));
	}
	if (vala_local_variable_get_initializer (self) != NULL) {
		if (vala_expression_get_value_type (vala_local_variable_get_initializer (self)) == NULL) {
			gboolean _tmp4;
			gboolean _tmp6;
			_tmp4 = FALSE;
			if (!VALA_IS_MEMBER_ACCESS (vala_local_variable_get_initializer (self))) {
				_tmp4 = !VALA_IS_LAMBDA_EXPRESSION (vala_local_variable_get_initializer (self));
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "expression type not allowed as initializer");
				return FALSE;
			}
			_tmp6 = FALSE;
			if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_local_variable_get_initializer (self)))) {
				_tmp6 = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (self));
			} else {
				_tmp6 = FALSE;
			}
			if (_tmp6) {
				ValaMethod* _tmp7;
				ValaMethod* m;
				ValaDelegateType* _tmp8;
				ValaDelegateType* dt;
				ValaDelegate* _tmp9;
				ValaDelegate* cb;
				_tmp7 = NULL;
				m = (_tmp7 = VALA_METHOD (vala_expression_get_symbol_reference (vala_local_variable_get_initializer (self))), (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
				_tmp8 = NULL;
				dt = (_tmp8 = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (self)), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
				_tmp9 = NULL;
				cb = (_tmp9 = vala_delegate_type_get_delegate_symbol (dt), (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9));
				/* check whether method matches callback type */
				if (!vala_delegate_matches_method (cb, m)) {
					char* _tmp12;
					char* _tmp11;
					char* _tmp10;
					gboolean _tmp13;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp12 = NULL;
					_tmp11 = NULL;
					_tmp10 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp12 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp10 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp11 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
					_tmp12 = (g_free (_tmp12), NULL);
					_tmp11 = (g_free (_tmp11), NULL);
					_tmp10 = (g_free (_tmp10), NULL);
					return (_tmp13 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), _tmp13);
				}
				vala_expression_set_value_type (vala_local_variable_get_initializer (self), vala_local_variable_get_variable_type (self));
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
				(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
				(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "expression type not allowed as initializer");
				return FALSE;
			}
		}
		if (!vala_data_type_compatible (vala_expression_get_value_type (vala_local_variable_get_initializer (self)), vala_local_variable_get_variable_type (self))) {
			char* _tmp17;
			char* _tmp16;
			char* _tmp15;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp17 = NULL;
			_tmp16 = NULL;
			_tmp15 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp17 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp15 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_local_variable_get_initializer (self))), _tmp16 = vala_code_node_to_string ((ValaCodeNode*) vala_local_variable_get_variable_type (self))));
			_tmp17 = (g_free (_tmp17), NULL);
			_tmp16 = (g_free (_tmp16), NULL);
			_tmp15 = (g_free (_tmp15), NULL);
			return FALSE;
		}
		if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_local_variable_get_initializer (self)))) {
			gboolean _tmp19;
			_tmp19 = FALSE;
			if (!VALA_IS_POINTER_TYPE (vala_local_variable_get_variable_type (self))) {
				_tmp19 = !vala_data_type_get_value_owned (vala_local_variable_get_variable_type (self));
			} else {
				_tmp19 = FALSE;
			}
			/* rhs transfers ownership of the expression */
			if (_tmp19) {
				/* lhs doesn't own the value */
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
				return FALSE;
			}
		}
	}
	vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_local_variable_get_variable_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_scope_add (vala_symbol_get_scope (vala_semantic_analyzer_get_current_symbol (analyzer)), vala_symbol_get_name ((ValaSymbol*) self), (ValaSymbol*) self);
	/* current_symbol is a Method if this is the `result'
	 variable used for postconditions*/
	_tmp22 = NULL;
	_tmp21 = NULL;
	block = (_tmp22 = (_tmp21 = vala_semantic_analyzer_get_current_symbol (analyzer), VALA_IS_BLOCK (_tmp21) ? ((ValaBlock*) _tmp21) : NULL), (_tmp22 == NULL) ? NULL : vala_code_node_ref (_tmp22));
	if (block != NULL) {
		vala_block_add_local_variable (block, self);
	}
	vala_symbol_set_active ((ValaSymbol*) self, TRUE);
	return (_tmp23 = !vala_code_node_get_error ((ValaCodeNode*) self), (block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL)), _tmp23);
}


ValaExpression* vala_local_variable_get_initializer (ValaLocalVariable* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_initializer;
}


void vala_local_variable_set_initializer (ValaLocalVariable* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_initializer == NULL) ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL)), _tmp2);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_initializer, (ValaCodeNode*) self);
	}
}


ValaDataType* vala_local_variable_get_variable_type (ValaLocalVariable* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_variable_type;
}


void vala_local_variable_set_variable_type (ValaLocalVariable* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_variable_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_variable_type == NULL) ? NULL : (self->priv->_variable_type = (vala_code_node_unref (self->priv->_variable_type), NULL)), _tmp2);
	if (self->priv->_variable_type != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_variable_type, (ValaCodeNode*) self);
	}
}


gboolean vala_local_variable_get_floating (ValaLocalVariable* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_floating;
}


void vala_local_variable_set_floating (ValaLocalVariable* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_floating = value;
}


static void vala_local_variable_class_init (ValaLocalVariableClass * klass) {
	vala_local_variable_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_local_variable_finalize;
	g_type_class_add_private (klass, sizeof (ValaLocalVariablePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_local_variable_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_local_variable_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_local_variable_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_local_variable_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_local_variable_real_check;
}


static void vala_local_variable_instance_init (ValaLocalVariable * self) {
	self->priv = VALA_LOCAL_VARIABLE_GET_PRIVATE (self);
}


static void vala_local_variable_finalize (ValaCodeNode* obj) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (obj);
	(self->priv->_initializer == NULL) ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL));
	(self->priv->_variable_type == NULL) ? NULL : (self->priv->_variable_type = (vala_code_node_unref (self->priv->_variable_type), NULL));
	VALA_CODE_NODE_CLASS (vala_local_variable_parent_class)->finalize (obj);
}


GType vala_local_variable_get_type (void) {
	static GType vala_local_variable_type_id = 0;
	if (vala_local_variable_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaLocalVariableClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_local_variable_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaLocalVariable), 0, (GInstanceInitFunc) vala_local_variable_instance_init, NULL };
		vala_local_variable_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaLocalVariable", &g_define_type_info, 0);
	}
	return vala_local_variable_type_id;
}




