/* valacastexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valacastexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasourcefile.h>




struct _ValaCastExpressionPrivate {
	gboolean _is_silent_cast;
	ValaExpression* _inner;
	ValaDataType* _data_type;
};

#define VALA_CAST_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CAST_EXPRESSION, ValaCastExpressionPrivate))
enum  {
	VALA_CAST_EXPRESSION_DUMMY_PROPERTY
};
static void vala_cast_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_cast_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_cast_expression_real_is_pure (ValaExpression* base);
static void vala_cast_expression_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_cast_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_cast_expression_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_cast_expression_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_cast_expression_parent_class = NULL;
static void vala_cast_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new cast expression.
 *
 * @param inner expression to be cast
 * @param type  target type
 * @return      newly created cast expression
 */
ValaCastExpression* vala_cast_expression_construct (GType object_type, ValaExpression* inner, ValaDataType* type_reference, ValaSourceReference* source_reference, gboolean is_silent_cast) {
	ValaCastExpression* self;
	g_return_val_if_fail (inner != NULL, NULL);
	g_return_val_if_fail (type_reference != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaCastExpression*) g_type_create_instance (object_type);
	vala_cast_expression_set_type_reference (self, type_reference);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_cast_expression_set_is_silent_cast (self, is_silent_cast);
	vala_cast_expression_set_inner (self, inner);
	return self;
}


ValaCastExpression* vala_cast_expression_new (ValaExpression* inner, ValaDataType* type_reference, ValaSourceReference* source_reference, gboolean is_silent_cast) {
	return vala_cast_expression_construct (VALA_TYPE_CAST_EXPRESSION, inner, type_reference, source_reference, is_silent_cast);
}


static void vala_cast_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaCastExpression * self;
	self = (ValaCastExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_cast_expression_get_inner (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_cast_expression_get_type_reference (self), visitor);
	vala_code_visitor_visit_cast_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_cast_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaCastExpression * self;
	self = (ValaCastExpression*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_cast_expression_get_inner (self) == old_node) {
		vala_cast_expression_set_inner (self, new_node);
	}
}


static gboolean vala_cast_expression_real_is_pure (ValaExpression* base) {
	ValaCastExpression * self;
	self = (ValaCastExpression*) base;
	return vala_expression_is_pure (vala_cast_expression_get_inner (self));
}


static void vala_cast_expression_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaCastExpression * self;
	self = (ValaCastExpression*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_cast_expression_get_type_reference (self) == old_type) {
		vala_cast_expression_set_type_reference (self, new_type);
	}
}


static gboolean vala_cast_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaCastExpression * self;
	ValaDataType* _tmp2;
	self = (ValaCastExpression*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_cast_expression_get_inner (self), analyzer)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	vala_code_node_check ((ValaCodeNode*) vala_cast_expression_get_type_reference (self), analyzer);
	/* FIXME: check whether cast is allowed*/
	vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_cast_expression_get_type_reference (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_expression_set_value_type ((ValaExpression*) self, vala_cast_expression_get_type_reference (self));
	vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), vala_data_type_get_value_owned (vala_expression_get_value_type (vala_cast_expression_get_inner (self))));
	_tmp2 = NULL;
	vala_expression_set_target_type (vala_cast_expression_get_inner (self), _tmp2 = vala_data_type_copy (vala_expression_get_value_type (vala_cast_expression_get_inner (self))));
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_cast_expression_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaCastExpression * self;
	self = (ValaCastExpression*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_cast_expression_get_inner (self), collection);
}


static void vala_cast_expression_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaCastExpression * self;
	self = (ValaCastExpression*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_cast_expression_get_inner (self), collection);
}


ValaExpression* vala_cast_expression_get_inner (ValaCastExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_inner;
}


void vala_cast_expression_set_inner (ValaCastExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_inner = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_inner, (ValaCodeNode*) self);
}


ValaDataType* vala_cast_expression_get_type_reference (ValaCastExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_cast_expression_set_type_reference (ValaCastExpression* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
}


gboolean vala_cast_expression_get_is_silent_cast (ValaCastExpression* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_silent_cast;
}


void vala_cast_expression_set_is_silent_cast (ValaCastExpression* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_silent_cast = value;
}


static void vala_cast_expression_class_init (ValaCastExpressionClass * klass) {
	vala_cast_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_cast_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaCastExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_cast_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_cast_expression_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_cast_expression_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_cast_expression_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_cast_expression_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_cast_expression_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_cast_expression_real_get_used_variables;
}


static void vala_cast_expression_instance_init (ValaCastExpression * self) {
	self->priv = VALA_CAST_EXPRESSION_GET_PRIVATE (self);
}


static void vala_cast_expression_finalize (ValaCodeNode* obj) {
	ValaCastExpression * self;
	self = VALA_CAST_EXPRESSION (obj);
	(self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL));
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	VALA_CODE_NODE_CLASS (vala_cast_expression_parent_class)->finalize (obj);
}


GType vala_cast_expression_get_type (void) {
	static GType vala_cast_expression_type_id = 0;
	if (vala_cast_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCastExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_cast_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCastExpression), 0, (GInstanceInitFunc) vala_cast_expression_instance_init, NULL };
		vala_cast_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaCastExpression", &g_define_type_info, 0);
	}
	return vala_cast_expression_type_id;
}




