/* valadelegate.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valadelegate.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypeparameter.h>
#include <vala/valaformalparameter.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valamethod.h>
#include <vala/valamember.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valareferencetype.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasourcefile.h>




struct _ValaDelegatePrivate {
	gboolean _has_target;
	double _cinstance_parameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	gboolean _no_array_length;
	gboolean _array_null_terminated;
	GeeList* type_parameters;
	GeeList* parameters;
	char* cname;
	ValaDataType* _return_type;
};

#define VALA_DELEGATE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DELEGATE, ValaDelegatePrivate))
enum  {
	VALA_DELEGATE_DUMMY_PROPERTY
};
static void vala_delegate_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_delegate_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_delegate_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static void vala_delegate_process_ccode_attribute (ValaDelegate* self, ValaAttribute* a);
static gboolean vala_delegate_real_is_reference_type (ValaTypeSymbol* base);
static char* vala_delegate_real_get_type_id (ValaTypeSymbol* base);
static char* vala_delegate_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_delegate_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_delegate_real_get_set_value_function (ValaTypeSymbol* base);
static void vala_delegate_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static char* vala_delegate_get_return_type_string (ValaDelegate* self);
static char* vala_delegate_get_parameters_string (ValaDelegate* self);
static gboolean vala_delegate_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_delegate_parent_class = NULL;
static void vala_delegate_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new delegate.
 *
 * @param name        delegate type name
 * @param return_type return type
 * @param source      reference to source code
 * @return            newly created delegate
 */
ValaDelegate* vala_delegate_construct (GType object_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaDelegate* self;
	g_return_val_if_fail (return_type != NULL, NULL);
	self = (ValaDelegate*) vala_typesymbol_construct (object_type, name, source_reference);
	vala_delegate_set_return_type (self, return_type);
	/* error is -1 (right of user_data)*/
	vala_delegate_set_cinstance_parameter_position (self, (double) (-2));
	vala_delegate_set_carray_length_parameter_position (self, (double) (-3));
	vala_delegate_set_cdelegate_target_parameter_position (self, (double) (-3));
	return self;
}


ValaDelegate* vala_delegate_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	return vala_delegate_construct (VALA_TYPE_DELEGATE, name, return_type, source_reference);
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_delegate_add_type_parameter (ValaDelegate* self, ValaTypeParameter* p) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (p != NULL);
	gee_collection_add ((GeeCollection*) self->priv->type_parameters, p);
	p->type = (ValaTypeSymbol*) self;
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) p), (ValaSymbol*) p);
}


/**
 * Appends paramater to this callback function.
 *
 * @param param a formal parameter
 */
void vala_delegate_add_parameter (ValaDelegate* self, ValaFormalParameter* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	/* default C parameter position*/
	vala_formal_parameter_set_cparameter_position (param, (double) (gee_collection_get_size ((GeeCollection*) self->priv->parameters) + 1));
	vala_formal_parameter_set_carray_length_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	vala_formal_parameter_set_cdelegate_target_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	gee_collection_add ((GeeCollection*) self->priv->parameters, param);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) param), (ValaSymbol*) param);
}


/**
 * Return copy of parameter list.
 *
 * @return parameter list
 */
GeeList* vala_delegate_get_parameters (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->parameters);
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this callback.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this callback
 */
gboolean vala_delegate_matches_method (ValaDelegate* self, ValaMethod* m) {
	GeeList* method_params;
	GeeIterator* method_params_it;
	gboolean first;
	gboolean _tmp5;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (m != NULL, FALSE);
	/* method is allowed to ensure stricter return type (stronger postcondition)*/
	if (!vala_data_type_stricter (vala_method_get_return_type (m), vala_delegate_get_return_type (self))) {
		return FALSE;
	}
	method_params = vala_method_get_parameters (m);
	method_params_it = gee_iterable_iterator ((GeeIterable*) method_params);
	first = TRUE;
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			gboolean _tmp1;
			gboolean _tmp2;
			ValaFormalParameter* method_param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			_tmp1 = FALSE;
			_tmp2 = FALSE;
			if (first) {
				_tmp2 = vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE;
			} else {
				_tmp2 = FALSE;
			}
			if (_tmp2) {
				_tmp1 = !self->priv->_has_target;
			} else {
				_tmp1 = FALSE;
			}
			/* use first callback parameter as instance parameter if
			 * an instance method is being compared to a static
			 * callback
			 */
			if (_tmp1) {
				first = FALSE;
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				continue;
			}
			/* method is allowed to accept less arguments */
			if (!gee_iterator_next (method_params_it)) {
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				break;
			}
			/* method is allowed to accept arguments of looser types (weaker precondition)*/
			method_param = (ValaFormalParameter*) gee_iterator_get (method_params_it);
			if (!vala_data_type_stricter (vala_formal_parameter_get_parameter_type (param), vala_formal_parameter_get_parameter_type (method_param))) {
				gboolean _tmp3;
				return (_tmp3 = FALSE, (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), (method_param == NULL) ? NULL : (method_param = (vala_code_node_unref (method_param), NULL)), (_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL)), (method_params == NULL) ? NULL : (method_params = (gee_collection_object_unref (method_params), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp3);
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			(method_param == NULL) ? NULL : (method_param = (vala_code_node_unref (method_param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	/* method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		gboolean _tmp4;
		return (_tmp4 = FALSE, (method_params == NULL) ? NULL : (method_params = (gee_collection_object_unref (method_params), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp4);
	}
	return (_tmp5 = TRUE, (method_params == NULL) ? NULL : (method_params = (gee_collection_object_unref (method_params), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp5);
}


static void vala_delegate_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_delegate (visitor, self);
}


static void vala_delegate_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _p_it;
		_p_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_parameters);
		while (gee_iterator_next (_p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_accept ((ValaCodeNode*) p, visitor);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	vala_code_node_accept ((ValaCodeNode*) vala_delegate_get_return_type (self), visitor);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_code_node_accept ((ValaCodeNode*) param, visitor);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _error_type_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_error_type_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_code_node_get_error_types ((ValaCodeNode*) self))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_error_type_it)) {
			ValaDataType* error_type;
			error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
			vala_code_node_accept ((ValaCodeNode*) error_type, visitor);
			(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
		}
		(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
	}
}


static char* vala_delegate_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaDelegate * self;
	const char* _tmp2;
	self = (ValaDelegate*) base;
	if (self->priv->cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = g_strdup_printf ("%s%s", _tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


/**
 * Sets the name of this callback as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_delegate_set_cname (ValaDelegate* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
}


static void vala_delegate_process_ccode_attribute (ValaDelegate* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_delegate_set_cname (self, _tmp0 = vala_attribute_get_string (a, "cname"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "instance_pos")) {
		vala_delegate_set_cinstance_parameter_position (self, vala_attribute_get_double (a, "instance_pos"));
	}
	if (vala_attribute_has_argument (a, "array_length")) {
		vala_delegate_set_no_array_length (self, !vala_attribute_get_bool (a, "array_length"));
	}
	if (vala_attribute_has_argument (a, "array_null_terminated")) {
		vala_delegate_set_array_null_terminated (self, vala_attribute_get_bool (a, "array_null_terminated"));
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_delegate_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_delegate_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp1;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp1 = NULL;
			filename_collection = _tmp1 = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp1);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp1); filename_it = filename_it + 1) {
				const char* _tmp2;
				char* filename;
				_tmp2 = NULL;
				filename = (_tmp2 = filename_collection[filename_it], (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_delegate_process_attributes (ValaDelegate* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_delegate_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_delegate_real_is_reference_type (ValaTypeSymbol* base) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	return FALSE;
}


static char* vala_delegate_real_get_type_id (ValaTypeSymbol* base) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	return g_strdup ("G_TYPE_POINTER");
}


static char* vala_delegate_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	return g_strdup ("POINTER");
}


static char* vala_delegate_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	return g_strdup ("g_value_get_pointer");
}


static char* vala_delegate_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaDelegate * self;
	self = (ValaDelegate*) base;
	return g_strdup ("g_value_set_pointer");
}


static void vala_delegate_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaDelegate * self;
	GeeList* error_types;
	self = (ValaDelegate*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_delegate_get_return_type (self) == old_type) {
		vala_delegate_set_return_type (self, new_type);
		return;
	}
	error_types = vala_code_node_get_error_types ((ValaCodeNode*) self);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) error_types); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get (error_types, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set (error_types, i, new_type);
				(error_types == NULL) ? NULL : (error_types = (gee_collection_object_unref (error_types), NULL));
				return;
			}
		}
	}
	(error_types == NULL) ? NULL : (error_types = (gee_collection_object_unref (error_types), NULL));
}


char* vala_delegate_get_prototype_string (ValaDelegate* self, const char* name) {
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s %s %s", _tmp0 = vala_delegate_get_return_type_string (self), name, _tmp1 = vala_delegate_get_parameters_string (self)), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static char* vala_delegate_get_return_type_string (ValaDelegate* self) {
	char* str;
	gboolean _tmp0;
	char* _tmp3;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_strdup ("");
	_tmp0 = FALSE;
	if (!vala_data_type_get_value_owned (vala_delegate_get_return_type (self))) {
		_tmp0 = VALA_IS_REFERENCE_TYPE (vala_delegate_get_return_type (self));
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		char* _tmp1;
		_tmp1 = NULL;
		str = (_tmp1 = g_strdup ("weak "), str = (g_free (str), NULL), _tmp1);
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	str = (_tmp3 = g_strconcat (str, _tmp2 = (vala_code_node_to_string ((ValaCodeNode*) vala_delegate_get_return_type (self))), NULL), str = (g_free (str), NULL), _tmp3);
	_tmp2 = (g_free (_tmp2), NULL);
	return str;
}


static char* vala_delegate_get_parameters_string (ValaDelegate* self) {
	char* str;
	gint i;
	char* _tmp9;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_strdup ("(");
	i = 1;
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			char* _tmp6;
			char* _tmp5;
			gboolean _tmp7;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			if (i > 1) {
				char* _tmp0;
				_tmp0 = NULL;
				str = (_tmp0 = g_strconcat (str, (", "), NULL), str = (g_free (str), NULL), _tmp0);
			}
			if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_IN) {
				gboolean _tmp3;
				if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_REF) {
					char* _tmp1;
					_tmp1 = NULL;
					str = (_tmp1 = g_strconcat (str, ("ref "), NULL), str = (g_free (str), NULL), _tmp1);
				} else {
					if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_OUT) {
						char* _tmp2;
						_tmp2 = NULL;
						str = (_tmp2 = g_strconcat (str, ("out "), NULL), str = (g_free (str), NULL), _tmp2);
					}
				}
				_tmp3 = FALSE;
				if (!vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
					_tmp3 = VALA_IS_REFERENCE_TYPE (vala_formal_parameter_get_parameter_type (param));
				} else {
					_tmp3 = FALSE;
				}
				if (_tmp3) {
					char* _tmp4;
					_tmp4 = NULL;
					str = (_tmp4 = g_strconcat (str, ("weak "), NULL), str = (g_free (str), NULL), _tmp4);
				}
			}
			_tmp6 = NULL;
			_tmp5 = NULL;
			str = (_tmp6 = g_strconcat (str, _tmp5 = (vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (param))), NULL), str = (g_free (str), NULL), _tmp6);
			_tmp5 = (g_free (_tmp5), NULL);
			_tmp7 = FALSE;
			if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN) {
				_tmp7 = vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param));
			} else {
				_tmp7 = FALSE;
			}
			if (_tmp7) {
				char* _tmp8;
				_tmp8 = NULL;
				str = (_tmp8 = g_strconcat (str, ("#"), NULL), str = (g_free (str), NULL), _tmp8);
			}
			i++;
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	_tmp9 = NULL;
	str = (_tmp9 = g_strconcat (str, (")"), NULL), str = (g_free (str), NULL), _tmp9);
	return str;
}


static gboolean vala_delegate_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaDelegate * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	gboolean _tmp4;
	self = (ValaDelegate*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_delegate_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	{
		GeeIterator* _p_it;
		_p_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_parameters);
		while (gee_iterator_next (_p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_check ((ValaCodeNode*) p, analyzer);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	vala_code_node_check ((ValaCodeNode*) vala_delegate_get_return_type (self), analyzer);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_code_node_check ((ValaCodeNode*) param, analyzer);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	{
		GeeList* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* _error_type_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		_error_type_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_code_node_get_error_types ((ValaCodeNode*) self))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (_error_type_it)) {
			ValaDataType* error_type;
			error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
			vala_code_node_check ((ValaCodeNode*) error_type, analyzer);
			(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
		}
		(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	return (_tmp4 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), _tmp4);
}


ValaDataType* vala_delegate_get_return_type (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_return_type;
}


void vala_delegate_set_return_type (ValaDelegate* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_return_type == NULL) ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_return_type, (ValaCodeNode*) self);
}


gboolean vala_delegate_get_has_target (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_target;
}


void vala_delegate_set_has_target (ValaDelegate* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_target = value;
}


double vala_delegate_get_cinstance_parameter_position (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cinstance_parameter_position;
}


void vala_delegate_set_cinstance_parameter_position (ValaDelegate* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cinstance_parameter_position = value;
}


double vala_delegate_get_carray_length_parameter_position (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_delegate_set_carray_length_parameter_position (ValaDelegate* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_carray_length_parameter_position = value;
}


double vala_delegate_get_cdelegate_target_parameter_position (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_delegate_set_cdelegate_target_parameter_position (ValaDelegate* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cdelegate_target_parameter_position = value;
}


gboolean vala_delegate_get_no_array_length (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_array_length;
}


void vala_delegate_set_no_array_length (ValaDelegate* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_array_length = value;
}


gboolean vala_delegate_get_array_null_terminated (ValaDelegate* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_array_null_terminated;
}


void vala_delegate_set_array_null_terminated (ValaDelegate* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_array_null_terminated = value;
}


static void vala_delegate_class_init (ValaDelegateClass * klass) {
	vala_delegate_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_delegate_finalize;
	g_type_class_add_private (klass, sizeof (ValaDelegatePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_delegate_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_delegate_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_delegate_real_get_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_delegate_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_delegate_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_delegate_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_delegate_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_delegate_real_get_set_value_function;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_delegate_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_delegate_real_check;
}


static void vala_delegate_instance_init (ValaDelegate * self) {
	self->priv = VALA_DELEGATE_GET_PRIVATE (self);
	self->priv->type_parameters = (GeeList*) gee_array_list_new (VALA_TYPE_TYPEPARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->parameters = (GeeList*) gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_delegate_finalize (ValaCodeNode* obj) {
	ValaDelegate * self;
	self = VALA_DELEGATE (obj);
	(self->priv->type_parameters == NULL) ? NULL : (self->priv->type_parameters = (gee_collection_object_unref (self->priv->type_parameters), NULL));
	(self->priv->parameters == NULL) ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_return_type == NULL) ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL));
	VALA_CODE_NODE_CLASS (vala_delegate_parent_class)->finalize (obj);
}


GType vala_delegate_get_type (void) {
	static GType vala_delegate_type_id = 0;
	if (vala_delegate_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDelegateClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_delegate_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDelegate), 0, (GInstanceInitFunc) vala_delegate_instance_init, NULL };
		vala_delegate_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaDelegate", &g_define_type_info, 0);
	}
	return vala_delegate_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




