/* valagirparser.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vapigen/valagirparser.h>
#include <vala/valanamespace.h>
#include <vala/valasourcelocation.h>
#include <gee/hashmap.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodenode.h>
#include <vala/valaclass.h>
#include <vala/valastruct.h>
#include <vala/valainterface.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valaenum.h>
#include <vala/valadelegate.h>
#include <vala/valamethod.h>
#include <vala/valaconstant.h>
#include <vala/valadatatype.h>
#include <vala/valaenumvalue.h>
#include <vala/valavoidtype.h>
#include <vala/valamember.h>
#include <gee/arraylist.h>
#include <vala/valaformalparameter.h>
#include <gee/collection.h>
#include <vala/valaarraytype.h>
#include <vala/valapointertype.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valafield.h>
#include <vala/valasignal.h>
#include <vala/valaproperty.h>
#include <vala/valaexpression.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valacreationmethod.h>
#include <vala/valaerrortype.h>
#include <vala/valaerrordomain.h>
#include <vala/valaerrorcode.h>
#include <glib/gstdio.h>
#include <gee/map.h>
#include "vapigen/valamarkupreader.h"




static glong string_get_length (const char* self);
struct _ValaGirParserPrivate {
	ValaMarkupReader* reader;
	ValaCodeContext* context;
	ValaNamespace* glib_ns;
	ValaSourceFile* current_source_file;
	ValaSourceLocation begin;
	ValaSourceLocation end;
	ValaMarkupTokenType current_token;
	GeeHashMap* attributes_map;
};

#define VALA_GIR_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIR_PARSER, ValaGirParserPrivate))
enum  {
	VALA_GIR_PARSER_DUMMY_PROPERTY
};
static void vala_gir_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gir_parser_next (ValaGirParser* self);
static void vala_gir_parser_start_element (ValaGirParser* self, const char* name);
static void vala_gir_parser_end_element (ValaGirParser* self, const char* name);
static ValaSourceReference* vala_gir_parser_get_current_src (ValaGirParser* self);
static void vala_gir_parser_parse_repository (ValaGirParser* self);
static void vala_gir_parser_parse_include (ValaGirParser* self);
static ValaNamespace* vala_gir_parser_parse_namespace (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_alias (ValaGirParser* self);
static ValaEnum* vala_gir_parser_parse_enumeration (ValaGirParser* self);
static ValaEnum* vala_gir_parser_parse_bitfield (ValaGirParser* self);
static ValaEnumValue* vala_gir_parser_parse_enumeration_member (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_function (ValaGirParser* self);
static ValaDataType* vala_gir_parser_parse_return_value (ValaGirParser* self);
static ValaFormalParameter* vala_gir_parser_parse_parameter (ValaGirParser* self, gint* array_length_idx);
static ValaDataType* vala_gir_parser_parse_type (ValaGirParser* self, gint* array_length_index);
static ValaDataType* vala_gir_parser_parse_type_from_name (ValaGirParser* self, const char* type_name);
static char* vala_gir_parser_transform_namespace_name (ValaGirParser* self, const char* gir_module_name);
static ValaStruct* vala_gir_parser_parse_record (ValaGirParser* self);
static ValaClass* vala_gir_parser_parse_class (ValaGirParser* self);
static ValaInterface* vala_gir_parser_parse_interface (ValaGirParser* self);
static ValaField* vala_gir_parser_parse_field (ValaGirParser* self);
static ValaProperty* vala_gir_parser_parse_property (ValaGirParser* self);
static ValaDelegate* vala_gir_parser_parse_callback (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_constructor (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_method (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_vmethod (ValaGirParser* self);
static ValaSignal* vala_gir_parser_parse_signal (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_boxed (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_union (ValaGirParser* self);
static ValaConstant* vala_gir_parser_parse_constant (ValaGirParser* self);
static char* vala_gir_parser_get_attribute (ValaGirParser* self, const char* node, const char* key);
static gpointer vala_gir_parser_parent_class = NULL;
static void vala_gir_parser_finalize (ValaCodeVisitor* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



static glong string_get_length (const char* self) {
	g_return_val_if_fail (self != NULL, 0L);
	return g_utf8_strlen (self, -1);
}


/**
 * Parses all .gir source files in the specified code
 * context and builds a code tree.
 *
 * @param context a code context
 */
void vala_gir_parser_parse (ValaGirParser* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	ValaNamespace* _tmp3;
	ValaSymbol* _tmp2;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL ? NULL : vala_code_context_ref (_tmp0))), (self->priv->context == NULL ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL))), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->glib_ns = (_tmp3 = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (vala_code_context_get_root (context)))), "GLib"), (VALA_IS_NAMESPACE (_tmp2) ? ((ValaNamespace*) (_tmp2)) : NULL)), (self->priv->glib_ns == NULL ? NULL : (self->priv->glib_ns = (vala_code_node_unref (self->priv->glib_ns), NULL))), _tmp3);
	vala_code_context_accept (context, ((ValaCodeVisitor*) (self)));
}


static void vala_gir_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGirParser * self;
	self = ((ValaGirParser*) (base));
	g_return_if_fail (source_file != NULL);
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gir")) {
		vala_gir_parser_parse_file (self, source_file);
	}
}


void vala_gir_parser_parse_file (ValaGirParser* self, ValaSourceFile* source_file) {
	ValaSourceFile* _tmp1;
	ValaSourceFile* _tmp0;
	ValaMarkupReader* _tmp2;
	ValaMarkupReader* _tmp15;
	ValaSourceFile* _tmp16;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_source_file = (_tmp1 = (_tmp0 = source_file, (_tmp0 == NULL ? NULL : vala_source_file_ref (_tmp0))), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL))), _tmp1);
	_tmp2 = NULL;
	self->priv->reader = (_tmp2 = vala_markup_reader_new (vala_source_file_get_filename (source_file)), (self->priv->reader == NULL ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL))), _tmp2);
	/* xml prolog*/
	vala_gir_parser_next (self);
	vala_gir_parser_next (self);
	vala_gir_parser_next (self);
	vala_gir_parser_parse_repository (self);
	{
		GeeList* node_collection;
		int node_it;
		node_collection = vala_source_file_get_nodes (source_file);
		for (node_it = 0; node_it < gee_collection_get_size (GEE_COLLECTION (node_collection)); node_it = node_it + 1) {
			ValaCodeNode* node;
			node = ((ValaCodeNode*) (gee_list_get (GEE_LIST (node_collection), node_it)));
			{
				if (VALA_IS_CLASS (node)) {
					ValaClass* _tmp3;
					ValaClass* cl;
					ValaNamespace* _tmp5;
					ValaSymbol* _tmp4;
					ValaNamespace* ns;
					char* _tmp6;
					ValaSymbol* _tmp7;
					ValaStruct* _tmp8;
					ValaStruct* class_struct;
					_tmp3 = NULL;
					cl = (_tmp3 = VALA_CLASS (node), (_tmp3 == NULL ? NULL : vala_code_node_ref (_tmp3)));
					_tmp5 = NULL;
					_tmp4 = NULL;
					ns = (_tmp5 = (_tmp4 = vala_symbol_get_parent_symbol (((ValaSymbol*) (cl))), (VALA_IS_NAMESPACE (_tmp4) ? ((ValaNamespace*) (_tmp4)) : NULL)), (_tmp5 == NULL ? NULL : vala_code_node_ref (_tmp5)));
					/* remove Class records*/
					_tmp6 = NULL;
					_tmp7 = NULL;
					_tmp8 = NULL;
					class_struct = (_tmp8 = (_tmp7 = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (ns))), (_tmp6 = g_strconcat (vala_symbol_get_name (((ValaSymbol*) (cl))), "Class", NULL))), (VALA_IS_STRUCT (_tmp7) ? ((ValaStruct*) (_tmp7)) : NULL)), (_tmp6 = (g_free (_tmp6), NULL)), _tmp8);
					if (class_struct != NULL) {
						vala_namespace_remove_struct (ns, VALA_STRUCT (class_struct));
						vala_source_file_remove_node (source_file, ((ValaCodeNode*) (class_struct)));
					}
					(cl == NULL ? NULL : (cl = (vala_code_node_unref (cl), NULL)));
					(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
					(class_struct == NULL ? NULL : (class_struct = (vala_code_node_unref (class_struct), NULL)));
				} else {
					if (VALA_IS_INTERFACE (node)) {
						ValaInterface* _tmp9;
						ValaInterface* iface;
						ValaNamespace* _tmp11;
						ValaSymbol* _tmp10;
						ValaNamespace* ns;
						char* _tmp12;
						ValaSymbol* _tmp13;
						ValaStruct* _tmp14;
						ValaStruct* iface_struct;
						_tmp9 = NULL;
						iface = (_tmp9 = VALA_INTERFACE (node), (_tmp9 == NULL ? NULL : vala_code_node_ref (_tmp9)));
						_tmp11 = NULL;
						_tmp10 = NULL;
						ns = (_tmp11 = (_tmp10 = vala_symbol_get_parent_symbol (((ValaSymbol*) (iface))), (VALA_IS_NAMESPACE (_tmp10) ? ((ValaNamespace*) (_tmp10)) : NULL)), (_tmp11 == NULL ? NULL : vala_code_node_ref (_tmp11)));
						/* remove Iface records*/
						_tmp12 = NULL;
						_tmp13 = NULL;
						_tmp14 = NULL;
						iface_struct = (_tmp14 = (_tmp13 = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (ns))), (_tmp12 = g_strconcat (vala_symbol_get_name (((ValaSymbol*) (iface))), "Iface", NULL))), (VALA_IS_STRUCT (_tmp13) ? ((ValaStruct*) (_tmp13)) : NULL)), (_tmp12 = (g_free (_tmp12), NULL)), _tmp14);
						if (iface_struct != NULL) {
							vala_namespace_remove_struct (ns, VALA_STRUCT (iface_struct));
							vala_source_file_remove_node (source_file, ((ValaCodeNode*) (iface_struct)));
						}
						(iface == NULL ? NULL : (iface = (vala_code_node_unref (iface), NULL)));
						(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
						(iface_struct == NULL ? NULL : (iface_struct = (vala_code_node_unref (iface_struct), NULL)));
					}
				}
				(node == NULL ? NULL : (node = (vala_code_node_unref (node), NULL)));
			}
		}
		(node_collection == NULL ? NULL : (node_collection = (gee_collection_object_unref (node_collection), NULL)));
	}
	_tmp15 = NULL;
	self->priv->reader = (_tmp15 = NULL, (self->priv->reader == NULL ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL))), _tmp15);
	_tmp16 = NULL;
	self->priv->current_source_file = (_tmp16 = NULL, (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL))), _tmp16);
}


static void vala_gir_parser_next (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	self->priv->current_token = vala_markup_reader_read_token (self->priv->reader, &self->priv->begin, &self->priv->end);
}


static void vala_gir_parser_start_element (ValaGirParser* self, const char* name) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	if (self->priv->current_token != VALA_MARKUP_TOKEN_TYPE_START_ELEMENT || _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), name) != 0) {
		char* _tmp1;
		ValaSourceReference* _tmp0;
		/* error*/
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error ((_tmp0 = vala_gir_parser_get_current_src (self)), (_tmp1 = g_strdup_printf ("expected start element of `%s'", name)));
		_tmp1 = (g_free (_tmp1), NULL);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
	}
}


static void vala_gir_parser_end_element (ValaGirParser* self, const char* name) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	if (self->priv->current_token != VALA_MARKUP_TOKEN_TYPE_END_ELEMENT || _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), name) != 0) {
		char* _tmp1;
		ValaSourceReference* _tmp0;
		/* error*/
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error ((_tmp0 = vala_gir_parser_get_current_src (self)), (_tmp1 = g_strdup_printf ("expected end element of `%s'", name)));
		_tmp1 = (g_free (_tmp1), NULL);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
	}
	vala_gir_parser_next (self);
}


static ValaSourceReference* vala_gir_parser_get_current_src (ValaGirParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return vala_source_reference_new (self->priv->current_source_file, self->priv->begin.line, self->priv->begin.column, self->priv->end.line, self->priv->end.column);
}


static void vala_gir_parser_parse_repository (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "repository");
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "namespace") == 0) {
			ValaNamespace* ns;
			ns = vala_gir_parser_parse_namespace (self);
			if (ns != NULL) {
				vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
			}
			(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "include") == 0) {
				vala_gir_parser_parse_include (self);
			} else {
				char* _tmp1;
				ValaSourceReference* _tmp0;
				/* error*/
				_tmp1 = NULL;
				_tmp0 = NULL;
				vala_report_error ((_tmp0 = vala_gir_parser_get_current_src (self)), (_tmp1 = g_strdup_printf ("unknown child element `%s' in `repository'", vala_markup_reader_get_name (self->priv->reader))));
				_tmp1 = (g_free (_tmp1), NULL);
				(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
				break;
			}
		}
	}
	vala_gir_parser_end_element (self, "repository");
}


static void vala_gir_parser_parse_include (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "include");
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "include");
}


static ValaNamespace* vala_gir_parser_parse_namespace (ValaGirParser* self) {
	gboolean new_namespace;
	char* _tmp0;
	char* _tmp1;
	char* namespace_name;
	ValaSymbol* _tmp2;
	ValaNamespace* ns;
	char* cheader;
	ValaNamespace* _tmp19;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "namespace");
	new_namespace = FALSE;
	_tmp0 = NULL;
	_tmp1 = NULL;
	namespace_name = (_tmp1 = vala_gir_parser_transform_namespace_name (self, (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	_tmp2 = NULL;
	ns = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (vala_code_context_get_root (self->priv->context)))), namespace_name), (VALA_IS_NAMESPACE (_tmp2) ? ((ValaNamespace*) (_tmp2)) : NULL));
	if (ns == NULL) {
		ValaNamespace* _tmp3;
		_tmp3 = NULL;
		ns = (_tmp3 = vala_namespace_new (namespace_name, NULL), (ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL))), _tmp3);
		new_namespace = TRUE;
	} else {
		ValaSourceReference* _tmp4;
		_tmp4 = NULL;
		vala_code_node_set_source_reference (((ValaCodeNode*) (ns)), (_tmp4 = vala_source_reference_new (self->priv->current_source_file, 0, 0, 0, 0)));
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)));
	}
	cheader = vala_gir_parser_get_attribute (self, vala_symbol_get_name (((ValaSymbol*) (ns))), "c:header-filename");
	if (cheader != NULL) {
		vala_namespace_set_cheader_filename (ns, cheader);
	}
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		ValaSymbol* sym;
		sym = NULL;
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "alias") == 0) {
			ValaSymbol* _tmp5;
			_tmp5 = NULL;
			sym = (_tmp5 = ((ValaSymbol*) (vala_gir_parser_parse_alias (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp5);
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "enumeration") == 0) {
				ValaSymbol* _tmp6;
				_tmp6 = NULL;
				sym = (_tmp6 = ((ValaSymbol*) (vala_gir_parser_parse_enumeration (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp6);
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "bitfield") == 0) {
					ValaSymbol* _tmp7;
					_tmp7 = NULL;
					sym = (_tmp7 = ((ValaSymbol*) (vala_gir_parser_parse_bitfield (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp7);
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
						ValaSymbol* _tmp8;
						_tmp8 = NULL;
						sym = (_tmp8 = ((ValaSymbol*) (vala_gir_parser_parse_function (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp8);
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
							ValaSymbol* _tmp9;
							_tmp9 = NULL;
							sym = (_tmp9 = ((ValaSymbol*) (vala_gir_parser_parse_callback (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp9);
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "record") == 0) {
								ValaSymbol* _tmp10;
								_tmp10 = NULL;
								sym = (_tmp10 = ((ValaSymbol*) (vala_gir_parser_parse_record (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp10);
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "class") == 0) {
									ValaSymbol* _tmp11;
									_tmp11 = NULL;
									sym = (_tmp11 = ((ValaSymbol*) (vala_gir_parser_parse_class (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp11);
								} else {
									if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "interface") == 0) {
										ValaSymbol* _tmp12;
										_tmp12 = NULL;
										sym = (_tmp12 = ((ValaSymbol*) (vala_gir_parser_parse_interface (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp12);
									} else {
										if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:boxed") == 0) {
											ValaStruct* _tmp13;
											_tmp13 = NULL;
											_tmp13 = vala_gir_parser_parse_boxed (self);
											(_tmp13 == NULL ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL)));
										} else {
											if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "union") == 0) {
												ValaStruct* _tmp14;
												_tmp14 = NULL;
												_tmp14 = vala_gir_parser_parse_union (self);
												(_tmp14 == NULL ? NULL : (_tmp14 = (vala_code_node_unref (_tmp14), NULL)));
											} else {
												if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constant") == 0) {
													ValaSymbol* _tmp15;
													_tmp15 = NULL;
													sym = (_tmp15 = ((ValaSymbol*) (vala_gir_parser_parse_constant (self))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp15);
												} else {
													char* _tmp17;
													ValaSourceReference* _tmp16;
													/* error*/
													_tmp17 = NULL;
													_tmp16 = NULL;
													vala_report_error ((_tmp16 = vala_gir_parser_get_current_src (self)), (_tmp17 = g_strdup_printf ("unknown child element `%s' in `namespace'", vala_markup_reader_get_name (self->priv->reader))));
													_tmp17 = (g_free (_tmp17), NULL);
													(_tmp16 == NULL ? NULL : (_tmp16 = (vala_source_reference_unref (_tmp16), NULL)));
													(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
													break;
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		if (VALA_IS_CLASS (sym)) {
			vala_namespace_add_class (ns, VALA_CLASS (sym));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				vala_namespace_add_interface (ns, VALA_INTERFACE (sym));
			} else {
				if (VALA_IS_STRUCT (sym)) {
					vala_namespace_add_struct (ns, VALA_STRUCT (sym));
				} else {
					if (VALA_IS_ENUM (sym)) {
						vala_namespace_add_enum (ns, VALA_ENUM (sym));
					} else {
						if (VALA_IS_DELEGATE (sym)) {
							vala_namespace_add_delegate (ns, VALA_DELEGATE (sym));
						} else {
							if (VALA_IS_METHOD (sym)) {
								vala_namespace_add_method (ns, VALA_METHOD (sym));
							} else {
								if (VALA_IS_CONSTANT (sym)) {
									vala_namespace_add_constant (ns, VALA_CONSTANT (sym));
								} else {
									if (sym == NULL) {
										(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
										continue;
									}
								}
							}
						}
					}
				}
			}
		}
		vala_source_file_add_node (self->priv->current_source_file, ((ValaCodeNode*) (sym)));
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
	}
	vala_gir_parser_end_element (self, "namespace");
	if (!new_namespace) {
		ValaNamespace* _tmp18;
		_tmp18 = NULL;
		ns = (_tmp18 = NULL, (ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL))), _tmp18);
	}
	_tmp19 = NULL;
	return (_tmp19 = ns, (namespace_name = (g_free (namespace_name), NULL)), (cheader = (g_free (cheader), NULL)), _tmp19);
}


static ValaStruct* vala_gir_parser_parse_alias (ValaGirParser* self) {
	char* _tmp0;
	ValaStruct* _tmp1;
	ValaStruct* st;
	ValaDataType* _tmp3;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "alias");
	_tmp0 = NULL;
	_tmp1 = NULL;
	st = (_tmp1 = vala_struct_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (st)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp3 = NULL;
	_tmp2 = NULL;
	vala_struct_add_base_type (st, (_tmp3 = vala_gir_parser_parse_type_from_name (self, (_tmp2 = vala_markup_reader_get_attribute (self->priv->reader, "target")))));
	(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
	_tmp2 = (g_free (_tmp2), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "alias");
	return st;
}


static ValaEnum* vala_gir_parser_parse_enumeration (ValaGirParser* self) {
	char* _tmp0;
	ValaEnum* _tmp1;
	ValaEnum* en;
	char* common_prefix;
	ValaEnum* _tmp7;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "enumeration");
	_tmp0 = NULL;
	_tmp1 = NULL;
	en = (_tmp1 = vala_enum_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (en)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	common_prefix = NULL;
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "member") == 0) {
			ValaEnumValue* ev;
			char* cname;
			ev = vala_gir_parser_parse_enumeration_member (self);
			vala_enum_add_value (en, ev);
			cname = vala_enum_value_get_cname (ev);
			if (common_prefix == NULL) {
				char* _tmp3;
				const char* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				common_prefix = (_tmp3 = (_tmp2 = cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (common_prefix = (g_free (common_prefix), NULL)), _tmp3);
				while (g_utf8_strlen (common_prefix, -1) > 0 && !g_str_has_suffix (common_prefix, "_")) {
					char* _tmp4;
					/* FIXME: could easily be made faster*/
					_tmp4 = NULL;
					common_prefix = (_tmp4 = g_strndup (common_prefix, ((gulong) (strlen (common_prefix) - 1))), (common_prefix = (g_free (common_prefix), NULL)), _tmp4);
				}
			} else {
				while (!g_str_has_prefix (cname, common_prefix)) {
					char* _tmp5;
					_tmp5 = NULL;
					common_prefix = (_tmp5 = g_strndup (common_prefix, ((gulong) (strlen (common_prefix) - 1))), (common_prefix = (g_free (common_prefix), NULL)), _tmp5);
				}
			}
			while (g_utf8_strlen (common_prefix, -1) > 0 && (!g_str_has_suffix (common_prefix, "_") || (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (cname, strlen (common_prefix))))) && (g_utf8_strlen (cname, -1) - g_utf8_strlen (common_prefix, -1)) <= 1)) {
				char* _tmp6;
				/* enum values may not consist solely of digits*/
				_tmp6 = NULL;
				common_prefix = (_tmp6 = g_strndup (common_prefix, ((gulong) (strlen (common_prefix) - 1))), (common_prefix = (g_free (common_prefix), NULL)), _tmp6);
			}
			(ev == NULL ? NULL : (ev = (vala_code_node_unref (ev), NULL)));
			cname = (g_free (cname), NULL);
		} else {
			/* error*/
			break;
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	vala_gir_parser_end_element (self, "enumeration");
	_tmp7 = NULL;
	return (_tmp7 = en, (common_prefix = (g_free (common_prefix), NULL)), _tmp7);
}


static ValaEnum* vala_gir_parser_parse_bitfield (ValaGirParser* self) {
	char* _tmp0;
	ValaEnum* _tmp1;
	ValaEnum* en;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "bitfield");
	_tmp0 = NULL;
	_tmp1 = NULL;
	en = (_tmp1 = vala_enum_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (en)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "member") == 0) {
			ValaEnumValue* _tmp2;
			_tmp2 = NULL;
			vala_enum_add_value (en, (_tmp2 = vala_gir_parser_parse_enumeration_member (self)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		} else {
			/* error*/
			break;
		}
	}
	vala_gir_parser_end_element (self, "bitfield");
	return en;
}


static ValaEnumValue* vala_gir_parser_parse_enumeration_member (ValaGirParser* self) {
	char* _tmp3;
	gint _tmp2_length1;
	char** _tmp2;
	char* _tmp1;
	char* _tmp0;
	ValaEnumValue* _tmp4;
	ValaEnumValue* ev;
	char* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "member");
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp4 = NULL;
	ev = (_tmp4 = vala_enum_value_new ((_tmp3 = g_strjoinv ("_", ((_tmp2 = g_strsplit ((_tmp1 = g_utf8_strup ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), -1)), "-", 0)), _tmp2_length1 = -1, _tmp2))), NULL), (_tmp3 = (g_free (_tmp3), NULL)), (_tmp2 = (_vala_array_free (_tmp2, _tmp2_length1, ((GDestroyNotify) (g_free))), NULL)), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp4);
	_tmp5 = NULL;
	vala_enum_value_set_cname (ev, (_tmp5 = vala_markup_reader_get_attribute (self->priv->reader, "c:identifier")));
	_tmp5 = (g_free (_tmp5), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "member");
	return ev;
}


static ValaMethod* vala_gir_parser_parse_function (ValaGirParser* self) {
	char* name;
	ValaDataType* return_type;
	ValaMethod* m;
	GeeArrayList* parameters;
	GeeArrayList* array_length_parameters;
	gint i;
	ValaMethod* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "function");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	return_type = NULL;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return_type = (_tmp0 = vala_gir_parser_parse_return_value (self), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp0);
	} else {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = ((ValaDataType*) (vala_void_type_new ())), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp1);
	}
	m = vala_method_new (name, return_type, NULL);
	vala_symbol_set_access (((ValaSymbol*) (m)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_method_set_binding (m, MEMBER_BINDING_STATIC);
	parameters = gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	array_length_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			gint array_length_idx;
			ValaFormalParameter* param;
			array_length_idx = -1;
			param = vala_gir_parser_parse_parameter (self, &array_length_idx);
			if (array_length_idx != -1) {
				gee_collection_add (((GeeCollection*) (array_length_parameters)), GINT_TO_POINTER (array_length_idx));
			}
			gee_collection_add (((GeeCollection*) (parameters)), param);
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	i = 0;
	{
		GeeArrayList* param_collection;
		int param_it;
		param_collection = parameters;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				if (!gee_collection_contains (((GeeCollection*) (array_length_parameters)), GINT_TO_POINTER (i))) {
					vala_method_add_parameter (m, param);
				}
				i++;
				(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
			}
		}
	}
	vala_gir_parser_end_element (self, "function");
	_tmp2 = NULL;
	return (_tmp2 = m, (name = (g_free (name), NULL)), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), (parameters == NULL ? NULL : (parameters = (gee_collection_object_unref (parameters), NULL))), (array_length_parameters == NULL ? NULL : (array_length_parameters = (gee_collection_object_unref (array_length_parameters), NULL))), _tmp2);
}


static ValaDataType* vala_gir_parser_parse_return_value (ValaGirParser* self) {
	char* transfer;
	ValaDataType* type;
	ValaDataType* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "return-value");
	transfer = vala_markup_reader_get_attribute (self->priv->reader, "transfer-ownership");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	if (_vala_strcmp0 (transfer, "full") == 0) {
		vala_data_type_set_value_owned (type, TRUE);
	}
	vala_gir_parser_end_element (self, "return-value");
	_tmp0 = NULL;
	return (_tmp0 = type, (transfer = (g_free (transfer), NULL)), _tmp0);
}


static ValaFormalParameter* vala_gir_parser_parse_parameter (ValaGirParser* self, gint* array_length_idx) {
	ValaFormalParameter* param;
	char* name;
	char* direction;
	char* transfer;
	ValaFormalParameter* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	param = NULL;
	vala_gir_parser_start_element (self, "parameter");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	direction = vala_markup_reader_get_attribute (self->priv->reader, "direction");
	transfer = vala_markup_reader_get_attribute (self->priv->reader, "transfer-ownership");
	vala_gir_parser_next (self);
	if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "varargs") == 0) {
		ValaFormalParameter* _tmp0;
		vala_gir_parser_start_element (self, "varargs");
		vala_gir_parser_next (self);
		_tmp0 = NULL;
		param = (_tmp0 = vala_formal_parameter_new_with_ellipsis (NULL), (param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL))), _tmp0);
		vala_gir_parser_end_element (self, "varargs");
	} else {
		ValaDataType* type;
		ValaFormalParameter* _tmp1;
		type = vala_gir_parser_parse_type (self, &(*array_length_idx));
		if (_vala_strcmp0 (transfer, "full") == 0) {
			vala_data_type_set_value_owned (type, TRUE);
		}
		_tmp1 = NULL;
		param = (_tmp1 = vala_formal_parameter_new (name, type, NULL), (param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL))), _tmp1);
		if (_vala_strcmp0 (direction, "out") == 0) {
			vala_formal_parameter_set_direction (param, VALA_PARAMETER_DIRECTION_OUT);
		} else {
			if (_vala_strcmp0 (direction, "inout") == 0) {
				vala_formal_parameter_set_direction (param, VALA_PARAMETER_DIRECTION_REF);
			}
		}
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
	}
	vala_gir_parser_end_element (self, "parameter");
	_tmp2 = NULL;
	return (_tmp2 = param, (name = (g_free (name), NULL)), (direction = (g_free (direction), NULL)), (transfer = (g_free (transfer), NULL)), _tmp2);
}


static ValaDataType* vala_gir_parser_parse_type (ValaGirParser* self, gint* array_length_index) {
	g_return_val_if_fail (self != NULL, NULL);
	if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "array") == 0) {
		char* _tmp0;
		gboolean _tmp1;
		ValaDataType* element_type;
		ValaDataType* _tmp3;
		vala_gir_parser_start_element (self, "array");
		_tmp0 = NULL;
		if ((_tmp1 = (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "length")) != NULL && &(*array_length_index) != NULL, (_tmp0 = (g_free (_tmp0), NULL)), _tmp1)) {
			char* _tmp2;
			_tmp2 = NULL;
			(*array_length_index) = atoi ((_tmp2 = vala_markup_reader_get_attribute (self->priv->reader, "length")));
			_tmp2 = (g_free (_tmp2), NULL);
		}
		vala_gir_parser_next (self);
		element_type = vala_gir_parser_parse_type (self, NULL);
		vala_gir_parser_end_element (self, "array");
		_tmp3 = NULL;
		return (_tmp3 = ((ValaDataType*) (vala_array_type_new (element_type, 1, NULL))), (element_type == NULL ? NULL : (element_type = (vala_code_node_unref (element_type), NULL))), _tmp3);
	} else {
		char* _tmp4;
		ValaDataType* _tmp5;
		ValaDataType* type;
		vala_gir_parser_start_element (self, "type");
		_tmp4 = NULL;
		_tmp5 = NULL;
		type = (_tmp5 = vala_gir_parser_parse_type_from_name (self, (_tmp4 = vala_markup_reader_get_attribute (self->priv->reader, "name"))), (_tmp4 = (g_free (_tmp4), NULL)), _tmp5);
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaDataType* _tmp6;
			_tmp6 = NULL;
			_tmp6 = vala_gir_parser_parse_type (self, NULL);
			(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
		}
		vala_gir_parser_end_element (self, "type");
		return type;
	}
}


static ValaDataType* vala_gir_parser_parse_type_from_name (ValaGirParser* self, const char* type_name) {
	ValaDataType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	type = NULL;
	if (_vala_strcmp0 (type_name, "none") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		type = (_tmp0 = ((ValaDataType*) (vala_void_type_new ())), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp0);
	} else {
		if (_vala_strcmp0 (type_name, "any") == 0) {
			ValaDataType* _tmp2;
			ValaVoidType* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			type = (_tmp2 = ((ValaDataType*) (vala_pointer_type_new (((ValaDataType*) ((_tmp1 = vala_void_type_new ()))), NULL))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp2);
			(_tmp1 == NULL ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)));
		} else {
			if (_vala_strcmp0 (type_name, "GObject.Strv") == 0) {
				ValaDataType* _tmp5;
				ValaUnresolvedType* _tmp4;
				ValaUnresolvedSymbol* _tmp3;
				_tmp5 = NULL;
				_tmp4 = NULL;
				_tmp3 = NULL;
				type = (_tmp5 = ((ValaDataType*) (vala_array_type_new (((ValaDataType*) ((_tmp4 = vala_unresolved_type_new_from_symbol ((_tmp3 = vala_unresolved_symbol_new (NULL, "string", NULL)), NULL)))), 1, NULL))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp5);
				(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
				(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
			} else {
				char** _tmp6;
				gint type_components_length1;
				char** type_components;
				if (_vala_strcmp0 (type_name, "utf8") == 0) {
					type_name = "string";
				} else {
					if (_vala_strcmp0 (type_name, "boolean") == 0) {
						type_name = "bool";
					} else {
						if (_vala_strcmp0 (type_name, "GLib.offset") == 0) {
							type_name = "int64";
						} else {
							if (_vala_strcmp0 (type_name, "GType") == 0) {
								type_name = "GLib.Type";
							} else {
								if (_vala_strcmp0 (type_name, "GObject.String") == 0) {
									type_name = "GLib.StringBuilder";
								} else {
									if (_vala_strcmp0 (type_name, "GObject.Class") == 0) {
										type_name = "GLib.ObjectClass";
									} else {
										if (_vala_strcmp0 (type_name, "GLib.unichar") == 0) {
											type_name = "unichar";
										} else {
											if (_vala_strcmp0 (type_name, "GLib.Data") == 0) {
												type_name = "GLib.Datalist";
											}
										}
									}
								}
							}
						}
					}
				}
				_tmp6 = NULL;
				type_components = (_tmp6 = g_strsplit (type_name, ".", 0), type_components_length1 = -1, _tmp6);
				if (type_components[1] != NULL) {
					char* namespace_name;
					const char* _tmp7;
					char* transformed_type_name;
					ValaDataType* _tmp10;
					ValaUnresolvedSymbol* _tmp9;
					ValaUnresolvedSymbol* _tmp8;
					/* namespaced name*/
					namespace_name = vala_gir_parser_transform_namespace_name (self, type_components[0]);
					_tmp7 = NULL;
					transformed_type_name = (_tmp7 = type_components[1], (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
					_tmp10 = NULL;
					_tmp9 = NULL;
					_tmp8 = NULL;
					type = (_tmp10 = ((ValaDataType*) (vala_unresolved_type_new_from_symbol ((_tmp9 = vala_unresolved_symbol_new ((_tmp8 = vala_unresolved_symbol_new (NULL, namespace_name, NULL)), transformed_type_name, NULL)), NULL))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp10);
					(_tmp9 == NULL ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL)));
					(_tmp8 == NULL ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)));
					namespace_name = (g_free (namespace_name), NULL);
					transformed_type_name = (g_free (transformed_type_name), NULL);
				} else {
					ValaDataType* _tmp12;
					ValaUnresolvedSymbol* _tmp11;
					_tmp12 = NULL;
					_tmp11 = NULL;
					type = (_tmp12 = ((ValaDataType*) (vala_unresolved_type_new_from_symbol ((_tmp11 = vala_unresolved_symbol_new (NULL, type_name, NULL)), NULL))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp12);
					(_tmp11 == NULL ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL)));
				}
				type_components = (_vala_array_free (type_components, type_components_length1, ((GDestroyNotify) (g_free))), NULL);
			}
		}
	}
	return type;
}


static char* vala_gir_parser_transform_namespace_name (ValaGirParser* self, const char* gir_module_name) {
	const char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (gir_module_name != NULL, NULL);
	if (_vala_strcmp0 (gir_module_name, "GObject") == 0) {
		return g_strdup ("GLib");
	} else {
		if (_vala_strcmp0 (gir_module_name, "Gio") == 0) {
			return g_strdup ("GLib");
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = gir_module_name, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static ValaStruct* vala_gir_parser_parse_record (ValaGirParser* self) {
	char* _tmp0;
	ValaStruct* _tmp1;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "record");
	_tmp0 = NULL;
	_tmp1 = NULL;
	st = (_tmp1 = vala_struct_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (st)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2;
			_tmp2 = NULL;
			vala_struct_add_field (st, (_tmp2 = vala_gir_parser_parse_field (self)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
				ValaDelegate* _tmp3;
				_tmp3 = NULL;
				_tmp3 = vala_gir_parser_parse_callback (self);
				(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
					ValaMethod* _tmp4;
					_tmp4 = NULL;
					_tmp4 = vala_gir_parser_parse_constructor (self);
					(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
						ValaMethod* _tmp5;
						_tmp5 = NULL;
						_tmp5 = vala_gir_parser_parse_method (self);
						(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
					} else {
						char* _tmp7;
						ValaSourceReference* _tmp6;
						/* error*/
						_tmp7 = NULL;
						_tmp6 = NULL;
						vala_report_error ((_tmp6 = vala_gir_parser_get_current_src (self)), (_tmp7 = g_strdup_printf ("unknown child element `%s' in `record'", vala_markup_reader_get_name (self->priv->reader))));
						_tmp7 = (g_free (_tmp7), NULL);
						(_tmp6 == NULL ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL)));
						break;
					}
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "record");
	return st;
}


static ValaClass* vala_gir_parser_parse_class (ValaGirParser* self) {
	char* _tmp0;
	ValaClass* _tmp1;
	ValaClass* cl;
	char* parent;
	GeeArrayList* signals;
	GeeArrayList* methods;
	GeeArrayList* vmethods;
	GeeArrayList* fields;
	ValaClass* _tmp21;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "class");
	_tmp0 = NULL;
	_tmp1 = NULL;
	cl = (_tmp1 = vala_class_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (cl)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	parent = vala_markup_reader_get_attribute (self->priv->reader, "parent");
	if (parent != NULL) {
		ValaDataType* _tmp2;
		_tmp2 = NULL;
		vala_class_add_base_type (cl, (_tmp2 = vala_gir_parser_parse_type_from_name (self, parent)));
		(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
	}
	vala_gir_parser_next (self);
	signals = gee_array_list_new (VALA_TYPE_SIGNAL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	methods = gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	vmethods = gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	fields = gee_array_list_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "implements") == 0) {
			ValaDataType* _tmp4;
			char* _tmp3;
			vala_gir_parser_start_element (self, "implements");
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_class_add_base_type (cl, (_tmp4 = vala_gir_parser_parse_type_from_name (self, (_tmp3 = vala_markup_reader_get_attribute (self->priv->reader, "name")))));
			(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
			_tmp3 = (g_free (_tmp3), NULL);
			vala_gir_parser_next (self);
			vala_gir_parser_end_element (self, "implements");
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
				ValaField* _tmp5;
				_tmp5 = NULL;
				gee_collection_add (((GeeCollection*) (fields)), (_tmp5 = vala_gir_parser_parse_field (self)));
				(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "property") == 0) {
					ValaProperty* _tmp6;
					_tmp6 = NULL;
					vala_class_add_property (cl, (_tmp6 = vala_gir_parser_parse_property (self)));
					(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
						ValaMethod* _tmp7;
						_tmp7 = NULL;
						vala_class_add_method (cl, (_tmp7 = vala_gir_parser_parse_constructor (self)));
						(_tmp7 == NULL ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL)));
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
							ValaMethod* _tmp8;
							_tmp8 = NULL;
							gee_collection_add (((GeeCollection*) (methods)), (_tmp8 = vala_gir_parser_parse_method (self)));
							(_tmp8 == NULL ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)));
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
								ValaMethod* _tmp9;
								_tmp9 = NULL;
								gee_collection_add (((GeeCollection*) (vmethods)), (_tmp9 = vala_gir_parser_parse_vmethod (self)));
								(_tmp9 == NULL ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL)));
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:signal") == 0) {
									ValaSignal* _tmp10;
									_tmp10 = NULL;
									gee_collection_add (((GeeCollection*) (signals)), (_tmp10 = vala_gir_parser_parse_signal (self)));
									(_tmp10 == NULL ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL)));
								} else {
									char* _tmp12;
									ValaSourceReference* _tmp11;
									/* error*/
									_tmp12 = NULL;
									_tmp11 = NULL;
									vala_report_error ((_tmp11 = vala_gir_parser_get_current_src (self)), (_tmp12 = g_strdup_printf ("unknown child element `%s' in `class'", vala_markup_reader_get_name (self->priv->reader))));
									_tmp12 = (g_free (_tmp12), NULL);
									(_tmp11 == NULL ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL)));
									break;
								}
							}
						}
					}
				}
			}
		}
	}
	/* signal merging*/
	{
		GeeArrayList* sig_collection;
		int sig_it;
		sig_collection = signals;
		for (sig_it = 0; sig_it < gee_collection_get_size (GEE_COLLECTION (sig_collection)); sig_it = sig_it + 1) {
			ValaSignal* sig;
			sig = ((ValaSignal*) (gee_list_get (GEE_LIST (sig_collection), sig_it)));
			{
				ValaSymbol* symbol;
				symbol = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (cl))), vala_symbol_get_name (((ValaSymbol*) (sig))));
				if (symbol == NULL) {
					vala_class_add_signal (cl, sig);
				} else {
					if (VALA_IS_PROPERTY (symbol)) {
					} else {
						char* _tmp14;
						ValaSourceReference* _tmp13;
						/* properties take precedence*/
						_tmp14 = NULL;
						_tmp13 = NULL;
						vala_report_error ((_tmp13 = vala_gir_parser_get_current_src (self)), (_tmp14 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name (((ValaSymbol*) (sig))), vala_symbol_get_name (((ValaSymbol*) (cl))))));
						_tmp14 = (g_free (_tmp14), NULL);
						(_tmp13 == NULL ? NULL : (_tmp13 = (vala_source_reference_unref (_tmp13), NULL)));
					}
				}
				(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
				(symbol == NULL ? NULL : (symbol = (vala_code_node_unref (symbol), NULL)));
			}
		}
	}
	/* virtual method merging*/
	{
		GeeArrayList* m_collection;
		int m_it;
		m_collection = vmethods;
		for (m_it = 0; m_it < gee_collection_get_size (GEE_COLLECTION (m_collection)); m_it = m_it + 1) {
			ValaMethod* m;
			m = ((ValaMethod*) (gee_list_get (GEE_LIST (m_collection), m_it)));
			{
				ValaSymbol* symbol;
				symbol = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (cl))), vala_symbol_get_name (((ValaSymbol*) (m))));
				if (symbol == NULL) {
					vala_class_add_method (cl, m);
				} else {
					if (VALA_IS_SIGNAL (symbol)) {
						ValaSignal* _tmp15;
						ValaSignal* sig;
						_tmp15 = NULL;
						sig = (_tmp15 = VALA_SIGNAL (symbol), (_tmp15 == NULL ? NULL : vala_code_node_ref (_tmp15)));
						vala_signal_set_is_virtual (sig, TRUE);
						(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
					} else {
						if (VALA_IS_PROPERTY (symbol) || VALA_IS_FIELD (symbol)) {
						} else {
							char* _tmp17;
							ValaSourceReference* _tmp16;
							/* assume method is getter for property/field ignore method*/
							_tmp17 = NULL;
							_tmp16 = NULL;
							vala_report_error ((_tmp16 = vala_gir_parser_get_current_src (self)), (_tmp17 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (cl))))));
							_tmp17 = (g_free (_tmp17), NULL);
							(_tmp16 == NULL ? NULL : (_tmp16 = (vala_source_reference_unref (_tmp16), NULL)));
						}
					}
				}
				(m == NULL ? NULL : (m = (vala_code_node_unref (m), NULL)));
				(symbol == NULL ? NULL : (symbol = (vala_code_node_unref (symbol), NULL)));
			}
		}
	}
	/* method merging*/
	{
		GeeArrayList* m_collection;
		int m_it;
		m_collection = methods;
		for (m_it = 0; m_it < gee_collection_get_size (GEE_COLLECTION (m_collection)); m_it = m_it + 1) {
			ValaMethod* m;
			m = ((ValaMethod*) (gee_list_get (GEE_LIST (m_collection), m_it)));
			{
				ValaSymbol* symbol;
				symbol = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (cl))), vala_symbol_get_name (((ValaSymbol*) (m))));
				if (symbol == NULL) {
					vala_class_add_method (cl, m);
				} else {
					if (VALA_IS_SIGNAL (symbol)) {
						ValaSignal* _tmp18;
						ValaSignal* sig;
						_tmp18 = NULL;
						sig = (_tmp18 = VALA_SIGNAL (symbol), (_tmp18 == NULL ? NULL : vala_code_node_ref (_tmp18)));
						vala_signal_set_has_emitter (sig, TRUE);
						(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
					} else {
						if (VALA_IS_PROPERTY (symbol) || VALA_IS_FIELD (symbol)) {
						} else {
							if (VALA_IS_METHOD (symbol)) {
							} else {
								char* _tmp20;
								ValaSourceReference* _tmp19;
								/* assume method is wrapper for virtual method*/
								_tmp20 = NULL;
								_tmp19 = NULL;
								vala_report_error ((_tmp19 = vala_gir_parser_get_current_src (self)), (_tmp20 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (cl))))));
								_tmp20 = (g_free (_tmp20), NULL);
								(_tmp19 == NULL ? NULL : (_tmp19 = (vala_source_reference_unref (_tmp19), NULL)));
							}
						}
					}
				}
				(m == NULL ? NULL : (m = (vala_code_node_unref (m), NULL)));
				(symbol == NULL ? NULL : (symbol = (vala_code_node_unref (symbol), NULL)));
			}
		}
	}
	/* fields have lowest priority*/
	{
		GeeArrayList* f_collection;
		int f_it;
		f_collection = fields;
		for (f_it = 0; f_it < gee_collection_get_size (GEE_COLLECTION (f_collection)); f_it = f_it + 1) {
			ValaField* f;
			f = ((ValaField*) (gee_list_get (GEE_LIST (f_collection), f_it)));
			{
				ValaSymbol* symbol;
				symbol = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (cl))), vala_symbol_get_name (((ValaSymbol*) (f))));
				if (symbol == NULL) {
					vala_class_add_field (cl, f);
				}
				(f == NULL ? NULL : (f = (vala_code_node_unref (f), NULL)));
				(symbol == NULL ? NULL : (symbol = (vala_code_node_unref (symbol), NULL)));
			}
		}
	}
	vala_gir_parser_end_element (self, "class");
	_tmp21 = NULL;
	return (_tmp21 = cl, (parent = (g_free (parent), NULL)), (signals == NULL ? NULL : (signals = (gee_collection_object_unref (signals), NULL))), (methods == NULL ? NULL : (methods = (gee_collection_object_unref (methods), NULL))), (vmethods == NULL ? NULL : (vmethods = (gee_collection_object_unref (vmethods), NULL))), (fields == NULL ? NULL : (fields = (gee_collection_object_unref (fields), NULL))), _tmp21);
}


static ValaInterface* vala_gir_parser_parse_interface (ValaGirParser* self) {
	char* _tmp0;
	ValaInterface* _tmp1;
	ValaInterface* iface;
	GeeArrayList* methods;
	ValaInterface* _tmp12;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "interface");
	_tmp0 = NULL;
	_tmp1 = NULL;
	iface = (_tmp1 = vala_interface_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (iface)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	methods = gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2;
			_tmp2 = NULL;
			_tmp2 = vala_gir_parser_parse_field (self);
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "property") == 0) {
				ValaProperty* _tmp3;
				_tmp3 = NULL;
				vala_interface_add_property (iface, (_tmp3 = vala_gir_parser_parse_property (self)));
				(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
					ValaDelegate* _tmp4;
					_tmp4 = NULL;
					_tmp4 = vala_gir_parser_parse_callback (self);
					(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
						ValaMethod* _tmp5;
						_tmp5 = NULL;
						gee_collection_add (((GeeCollection*) (methods)), (_tmp5 = vala_gir_parser_parse_method (self)));
						(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:signal") == 0) {
							ValaSignal* _tmp6;
							_tmp6 = NULL;
							vala_interface_add_signal (iface, (_tmp6 = vala_gir_parser_parse_signal (self)));
							(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
						} else {
							char* _tmp8;
							ValaSourceReference* _tmp7;
							/* error*/
							_tmp8 = NULL;
							_tmp7 = NULL;
							vala_report_error ((_tmp7 = vala_gir_parser_get_current_src (self)), (_tmp8 = g_strdup_printf ("unknown child element `%s' in `interface'", vala_markup_reader_get_name (self->priv->reader))));
							_tmp8 = (g_free (_tmp8), NULL);
							(_tmp7 == NULL ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL)));
							break;
						}
					}
				}
			}
		}
	}
	/* method merging*/
	{
		GeeArrayList* m_collection;
		int m_it;
		m_collection = methods;
		for (m_it = 0; m_it < gee_collection_get_size (GEE_COLLECTION (m_collection)); m_it = m_it + 1) {
			ValaMethod* m;
			m = ((ValaMethod*) (gee_list_get (GEE_LIST (m_collection), m_it)));
			{
				ValaSymbol* symbol;
				symbol = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (iface))), vala_symbol_get_name (((ValaSymbol*) (m))));
				if (symbol == NULL) {
					vala_interface_add_method (iface, m);
				} else {
					if (VALA_IS_SIGNAL (symbol)) {
						ValaSignal* _tmp9;
						ValaSignal* sig;
						_tmp9 = NULL;
						sig = (_tmp9 = VALA_SIGNAL (symbol), (_tmp9 == NULL ? NULL : vala_code_node_ref (_tmp9)));
						vala_signal_set_has_emitter (sig, TRUE);
						(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
					} else {
						char* _tmp11;
						ValaSourceReference* _tmp10;
						_tmp11 = NULL;
						_tmp10 = NULL;
						vala_report_error ((_tmp10 = vala_gir_parser_get_current_src (self)), (_tmp11 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name (((ValaSymbol*) (m))), vala_symbol_get_name (((ValaSymbol*) (iface))))));
						_tmp11 = (g_free (_tmp11), NULL);
						(_tmp10 == NULL ? NULL : (_tmp10 = (vala_source_reference_unref (_tmp10), NULL)));
					}
				}
				(m == NULL ? NULL : (m = (vala_code_node_unref (m), NULL)));
				(symbol == NULL ? NULL : (symbol = (vala_code_node_unref (symbol), NULL)));
			}
		}
	}
	vala_gir_parser_end_element (self, "interface");
	_tmp12 = NULL;
	return (_tmp12 = iface, (methods == NULL ? NULL : (methods = (gee_collection_object_unref (methods), NULL))), _tmp12);
}


static ValaField* vala_gir_parser_parse_field (ValaGirParser* self) {
	char* name;
	ValaDataType* type;
	ValaField* field;
	ValaField* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "field");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	field = vala_field_new (name, type, NULL, NULL);
	vala_symbol_set_access (((ValaSymbol*) (field)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_end_element (self, "field");
	_tmp0 = NULL;
	return (_tmp0 = field, (name = (g_free (name), NULL)), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp0);
}


static ValaProperty* vala_gir_parser_parse_property (ValaGirParser* self) {
	gint _tmp1_length1;
	char** _tmp1;
	char* _tmp0;
	char* _tmp2;
	char* name;
	ValaDataType* type;
	ValaProperty* prop;
	ValaPropertyAccessor* _tmp3;
	ValaPropertyAccessor* _tmp4;
	ValaProperty* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "property");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	name = (_tmp2 = g_strjoinv ("_", ((_tmp1 = g_strsplit ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), "-", 0)), _tmp1_length1 = -1, _tmp1)), (_tmp1 = (_vala_array_free (_tmp1, _tmp1_length1, ((GDestroyNotify) (g_free))), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	prop = vala_property_new (name, type, NULL, NULL, NULL);
	vala_symbol_set_access (((ValaSymbol*) (prop)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp3 = NULL;
	vala_property_set_get_accessor (prop, (_tmp3 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL)));
	(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
	_tmp4 = NULL;
	vala_property_set_set_accessor (prop, (_tmp4 = vala_property_accessor_new (FALSE, TRUE, FALSE, NULL, NULL)));
	(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
	vala_gir_parser_end_element (self, "property");
	_tmp5 = NULL;
	return (_tmp5 = prop, (name = (g_free (name), NULL)), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp5);
}


static ValaDelegate* vala_gir_parser_parse_callback (ValaGirParser* self) {
	char* name;
	ValaDataType* return_type;
	ValaDelegate* d;
	ValaDelegate* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "callback");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	return_type = NULL;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return_type = (_tmp0 = vala_gir_parser_parse_return_value (self), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp0);
	} else {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = ((ValaDataType*) (vala_void_type_new ())), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp1);
	}
	d = vala_delegate_new (name, return_type, NULL);
	vala_symbol_set_access (((ValaSymbol*) (d)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp2;
			_tmp2 = NULL;
			vala_delegate_add_parameter (d, (_tmp2 = vala_gir_parser_parse_parameter (self, NULL)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "callback");
	_tmp3 = NULL;
	return (_tmp3 = d, (name = (g_free (name), NULL)), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp3);
}


static ValaMethod* vala_gir_parser_parse_constructor (ValaGirParser* self) {
	char* name;
	ValaDataType* return_type;
	ValaCreationMethod* m;
	ValaMethod* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "constructor");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	return_type = vala_gir_parser_parse_return_value (self);
	m = vala_creation_method_new (NULL, name, NULL);
	vala_symbol_set_access (((ValaSymbol*) (m)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_method_set_has_construct_function (((ValaMethod*) (m)), FALSE);
	if (g_str_has_prefix (vala_symbol_get_name (((ValaSymbol*) (m))), "new_")) {
		vala_symbol_set_name (((ValaSymbol*) (m)), g_utf8_offset_to_pointer (vala_symbol_get_name (((ValaSymbol*) (m))), g_utf8_strlen ("new_", -1)));
	}
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp0;
			_tmp0 = NULL;
			vala_method_add_parameter (((ValaMethod*) (m)), (_tmp0 = vala_gir_parser_parse_parameter (self, NULL)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "constructor");
	_tmp1 = NULL;
	return (_tmp1 = ((ValaMethod*) (m)), (name = (g_free (name), NULL)), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp1);
}


static ValaMethod* vala_gir_parser_parse_method (ValaGirParser* self) {
	char* name;
	char* throws_string;
	ValaDataType* return_type;
	ValaMethod* m;
	ValaMethod* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "method");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	throws_string = vala_markup_reader_get_attribute (self->priv->reader, "throws");
	vala_gir_parser_next (self);
	return_type = NULL;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return_type = (_tmp0 = vala_gir_parser_parse_return_value (self), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp0);
	} else {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = ((ValaDataType*) (vala_void_type_new ())), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp1);
	}
	m = vala_method_new (name, return_type, NULL);
	vala_symbol_set_access (((ValaSymbol*) (m)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp2;
			_tmp2 = NULL;
			vala_method_add_parameter (m, (_tmp2 = vala_gir_parser_parse_parameter (self, NULL)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	if (_vala_strcmp0 (throws_string, "1") == 0) {
		ValaErrorType* _tmp3;
		_tmp3 = NULL;
		vala_code_node_add_error_type (((ValaCodeNode*) (m)), ((ValaDataType*) ((_tmp3 = vala_error_type_new (NULL, NULL, NULL)))));
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
	}
	vala_gir_parser_end_element (self, "method");
	_tmp4 = NULL;
	return (_tmp4 = m, (name = (g_free (name), NULL)), (throws_string = (g_free (throws_string), NULL)), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp4);
}


static ValaMethod* vala_gir_parser_parse_vmethod (ValaGirParser* self) {
	char* name;
	char* throws_string;
	ValaDataType* return_type;
	ValaMethod* m;
	ValaMethod* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "callback");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	throws_string = vala_markup_reader_get_attribute (self->priv->reader, "throws");
	vala_gir_parser_next (self);
	return_type = NULL;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return_type = (_tmp0 = vala_gir_parser_parse_return_value (self), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp0);
	} else {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = ((ValaDataType*) (vala_void_type_new ())), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp1);
	}
	m = vala_method_new (name, return_type, NULL);
	vala_symbol_set_access (((ValaSymbol*) (m)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_method_set_is_virtual (m, TRUE);
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0) {
		gboolean first;
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		first = TRUE;
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* param;
			param = vala_gir_parser_parse_parameter (self, NULL);
			/* first parameter is instance pointer, ignore*/
			if (!first) {
				vala_method_add_parameter (m, param);
			} else {
				first = FALSE;
			}
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	if (_vala_strcmp0 (throws_string, "1") == 0) {
		ValaErrorType* _tmp2;
		_tmp2 = NULL;
		vala_code_node_add_error_type (((ValaCodeNode*) (m)), ((ValaDataType*) ((_tmp2 = vala_error_type_new (NULL, NULL, NULL)))));
		(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
	}
	vala_gir_parser_end_element (self, "callback");
	_tmp3 = NULL;
	return (_tmp3 = m, (name = (g_free (name), NULL)), (throws_string = (g_free (throws_string), NULL)), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp3);
}


static ValaSignal* vala_gir_parser_parse_signal (ValaGirParser* self) {
	gint _tmp1_length1;
	char** _tmp1;
	char* _tmp0;
	char* _tmp2;
	char* name;
	ValaDataType* return_type;
	ValaSignal* sig;
	ValaSignal* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "glib:signal");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	name = (_tmp2 = g_strjoinv ("_", ((_tmp1 = g_strsplit ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), "-", 0)), _tmp1_length1 = -1, _tmp1)), (_tmp1 = (_vala_array_free (_tmp1, _tmp1_length1, ((GDestroyNotify) (g_free))), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
	vala_gir_parser_next (self);
	return_type = NULL;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0) {
		ValaDataType* _tmp3;
		_tmp3 = NULL;
		return_type = (_tmp3 = vala_gir_parser_parse_return_value (self), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp3);
	} else {
		ValaDataType* _tmp4;
		_tmp4 = NULL;
		return_type = (_tmp4 = ((ValaDataType*) (vala_void_type_new ())), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp4);
	}
	sig = vala_signal_new (name, return_type, NULL);
	vala_symbol_set_access (((ValaSymbol*) (sig)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT && _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp5;
			_tmp5 = NULL;
			vala_signal_add_parameter (sig, (_tmp5 = vala_gir_parser_parse_parameter (self, NULL)));
			(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "glib:signal");
	_tmp6 = NULL;
	return (_tmp6 = sig, (name = (g_free (name), NULL)), (return_type == NULL ? NULL : (return_type = (vala_code_node_unref (return_type), NULL))), _tmp6);
}


static ValaStruct* vala_gir_parser_parse_boxed (ValaGirParser* self) {
	char* _tmp0;
	ValaStruct* _tmp1;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "glib:boxed");
	_tmp0 = NULL;
	_tmp1 = NULL;
	st = (_tmp1 = vala_struct_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "glib:name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (st)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2;
			_tmp2 = NULL;
			vala_struct_add_field (st, (_tmp2 = vala_gir_parser_parse_field (self)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
				ValaMethod* _tmp3;
				_tmp3 = NULL;
				_tmp3 = vala_gir_parser_parse_constructor (self);
				(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
					ValaMethod* _tmp4;
					_tmp4 = NULL;
					vala_struct_add_method (st, (_tmp4 = vala_gir_parser_parse_method (self)));
					(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
				} else {
					char* _tmp6;
					ValaSourceReference* _tmp5;
					/* error*/
					_tmp6 = NULL;
					_tmp5 = NULL;
					vala_report_error ((_tmp5 = vala_gir_parser_get_current_src (self)), (_tmp6 = g_strdup_printf ("unknown child element `%s' in `class'", vala_markup_reader_get_name (self->priv->reader))));
					_tmp6 = (g_free (_tmp6), NULL);
					(_tmp5 == NULL ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)));
					break;
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "glib:boxed");
	return st;
}


static ValaStruct* vala_gir_parser_parse_union (ValaGirParser* self) {
	char* _tmp0;
	ValaStruct* _tmp1;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "union");
	_tmp0 = NULL;
	_tmp1 = NULL;
	st = (_tmp1 = vala_struct_new ((_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access (((ValaSymbol*) (st)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2;
			_tmp2 = NULL;
			vala_struct_add_field (st, (_tmp2 = vala_gir_parser_parse_field (self)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
				ValaMethod* _tmp3;
				_tmp3 = NULL;
				_tmp3 = vala_gir_parser_parse_constructor (self);
				(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
					ValaMethod* _tmp4;
					_tmp4 = NULL;
					vala_struct_add_method (st, (_tmp4 = vala_gir_parser_parse_method (self)));
					(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
				} else {
					char* _tmp6;
					ValaSourceReference* _tmp5;
					/* error*/
					_tmp6 = NULL;
					_tmp5 = NULL;
					vala_report_error ((_tmp5 = vala_gir_parser_get_current_src (self)), (_tmp6 = g_strdup_printf ("unknown child element `%s' in `union'", vala_markup_reader_get_name (self->priv->reader))));
					_tmp6 = (g_free (_tmp6), NULL);
					(_tmp5 == NULL ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)));
					break;
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "union");
	return st;
}


static ValaConstant* vala_gir_parser_parse_constant (ValaGirParser* self) {
	char* name;
	ValaDataType* type;
	ValaConstant* c;
	ValaConstant* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "constant");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	c = vala_constant_new (name, type, NULL, NULL);
	vala_symbol_set_access (((ValaSymbol*) (c)), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_end_element (self, "constant");
	_tmp0 = NULL;
	return (_tmp0 = c, (name = (g_free (name), NULL)), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp0);
}


void vala_gir_parser_parse_metadata (ValaGirParser* self, const char* metadata_filename) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	g_return_if_fail (metadata_filename != NULL);
	inner_error = NULL;
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			gulong metadata_len;
			char* _tmp2;
			gboolean _tmp1;
			char* _tmp0;
			metadata = NULL;
			metadata_len = 0UL;
			_tmp2 = NULL;
			_tmp0 = NULL;
			_tmp1 = g_file_get_contents (metadata_filename, &_tmp0, &metadata_len, &inner_error);
			metadata = (_tmp2 = _tmp0, (metadata = (g_free (metadata), NULL)), _tmp2);
			_tmp1;
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch2_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** line_collection;
				int line_collection_length1;
				char** line_it;
				line_collection = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = -1;
				for (line_it = line_collection; *line_it != NULL; line_it = line_it + 1) {
					const char* _tmp8;
					char* line;
					_tmp8 = NULL;
					line = (_tmp8 = *line_it, (_tmp8 == NULL ? NULL : g_strdup (_tmp8)));
					{
						char** _tmp3;
						gint tokens_length1;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							/* ignore comment lines*/
							line = (g_free (line), NULL);
							continue;
						}
						_tmp3 = NULL;
						tokens = (_tmp3 = g_strsplit (line, " ", 2), tokens_length1 = -1, _tmp3);
						if (NULL == tokens[0]) {
							line = (g_free (line), NULL);
							tokens = (_vala_array_free (tokens, tokens_length1, ((GDestroyNotify) (g_free))), NULL);
							continue;
						}
						{
							char** attribute_collection;
							int attribute_collection_length1;
							char** attribute_it;
							attribute_collection = g_strsplit (tokens[1], " ", 0);
							attribute_collection_length1 = -1;
							for (attribute_it = attribute_collection; *attribute_it != NULL; attribute_it = attribute_it + 1) {
								const char* _tmp7;
								char* attribute;
								_tmp7 = NULL;
								attribute = (_tmp7 = *attribute_it, (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
								{
									char** _tmp4;
									gint pair_length1;
									char** pair;
									char* key;
									char* _tmp6;
									char* _tmp5;
									_tmp4 = NULL;
									pair = (_tmp4 = g_strsplit (attribute, "=", 2), pair_length1 = -1, _tmp4);
									key = g_strdup_printf ("%s/@%s", tokens[0], pair[0]);
									_tmp6 = NULL;
									_tmp5 = NULL;
									gee_map_set (((GeeMap*) (self->priv->attributes_map)), key, (_tmp6 = (_tmp5 = g_utf8_offset_to_pointer (pair[1], ((glong) (1))), g_strndup (_tmp5, g_utf8_offset_to_pointer (_tmp5, string_get_length (pair[1]) - 2) - _tmp5))));
									_tmp6 = (g_free (_tmp6), NULL);
									attribute = (g_free (attribute), NULL);
									pair = (_vala_array_free (pair, pair_length1, ((GDestroyNotify) (g_free))), NULL);
									key = (g_free (key), NULL);
								}
							}
							attribute_collection = (_vala_array_free (attribute_collection, attribute_collection_length1, ((GDestroyNotify) (g_free))), NULL);
						}
						line = (g_free (line), NULL);
						tokens = (_vala_array_free (tokens, tokens_length1, ((GDestroyNotify) (g_free))), NULL);
					}
				}
				line_collection = (_vala_array_free (line_collection, line_collection_length1, ((GDestroyNotify) (g_free))), NULL);
			}
			metadata = (g_free (metadata), NULL);
		}
		goto __finally2;
		__catch2_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp9;
				_tmp9 = NULL;
				vala_report_error (NULL, (_tmp9 = g_strdup_printf ("Unable to read metadata file: %s", e->message)));
				_tmp9 = (g_free (_tmp9), NULL);
				(e == NULL ? NULL : (e = (g_error_free (e), NULL)));
			}
		}
		__finally2:
		;
	} else {
		char* _tmp10;
		_tmp10 = NULL;
		vala_report_error (NULL, (_tmp10 = g_strdup_printf ("Metadata file `%s' not found", metadata_filename)));
		_tmp10 = (g_free (_tmp10), NULL);
	}
}


static char* vala_gir_parser_get_attribute (ValaGirParser* self, const char* node, const char* key) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (node != NULL, NULL);
	g_return_val_if_fail (key != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = ((char*) (gee_map_get (((GeeMap*) (self->priv->attributes_map)), (_tmp0 = g_strdup_printf ("%s/@%s", node, key))))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


/**
 * Code visitor parsing all Vala source files.
 */
ValaGirParser* vala_gir_parser_construct (GType object_type) {
	ValaGirParser* self;
	self = ((ValaGirParser*) (g_type_create_instance (object_type)));
	return self;
}


ValaGirParser* vala_gir_parser_new (void) {
	return vala_gir_parser_construct (VALA_TYPE_GIR_PARSER);
}


static void vala_gir_parser_class_init (ValaGirParserClass * klass) {
	vala_gir_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gir_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaGirParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gir_parser_real_visit_source_file;
}


static void vala_gir_parser_instance_init (ValaGirParser * self) {
	self->priv = VALA_GIR_PARSER_GET_PRIVATE (self);
	self->priv->attributes_map = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_gir_parser_finalize (ValaCodeVisitor* obj) {
	ValaGirParser * self;
	self = VALA_GIR_PARSER (obj);
	(self->priv->reader == NULL ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL)));
	(self->priv->context == NULL ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)));
	(self->priv->glib_ns == NULL ? NULL : (self->priv->glib_ns = (vala_code_node_unref (self->priv->glib_ns), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)));
	(self->priv->attributes_map == NULL ? NULL : (self->priv->attributes_map = (gee_collection_object_unref (self->priv->attributes_map), NULL)));
	VALA_CODE_VISITOR_CLASS (vala_gir_parser_parent_class)->finalize (obj);
}


GType vala_gir_parser_get_type (void) {
	static GType vala_gir_parser_type_id = 0;
	if (vala_gir_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGirParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gir_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGirParser), 0, (GInstanceInitFunc) vala_gir_parser_instance_init, NULL };
		vala_gir_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGirParser", &g_define_type_info, 0);
	}
	return vala_gir_parser_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




