/* valasignal.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valasignal.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valatypereference.h>
#include <vala/valacallback.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valasignal.h>
#include <vala/valaformalparameter.h>
#include <vala/valascope.h>
#include <vala/valadatatype.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaSignalPrivate {
	ValaTypeReference* _return_type;
	ValaMemberAccessibility _access;
	gboolean _has_emitter;
	GeeList* parameters;
	ValaCallback* generated_callback;
	char* cname;
	gboolean lock_used;
};
#define VALA_SIGNAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SIGNAL, ValaSignalPrivate))
enum  {
	VALA_SIGNAL_DUMMY_PROPERTY,
	VALA_SIGNAL_RETURN_TYPE,
	VALA_SIGNAL_ACCESS,
	VALA_SIGNAL_HAS_EMITTER
};
static GeeCollection* vala_signal_real_get_parameters (ValaInvokable* base);
static ValaTypeReference* vala_signal_real_get_return_type (ValaInvokable* base);
static gboolean vala_signal_real_is_invokable (ValaInvokable* base);
static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_signal_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_signal_real_get_lock_used (ValaLockable* base);
static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used);
static gpointer vala_signal_parent_class = NULL;
static ValaInvokableIface* vala_signal_vala_invokable_parent_iface = NULL;
static ValaLockableIface* vala_signal_vala_lockable_parent_iface = NULL;
static void vala_signal_dispose (GObject * obj);


/**
 * Creates a new signal.
 *
 * @param name        signal name
 * @param return_type signal return type
 * @param source      reference to source code
 * @return            newly created signal
 */
ValaSignal* vala_signal_new (const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference)
{
	ValaSignal * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_SIGNAL, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_signal_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends parameter to signal handler.
 *
 * @param param a formal parameter
 */
void vala_signal_add_parameter (ValaSignal* self, ValaFormalParameter* param)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) param)), VALA_SYMBOL (param));
}


static GeeCollection* vala_signal_real_get_parameters (ValaInvokable* base)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->parameters)));
}


static ValaTypeReference* vala_signal_real_get_return_type (ValaInvokable* base)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	return g_object_ref (vala_signal_get_return_type (self));
}


static gboolean vala_signal_real_is_invokable (ValaInvokable* base)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	return TRUE;
}


/**
 * Returns generated callback to be used for signal handlers.
 *
 * @return callback
 */
ValaCallback* vala_signal_get_callback (ValaSignal* self)
{
	ValaCallback* __temp4;
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	if (self->priv->generated_callback == NULL) {
		ValaCallback* __temp3;
		ValaTypeReference* sender_type;
		ValaFormalParameter* sender_param;
		__temp3 = NULL;
		self->priv->generated_callback = (__temp3 = vala_callback_new (NULL, vala_signal_get_return_type (self), NULL), (self->priv->generated_callback == NULL ? NULL : (self->priv->generated_callback = (g_object_unref (self->priv->generated_callback), NULL))), __temp3);
		vala_callback_set_instance (self->priv->generated_callback, TRUE);
		sender_type = vala_type_reference_new ();
		vala_type_reference_set_data_type (sender_type, VALA_DATA_TYPE (vala_symbol_get_parent_symbol (((ValaSymbol*) self))));
		sender_param = vala_formal_parameter_new ("sender", sender_type, NULL);
		vala_callback_add_parameter (self->priv->generated_callback, sender_param);
		{
			GeeList* param_collection;
			GeeIterator* param_it;
			param_collection = self->priv->parameters;
			param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
			while (gee_iterator_next (param_it)) {
				ValaFormalParameter* param;
				param = gee_iterator_get (param_it);
				{
					vala_callback_add_parameter (self->priv->generated_callback, param);
					(param = (g_object_unref (param), NULL));
				}
			}
			(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
		}
		(sender_type == NULL ? NULL : (sender_type = (g_object_unref (sender_type), NULL)));
		(sender_param == NULL ? NULL : (sender_param = (g_object_unref (sender_param), NULL)));
	}
	__temp4 = NULL;
	return (__temp4 = self->priv->generated_callback, (__temp4 == NULL ? NULL : g_object_ref (__temp4)));
}


/**
 * Returns the name of this signal as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_signal_get_cname (ValaSignal* self)
{
	const char* __temp8;
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	if (self->priv->cname == NULL) {
		char* __temp7;
		const char* __temp6;
		__temp7 = NULL;
		__temp6 = NULL;
		self->priv->cname = (__temp7 = (__temp6 = vala_symbol_get_name (((ValaSymbol*) self)), (__temp6 == NULL ? NULL : g_strdup (__temp6))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp7);
	}
	__temp8 = NULL;
	return (__temp8 = self->priv->cname, (__temp8 == NULL ? NULL : g_strdup (__temp8)));
}


void vala_signal_set_cname (ValaSignal* self, const char* cname)
{
	char* __temp11;
	const char* __temp10;
	g_return_if_fail (VALA_IS_SIGNAL (self));
	__temp11 = NULL;
	__temp10 = NULL;
	self->priv->cname = (__temp11 = (__temp10 = cname, (__temp10 == NULL ? NULL : g_strdup (__temp10))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp11);
}


/**
 * Returns the string literal of this signal to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_signal_get_canonical_cconstant (ValaSignal* self)
{
	GString* str;
	const char* __temp12;
	char* i;
	ValaCCodeConstant* __temp15;
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	str = g_string_new ("\"");
	__temp12 = NULL;
	i = (__temp12 = vala_symbol_get_name (((ValaSymbol*) self)), (__temp12 == NULL ? NULL : g_strdup (__temp12)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* __temp14;
		const char* __temp13;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		__temp14 = NULL;
		__temp13 = NULL;
		i = (__temp14 = (__temp13 = g_utf8_next_char (i), (__temp13 == NULL ? NULL : g_strdup (__temp13))), (i = (g_free (i), NULL)), __temp14);
	}
	g_string_append_c (str, '"');
	__temp15 = NULL;
	return (__temp15 = vala_ccode_constant_new (str->str), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), __temp15);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_signal (visitor, self);
}


static void vala_signal_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_signal_get_return_type (self)), visitor);
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


/**
 * Process all associated attributes.
 */
void vala_signal_process_attributes (ValaSignal* self)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* __temp16;
			ValaAttribute* a;
			__temp16 = NULL;
			a = (__temp16 = a_it->data, (__temp16 == NULL ? NULL : g_object_ref (__temp16)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "HasEmitter") == 0) {
					vala_signal_set_has_emitter (self, TRUE);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_signal_real_get_lock_used (ValaLockable* base)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	return self->priv->lock_used;
}


static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used)
{
	ValaSignal * self;
	self = ((ValaSignal*) base);
	self->priv->lock_used = used;
}


ValaTypeReference* vala_signal_get_return_type (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	return self->priv->_return_type;
}


void vala_signal_set_return_type (ValaSignal* self, ValaTypeReference* value)
{
	ValaTypeReference* __temp19;
	g_return_if_fail (VALA_IS_SIGNAL (self));
	__temp19 = NULL;
	self->priv->_return_type = (__temp19 = g_object_ref (value), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), __temp19);
}


ValaMemberAccessibility vala_signal_get_access (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), 0);
	return self->priv->_access;
}


void vala_signal_set_access (ValaSignal* self, ValaMemberAccessibility value)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	self->priv->_access = value;
}


gboolean vala_signal_get_has_emitter (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), FALSE);
	return self->priv->_has_emitter;
}


void vala_signal_set_has_emitter (ValaSignal* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	self->priv->_has_emitter = value;
}


static void vala_signal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSignal * self;
	self = VALA_SIGNAL (object);
	switch (property_id) {
		case VALA_SIGNAL_RETURN_TYPE:
		g_value_set_object (value, vala_signal_get_return_type (self));
		break;
		case VALA_SIGNAL_ACCESS:
		g_value_set_int (value, vala_signal_get_access (self));
		break;
		case VALA_SIGNAL_HAS_EMITTER:
		g_value_set_boolean (value, vala_signal_get_has_emitter (self));
		break;
	}
}


static void vala_signal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSignal * self;
	self = VALA_SIGNAL (object);
	switch (property_id) {
		case VALA_SIGNAL_RETURN_TYPE:
		vala_signal_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_SIGNAL_ACCESS:
		vala_signal_set_access (self, g_value_get_int (value));
		break;
		case VALA_SIGNAL_HAS_EMITTER:
		vala_signal_set_has_emitter (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_signal_class_init (ValaSignalClass * klass)
{
	vala_signal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSignalPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_signal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_signal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_signal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_signal_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_signal_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_RETURN_TYPE, g_param_spec_object ("return-type", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_ACCESS, g_param_spec_int ("access", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_HAS_EMITTER, g_param_spec_boolean ("has-emitter", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_signal_vala_invokable_interface_init (ValaInvokableIface * iface)
{
	vala_signal_vala_invokable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_parameters = vala_signal_real_get_parameters;
	iface->get_return_type = vala_signal_real_get_return_type;
	iface->is_invokable = vala_signal_real_is_invokable;
}


static void vala_signal_vala_lockable_interface_init (ValaLockableIface * iface)
{
	vala_signal_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_signal_real_get_lock_used;
	iface->set_lock_used = vala_signal_real_set_lock_used;
}


static void vala_signal_init (ValaSignal * self)
{
	self->priv = VALA_SIGNAL_GET_PRIVATE (self);
	self->priv->parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->lock_used = FALSE;
}


static void vala_signal_dispose (GObject * obj)
{
	ValaSignal * self;
	ValaSignalClass * klass;
	GObjectClass * parent_class;
	self = VALA_SIGNAL (obj);
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	(self->priv->generated_callback == NULL ? NULL : (self->priv->generated_callback = (g_object_unref (self->priv->generated_callback), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	klass = VALA_SIGNAL_CLASS (g_type_class_peek (VALA_TYPE_SIGNAL));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_signal_get_type (void)
{
	static GType vala_signal_type_id = 0;
	if (G_UNLIKELY (vala_signal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSignalClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_signal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSignal), 0, (GInstanceInitFunc) vala_signal_init };
		static const GInterfaceInfo vala_invokable_info = { (GInterfaceInitFunc) vala_signal_vala_invokable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_signal_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_signal_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaSignal", &g_define_type_info, 0);
		g_type_add_interface_static (vala_signal_type_id, VALA_TYPE_INVOKABLE, &vala_invokable_info);
		g_type_add_interface_static (vala_signal_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_signal_type_id;
}




