/* valamethod.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include "valamethod.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypereference.h>
#include <vala/valablock.h>
#include <vala/valamethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaMethodPrivate {
	ValaTypeReference* _return_type;
	ValaBlock* _body;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	gboolean _returns_modified_pointer;
	gboolean _instance_last;
	gboolean _instance_by_reference;
	ValaMethod* _base_method;
	ValaMethod* _base_interface_method;
	ValaFormalParameter* _this_parameter;
	gboolean _printf_format;
	gboolean _instance;
	GeeList* parameters;
	char* cname;
	gboolean _no_array_length;
	GeeList* error_domains;
};
#define VALA_METHOD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD, ValaMethodPrivate))
enum  {
	VALA_METHOD_DUMMY_PROPERTY,
	VALA_METHOD_RETURN_TYPE,
	VALA_METHOD_BODY,
	VALA_METHOD_INSTANCE,
	VALA_METHOD_IS_ABSTRACT,
	VALA_METHOD_IS_VIRTUAL,
	VALA_METHOD_OVERRIDES,
	VALA_METHOD_RETURNS_MODIFIED_POINTER,
	VALA_METHOD_INSTANCE_LAST,
	VALA_METHOD_INSTANCE_BY_REFERENCE,
	VALA_METHOD_BASE_METHOD,
	VALA_METHOD_BASE_INTERFACE_METHOD,
	VALA_METHOD_THIS_PARAMETER,
	VALA_METHOD_NO_ARRAY_LENGTH,
	VALA_METHOD_PRINTF_FORMAT
};
static GeeCollection* vala_method_real_get_parameters (ValaInvokable* base);
static ValaTypeReference* vala_method_real_get_return_type (ValaInvokable* base);
static gboolean vala_method_real_is_invokable (ValaInvokable* base);
static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_method_real_get_default_cname (ValaMethod* self);
static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a);
static gpointer vala_method_parent_class = NULL;
static ValaInvokableIface* vala_method_vala_invokable_parent_iface = NULL;
static void vala_method_dispose (GObject * obj);


/**
 * Creates a new method.
 *
 * @param name        method name
 * @param return_type method return type
 * @param source      reference to source code
 * @return            newly created method
 */
ValaMethod* vala_method_new (const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference)
{
	ValaMethod * self;
	g_return_val_if_fail (return_type == NULL || VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_METHOD, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_method_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends parameter to this method.
 *
 * @param param a formal parameter
 */
void vala_method_add_parameter (ValaMethod* self, ValaFormalParameter* param)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	if (vala_method_get_no_array_length (self)) {
		vala_formal_parameter_set_no_array_length (param, TRUE);
	}
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	if (!vala_formal_parameter_get_ellipsis (param)) {
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) param)), VALA_SYMBOL (param));
	}
}


static GeeCollection* vala_method_real_get_parameters (ValaInvokable* base)
{
	ValaMethod * self;
	self = ((ValaMethod*) base);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->parameters)));
}


static ValaTypeReference* vala_method_real_get_return_type (ValaInvokable* base)
{
	ValaMethod * self;
	ValaTypeReference* __temp1;
	self = ((ValaMethod*) base);
	__temp1 = NULL;
	return (__temp1 = vala_method_get_return_type (self), (__temp1 == NULL ? NULL : g_object_ref (__temp1)));
}


static gboolean vala_method_real_is_invokable (ValaInvokable* base)
{
	ValaMethod * self;
	self = ((ValaMethod*) base);
	return TRUE;
}


static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaMethod * self;
	self = ((ValaMethod*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_method (visitor, self);
}


static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaMethod * self;
	self = ((ValaMethod*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_return_type (self)), visitor);
	}
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	{
		GeeList* error_domain_collection;
		GeeIterator* error_domain_it;
		error_domain_collection = self->priv->error_domains;
		error_domain_it = gee_iterable_iterator (GEE_ITERABLE (error_domain_collection));
		while (gee_iterator_next (error_domain_it)) {
			ValaTypeReference* error_domain;
			error_domain = gee_iterator_get (error_domain_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (error_domain), visitor);
				(error_domain == NULL ? NULL : (error_domain = (g_object_unref (error_domain), NULL)));
			}
		}
		(error_domain_it == NULL ? NULL : (error_domain_it = (g_object_unref (error_domain_it), NULL)));
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_body (self)), visitor);
	}
}


/**
 * Returns the interface name of this method as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_cname (ValaMethod* self)
{
	const char* __temp5;
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->cname == NULL) {
		char* __temp4;
		__temp4 = NULL;
		self->priv->cname = (__temp4 = vala_method_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp4);
	}
	__temp5 = NULL;
	return (__temp5 = self->priv->cname, (__temp5 == NULL ? NULL : g_strdup (__temp5)));
}


/**
 * Returns the default interface name of this method as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
static char* vala_method_real_get_default_cname (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (g_str_has_prefix (vala_symbol_get_name (((ValaSymbol*) self)), "_")) {
		char* __temp7;
		char* __temp8;
		__temp7 = NULL;
		__temp8 = NULL;
		return (__temp8 = g_strdup_printf ("_%s%s", (__temp7 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), g_utf8_offset_to_pointer (vala_symbol_get_name (((ValaSymbol*) self)), 1)), (__temp7 = (g_free (__temp7), NULL)), __temp8);
	} else {
		char* __temp10;
		char* __temp11;
		__temp10 = NULL;
		__temp11 = NULL;
		return (__temp11 = g_strdup_printf ("%s%s", (__temp10 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), vala_symbol_get_name (((ValaSymbol*) self))), (__temp10 = (g_free (__temp10), NULL)), __temp11);
	}
}


char* vala_method_get_default_cname (ValaMethod* self)
{
	return VALA_METHOD_GET_CLASS (self)->get_default_cname (self);
}


/**
 * Returns the implementation name of this data type as it is used in C
 * code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_real_cname (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (vala_method_get_base_method (self) != NULL || vala_method_get_base_interface_method (self) != NULL) {
		char* __temp13;
		char* __temp14;
		__temp13 = NULL;
		__temp14 = NULL;
		return (__temp14 = g_strdup_printf ("%s_real_%s", (__temp13 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol (((ValaSymbol*) self)), NULL)), vala_symbol_get_name (((ValaSymbol*) self))), (__temp13 = (g_free (__temp13), NULL)), __temp14);
	} else {
		return vala_method_get_cname (self);
	}
}


/**
 * Sets the name of this method as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_method_set_cname (ValaMethod* self, const char* cname)
{
	char* __temp18;
	const char* __temp17;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp18 = NULL;
	__temp17 = NULL;
	self->priv->cname = (__temp18 = (__temp17 = cname, (__temp17 == NULL ? NULL : g_strdup (__temp17))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp18);
}


static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp19;
		__temp19 = NULL;
		vala_method_set_cname (self, (__temp19 = vala_attribute_get_string (a, "cname")));
		(__temp19 = (g_free (__temp19), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* __temp20;
				char* filename;
				__temp20 = NULL;
				filename = (__temp20 = *filename_it, (__temp20 == NULL ? NULL : g_strdup (__temp20)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (g_strfreev (filename_collection), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_method_process_attributes (ValaMethod* self)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* __temp21;
			ValaAttribute* a;
			__temp21 = NULL;
			a = (__temp21 = a_it->data, (__temp21 == NULL ? NULL : g_object_ref (__temp21)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_method_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "ReturnsModifiedPointer") == 0) {
						vala_method_set_returns_modified_pointer (self, TRUE);
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "InstanceLast") == 0) {
							vala_method_set_instance_last (self, TRUE);
						} else {
							if (g_utf8_collate (vala_attribute_get_name (a), "InstanceByReference") == 0) {
								vala_method_set_instance_by_reference (self, TRUE);
							} else {
								if (g_utf8_collate (vala_attribute_get_name (a), "FloatingReference") == 0) {
									vala_type_reference_set_floating_reference (vala_method_get_return_type (self), TRUE);
								} else {
									if (g_utf8_collate (vala_attribute_get_name (a), "NoArrayLength") == 0) {
										vala_method_set_no_array_length (self, TRUE);
									} else {
										if (g_utf8_collate (vala_attribute_get_name (a), "PrintfFormat") == 0) {
											vala_method_set_printf_format (self, TRUE);
										}
									}
								}
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this method.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this method
 */
gboolean vala_method_equals (ValaMethod* self, ValaMethod* m2)
{
	GeeCollection* __temp23;
	GeeIterator* __temp24;
	GeeIterator* method_params_it;
	gboolean __temp30;
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (m2), FALSE);
	if (!vala_type_reference_equals (vala_method_get_return_type (m2), vala_method_get_return_type (self))) {
		return FALSE;
	}
	__temp23 = NULL;
	__temp24 = NULL;
	method_params_it = (__temp24 = gee_iterable_iterator (GEE_ITERABLE ((__temp23 = vala_invokable_get_parameters (VALA_INVOKABLE (m2))))), (__temp23 == NULL ? NULL : (__temp23 = (g_object_unref (__temp23), NULL))), __temp24);
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				ValaFormalParameter* __temp26;
				gboolean __temp27;
				/* method may not expect less arguments */
				if (!gee_iterator_next (method_params_it)) {
					gboolean __temp25;
					return (__temp25 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp25);
				}
				__temp26 = NULL;
				if ((__temp27 = !vala_type_reference_equals (vala_formal_parameter_get_type_reference ((__temp26 = gee_iterator_get (method_params_it))), vala_formal_parameter_get_type_reference (param)), (__temp26 == NULL ? NULL : (__temp26 = (g_object_unref (__temp26), NULL))), __temp27)) {
					gboolean __temp28;
					return (__temp28 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp28);
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	/* method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		gboolean __temp29;
		return (__temp29 = FALSE, (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp29);
	}
	return (__temp30 = TRUE, (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp30);
	(method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL)));
}


/**
 * Adds an error domain to this method.
 *
 * @param error_domain an error domain
 */
void vala_method_add_error_domain (ValaMethod* self, ValaTypeReference* error_domain)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (error_domain));
	gee_collection_add (GEE_COLLECTION (self->priv->error_domains), error_domain);
}


/**
 * Returns a copy of the list of error domains of this method.
 *
 * @return list of error domains
 */
GeeCollection* vala_method_get_error_domains (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->error_domains)));
}


ValaTypeReference* vala_method_get_return_type (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_return_type;
}


void vala_method_set_return_type (ValaMethod* self, ValaTypeReference* value)
{
	ValaTypeReference* __temp34;
	ValaTypeReference* __temp33;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp34 = NULL;
	__temp33 = NULL;
	self->priv->_return_type = (__temp34 = (__temp33 = value, (__temp33 == NULL ? NULL : g_object_ref (__temp33))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), __temp34);
}


ValaBlock* vala_method_get_body (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_body;
}


void vala_method_set_body (ValaMethod* self, ValaBlock* value)
{
	ValaBlock* __temp37;
	ValaBlock* __temp36;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp37 = NULL;
	__temp36 = NULL;
	self->priv->_body = (__temp37 = (__temp36 = value, (__temp36 == NULL ? NULL : g_object_ref (__temp36))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), __temp37);
}


gboolean vala_method_get_instance (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance;
}


void vala_method_set_instance (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance = value;
}


gboolean vala_method_get_is_abstract (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_method_set_is_abstract (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_abstract = value;
}


gboolean vala_method_get_is_virtual (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_method_set_is_virtual (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_virtual = value;
}


gboolean vala_method_get_overrides (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_overrides;
}


void vala_method_set_overrides (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_overrides = value;
}


gboolean vala_method_get_returns_modified_pointer (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_returns_modified_pointer;
}


void vala_method_set_returns_modified_pointer (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_returns_modified_pointer = value;
}


gboolean vala_method_get_instance_last (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance_last;
}


void vala_method_set_instance_last (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance_last = value;
}


gboolean vala_method_get_instance_by_reference (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance_by_reference;
}


void vala_method_set_instance_by_reference (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance_by_reference = value;
}


ValaMethod* vala_method_get_base_method (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_method;
}


void vala_method_set_base_method (ValaMethod* self, ValaMethod* value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_base_method = value;
}


ValaMethod* vala_method_get_base_interface_method (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_interface_method;
}


void vala_method_set_base_interface_method (ValaMethod* self, ValaMethod* value)
{
	ValaMethod* __temp48;
	ValaMethod* __temp47;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp48 = NULL;
	__temp47 = NULL;
	self->priv->_base_interface_method = (__temp48 = (__temp47 = value, (__temp47 == NULL ? NULL : g_object_ref (__temp47))), (self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL))), __temp48);
}


ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_this_parameter;
}


void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value)
{
	ValaFormalParameter* __temp51;
	ValaFormalParameter* __temp50;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp51 = NULL;
	__temp50 = NULL;
	self->priv->_this_parameter = (__temp51 = (__temp50 = value, (__temp50 == NULL ? NULL : g_object_ref (__temp50))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL))), __temp51);
}


gboolean vala_method_get_no_array_length (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_method_set_no_array_length (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_no_array_length = value;
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_formal_parameter_set_no_array_length (param, value);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


gboolean vala_method_get_printf_format (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_printf_format;
}


void vala_method_set_printf_format (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_printf_format = value;
}


static void vala_method_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_RETURN_TYPE:
		g_value_set_object (value, vala_method_get_return_type (self));
		break;
		case VALA_METHOD_BODY:
		g_value_set_object (value, vala_method_get_body (self));
		break;
		case VALA_METHOD_INSTANCE:
		g_value_set_boolean (value, vala_method_get_instance (self));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		g_value_set_boolean (value, vala_method_get_is_abstract (self));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		g_value_set_boolean (value, vala_method_get_is_virtual (self));
		break;
		case VALA_METHOD_OVERRIDES:
		g_value_set_boolean (value, vala_method_get_overrides (self));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		g_value_set_boolean (value, vala_method_get_returns_modified_pointer (self));
		break;
		case VALA_METHOD_INSTANCE_LAST:
		g_value_set_boolean (value, vala_method_get_instance_last (self));
		break;
		case VALA_METHOD_INSTANCE_BY_REFERENCE:
		g_value_set_boolean (value, vala_method_get_instance_by_reference (self));
		break;
		case VALA_METHOD_BASE_METHOD:
		g_value_set_object (value, vala_method_get_base_method (self));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		g_value_set_object (value, vala_method_get_base_interface_method (self));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		g_value_set_object (value, vala_method_get_this_parameter (self));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_method_get_no_array_length (self));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		g_value_set_boolean (value, vala_method_get_printf_format (self));
		break;
	}
}


static void vala_method_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_RETURN_TYPE:
		vala_method_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BODY:
		vala_method_set_body (self, g_value_get_object (value));
		break;
		case VALA_METHOD_INSTANCE:
		vala_method_set_instance (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		vala_method_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		vala_method_set_is_virtual (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_OVERRIDES:
		vala_method_set_overrides (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		vala_method_set_returns_modified_pointer (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_INSTANCE_LAST:
		vala_method_set_instance_last (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_INSTANCE_BY_REFERENCE:
		vala_method_set_instance_by_reference (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_BASE_METHOD:
		vala_method_set_base_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		vala_method_set_base_interface_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		vala_method_set_this_parameter (self, g_value_get_object (value));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		vala_method_set_no_array_length (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		vala_method_set_printf_format (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_method_class_init (ValaMethodClass * klass)
{
	vala_method_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMethodPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_method_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_method_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_method_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_real_accept_children;
	VALA_METHOD_CLASS (klass)->get_default_cname = vala_method_real_get_default_cname;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURN_TYPE, g_param_spec_object ("return-type", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BODY, g_param_spec_object ("body", "foo", "bar", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE, g_param_spec_boolean ("instance", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_VIRTUAL, g_param_spec_boolean ("is-virtual", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_OVERRIDES, g_param_spec_boolean ("overrides", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURNS_MODIFIED_POINTER, g_param_spec_boolean ("returns-modified-pointer", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE_LAST, g_param_spec_boolean ("instance-last", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE_BY_REFERENCE, g_param_spec_boolean ("instance-by-reference", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_METHOD, g_param_spec_object ("base-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_INTERFACE_METHOD, g_param_spec_object ("base-interface-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_THIS_PARAMETER, g_param_spec_object ("this-parameter", "foo", "bar", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_PRINTF_FORMAT, g_param_spec_boolean ("printf-format", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_method_vala_invokable_interface_init (ValaInvokableIface * iface)
{
	vala_method_vala_invokable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_parameters = vala_method_real_get_parameters;
	iface->get_return_type = vala_method_real_get_return_type;
	iface->is_invokable = vala_method_real_is_invokable;
}


static void vala_method_init (ValaMethod * self)
{
	self->priv = VALA_METHOD_GET_PRIVATE (self);
	self->priv->_instance = TRUE;
	self->priv->parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->error_domains = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_method_dispose (GObject * obj)
{
	ValaMethod * self;
	ValaMethodClass * klass;
	GObjectClass * parent_class;
	self = VALA_METHOD (obj);
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	(self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->error_domains == NULL ? NULL : (self->priv->error_domains = (g_object_unref (self->priv->error_domains), NULL)));
	klass = VALA_METHOD_CLASS (g_type_class_peek (VALA_TYPE_METHOD));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_method_get_type (void)
{
	static GType vala_method_type_id = 0;
	if (G_UNLIKELY (vala_method_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethod), 0, (GInstanceInitFunc) vala_method_init };
		static const GInterfaceInfo vala_invokable_info = { (GInterfaceInitFunc) vala_method_vala_invokable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_method_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaMethod", &g_define_type_info, 0);
		g_type_add_interface_static (vala_method_type_id, VALA_TYPE_INVOKABLE, &vala_invokable_info);
	}
	return vala_method_type_id;
}




