/* valacompiler.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valacompiler.h"
#include <vala/valareport.h>
#include <stdio.h>
#include <glib/gstdio.h>
#include <config.h>
#include <vala/valasourcefile.h>
#include <vala/valasourcereference.h>
#include <vala/valaparser.h>
#include <vala/valaattributeprocessor.h>
#include <vala/valasymbolresolver.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valamemorymanager.h>
#include <gobject/valacodegenerator.h>
#include <vala/valainterfacewriter.h>
#include <gobject/valaccodecompiler.h>

typedef struct _ValaCompiler ValaCompiler;
typedef struct _ValaCompilerClass ValaCompilerClass;
typedef struct _ValaCompilerPrivate ValaCompilerPrivate;

#define VALA_TYPE_COMPILER (vala_compiler_get_type ())
#define VALA_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMPILER, ValaCompiler))
#define VALA_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMPILER, ValaCompilerClass))
#define VALA_IS_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMPILER))
#define VALA_IS_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMPILER))
#define VALA_COMPILER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMPILER, ValaCompilerClass))

struct _ValaCompiler {
	GObject parent;
	ValaCompilerPrivate * priv;
};
struct _ValaCompilerClass {
	GObjectClass parent;
};
struct _ValaCompilerPrivate {
	ValaCodeContext* context;
};
#define VALA_COMPILER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_COMPILER, ValaCompilerPrivate))
enum  {
	VALA_COMPILER_DUMMY_PROPERTY
};
static char* vala_compiler_directory;
static gboolean vala_compiler_version;
static char** vala_compiler_sources;
static char** vala_compiler_vapi_directories;
static char* vala_compiler_library;
static char** vala_compiler_packages;
static gboolean vala_compiler_disable_memory_management;
static gboolean vala_compiler_ccode_only;
static gboolean vala_compiler_compile_only;
static char* vala_compiler_output;
static gboolean vala_compiler_debug;
static gboolean vala_compiler_thread;
static gint vala_compiler_optlevel;
static gboolean vala_compiler_disable_assert;
static gboolean vala_compiler_enable_checking;
static char* vala_compiler_cc_command;
static char** vala_compiler_cc_options;
static const GOptionEntry VALA_COMPILER_options[] = {{"vapidir", 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", 0, 0, G_OPTION_ARG_STRING, &vala_compiler_library, "Library name", "NAME"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_compiler_directory, "Output directory", "DIRECTORY"}, {"version", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_version, "Display version number", NULL}, {"disable-memory-management", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_memory_management, "Disable memory management", NULL}, {"ccode", 'C', 0, G_OPTION_ARG_NONE, &vala_compiler_ccode_only, "Output C code", NULL}, {"compile", 'c', 0, G_OPTION_ARG_NONE, &vala_compiler_compile_only, "Compile but do not link", NULL}, {"output", 'o', 0, G_OPTION_ARG_FILENAME, &vala_compiler_output, "Place output in file FILE", "FILE"}, {"debug", 'g', 0, G_OPTION_ARG_NONE, &vala_compiler_debug, "Produce debug information", NULL}, {"thread", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_thread, "Enable multithreading support", NULL}, {"optimize", 'O', 0, G_OPTION_ARG_INT, &vala_compiler_optlevel, "Optimization level", "OPTLEVEL"}, {"disable-assert", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_assert, "Disable assertions", NULL}, {"enable-checking", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_enable_checking, "Enable run-time checks", NULL}, {"cc", 0, 0, G_OPTION_ARG_STRING, &vala_compiler_cc_command, "Use COMMAND as C compiler command", "COMMAND"}, {"Xcc", 'X', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_cc_options, "Pass OPTION to the C compiler", "OPTION..."}, {"", 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_sources, NULL, "FILE..."}, {NULL}};
static gint vala_compiler_quit (ValaCompiler* self);
static char* vala_compiler_get_package_path (ValaCompiler* self, const char* pkg);
static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg);
static gint vala_compiler_run (ValaCompiler* self);
static gint vala_compiler_main (int args_length1, char** args);
static gpointer vala_compiler_parent_class = NULL;
static void vala_compiler_dispose (GObject * obj);
GType vala_compiler_get_type (void);


static gint vala_compiler_quit (ValaCompiler* self)
{
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	if (vala_report_get_errors () == 0) {
		fprintf (stdout, "Compilation succeeded - %d warning(s)\n", vala_report_get_warnings ());
		return 0;
	} else {
		fprintf (stdout, "Compilation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (), vala_report_get_warnings ());
		return 1;
	}
}


static char* vala_compiler_get_package_path (ValaCompiler* self, const char* pkg)
{
	char* basename;
	char* filename;
	char* __temp5;
	char* __temp7;
	gpointer __temp9;
	g_return_val_if_fail (VALA_IS_COMPILER (self), NULL);
	g_return_val_if_fail (pkg != NULL, NULL);
	basename = g_strdup_printf ("%s.vala", pkg);
	if (vala_compiler_vapi_directories != NULL) {
		{
			char** vapidir_collection;
			char** vapidir_it;
			vapidir_collection = vala_compiler_vapi_directories;
			for (vapidir_it = vapidir_collection; *vapidir_it != NULL; vapidir_it = vapidir_it + 1) {
				const char* __temp3;
				char* vapidir;
				__temp3 = NULL;
				vapidir = (__temp3 = *vapidir_it, (__temp3 == NULL ? NULL : g_strdup (__temp3)));
				{
					char* filename;
					filename = g_build_filename (vapidir, basename, NULL);
					if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
						char* __temp2;
						__temp2 = NULL;
						return (__temp2 = filename, (vapidir = (g_free (vapidir), NULL)), (basename = (g_free (basename), NULL)), __temp2);
					}
					(vapidir = (g_free (vapidir), NULL));
					(filename = (g_free (filename), NULL));
				}
			}
		}
	}
	filename = g_build_filename (PACKAGE_DATADIR, "vapi", basename, NULL);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* __temp4;
		__temp4 = NULL;
		return (__temp4 = filename, (basename = (g_free (basename), NULL)), __temp4);
	}
	__temp5 = NULL;
	filename = (__temp5 = g_build_filename ("/usr/local/share/vala/vapi", basename, NULL), (filename = (g_free (filename), NULL)), __temp5);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* __temp6;
		__temp6 = NULL;
		return (__temp6 = filename, (basename = (g_free (basename), NULL)), __temp6);
	}
	__temp7 = NULL;
	filename = (__temp7 = g_build_filename ("/usr/share/vala/vapi", basename, NULL), (filename = (g_free (filename), NULL)), __temp7);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* __temp8;
		__temp8 = NULL;
		return (__temp8 = filename, (basename = (g_free (basename), NULL)), __temp8);
	}
	return (__temp9 = NULL, (basename = (g_free (basename), NULL)), (filename = (g_free (filename), NULL)), __temp9);
	(basename = (g_free (basename), NULL));
	(filename = (g_free (filename), NULL));
}


static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg)
{
	GError * inner_error;
	char* package_path;
	ValaSourceFile* __temp12;
	char* __temp14;
	char* __temp13;
	char* __temp15;
	char* deps_filename;
	gboolean __temp19;
	g_return_val_if_fail (VALA_IS_COMPILER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (context), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	inner_error = NULL;
	if (vala_code_context_has_package (context, pkg)) {
		return TRUE;
	}
	/* ignore multiple occurences of the same package*/
	package_path = vala_compiler_get_package_path (self, pkg);
	if (package_path == NULL) {
		gboolean __temp11;
		return (__temp11 = FALSE, (package_path = (g_free (package_path), NULL)), __temp11);
	}
	vala_code_context_add_package (context, pkg);
	__temp12 = NULL;
	vala_code_context_add_source_file (context, (__temp12 = vala_source_file_new (context, package_path, TRUE)));
	(__temp12 == NULL ? NULL : (__temp12 = (g_object_unref (__temp12), NULL)));
	__temp14 = NULL;
	__temp13 = NULL;
	__temp15 = NULL;
	deps_filename = (__temp15 = g_build_filename ((__temp13 = g_path_get_dirname (package_path)), (__temp14 = g_strdup_printf ("%s.deps", pkg)), NULL), (__temp14 = (g_free (__temp14), NULL)), (__temp13 = (g_free (__temp13), NULL)), __temp15);
	if (g_file_test (deps_filename, G_FILE_TEST_EXISTS)) {
		{
			char* deps_content;
			glong deps_len;
			deps_content = NULL;
			deps_len = 0L;
			g_file_get_contents (deps_filename, &deps_content, &deps_len, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** dep_collection;
				char** dep_it;
				dep_collection = g_strsplit (deps_content, "\n", 0);
				for (dep_it = dep_collection; *dep_it != NULL; dep_it = dep_it + 1) {
					const char* __temp17;
					char* dep;
					__temp17 = NULL;
					dep = (__temp17 = *dep_it, (__temp17 == NULL ? NULL : g_strdup (__temp17)));
					{
						if (g_utf8_collate (dep, "") != 0) {
							if (!vala_compiler_add_package (self, context, dep)) {
								char* __temp16;
								__temp16 = NULL;
								vala_report_error (NULL, (__temp16 = g_strdup_printf ("%s, dependency of %s, not found in specified Vala API directories", dep, pkg)));
								(__temp16 = (g_free (__temp16), NULL));
							}
						}
						(dep = (g_free (dep), NULL));
					}
				}
				(dep_collection = (g_strfreev (dep_collection), NULL));
			}
			(deps_content = (g_free (deps_content), NULL));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* __temp18;
				__temp18 = NULL;
				vala_report_error (NULL, (__temp18 = g_strdup_printf ("Unable to read dependency file: %s", e->message)));
				(__temp18 = (g_free (__temp18), NULL));
			}
		}
		__finally0:
		;
	}
	return (__temp19 = TRUE, (package_path = (g_free (package_path), NULL)), (deps_filename = (g_free (deps_filename), NULL)), __temp19);
	(package_path = (g_free (package_path), NULL));
	(deps_filename = (g_free (deps_filename), NULL));
}


static gint vala_compiler_run (ValaCompiler* self)
{
	ValaCodeContext* __temp20;
	char** __temp28;
	ValaParser* parser;
	ValaAttributeProcessor* attributeprocessor;
	ValaSymbolResolver* resolver;
	ValaSemanticAnalyzer* analyzer;
	ValaCodeGenerator* code_generator;
	gint __temp38;
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	__temp20 = NULL;
	self->priv->context = (__temp20 = g_object_new (VALA_TYPE_CODE_CONTEXT, NULL), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), __temp20);
	/* support old command line interface */
	if (!vala_compiler_ccode_only && !vala_compiler_compile_only && vala_compiler_output == NULL) {
		vala_compiler_ccode_only = TRUE;
	}
	vala_code_context_set_library (self->priv->context, vala_compiler_library);
	vala_code_context_set_memory_management (self->priv->context, !vala_compiler_disable_memory_management);
	vala_code_context_set_assert (self->priv->context, !vala_compiler_disable_assert);
	vala_code_context_set_checking (self->priv->context, vala_compiler_enable_checking);
	vala_code_context_set_ccode_only (self->priv->context, vala_compiler_ccode_only);
	vala_code_context_set_compile_only (self->priv->context, vala_compiler_compile_only);
	vala_code_context_set_output (self->priv->context, vala_compiler_output);
	vala_code_context_set_debug (self->priv->context, vala_compiler_debug);
	vala_code_context_set_thread (self->priv->context, vala_compiler_thread);
	vala_code_context_set_optlevel (self->priv->context, vala_compiler_optlevel);
	/* default package */
	if (!vala_compiler_add_package (self, self->priv->context, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (vala_compiler_packages != NULL) {
		char** __temp23;
		{
			char** package_collection;
			char** package_it;
			package_collection = vala_compiler_packages;
			for (package_it = package_collection; *package_it != NULL; package_it = package_it + 1) {
				const char* __temp22;
				char* package;
				__temp22 = NULL;
				package = (__temp22 = *package_it, (__temp22 == NULL ? NULL : g_strdup (__temp22)));
				{
					if (!vala_compiler_add_package (self, self->priv->context, package)) {
						char* __temp21;
						__temp21 = NULL;
						vala_report_error (NULL, (__temp21 = g_strdup_printf ("%s not found in specified Vala API directories", package)));
						(__temp21 = (g_free (__temp21), NULL));
					}
					(package = (g_free (package), NULL));
				}
			}
		}
		__temp23 = NULL;
		vala_compiler_packages = (__temp23 = NULL, (vala_compiler_packages = (g_strfreev (vala_compiler_packages), NULL)), __temp23);
	}
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	{
		char** source_collection;
		char** source_it;
		source_collection = vala_compiler_sources;
		for (source_it = source_collection; *source_it != NULL; source_it = source_it + 1) {
			const char* __temp27;
			char* source;
			__temp27 = NULL;
			source = (__temp27 = *source_it, (__temp27 == NULL ? NULL : g_strdup (__temp27)));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					ValaSourceFile* __temp25;
					__temp25 = NULL;
					vala_code_context_add_source_file (self->priv->context, (__temp25 = vala_source_file_new (self->priv->context, source, FALSE)));
					(__temp25 == NULL ? NULL : (__temp25 = (g_object_unref (__temp25), NULL)));
				} else {
					char* __temp26;
					__temp26 = NULL;
					vala_report_error (NULL, (__temp26 = g_strdup_printf ("%s not found", source)));
					(__temp26 = (g_free (__temp26), NULL));
				}
				(source = (g_free (source), NULL));
			}
		}
	}
	__temp28 = NULL;
	vala_compiler_sources = (__temp28 = NULL, (vala_compiler_sources = (g_strfreev (vala_compiler_sources), NULL)), __temp28);
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	parser = g_object_new (VALA_TYPE_PARSER, NULL);
	vala_parser_parse (parser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp30;
		return (__temp30 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), __temp30);
	}
	attributeprocessor = g_object_new (VALA_TYPE_ATTRIBUTE_PROCESSOR, NULL);
	vala_attribute_processor_process (attributeprocessor, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp31;
		return (__temp31 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), __temp31);
	}
	resolver = g_object_new (VALA_TYPE_SYMBOL_RESOLVER, NULL);
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp32;
		return (__temp32 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), __temp32);
	}
	analyzer = vala_semantic_analyzer_new (!vala_compiler_disable_memory_management);
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp33;
		return (__temp33 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), __temp33);
	}
	if (!vala_compiler_disable_memory_management) {
		ValaMemoryManager* memory_manager;
		memory_manager = g_object_new (VALA_TYPE_MEMORY_MANAGER, NULL);
		vala_memory_manager_analyze (memory_manager, self->priv->context);
		if (vala_report_get_errors () > 0) {
			gint __temp34;
			return (__temp34 = vala_compiler_quit (self), (memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL))), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), __temp34);
		}
		(memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL)));
	}
	code_generator = vala_code_generator_new (!vala_compiler_disable_memory_management);
	vala_code_generator_emit (code_generator, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp35;
		return (__temp35 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL))), __temp35);
	}
	if (vala_compiler_library != NULL) {
		ValaInterfaceWriter* interface_writer;
		char* __temp36;
		char* __temp37;
		interface_writer = g_object_new (VALA_TYPE_INTERFACE_WRITER, NULL);
		__temp36 = NULL;
		vala_interface_writer_write_file (interface_writer, self->priv->context, (__temp36 = g_strdup_printf ("%s.vala", vala_compiler_library)));
		(__temp36 = (g_free (__temp36), NULL));
		__temp37 = NULL;
		vala_compiler_library = (__temp37 = NULL, (vala_compiler_library = (g_free (vala_compiler_library), NULL)), __temp37);
		(interface_writer == NULL ? NULL : (interface_writer = (g_object_unref (interface_writer), NULL)));
	}
	if (!vala_compiler_ccode_only) {
		ValaCCodeCompiler* ccompiler;
		ccompiler = vala_ccode_compiler_new ();
		vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, vala_compiler_cc_options);
		(ccompiler == NULL ? NULL : (ccompiler = (g_object_unref (ccompiler), NULL)));
	}
	return (__temp38 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL))), __temp38);
	(parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL)));
	(attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL)));
	(resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL)));
	(analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL)));
	(code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL)));
}


static gint vala_compiler_main (int args_length1, char** args)
{
	GError * inner_error;
	ValaCompiler* compiler;
	gint __temp44;
	inner_error = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala Compiler");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_COMPILER_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_OPTION_ERROR) {
				goto __catch1_g_option_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		(opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL)));
	}
	goto __finally1;
	__catch1_g_option_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			return 1;
		}
	}
	__finally1:
	;
	if (vala_compiler_version) {
		fprintf (stdout, "Vala %s\n", PACKAGE_VERSION);
		return 0;
	}
	if (vala_compiler_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		return 1;
	}
	{
		char** source_collection;
		char** source_it;
		source_collection = vala_compiler_sources;
		for (source_it = source_collection; *source_it != NULL; source_it = source_it + 1) {
			const char* __temp43;
			char* source;
			__temp43 = NULL;
			source = (__temp43 = *source_it, (__temp43 == NULL ? NULL : g_strdup (__temp43)));
			{
				if (!g_str_has_suffix (source, ".vala")) {
					gint __temp42;
					fprintf (stderr, "Only .vala source files supported.\n");
					return (__temp42 = 1, (source = (g_free (source), NULL)), __temp42);
				}
				(source = (g_free (source), NULL));
			}
		}
	}
	compiler = g_object_new (VALA_TYPE_COMPILER, NULL);
	return (__temp44 = vala_compiler_run (compiler), (compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL))), __temp44);
	(compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL)));
}


int main (int argc, char ** argv)
{
	g_type_init ();
	return vala_compiler_main (argc, argv);
}


static void vala_compiler_class_init (ValaCompilerClass * klass)
{
	vala_compiler_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCompilerPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_compiler_dispose;
}


static void vala_compiler_init (ValaCompiler * self)
{
	self->priv = VALA_COMPILER_GET_PRIVATE (self);
}


static void vala_compiler_dispose (GObject * obj)
{
	ValaCompiler * self;
	ValaCompilerClass * klass;
	GObjectClass * parent_class;
	self = VALA_COMPILER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	klass = VALA_COMPILER_CLASS (g_type_class_peek (VALA_TYPE_COMPILER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_compiler_get_type (void)
{
	static GType vala_compiler_type_id = 0;
	if (G_UNLIKELY (vala_compiler_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCompiler), 0, (GInstanceInitFunc) vala_compiler_init };
		vala_compiler_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCompiler", &g_define_type_info, 0);
	}
	return vala_compiler_type_id;
}




