/* valaproperty.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valaproperty.h"
#include <vala/valatypereference.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valaformalparameter.h>
#include <vala/valaproperty.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasymbol.h>
#include <vala/valadatatype.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>

struct _ValaPropertyPrivate {
	char* _name;
	ValaTypeReference* _type_reference;
	ValaPropertyAccessor* _get_accessor;
	ValaPropertyAccessor* _set_accessor;
	ValaMemberAccessibility _access;
	ValaFormalParameter* _this_parameter;
	gboolean _no_accessor_method;
	gboolean _interface_only;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	ValaProperty* _base_property;
	ValaProperty* _base_interface_property;
	gboolean lock_used;
};
#define VALA_PROPERTY_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY, ValaPropertyPrivate))
enum  {
	VALA_PROPERTY_DUMMY_PROPERTY,
	VALA_PROPERTY_NAME,
	VALA_PROPERTY_TYPE_REFERENCE,
	VALA_PROPERTY_GET_ACCESSOR,
	VALA_PROPERTY_SET_ACCESSOR,
	VALA_PROPERTY_ACCESS,
	VALA_PROPERTY_THIS_PARAMETER,
	VALA_PROPERTY_NO_ACCESSOR_METHOD,
	VALA_PROPERTY_INTERFACE_ONLY,
	VALA_PROPERTY_IS_ABSTRACT,
	VALA_PROPERTY_IS_VIRTUAL,
	VALA_PROPERTY_OVERRIDES,
	VALA_PROPERTY_BASE_PROPERTY,
	VALA_PROPERTY_BASE_INTERFACE_PROPERTY,
};
static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_property_real_get_lock_used (ValaLockable* base);
static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used);
static gpointer vala_property_parent_class = NULL;
static ValaLockableIface* vala_property_vala_lockable_parent_iface = NULL;
static void vala_property_dispose (GObject * obj);


/**
 * Creates a new property.
 *
 * @param name         property name
 * @param type         property type
 * @param get_accessor get accessor
 * @param set_accessor set/construct accessor
 * @param source       reference to source code
 * @return             newly created property
 */
ValaProperty* vala_property_new (const char* _name, ValaTypeReference* type, ValaPropertyAccessor* _get_accessor, ValaPropertyAccessor* _set_accessor, ValaSourceReference* source)
{
	g_return_val_if_fail (_name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), NULL);
	g_return_val_if_fail (_get_accessor == NULL || VALA_IS_PROPERTY_ACCESSOR (_get_accessor), NULL);
	g_return_val_if_fail (_set_accessor == NULL || VALA_IS_PROPERTY_ACCESSOR (_set_accessor), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	GParameter * __params = g_new0 (GParameter, 5);
	GParameter * __params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "type-reference", g_value_init (&__params_it->value, VALA_TYPE_TYPE_REFERENCE), g_value_set_object (&__params_it->value, type), __params_it++);
	(__params_it->name = "get-accessor", g_value_init (&__params_it->value, VALA_TYPE_PROPERTY_ACCESSOR), g_value_set_object (&__params_it->value, _get_accessor), __params_it++);
	(__params_it->name = "set-accessor", g_value_init (&__params_it->value, VALA_TYPE_PROPERTY_ACCESSOR), g_value_set_object (&__params_it->value, _set_accessor), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	ValaProperty * self = g_object_newv (VALA_TYPE_PROPERTY, __params_it - __params, __params);
	return self;
}


static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaProperty * self = VALA_PROPERTY (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_begin_property (visitor, self);
	vala_code_node_accept (VALA_CODE_NODE (vala_property_get_type_reference (self)), visitor);
	if (vala_property_get_get_accessor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_property_get_get_accessor (self)), visitor);
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_property_get_set_accessor (self)), visitor);
	}
	vala_code_visitor_visit_end_property (visitor, self);
}


/**
 * Returns the C name of this property in upper case. Words are
 * separated by underscores. The upper case C name of the class is
 * prefix of the result.
 *
 * @return the upper case name to be used in C code
 */
char* vala_property_get_upper_case_cname (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	char* __temp2 = NULL;
	char* __temp1 = NULL;
	char* __temp0 = NULL;
	char* __temp3 = NULL;
	return (__temp3 = g_utf8_strup ((__temp2 = g_strdup_printf ("%s_%s", (__temp0 = vala_data_type_get_lower_case_cname ((VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))), NULL)), (__temp1 = vala_namespace_camel_case_to_lower_case (vala_property_get_name (self))))), -1), (__temp2 = (g_free (__temp2), NULL)), (__temp1 = (g_free (__temp1), NULL)), (__temp0 = (g_free (__temp0), NULL)), __temp3);
}


/**
 * Returns the string literal of this property to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_property_get_canonical_cconstant (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	GString* str = g_string_new ("\"");
	char* i = g_strdup (vala_property_get_name (self));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		char* __temp6 = NULL;
		const char* __temp5 = NULL;
		i = (__temp6 = (__temp5 = g_utf8_next_char (i), (__temp5 == NULL ? NULL : g_strdup (__temp5))), (i = (g_free (i), NULL)), __temp6);
	}
	g_string_append_c (str, '"');
	ValaCCodeConstant* __temp7 = NULL;
	return (__temp7 = vala_ccode_constant_new (str->str), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), __temp7);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


/**
 * Process all associated attributes.
 */
void vala_property_process_attributes (ValaProperty* self)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	{
		GList* __temp8 = NULL;
		__temp8 = VALA_CODE_NODE (self)->attributes;
		GList* a_it;
		for (a_it = __temp8; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "NoAccessorMethod") == 0) {
					(vala_property_set_no_accessor_method (self, TRUE), vala_property_get_no_accessor_method (self));
				}
			}
		}
	}
}


static gboolean vala_property_real_get_lock_used (ValaLockable* base)
{
	ValaProperty * self = VALA_PROPERTY (base);
	return self->priv->lock_used;
}


static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used)
{
	ValaProperty * self = VALA_PROPERTY (base);
	self->priv->lock_used = used;
}


/**
 * Checks whether the accessors and type of the specified property
 * matches this property.
 *
 * @param prop a property
 * @return     true if the specified property is compatible to this
 *             property
 */
gboolean vala_property_equals (ValaProperty* self, ValaProperty* prop2)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	g_return_val_if_fail (VALA_IS_PROPERTY (prop2), FALSE);
	if (!vala_type_reference_equals (vala_property_get_type_reference (prop2), vala_property_get_type_reference (self))) {
		return FALSE;
	}
	if ((vala_property_get_get_accessor (self) == NULL && vala_property_get_get_accessor (prop2) != NULL) || (vala_property_get_get_accessor (self) != NULL && vala_property_get_get_accessor (prop2) == NULL)) {
		return FALSE;
	}
	if ((vala_property_get_set_accessor (self) == NULL && vala_property_get_set_accessor (prop2) != NULL) || (vala_property_get_set_accessor (self) != NULL && vala_property_get_set_accessor (prop2) == NULL)) {
		return FALSE;
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (self)) != vala_property_accessor_get_writable (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (self)) != vala_property_accessor_get_construction (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
	}
	return TRUE;
}


char* vala_property_get_name (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_name;
}


void vala_property_set_name (ValaProperty* self, const char* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	char* __temp17 = NULL;
	self->priv->_name = (__temp17 = g_strdup (value), (self->priv->_name = (g_free (self->priv->_name), NULL)), __temp17);
}


ValaTypeReference* vala_property_get_type_reference (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_type_reference;
}


void vala_property_set_type_reference (ValaProperty* self, ValaTypeReference* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	ValaTypeReference* __temp19 = NULL;
	self->priv->_type_reference = (__temp19 = g_object_ref (value), (self->priv->_type_reference == NULL ? NULL : (self->priv->_type_reference = (g_object_unref (self->priv->_type_reference), NULL))), __temp19);
}


ValaPropertyAccessor* vala_property_get_get_accessor (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_get_accessor;
}


void vala_property_set_get_accessor (ValaProperty* self, ValaPropertyAccessor* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	ValaPropertyAccessor* __temp22 = NULL;
	ValaPropertyAccessor* __temp21 = NULL;
	self->priv->_get_accessor = (__temp22 = (__temp21 = value, (__temp21 == NULL ? NULL : g_object_ref (__temp21))), (self->priv->_get_accessor == NULL ? NULL : (self->priv->_get_accessor = (g_object_unref (self->priv->_get_accessor), NULL))), __temp22);
}


ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_set_accessor;
}


void vala_property_set_set_accessor (ValaProperty* self, ValaPropertyAccessor* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	ValaPropertyAccessor* __temp25 = NULL;
	ValaPropertyAccessor* __temp24 = NULL;
	self->priv->_set_accessor = (__temp25 = (__temp24 = value, (__temp24 == NULL ? NULL : g_object_ref (__temp24))), (self->priv->_set_accessor == NULL ? NULL : (self->priv->_set_accessor = (g_object_unref (self->priv->_set_accessor), NULL))), __temp25);
}


ValaMemberAccessibility vala_property_get_access (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), 0);
	return self->priv->_access;
}


void vala_property_set_access (ValaProperty* self, ValaMemberAccessibility value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_access = value;
}


ValaFormalParameter* vala_property_get_this_parameter (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_this_parameter;
}


void vala_property_set_this_parameter (ValaProperty* self, ValaFormalParameter* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	ValaFormalParameter* __temp29 = NULL;
	ValaFormalParameter* __temp28 = NULL;
	self->priv->_this_parameter = (__temp29 = (__temp28 = value, (__temp28 == NULL ? NULL : g_object_ref (__temp28))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL))), __temp29);
}


gboolean vala_property_get_no_accessor_method (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_no_accessor_method;
}


void vala_property_set_no_accessor_method (ValaProperty* self, gboolean value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_no_accessor_method = value;
}


gboolean vala_property_get_interface_only (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_interface_only;
}


void vala_property_set_interface_only (ValaProperty* self, gboolean value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_interface_only = value;
}


gboolean vala_property_get_is_abstract (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_property_set_is_abstract (ValaProperty* self, gboolean value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_is_abstract = value;
}


gboolean vala_property_get_is_virtual (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_property_set_is_virtual (ValaProperty* self, gboolean value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_is_virtual = value;
}


gboolean vala_property_get_overrides (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_overrides;
}


void vala_property_set_overrides (ValaProperty* self, gboolean value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_overrides = value;
}


ValaProperty* vala_property_get_base_property (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_base_property;
}


void vala_property_set_base_property (ValaProperty* self, ValaProperty* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_base_property = value;
}


ValaProperty* vala_property_get_base_interface_property (ValaProperty* self)
{
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_base_interface_property;
}


void vala_property_set_base_interface_property (ValaProperty* self, ValaProperty* value)
{
	g_return_if_fail (VALA_IS_PROPERTY (self));
	ValaProperty* __temp38 = NULL;
	ValaProperty* __temp37 = NULL;
	self->priv->_base_interface_property = (__temp38 = (__temp37 = value, (__temp37 == NULL ? NULL : g_object_ref (__temp37))), (self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (g_object_unref (self->priv->_base_interface_property), NULL))), __temp38);
}


static void vala_property_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaProperty * self = VALA_PROPERTY (object);
	switch (property_id) {
		case VALA_PROPERTY_NAME:
		g_value_set_string (value, vala_property_get_name (self));
		break;
		case VALA_PROPERTY_TYPE_REFERENCE:
		g_value_set_object (value, vala_property_get_type_reference (self));
		break;
		case VALA_PROPERTY_GET_ACCESSOR:
		g_value_set_object (value, vala_property_get_get_accessor (self));
		break;
		case VALA_PROPERTY_SET_ACCESSOR:
		g_value_set_object (value, vala_property_get_set_accessor (self));
		break;
		case VALA_PROPERTY_ACCESS:
		g_value_set_int (value, vala_property_get_access (self));
		break;
		case VALA_PROPERTY_THIS_PARAMETER:
		g_value_set_object (value, vala_property_get_this_parameter (self));
		break;
		case VALA_PROPERTY_NO_ACCESSOR_METHOD:
		g_value_set_boolean (value, vala_property_get_no_accessor_method (self));
		break;
		case VALA_PROPERTY_INTERFACE_ONLY:
		g_value_set_boolean (value, vala_property_get_interface_only (self));
		break;
		case VALA_PROPERTY_IS_ABSTRACT:
		g_value_set_boolean (value, vala_property_get_is_abstract (self));
		break;
		case VALA_PROPERTY_IS_VIRTUAL:
		g_value_set_boolean (value, vala_property_get_is_virtual (self));
		break;
		case VALA_PROPERTY_OVERRIDES:
		g_value_set_boolean (value, vala_property_get_overrides (self));
		break;
		case VALA_PROPERTY_BASE_PROPERTY:
		g_value_set_object (value, vala_property_get_base_property (self));
		break;
		case VALA_PROPERTY_BASE_INTERFACE_PROPERTY:
		g_value_set_object (value, vala_property_get_base_interface_property (self));
		break;
	}
}


static void vala_property_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaProperty * self = VALA_PROPERTY (object);
	switch (property_id) {
		case VALA_PROPERTY_NAME:
		vala_property_set_name (self, g_value_get_string (value));
		break;
		case VALA_PROPERTY_TYPE_REFERENCE:
		vala_property_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_GET_ACCESSOR:
		vala_property_set_get_accessor (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_SET_ACCESSOR:
		vala_property_set_set_accessor (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_ACCESS:
		vala_property_set_access (self, g_value_get_int (value));
		break;
		case VALA_PROPERTY_THIS_PARAMETER:
		vala_property_set_this_parameter (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_NO_ACCESSOR_METHOD:
		vala_property_set_no_accessor_method (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_INTERFACE_ONLY:
		vala_property_set_interface_only (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_IS_ABSTRACT:
		vala_property_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_IS_VIRTUAL:
		vala_property_set_is_virtual (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_OVERRIDES:
		vala_property_set_overrides (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_BASE_PROPERTY:
		vala_property_set_base_property (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_BASE_INTERFACE_PROPERTY:
		vala_property_set_base_interface_property (self, g_value_get_object (value));
		break;
	}
}


static void vala_property_class_init (ValaPropertyClass * klass)
{
	vala_property_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaPropertyPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_property_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_property_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_property_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_NAME, g_param_spec_string ("name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_TYPE_REFERENCE, g_param_spec_object ("type-reference", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_GET_ACCESSOR, g_param_spec_object ("get-accessor", "foo", "bar", VALA_TYPE_PROPERTY_ACCESSOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_SET_ACCESSOR, g_param_spec_object ("set-accessor", "foo", "bar", VALA_TYPE_PROPERTY_ACCESSOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_ACCESS, g_param_spec_int ("access", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_THIS_PARAMETER, g_param_spec_object ("this-parameter", "foo", "bar", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_NO_ACCESSOR_METHOD, g_param_spec_boolean ("no-accessor-method", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_INTERFACE_ONLY, g_param_spec_boolean ("interface-only", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_IS_VIRTUAL, g_param_spec_boolean ("is-virtual", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_OVERRIDES, g_param_spec_boolean ("overrides", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_BASE_PROPERTY, g_param_spec_object ("base-property", "foo", "bar", VALA_TYPE_PROPERTY, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_BASE_INTERFACE_PROPERTY, g_param_spec_object ("base-interface-property", "foo", "bar", VALA_TYPE_PROPERTY, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_property_vala_lockable_interface_init (ValaLockableIface * iface)
{
	vala_property_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_property_real_get_lock_used;
	iface->set_lock_used = vala_property_real_set_lock_used;
}


static void vala_property_init (ValaProperty * self)
{
	self->priv = VALA_PROPERTY_GET_PRIVATE (self);
	self->priv->lock_used = FALSE;
}


static void vala_property_dispose (GObject * obj)
{
	ValaProperty * self = VALA_PROPERTY (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->_type_reference == NULL ? NULL : (self->priv->_type_reference = (g_object_unref (self->priv->_type_reference), NULL)));
	(self->priv->_get_accessor == NULL ? NULL : (self->priv->_get_accessor = (g_object_unref (self->priv->_get_accessor), NULL)));
	(self->priv->_set_accessor == NULL ? NULL : (self->priv->_set_accessor = (g_object_unref (self->priv->_set_accessor), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL)));
	(self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (g_object_unref (self->priv->_base_interface_property), NULL)));
	ValaPropertyClass * klass;
	GObjectClass * parent_class;
	klass = VALA_PROPERTY_CLASS (g_type_class_peek (VALA_TYPE_PROPERTY));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_property_get_type ()
{
	static GType vala_property_type_id = 0;
	if (G_UNLIKELY (vala_property_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaProperty), 0, (GInstanceInitFunc) vala_property_init };
		vala_property_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaProperty", &g_define_type_info, 0);
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_property_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		g_type_add_interface_static (vala_property_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_property_type_id;
}




