/* valamemorymanager.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valamemorymanager.h"
#include <stdlib.h>
#include <string.h>
#include <vala/valasymbol.h>
#include <vala/valamemorymanager.h>
#include <vala/valatypereference.h>
#include <vala/valadatatype.h>
#include <vala/valacodenode.h>
#include <vala/valacallback.h>
#include <vala/valaformalparameter.h>
#include <vala/valasignal.h>
#include <vala/valaclass.h>
#include <vala/valainterface.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valatypeparameter.h>
#include <vala/valapointerindirection.h>

struct _ValaMemoryManagerPrivate {
	ValaSymbol* current_symbol;
};
#define VALA_MEMORY_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMORY_MANAGER, ValaMemoryManagerPrivate))
enum  {
	VALA_MEMORY_MANAGER_DUMMY_PROPERTY,
};
static void vala_memory_manager_visit_possibly_leaked_expression (ValaMemoryManager* self, ValaExpression* expr);
static void vala_memory_manager_visit_possibly_missing_copy_expression (ValaMemoryManager* self, ValaExpression* expr);
static void vala_memory_manager_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_memory_manager_real_visit_begin_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_memory_manager_real_visit_begin_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_memory_manager_real_visit_begin_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_memory_manager_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_memory_manager_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_memory_manager_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_memory_manager_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_memory_manager_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static void vala_memory_manager_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_memory_manager_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_memory_manager_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_memory_manager_parent_class = NULL;
static void vala_memory_manager_dispose (GObject * obj);


/**
 * Analyze memory usage in the specified code context.
 *
 * @param context a code context
 */
void vala_memory_manager_analyze (ValaMemoryManager* self, ValaCodeContext* context)
{
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_visit_possibly_leaked_expression (ValaMemoryManager* self, ValaExpression* expr)
{
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
	if (vala_expression_get_static_type (expr) != NULL && ((vala_type_reference_get_data_type (vala_expression_get_static_type (expr)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_expression_get_static_type (expr)))) || vala_type_reference_get_type_parameter (vala_expression_get_static_type (expr)) != NULL) && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (expr))) {
		/* mark reference as leaked */
		(vala_expression_set_ref_leaked (expr, TRUE), vala_expression_get_ref_leaked (expr));
	}
}


static void vala_memory_manager_visit_possibly_missing_copy_expression (ValaMemoryManager* self, ValaExpression* expr)
{
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
	if (vala_expression_get_static_type (expr) != NULL && ((vala_type_reference_get_data_type (vala_expression_get_static_type (expr)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_expression_get_static_type (expr)))) || vala_type_reference_get_type_parameter (vala_expression_get_static_type (expr)) != NULL) && !vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (expr))) {
		/* mark reference as missing */
		(vala_expression_set_ref_missing (expr, TRUE), vala_expression_get_ref_missing (expr));
	}
}


static void vala_memory_manager_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (vala_field_get_initializer (f) != NULL) {
		if (vala_type_reference_get_takes_ownership (vala_field_get_type_reference (f))) {
			vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_field_get_initializer (f));
		} else {
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_field_get_initializer (f));
		}
	}
}


static void vala_memory_manager_real_visit_begin_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	ValaSymbol* __temp1 = NULL;
	ValaSymbol* __temp0 = NULL;
	self->priv->current_symbol = (__temp1 = (__temp0 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp0 == NULL ? NULL : g_object_ref (__temp0))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp1);
}


static void vala_memory_manager_real_visit_begin_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	ValaSymbol* __temp3 = NULL;
	ValaSymbol* __temp2 = NULL;
	self->priv->current_symbol = (__temp3 = (__temp2 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp2 == NULL ? NULL : g_object_ref (__temp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp3);
}


static void vala_memory_manager_real_visit_begin_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	ValaSymbol* __temp5 = NULL;
	ValaSymbol* __temp4 = NULL;
	self->priv->current_symbol = (__temp5 = (__temp4 = vala_code_node_get_symbol (VALA_CODE_NODE (prop)), (__temp4 == NULL ? NULL : g_object_ref (__temp4))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp5);
}


static void vala_memory_manager_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_named_argument_get_argument (n));
}


static void vala_memory_manager_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
			vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_variable_declarator_get_initializer (decl));
		} else {
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_variable_declarator_get_initializer (decl));
		}
	}
}


static void vala_memory_manager_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_expression_statement_get_expression (stmt));
}


static void vala_memory_manager_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		if (VALA_IS_METHOD (vala_symbol_get_node (self->priv->current_symbol))) {
			ValaMethod* __temp6 = NULL;
			ValaMethod* m = (__temp6 = VALA_METHOD (vala_symbol_get_node (self->priv->current_symbol)), (__temp6 == NULL ? NULL : g_object_ref (__temp6)));
			if (vala_type_reference_get_transfers_ownership (vala_method_get_return_type (m))) {
				vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_return_statement_get_return_expression (stmt));
			} else {
				vala_memory_manager_visit_possibly_leaked_expression (self, vala_return_statement_get_return_expression (stmt));
			}
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			/* property get accessor */
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_return_statement_get_return_expression (stmt));
		}
	}
}


static void vala_memory_manager_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	if (vala_member_access_get_inner (expr) != NULL) {
		vala_memory_manager_visit_possibly_leaked_expression (self, vala_member_access_get_inner (expr));
	}
}


static void vala_memory_manager_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	GList* params = NULL;
	ValaSymbol* __temp7 = NULL;
	ValaSymbol* msym = (__temp7 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp7 == NULL ? NULL : g_object_ref (__temp7)));
	if (VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (msym))) {
		ValaVariableDeclarator* __temp8 = NULL;
		ValaVariableDeclarator* decl = (__temp8 = VALA_VARIABLE_DECLARATOR (vala_symbol_get_node (msym)), (__temp8 == NULL ? NULL : g_object_ref (__temp8)));
		ValaCallback* __temp9 = NULL;
		ValaCallback* cb = (__temp9 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp9 == NULL ? NULL : g_object_ref (__temp9)));
		GList* __temp10 = NULL;
		params = (__temp10 = vala_callback_get_parameters (cb), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp10);
		(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	} else {
		if (VALA_IS_FORMAL_PARAMETER (vala_symbol_get_node (msym))) {
			ValaFormalParameter* __temp11 = NULL;
			ValaFormalParameter* param = (__temp11 = VALA_FORMAL_PARAMETER (vala_symbol_get_node (msym)), (__temp11 == NULL ? NULL : g_object_ref (__temp11)));
			ValaCallback* __temp12 = NULL;
			ValaCallback* cb = (__temp12 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))), (__temp12 == NULL ? NULL : g_object_ref (__temp12)));
			GList* __temp13 = NULL;
			params = (__temp13 = vala_callback_get_parameters (cb), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp13);
			(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		} else {
			if (VALA_IS_FIELD (vala_symbol_get_node (msym))) {
				ValaField* __temp14 = NULL;
				ValaField* f = (__temp14 = VALA_FIELD (vala_symbol_get_node (msym)), (__temp14 == NULL ? NULL : g_object_ref (__temp14)));
				ValaCallback* __temp15 = NULL;
				ValaCallback* cb = (__temp15 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (__temp15 == NULL ? NULL : g_object_ref (__temp15)));
				GList* __temp16 = NULL;
				params = (__temp16 = vala_callback_get_parameters (cb), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp16);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				if (VALA_IS_METHOD (vala_symbol_get_node (msym))) {
					ValaMethod* __temp17 = NULL;
					ValaMethod* m = (__temp17 = VALA_METHOD (vala_symbol_get_node (msym)), (__temp17 == NULL ? NULL : g_object_ref (__temp17)));
					GList* __temp18 = NULL;
					params = (__temp18 = vala_invokable_get_parameters (VALA_INVOKABLE (m)), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp18);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (VALA_IS_SIGNAL (vala_symbol_get_node (msym))) {
						ValaSignal* __temp19 = NULL;
						ValaSignal* sig = (__temp19 = VALA_SIGNAL (vala_symbol_get_node (msym)), (__temp19 == NULL ? NULL : g_object_ref (__temp19)));
						GList* __temp20 = NULL;
						params = (__temp20 = vala_invokable_get_parameters (VALA_INVOKABLE (sig)), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp20);
						(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
					}
				}
			}
		}
	}
	GList* params_it = params;
	{
		GList* __temp41 = NULL;
		__temp41 = vala_invocation_expression_get_argument_list (expr);
		GList* arg_it;
		for (arg_it = __temp41; arg_it != NULL; arg_it = arg_it->next) {
			ValaExpression* arg = arg_it->data;
			{
				if (params_it != NULL) {
					ValaFormalParameter* __temp21 = NULL;
					ValaFormalParameter* param = (__temp21 = VALA_FORMAL_PARAMETER (params_it->data), (__temp21 == NULL ? NULL : g_object_ref (__temp21)));
					if (!vala_formal_parameter_get_ellipsis (param) && ((vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) || vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param)) != NULL)) {
						gboolean is_ref = vala_type_reference_get_takes_ownership (vala_formal_parameter_get_type_reference (param));
						if (is_ref && vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param)) != NULL) {
							/* TODO move this to semantic analyzer*/
							if (VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr))) {
								ValaMemberAccess* __temp22 = NULL;
								ValaMemberAccess* ma = (__temp22 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (__temp22 == NULL ? NULL : g_object_ref (__temp22)));
								ValaTypeReference* __temp23 = NULL;
								ValaTypeReference* instance_type = (__temp23 = vala_expression_get_static_type (vala_member_access_get_inner (ma)), (__temp23 == NULL ? NULL : g_object_ref (__temp23)));
								while (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
									/* trace type arguments back to the datatype where the method has been declared*/
									GList* base_types = NULL;
									if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
										ValaClass* __temp24 = NULL;
										ValaClass* cl = (__temp24 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (__temp24 == NULL ? NULL : g_object_ref (__temp24)));
										GList* __temp25 = NULL;
										base_types = (__temp25 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp25);
										(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
									} else {
										if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
											ValaInterface* __temp26 = NULL;
											ValaInterface* iface = (__temp26 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (__temp26 == NULL ? NULL : g_object_ref (__temp26)));
											GList* __temp27 = NULL;
											base_types = (__temp27 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp27);
											(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
										} else {
											vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported generic type");
											(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
											(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
											(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
											(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
											(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
											(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
											(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
											return;
										}
									}
									{
										GList* __temp35 = NULL;
										__temp35 = base_types;
										GList* base_type_it;
										for (base_type_it = __temp35; base_type_it != NULL; base_type_it = base_type_it->next) {
											ValaTypeReference* base_type = base_type_it->data;
											{
												ValaSymbol* __temp28 = NULL;
												gboolean __temp29;
												if ((__temp29 = (__temp28 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), vala_symbol_get_name (msym))) != NULL, (__temp28 == NULL ? NULL : (__temp28 = (g_object_unref (__temp28), NULL))), __temp29)) {
													/* construct a new type reference for the base type with correctly linked type arguments*/
													ValaTypeReference* instance_base_type = vala_type_reference_new ();
													(vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type)), vala_type_reference_get_data_type (instance_base_type));
													{
														GList* __temp32 = NULL;
														__temp32 = vala_type_reference_get_type_arguments (base_type);
														GList* type_arg_it;
														for (type_arg_it = __temp32; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
															ValaTypeReference* type_arg = type_arg_it->data;
															{
																if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
																	/* link to type argument of derived type*/
																	gint param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)));
																	if (param_index == -1) {
																		char* __temp30 = NULL;
																		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp30 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)))));
																		(__temp30 = (g_free (__temp30), NULL));
																		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
																		(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
																		(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
																		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
																		(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
																		(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
																		(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
																		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
																		return;
																	}
																	GList* __temp31 = NULL;
																	type_arg = g_list_nth_data ((__temp31 = vala_type_reference_get_type_arguments (instance_type)), param_index);
																	(__temp31 == NULL ? NULL : (__temp31 = (g_list_free (__temp31), NULL)));
																}
																vala_type_reference_add_type_argument (instance_base_type, type_arg);
															}
														}
														(__temp32 == NULL ? NULL : (__temp32 = (g_list_free (__temp32), NULL)));
													}
													ValaTypeReference* __temp34 = NULL;
													ValaTypeReference* __temp33 = NULL;
													instance_type = (__temp34 = (__temp33 = instance_base_type, (__temp33 == NULL ? NULL : g_object_ref (__temp33))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp34);
													(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
												}
											}
										}
									}
									(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
								}
								if (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: generic type parameter tracing not supported yet");
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
									(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
									(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
									(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
									return;
								}
								gint param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param))));
								if (param_index == -1) {
									char* __temp36 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp36 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param))))));
									(__temp36 = (g_free (__temp36), NULL));
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
									(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
									(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
									(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
									return;
								}
								ValaTypeReference* __temp38 = NULL;
								GList* __temp37 = NULL;
								ValaTypeReference* __temp39 = NULL;
								ValaTypeReference* param_type = (__temp39 = (__temp38 = VALA_TYPE_REFERENCE (g_list_nth_data ((__temp37 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (__temp38 == NULL ? NULL : g_object_ref (__temp38))), (__temp37 == NULL ? NULL : (__temp37 = (g_list_free (__temp37), NULL))), __temp39);
								if (param_type == NULL) {
									char* __temp40 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp40 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param))))));
									(__temp40 = (g_free (__temp40), NULL));
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
									(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
									(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
									(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
									(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
									return;
								}
								is_ref = vala_type_reference_get_takes_ownership (param_type);
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
								(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
							}
						}
						if (is_ref) {
							vala_memory_manager_visit_possibly_missing_copy_expression (self, arg);
						} else {
							vala_memory_manager_visit_possibly_leaked_expression (self, arg);
						}
					} else {
						vala_memory_manager_visit_possibly_leaked_expression (self, arg);
					}
					params_it = params_it->next;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				} else {
					vala_memory_manager_visit_possibly_leaked_expression (self, arg);
				}
			}
		}
		(__temp41 == NULL ? NULL : (__temp41 = (g_list_free (__temp41), NULL)));
	}
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
}


static void vala_memory_manager_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_binary_expression_get_left (expr));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_binary_expression_get_right (expr));
}


static void vala_memory_manager_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a)) || (vala_expression_get_symbol_reference (vala_assignment_get_left (a)) != NULL && VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_assignment_get_left (a)))))) {
	} else {
		if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
			vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_assignment_get_right (a));
		} else {
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_assignment_get_right (a));
		}
	}
}


static void vala_memory_manager_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (object);
	switch (property_id) {
	}
}


static void vala_memory_manager_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (object);
	switch (property_id) {
	}
}


static void vala_memory_manager_class_init (ValaMemoryManagerClass * klass)
{
	vala_memory_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMemoryManagerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_memory_manager_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_memory_manager_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_memory_manager_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_memory_manager_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_method = vala_memory_manager_real_visit_begin_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_creation_method = vala_memory_manager_real_visit_begin_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_property = vala_memory_manager_real_visit_begin_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_memory_manager_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_memory_manager_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_memory_manager_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_memory_manager_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_memory_manager_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_memory_manager_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_memory_manager_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_memory_manager_real_visit_end_assignment;
}


static void vala_memory_manager_init (ValaMemoryManager * self)
{
	self->priv = VALA_MEMORY_MANAGER_GET_PRIVATE (self);
}


static void vala_memory_manager_dispose (GObject * obj)
{
	ValaMemoryManager * self = VALA_MEMORY_MANAGER (obj);
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	ValaMemoryManagerClass * klass;
	GObjectClass * parent_class;
	klass = VALA_MEMORY_MANAGER_CLASS (g_type_class_peek (VALA_TYPE_MEMORY_MANAGER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_memory_manager_get_type ()
{
	static GType vala_memory_manager_type_id = 0;
	if (G_UNLIKELY (vala_memory_manager_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemoryManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_memory_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemoryManager), 0, (GInstanceInitFunc) vala_memory_manager_init };
		vala_memory_manager_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaMemoryManager", &g_define_type_info, 0);
	}
	return vala_memory_manager_type_id;
}




