/* valaassignment.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valaassignment.h"
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaAssignmentPrivate {
	ValaAssignmentOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};
#define VALA_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ASSIGNMENT, ValaAssignmentPrivate))
enum  {
	VALA_ASSIGNMENT_DUMMY_PROPERTY,
	VALA_ASSIGNMENT_LEFT,
	VALA_ASSIGNMENT_OPERATOR,
	VALA_ASSIGNMENT_RIGHT,
};
static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node);
static gpointer vala_assignment_parent_class = NULL;
static void vala_assignment_dispose (GObject * obj);


/**
 * Creates a new assignment.
 *
 * @param left left hand side
 * @param op assignment operator
 * @param right right hand side
 * @param source reference to source code
 * @return newly created assignment
 */
ValaAssignment* vala_assignment_new (ValaExpression* _left, ValaExpression* _right, ValaAssignmentOperator _op, ValaSourceReference* _source)
{
	g_return_val_if_fail (VALA_IS_EXPRESSION (_left), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (_right), NULL);
	g_return_val_if_fail (_source == NULL || VALA_IS_SOURCE_REFERENCE (_source), NULL);
	GParameter * __params = g_new0 (GParameter, 4);
	GParameter * __params_it = __params;
	(__params_it->name = "left", g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION), g_value_set_object (&__params_it->value, _left), __params_it++);
	(__params_it->name = "operator", g_value_init (&__params_it->value, G_TYPE_INT), g_value_set_int (&__params_it->value, _op), __params_it++);
	(__params_it->name = "right", g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION), g_value_set_object (&__params_it->value, _right), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, _source), __params_it++);
	ValaAssignment * self = g_object_newv (VALA_TYPE_ASSIGNMENT, __params_it - __params, __params);
	return self;
}


static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaAssignment * self = VALA_ASSIGNMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_left (self)), visitor);
	vala_code_visitor_visit_begin_assignment (visitor, self);
	vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (self)), visitor);
	vala_code_visitor_visit_end_assignment (visitor, self);
}


static void vala_assignment_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node)
{
	ValaAssignment * self = VALA_ASSIGNMENT (base);
	g_return_if_fail (VALA_IS_CODE_NODE (old_node));
	g_return_if_fail (VALA_IS_CODE_NODE (new_node));
	if (VALA_CODE_NODE (vala_assignment_get_left (self)) == old_node) {
		(vala_assignment_set_left (self, VALA_EXPRESSION (new_node)), vala_assignment_get_left (self));
	}
	if (VALA_CODE_NODE (vala_assignment_get_right (self)) == old_node) {
		(vala_assignment_set_right (self, VALA_EXPRESSION (new_node)), vala_assignment_get_right (self));
	}
}


ValaExpression* vala_assignment_get_left (ValaAssignment* self)
{
	g_return_val_if_fail (VALA_IS_ASSIGNMENT (self), NULL);
	return self->priv->_left;
}


void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value)
{
	g_return_if_fail (VALA_IS_ASSIGNMENT (self));
	ValaExpression* __temp1 = NULL;
	self->priv->_left = (__temp1 = g_object_ref (value), (self->priv->_left == NULL ? NULL : (self->priv->_left = (g_object_unref (self->priv->_left), NULL))), __temp1);
	(vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_left), VALA_CODE_NODE (self)), vala_code_node_get_parent_node (VALA_CODE_NODE (self->priv->_left)));
}


ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self)
{
	g_return_val_if_fail (VALA_IS_ASSIGNMENT (self), 0);
	return self->priv->_operator;
}


void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value)
{
	g_return_if_fail (VALA_IS_ASSIGNMENT (self));
	self->priv->_operator = value;
}


ValaExpression* vala_assignment_get_right (ValaAssignment* self)
{
	g_return_val_if_fail (VALA_IS_ASSIGNMENT (self), NULL);
	return self->priv->_right;
}


void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value)
{
	g_return_if_fail (VALA_IS_ASSIGNMENT (self));
	ValaExpression* __temp4 = NULL;
	self->priv->_right = (__temp4 = g_object_ref (value), (self->priv->_right == NULL ? NULL : (self->priv->_right = (g_object_unref (self->priv->_right), NULL))), __temp4);
	(vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_right), VALA_CODE_NODE (self)), vala_code_node_get_parent_node (VALA_CODE_NODE (self->priv->_right)));
}


static void vala_assignment_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaAssignment * self = VALA_ASSIGNMENT (object);
	switch (property_id) {
		case VALA_ASSIGNMENT_LEFT:
		g_value_set_object (value, vala_assignment_get_left (self));
		break;
		case VALA_ASSIGNMENT_OPERATOR:
		g_value_set_int (value, vala_assignment_get_operator (self));
		break;
		case VALA_ASSIGNMENT_RIGHT:
		g_value_set_object (value, vala_assignment_get_right (self));
		break;
	}
}


static void vala_assignment_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaAssignment * self = VALA_ASSIGNMENT (object);
	switch (property_id) {
		case VALA_ASSIGNMENT_LEFT:
		vala_assignment_set_left (self, g_value_get_object (value));
		break;
		case VALA_ASSIGNMENT_OPERATOR:
		vala_assignment_set_operator (self, g_value_get_int (value));
		break;
		case VALA_ASSIGNMENT_RIGHT:
		vala_assignment_set_right (self, g_value_get_object (value));
		break;
	}
}


static void vala_assignment_class_init (ValaAssignmentClass * klass)
{
	vala_assignment_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaAssignmentPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_assignment_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_assignment_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_assignment_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_assignment_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace = vala_assignment_real_replace;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ASSIGNMENT_LEFT, g_param_spec_object ("left", "foo", "bar", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ASSIGNMENT_OPERATOR, g_param_spec_int ("operator", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ASSIGNMENT_RIGHT, g_param_spec_object ("right", "foo", "bar", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}


static void vala_assignment_init (ValaAssignment * self)
{
	self->priv = VALA_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_assignment_dispose (GObject * obj)
{
	ValaAssignment * self = VALA_ASSIGNMENT (obj);
	(self->priv->_left == NULL ? NULL : (self->priv->_left = (g_object_unref (self->priv->_left), NULL)));
	(self->priv->_right == NULL ? NULL : (self->priv->_right = (g_object_unref (self->priv->_right), NULL)));
	ValaAssignmentClass * klass;
	GObjectClass * parent_class;
	klass = VALA_ASSIGNMENT_CLASS (g_type_class_peek (VALA_TYPE_ASSIGNMENT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_assignment_get_type ()
{
	static GType vala_assignment_type_id = 0;
	if (G_UNLIKELY (vala_assignment_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAssignment), 0, (GInstanceInitFunc) vala_assignment_init };
		vala_assignment_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaAssignment", &g_define_type_info, 0);
	}
	return vala_assignment_type_id;
}




