/*
 * Copyright (C) 2008, 2009 Nokia Corporation, all rights reserved.
 * Copyright (C) 2006, 2007, 2008 OpenedHand Ltd.
 *
 * Author: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *                               <zeeshan.ali@nokia.com>
 *         Jorn Baayen <jorn.baayen@gmail.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "rygel-http-request.h"
#include <stdlib.h>
#include <string.h>
#include <gst/gst.h>
#include <gee/collection.h>
#include <gee/list.h>
#include "rygel-http-response.h"
#include "rygel-media-item.h"
#include "rygel-media-object.h"
#include "rygel-seekable-response.h"
#include "rygel-live-response.h"




struct _RygelHTTPRequestPrivate {
	RygelMediaContainer* root_container;
	SoupServer* server;
	SoupMessage* msg;
	GHashTable* query;
	RygelHTTPResponse* response;
	char* item_id;
	RygelMediaItem* item;
	RygelSeek* seek;
	GCancellable* cancellable;
};

#define RYGEL_HTTP_REQUEST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), RYGEL_TYPE_HTTP_REQUEST, RygelHTTPRequestPrivate))
enum  {
	RYGEL_HTTP_REQUEST_DUMMY_PROPERTY
};
static void _rygel_http_request_on_item_found_gasync_ready_callback (GObject* source_object, GAsyncResult* res, gpointer self);
static void rygel_http_request_real_run (RygelStateMachine* base, GCancellable* cancellable);
static void _rygel_http_request_on_response_completed_rygel_state_machine_completed (RygelLiveResponse* _sender, gpointer self);
static void rygel_http_request_stream_from_gst_source (RygelHTTPRequest* self, GstElement* src, GError** error);
static void rygel_http_request_serve_uri (RygelHTTPRequest* self, const char* uri, gsize size);
static void rygel_http_request_on_response_completed (RygelHTTPRequest* self, RygelHTTPResponse* response);
static void rygel_http_request_handle_item_request (RygelHTTPRequest* self);
static void rygel_http_request_add_item_headers (RygelHTTPRequest* self);
static inline void _dynamic_set_tcp_timeout0 (GstElement* obj, gint64 value);
static void rygel_http_request_handle_streaming_item (RygelHTTPRequest* self, const char* uri);
static void rygel_http_request_handle_interactive_item (RygelHTTPRequest* self, const char* uri);
static void rygel_http_request_parse_range (RygelHTTPRequest* self, GError** error);
static void rygel_http_request_on_item_found (RygelHTTPRequest* self, GObject* source_object, GAsyncResult* res);
static void rygel_http_request_handle_error (RygelHTTPRequest* self, GError* error);
static gpointer rygel_http_request_parent_class = NULL;
static RygelStateMachineIface* rygel_http_request_rygel_state_machine_parent_iface = NULL;
static void rygel_http_request_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



GQuark rygel_http_request_error_quark (void) {
	return g_quark_from_static_string ("rygel_http_request_error-quark");
}


RygelHTTPRequest* rygel_http_request_construct (GType object_type, RygelMediaContainer* root_container, SoupServer* server, SoupMessage* msg, GHashTable* query) {
	RygelHTTPRequest * self;
	RygelMediaContainer* _tmp1;
	RygelMediaContainer* _tmp0;
	SoupServer* _tmp3;
	SoupServer* _tmp2;
	SoupMessage* _tmp5;
	SoupMessage* _tmp4;
	GHashTable* _tmp7;
	GHashTable* _tmp6;
	g_return_val_if_fail (root_container != NULL, NULL);
	g_return_val_if_fail (server != NULL, NULL);
	g_return_val_if_fail (msg != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->root_container = (_tmp1 = (_tmp0 = root_container, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->root_container == NULL) ? NULL : (self->priv->root_container = (g_object_unref (self->priv->root_container), NULL)), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->server = (_tmp3 = (_tmp2 = server, (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (self->priv->server == NULL) ? NULL : (self->priv->server = (g_object_unref (self->priv->server), NULL)), _tmp3);
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->msg = (_tmp5 = (_tmp4 = msg, (_tmp4 == NULL) ? NULL : g_object_ref (_tmp4)), (self->priv->msg == NULL) ? NULL : (self->priv->msg = (g_object_unref (self->priv->msg), NULL)), _tmp5);
	_tmp7 = NULL;
	_tmp6 = NULL;
	self->priv->query = (_tmp7 = (_tmp6 = query, (_tmp6 == NULL) ? NULL : g_hash_table_ref (_tmp6)), (self->priv->query == NULL) ? NULL : (self->priv->query = (g_hash_table_unref (self->priv->query), NULL)), _tmp7);
	soup_server_pause_message (self->priv->server, self->priv->msg);
	return self;
}


RygelHTTPRequest* rygel_http_request_new (RygelMediaContainer* root_container, SoupServer* server, SoupMessage* msg, GHashTable* query) {
	return rygel_http_request_construct (RYGEL_TYPE_HTTP_REQUEST, root_container, server, msg, query);
}


static void _rygel_http_request_on_item_found_gasync_ready_callback (GObject* source_object, GAsyncResult* res, gpointer self) {
	rygel_http_request_on_item_found (self, source_object, res);
}


static void rygel_http_request_real_run (RygelStateMachine* base, GCancellable* cancellable) {
	RygelHTTPRequest * self;
	GCancellable* _tmp1;
	GCancellable* _tmp0;
	gboolean _tmp2;
	char* _tmp4;
	char* _tmp3;
	gboolean _tmp5;
	self = (RygelHTTPRequest*) base;
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cancellable = (_tmp1 = (_tmp0 = cancellable, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->cancellable == NULL) ? NULL : (self->priv->cancellable = (g_object_unref (self->priv->cancellable), NULL)), _tmp1);
	_tmp2 = FALSE;
	_tmp4 = NULL;
	_tmp3 = NULL;
	if ((_tmp5 = _vala_strcmp0 (_tmp4 = (g_object_get (self->priv->msg, "method", &_tmp3, NULL), _tmp3), "HEAD") != 0, _tmp4 = (g_free (_tmp4), NULL), _tmp5)) {
		char* _tmp7;
		char* _tmp6;
		_tmp7 = NULL;
		_tmp6 = NULL;
		_tmp2 = _vala_strcmp0 (_tmp7 = (g_object_get (self->priv->msg, "method", &_tmp6, NULL), _tmp6), "GET") != 0;
		_tmp7 = (g_free (_tmp7), NULL);
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		GError* _tmp8;
		/* We only entertain 'HEAD' and 'GET' requests */
		_tmp8 = NULL;
		rygel_http_request_handle_error (self, _tmp8 = g_error_new_literal (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_BAD_REQUEST, "Invalid Request"));
		(_tmp8 == NULL) ? NULL : (_tmp8 = (g_error_free (_tmp8), NULL));
		return;
	}
	if (self->priv->query != NULL) {
		char* _tmp10;
		const char* _tmp9;
		_tmp10 = NULL;
		_tmp9 = NULL;
		self->priv->item_id = (_tmp10 = (_tmp9 = (const char*) g_hash_table_lookup (self->priv->query, "itemid"), (_tmp9 == NULL) ? NULL : g_strdup (_tmp9)), self->priv->item_id = (g_free (self->priv->item_id), NULL), _tmp10);
	}
	if (self->priv->item_id == NULL) {
		GError* _tmp11;
		_tmp11 = NULL;
		rygel_http_request_handle_error (self, _tmp11 = g_error_new_literal (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_NOT_FOUND, "Not Found"));
		(_tmp11 == NULL) ? NULL : (_tmp11 = (g_error_free (_tmp11), NULL));
		return;
	}
	/* Fetch the requested item*/
	rygel_media_container_find_object (self->priv->root_container, self->priv->item_id, NULL, _rygel_http_request_on_item_found_gasync_ready_callback, self);
}


static void _rygel_http_request_on_response_completed_rygel_state_machine_completed (RygelLiveResponse* _sender, gpointer self) {
	rygel_http_request_on_response_completed (self, _sender);
}


static void rygel_http_request_stream_from_gst_source (RygelHTTPRequest* self, GstElement* src, GError** error) {
	GError * inner_error;
	RygelLiveResponse* response;
	RygelHTTPResponse* _tmp1;
	RygelHTTPResponse* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (src != NULL);
	inner_error = NULL;
	response = rygel_live_response_new (self->priv->server, self->priv->msg, "RygelLiveResponse", src, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
		return;
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->response = (_tmp1 = (_tmp0 = (RygelHTTPResponse*) response, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->response == NULL) ? NULL : (self->priv->response = (g_object_unref (self->priv->response), NULL)), _tmp1);
	g_signal_connect_object ((RygelStateMachine*) response, "completed", (GCallback) _rygel_http_request_on_response_completed_rygel_state_machine_completed, self, 0);
	rygel_http_response_run ((RygelHTTPResponse*) response, self->priv->cancellable);
	(response == NULL) ? NULL : (response = (g_object_unref (response), NULL));
	(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
}


static void rygel_http_request_serve_uri (RygelHTTPRequest* self, const char* uri, gsize size) {
	RygelSeekableResponse* response;
	RygelHTTPResponse* _tmp1;
	RygelHTTPResponse* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (uri != NULL);
	response = rygel_seekable_response_new (self->priv->server, self->priv->msg, uri, self->priv->seek, size);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->response = (_tmp1 = (_tmp0 = (RygelHTTPResponse*) response, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->response == NULL) ? NULL : (self->priv->response = (g_object_unref (self->priv->response), NULL)), _tmp1);
	g_signal_connect_object ((RygelStateMachine*) response, "completed", (GCallback) _rygel_http_request_on_response_completed_rygel_state_machine_completed, self, 0);
	rygel_http_response_run ((RygelHTTPResponse*) response, self->priv->cancellable);
	(response == NULL) ? NULL : (response = (g_object_unref (response), NULL));
}


static void rygel_http_request_on_response_completed (RygelHTTPRequest* self, RygelHTTPResponse* response) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (response != NULL);
	rygel_http_request_end (self, (guint) SOUP_STATUS_NONE);
}


static void rygel_http_request_handle_item_request (RygelHTTPRequest* self) {
	GError * inner_error;
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	char* uri;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	{
		rygel_http_request_parse_range (self, &inner_error);
		if (inner_error != NULL) {
			goto __catch0_g_error;
			goto __finally0;
		}
	}
	goto __finally0;
	__catch0_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			rygel_http_request_handle_error (self, error);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
			return;
		}
	}
	__finally0:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	/* Add headers*/
	rygel_http_request_add_item_headers (self);
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = _vala_strcmp0 (_tmp1 = (g_object_get (self->priv->msg, "method", &_tmp0, NULL), _tmp0), "HEAD") == 0, _tmp1 = (g_free (_tmp1), NULL), _tmp2)) {
		/* Only headers requested, no need to send contents*/
		rygel_http_request_end (self, (guint) SOUP_STATUS_OK);
		return;
	}
	/* Just use the first URI available*/
	uri = NULL;
	if (gee_collection_get_size ((GeeCollection*) self->priv->item->uris) != 0) {
		char* _tmp3;
		_tmp3 = NULL;
		uri = (_tmp3 = (char*) gee_list_get ((GeeList*) self->priv->item->uris, 0), uri = (g_free (uri), NULL), _tmp3);
	}
	if (self->priv->item->size > 0) {
		rygel_http_request_handle_interactive_item (self, uri);
	} else {
		rygel_http_request_handle_streaming_item (self, uri);
	}
	uri = (g_free (uri), NULL);
}


static void rygel_http_request_add_item_headers (RygelHTTPRequest* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->item->mime_type != NULL) {
		soup_message_headers_append (self->priv->msg->response_headers, "Content-Type", self->priv->item->mime_type);
	}
	if (self->priv->item->size >= 0) {
		soup_message_headers_set_content_length (self->priv->msg->response_headers, (gint64) self->priv->item->size);
	}
	if (self->priv->item->size > 0) {
		gint64 first_byte;
		gint64 last_byte;
		char* _tmp6;
		char* _tmp5;
		char* _tmp4;
		char* _tmp3;
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		char* _tmp7;
		char* content_range;
		first_byte = 0LL;
		last_byte = 0LL;
		if (self->priv->seek != NULL) {
			first_byte = rygel_seek_get_start (self->priv->seek);
			last_byte = rygel_seek_get_stop (self->priv->seek);
		} else {
			first_byte = (gint64) 0;
			last_byte = (gint64) (self->priv->item->size - 1);
		}
		/* Content-Range: bytes START_BYTE-STOP_BYTE/TOTAL_LENGTH*/
		_tmp6 = NULL;
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp3 = NULL;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		_tmp7 = NULL;
		content_range = (_tmp7 = g_strconcat (_tmp5 = g_strconcat (_tmp4 = g_strconcat (_tmp2 = g_strconcat (_tmp1 = g_strconcat ("bytes ", _tmp0 = g_strdup_printf ("%lli", first_byte), NULL), "-", NULL), _tmp3 = g_strdup_printf ("%lli", last_byte), NULL), "/", NULL), _tmp6 = g_strdup_printf ("%li", self->priv->item->size), NULL), _tmp6 = (g_free (_tmp6), NULL), _tmp5 = (g_free (_tmp5), NULL), _tmp4 = (g_free (_tmp4), NULL), _tmp3 = (g_free (_tmp3), NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp7);
		soup_message_headers_append (self->priv->msg->response_headers, "Content-Range", content_range);
		soup_message_headers_append (self->priv->msg->response_headers, "Accept-Ranges", "bytes");
		content_range = (g_free (content_range), NULL);
	}
}


static inline void _dynamic_set_tcp_timeout0 (GstElement* obj, gint64 value) {
	g_object_set (obj, "tcp-timeout", value, NULL);
}


static void rygel_http_request_handle_streaming_item (RygelHTTPRequest* self, const char* uri) {
	GError * inner_error;
	GstElement* src;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	src = NULL;
	if (uri != NULL) {
		GstElement* _tmp0;
		/* URI provided, try to create source element from it*/
		_tmp0 = NULL;
		src = (_tmp0 = gst_element_make_from_uri (GST_URI_SRC, uri, NULL), (src == NULL) ? NULL : (src = (gst_object_unref (src), NULL)), _tmp0);
	} else {
		GstElement* _tmp1;
		/* No URI provided, ask for source element*/
		_tmp1 = NULL;
		src = (_tmp1 = rygel_media_item_create_stream_source (self->priv->item), (src == NULL) ? NULL : (src = (gst_object_unref (src), NULL)), _tmp1);
	}
	if (src == NULL) {
		GError* _tmp2;
		_tmp2 = NULL;
		rygel_http_request_handle_error (self, _tmp2 = g_error_new_literal (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_NOT_FOUND, "Not Found"));
		(_tmp2 == NULL) ? NULL : (_tmp2 = (g_error_free (_tmp2), NULL));
		(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
		return;
	}
	/* For rtspsrc since some RTSP sources takes a while to start
	 transmitting*/
	_dynamic_set_tcp_timeout0 (src, (gint64) 60000000);
	{
		GstElement* _tmp3;
		/* Then start the gst stream*/
		_tmp3 = NULL;
		rygel_http_request_stream_from_gst_source (self, (_tmp3 = src, (_tmp3 == NULL) ? NULL : gst_object_ref (_tmp3)), &inner_error);
		if (inner_error != NULL) {
			goto __catch1_g_error;
			goto __finally1;
		}
	}
	goto __finally1;
	__catch1_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			rygel_http_request_handle_error (self, error);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
			(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
			return;
		}
	}
	__finally1:
	if (inner_error != NULL) {
		(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
}


static void rygel_http_request_handle_interactive_item (RygelHTTPRequest* self, const char* uri) {
	g_return_if_fail (self != NULL);
	if (uri == NULL) {
		GError* error;
		error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_NOT_FOUND, "Requested item '%s' didn't provide a URI\n", ((RygelMediaObject*) self->priv->item)->id);
		rygel_http_request_handle_error (self, error);
		(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		return;
	}
	rygel_http_request_serve_uri (self, uri, (gsize) self->priv->item->size);
}


static void rygel_http_request_parse_range (RygelHTTPRequest* self, GError** error) {
	GError * inner_error;
	char* range;
	gint range_tokens_size;
	gint range_tokens_length1;
	char** range_tokens;
	char* _tmp1;
	const char* _tmp0;
	char** _tmp3;
	char** _tmp2;
	gboolean _tmp4;
	RygelSeek* _tmp5;
	const char* _tmp6;
	char* first_byte;
	const char* _tmp7;
	char* last_byte;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	range = NULL;
	range_tokens = (range_tokens_length1 = 0, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	range = (_tmp1 = (_tmp0 = soup_message_headers_get (self->priv->msg->request_headers, "Range"), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), range = (g_free (range), NULL), _tmp1);
	if (range == NULL) {
		range = (g_free (range), NULL);
		range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
		return;
	}
	/* We have a Range header. Parse.*/
	if (!g_str_has_prefix (range, "bytes=")) {
		inner_error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			range = (g_free (range), NULL);
			range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
			return;
		}
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	range_tokens = (_tmp3 = _tmp2 = g_strsplit (g_utf8_offset_to_pointer (range, (glong) 6), "-", 2), range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL), range_tokens_length1 = _vala_array_length (_tmp2), range_tokens_size = range_tokens_length1, _tmp3);
	_tmp4 = FALSE;
	if (range_tokens[0] == NULL) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = range_tokens[1] == NULL;
	}
	if (_tmp4) {
		inner_error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			range = (g_free (range), NULL);
			range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
			return;
		}
	}
	_tmp5 = NULL;
	self->priv->seek = (_tmp5 = rygel_seek_new (GST_FORMAT_BYTES, (gint64) 0, (gint64) (self->priv->item->size - 1)), (self->priv->seek == NULL) ? NULL : (self->priv->seek = (g_object_unref (self->priv->seek), NULL)), _tmp5);
	/* Get first byte position*/
	_tmp6 = NULL;
	first_byte = (_tmp6 = range_tokens[0], (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
	if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (first_byte, 0)))) {
		rygel_seek_set_start (self->priv->seek, g_ascii_strtoll (first_byte, NULL, 0));
	} else {
		if (_vala_strcmp0 (first_byte, "") != 0) {
			inner_error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				first_byte = (g_free (first_byte), NULL);
				return;
			}
		}
	}
	/* Get last byte position if specified*/
	_tmp7 = NULL;
	last_byte = (_tmp7 = range_tokens[1], (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
	if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (last_byte, 0)))) {
		rygel_seek_set_stop (self->priv->seek, g_ascii_strtoll (last_byte, NULL, 0));
	} else {
		if (_vala_strcmp0 (last_byte, "") != 0) {
			inner_error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				first_byte = (g_free (first_byte), NULL);
				last_byte = (g_free (last_byte), NULL);
				return;
			}
		}
	}
	if (self->priv->item->size > 0) {
		gboolean _tmp8;
		gboolean _tmp9;
		_tmp8 = FALSE;
		if (rygel_seek_get_start (self->priv->seek) > self->priv->item->size) {
			_tmp8 = TRUE;
		} else {
			_tmp8 = rygel_seek_get_length (self->priv->seek) > self->priv->item->size;
		}
		/* shouldn't go beyond actual length of media*/
		if (_tmp8) {
			inner_error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_OUT_OF_RANGE, "Range '%s' not setsifiable", range);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				first_byte = (g_free (first_byte), NULL);
				last_byte = (g_free (last_byte), NULL);
				return;
			}
		}
		_tmp9 = FALSE;
		if (rygel_seek_get_start (self->priv->seek) == 0) {
			_tmp9 = rygel_seek_get_length (self->priv->seek) == self->priv->item->size;
		} else {
			_tmp9 = FALSE;
		}
		/* No need to seek if whole stream is requested*/
		if (_tmp9) {
			self->priv->seek == NULL;
			range = (g_free (range), NULL);
			range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
			first_byte = (g_free (first_byte), NULL);
			last_byte = (g_free (last_byte), NULL);
			return;
		}
	} else {
		if (rygel_seek_get_start (self->priv->seek) == 0) {
			/* Might be an attempt to get the size, in which case it's not
			 an error. Just don't seek.*/
			self->priv->seek == NULL;
			range = (g_free (range), NULL);
			range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
			first_byte = (g_free (first_byte), NULL);
			last_byte = (g_free (last_byte), NULL);
			return;
		} else {
			inner_error = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_UNACCEPTABLE, "Partial download not applicable for item %s", ((RygelMediaObject*) self->priv->item)->id);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				first_byte = (g_free (first_byte), NULL);
				last_byte = (g_free (last_byte), NULL);
				return;
			}
		}
	}
	range = (g_free (range), NULL);
	range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
	first_byte = (g_free (first_byte), NULL);
	last_byte = (g_free (last_byte), NULL);
}


static void rygel_http_request_on_item_found (RygelHTTPRequest* self, GObject* source_object, GAsyncResult* res) {
	GError * inner_error;
	RygelMediaContainer* _tmp0;
	RygelMediaContainer* container;
	RygelMediaObject* media_object;
	gboolean _tmp3;
	RygelMediaItem* _tmp6;
	RygelMediaItem* _tmp5;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_object != NULL);
	g_return_if_fail (res != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	container = (_tmp0 = RYGEL_MEDIA_CONTAINER (source_object), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	media_object = NULL;
	{
		RygelMediaObject* _tmp1;
		RygelMediaObject* _tmp2;
		_tmp1 = rygel_media_container_find_object_finish (container, res, &inner_error);
		if (inner_error != NULL) {
			goto __catch2_g_error;
			goto __finally2;
		}
		_tmp2 = NULL;
		media_object = (_tmp2 = _tmp1, (media_object == NULL) ? NULL : (media_object = (g_object_unref (media_object), NULL)), _tmp2);
	}
	goto __finally2;
	__catch2_g_error:
	{
		GError * err;
		err = inner_error;
		inner_error = NULL;
		{
			rygel_http_request_handle_error (self, err);
			(err == NULL) ? NULL : (err = (g_error_free (err), NULL));
			(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
			(media_object == NULL) ? NULL : (media_object = (g_object_unref (media_object), NULL));
			return;
		}
	}
	__finally2:
	if (inner_error != NULL) {
		(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
		(media_object == NULL) ? NULL : (media_object = (g_object_unref (media_object), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	_tmp3 = FALSE;
	if (media_object == NULL) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = !RYGEL_IS_MEDIA_ITEM (media_object);
	}
	if (_tmp3) {
		GError* _tmp4;
		_tmp4 = NULL;
		rygel_http_request_handle_error (self, _tmp4 = g_error_new (RYGEL_HTTP_REQUEST_ERROR, RYGEL_HTTP_REQUEST_ERROR_NOT_FOUND, "requested item '%s' not found", self->priv->item_id));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (g_error_free (_tmp4), NULL));
		(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
		(media_object == NULL) ? NULL : (media_object = (g_object_unref (media_object), NULL));
		return;
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->item = (_tmp6 = (_tmp5 = RYGEL_MEDIA_ITEM (media_object), (_tmp5 == NULL) ? NULL : g_object_ref (_tmp5)), (self->priv->item == NULL) ? NULL : (self->priv->item = (g_object_unref (self->priv->item), NULL)), _tmp6);
	rygel_http_request_handle_item_request (self);
	(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
	(media_object == NULL) ? NULL : (media_object = (g_object_unref (media_object), NULL));
}


static void rygel_http_request_handle_error (RygelHTTPRequest* self, GError* error) {
	guint status;
	g_return_if_fail (self != NULL);
	g_warning ("rygel-http-request.vala:314: %s", error->message);
	status = 0U;
	if (error->domain == RYGEL_HTTP_REQUEST_ERROR) {
		status = (guint) error->code;
	} else {
		status = (guint) SOUP_STATUS_NOT_FOUND;
	}
	rygel_http_request_end (self, status);
}


void rygel_http_request_end (RygelHTTPRequest* self, guint status) {
	g_return_if_fail (self != NULL);
	if (status != SOUP_STATUS_NONE) {
		soup_message_set_status (self->priv->msg, status);
	}
	g_signal_emit_by_name ((RygelStateMachine*) self, "completed");
}


static void rygel_http_request_class_init (RygelHTTPRequestClass * klass) {
	rygel_http_request_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (RygelHTTPRequestPrivate));
	G_OBJECT_CLASS (klass)->finalize = rygel_http_request_finalize;
}


static void rygel_http_request_rygel_state_machine_interface_init (RygelStateMachineIface * iface) {
	rygel_http_request_rygel_state_machine_parent_iface = g_type_interface_peek_parent (iface);
	iface->run = rygel_http_request_real_run;
}


static void rygel_http_request_instance_init (RygelHTTPRequest * self) {
	self->priv = RYGEL_HTTP_REQUEST_GET_PRIVATE (self);
}


static void rygel_http_request_finalize (GObject* obj) {
	RygelHTTPRequest * self;
	self = RYGEL_HTTP_REQUEST (obj);
	(self->priv->root_container == NULL) ? NULL : (self->priv->root_container = (g_object_unref (self->priv->root_container), NULL));
	(self->priv->server == NULL) ? NULL : (self->priv->server = (g_object_unref (self->priv->server), NULL));
	(self->priv->msg == NULL) ? NULL : (self->priv->msg = (g_object_unref (self->priv->msg), NULL));
	(self->priv->query == NULL) ? NULL : (self->priv->query = (g_hash_table_unref (self->priv->query), NULL));
	(self->priv->response == NULL) ? NULL : (self->priv->response = (g_object_unref (self->priv->response), NULL));
	self->priv->item_id = (g_free (self->priv->item_id), NULL);
	(self->priv->item == NULL) ? NULL : (self->priv->item = (g_object_unref (self->priv->item), NULL));
	(self->priv->seek == NULL) ? NULL : (self->priv->seek = (g_object_unref (self->priv->seek), NULL));
	(self->priv->cancellable == NULL) ? NULL : (self->priv->cancellable = (g_object_unref (self->priv->cancellable), NULL));
	G_OBJECT_CLASS (rygel_http_request_parent_class)->finalize (obj);
}


GType rygel_http_request_get_type (void) {
	static GType rygel_http_request_type_id = 0;
	if (rygel_http_request_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (RygelHTTPRequestClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_http_request_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelHTTPRequest), 0, (GInstanceInitFunc) rygel_http_request_instance_init, NULL };
		static const GInterfaceInfo rygel_state_machine_info = { (GInterfaceInitFunc) rygel_http_request_rygel_state_machine_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		rygel_http_request_type_id = g_type_register_static (G_TYPE_OBJECT, "RygelHTTPRequest", &g_define_type_info, 0);
		g_type_add_interface_static (rygel_http_request_type_id, RYGEL_TYPE_STATE_MACHINE, &rygel_state_machine_info);
	}
	return rygel_http_request_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




